<?php
ob_start();
$pageTitle = $pageTitle ?? 'Pedidos';
$activeMenu = $activeMenu ?? 'pedidos';
?>

<div class="row mb-3">
    <div class="col-12">
        <nav aria-label="breadcrumb" class="mb-3">
            <ol class="breadcrumb">
                <li class="breadcrumb-item"><a href="<?= url('/dashboard') ?>">Dashboard</a></li>
                <li class="breadcrumb-item active">Pedidos</li>
            </ol>
        </nav>
        <div class="d-flex justify-content-between align-items-center flex-wrap gap-2">
            <div>
                <h2><i class="bi bi-clipboard-check me-2"></i> Pedidos</h2>
                <p class="text-muted mb-0">Gestão de pedidos por status</p>
            </div>
            <div>
                <span class="badge bg-secondary fs-6">
                    <i class="bi bi-box-seam me-1"></i> Total: <?= number_format($totalPedidos, 0, ',', '.') ?>
                </span>
            </div>
        </div>
    </div>
</div>

<!-- Cards de Status - Layout Responsivo -->
<div class="row g-3" id="pedidos-cards">
    <?php foreach ($pedidosPorStatus as $cardKey => $cardData): ?>
        <?php $config = $cardData['config']; ?>
        <?php $pedidos = $cardData['pedidos']; ?>
        <?php $total = $cardData['total']; ?>

        <div class="col-12 col-sm-6 col-lg-4 col-xl-4 pedido-card-column">
            <div class="card h-100 shadow-sm pedido-status-card" data-status="<?= htmlspecialchars($cardKey) ?>">
                <!-- Card Header -->
                <div class="card-header bg-<?= htmlspecialchars($config['color']) ?> text-white d-flex justify-content-between align-items-center">
                    <div class="d-flex align-items-center gap-2">
                        <i class="<?= htmlspecialchars($config['icon']) ?> fs-5"></i>
                        <h6 class="mb-0 fw-bold"><?= htmlspecialchars($config['title']) ?></h6>
                    </div>
                    <span class="badge bg-light text-dark"><?= $total ?></span>
                </div>

                <!-- Card Body - Lista de Pedidos -->
                <div class="card-body p-2 pedido-list-container" style="max-height: 600px; overflow-y: auto;">
                    <?php if ($cardKey === 'entrada'): ?>
                        <!-- Botão para criar novo pedido -->
                        <div class="mb-2 p-2 border rounded bg-light cursor-pointer text-center"
                             onclick="criarNovoPedido()"
                             style="border: 2px dashed #0d6efd !important; background-color: #e7f3ff !important;">
                            <i class="bi bi-plus-circle text-primary fs-4"></i>
                            <p class="mb-0 mt-2 small text-primary fw-bold">Novo Pedido</p>
                        </div>
                    <?php endif; ?>

                    <?php if (empty($pedidos)): ?>
                        <div class="text-center text-muted py-4">
                            <i class="bi bi-inbox fs-1 opacity-25"></i>
                            <p class="mb-0 mt-2 small">Nenhum pedido</p>
                        </div>
                    <?php else: ?>
                        <?php foreach ($pedidos as $pedido): ?>
                            <?php
                                $timelineCard = array_map('strtolower', $config['timeline'] ?? []);
                                $mostrarValor = in_array('pedido', $timelineCard, true) || in_array('faturamento', $timelineCard, true);
                            ?>
                            <div class="pedido-item mb-2 p-2 border rounded cursor-pointer"
                                 data-pedido-id="<?= $pedido['id'] ?>"
                                 data-card-key="<?= htmlspecialchars($cardKey) ?>"
                                 onclick="abrirPedido(<?= $pedido['id'] ?>, '<?= $cardKey ?>')">
                                <!-- Número do Pedido -->
                                <div class="d-flex justify-content-between align-items-start mb-2">
                                    <div>
                                        <strong class="text-primary"><?= htmlspecialchars($pedido['sale_number']) ?></strong>
                                    </div>
                                    <?php if ($mostrarValor): ?>
                                        <span class="badge bg-light text-dark">
                                            <?= formatCurrency($pedido['total'] ?? 0) ?>
                                        </span>
                                    <?php endif; ?>
                                </div>

                                <!-- Cliente -->
                                <div class="mb-2">
                                    <i class="bi bi-person me-1 text-muted"></i>
                                    <small class="text-dark">
                                        <?= htmlspecialchars($pedido['customer_name'] ?? 'Cliente não informado') ?>
                                    </small>
                                </div>

                                <!-- Data -->
                                <div class="mb-2">
                                    <i class="bi bi-calendar me-1 text-muted"></i>
                                    <small class="text-muted">
                                        <?= date('d/m/Y', strtotime($pedido['sale_date'] ?? 'now')) ?>
                                    </small>
                                </div>

                                <!-- Método de Pagamento -->
                            <?php if ((!empty($pedido['payment_method_name'])) && ($cardKey === 'pedido' || $cardKey === 'finalizado')): ?>
                                <div class="mb-2">
                                    <i class="bi bi-credit-card me-1 text-muted"></i>
                                    <small class="text-muted">
                                        <?= htmlspecialchars($pedido['payment_method_name']) ?>
                                    </small>
                                </div>
                                <?php endif; ?>

                                <!-- Status Atual (badge pequeno) -->
                            </div>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    <?php endforeach; ?>
</div>

<style>
/* Cards Responsivos */
.pedido-status-card {
    transition: transform 0.2s, box-shadow 0.2s;
    border: 1px solid #dee2e6;
}

.pedido-status-card:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1) !important;
}

.pedido-list-container {
    background-color: #f8f9fa;
    scrollbar-width: thin;
    scrollbar-color: #cbd5e0 #f8f9fa;
}

.pedido-list-container::-webkit-scrollbar {
    width: 6px;
}

.pedido-list-container::-webkit-scrollbar-track {
    background: #f8f9fa;
}

.pedido-list-container::-webkit-scrollbar-thumb {
    background-color: #cbd5e0;
    border-radius: 3px;
}

.pedido-list-container::-webkit-scrollbar-thumb:hover {
    background-color: #a0aec0;
}

/* Item de Pedido */
.pedido-item {
    background: white;
    transition: all 0.2s;
    cursor: pointer;
}

.pedido-item:hover {
    background-color: #e7f3ff;
    border-color: #0d6efd !important;
    transform: translateX(4px);
}

.cursor-pointer {
    cursor: pointer;
}

/* Responsividade Mobile */
@media (max-width: 575.98px) {
    .pedido-card-column {
        padding: 0.5rem !important;
    }

    .pedido-status-card .card-header {
        padding: 0.75rem;
    }

    .pedido-status-card .card-header h6 {
        font-size: 0.9rem;
    }

    .pedido-item {
        padding: 0.75rem !important;
    }

    .pedido-list-container {
        max-height: 400px !important;
    }
}

/* Responsividade Tablet */
@media (min-width: 576px) and (max-width: 991.98px) {
    .pedido-card-column {
        padding: 0.75rem;
    }

    .pedido-list-container {
        max-height: 500px !important;
    }
}

/* Responsividade Desktop */
@media (min-width: 992px) {
    .pedido-list-container {
        max-height: 600px;
    }
}

/* Melhorias de UX */
.pedido-item strong {
    font-size: 0.95rem;
}

.pedido-item small {
    font-size: 0.8rem;
}

.card-header {
    position: sticky;
    top: 0;
    z-index: 10;
}

/* Animação de carregamento */
@keyframes fadeIn {
    from {
        opacity: 0;
        transform: translateY(10px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

.pedido-item {
    animation: fadeIn 0.3s ease-out;
}

/* Botão criar novo pedido */
.criar-pedido-btn {
    transition: all 0.2s;
}

.criar-pedido-btn:hover {
    background-color: #0d6efd !important;
    color: white !important;
    transform: scale(1.02);
    border-color: #0d6efd !important;
}

.criar-pedido-btn:hover i,
.criar-pedido-btn:hover p {
    color: white !important;
}
</style>

<script>
const moedaFormatter = new Intl.NumberFormat('pt-BR', { style: 'currency', currency: 'BRL' });
const numberFormatters = {};

function criarNovoPedido() {
    window.location.href = appUrl('/pedidos/criar');
}

function abrirPedido(id, cardKey) {
    if (typeof Swal !== 'undefined') {
        if (cardKey === 'pedido') {
            abrirResumoFaturamento(id);
            return;
        }

        if (cardKey === 'envaze') {
            processarEnvaze(id);
            return;
        }

        if (cardKey === 'expedicao') {
            confirmarEntrega(id);
            return;
        }
    }
    window.location.href = appUrl('/pedidos/editar?id=' + id);
}

function formatCurrencyBRL(value) {
    return moedaFormatter.format(Number(value) || 0);
}

function formatNumberBR(value, fractionDigits = 2) {
    const key = String(fractionDigits);
    if (!numberFormatters[key]) {
        numberFormatters[key] = new Intl.NumberFormat('pt-BR', {
            minimumFractionDigits: fractionDigits,
            maximumFractionDigits: fractionDigits
        });
    }
    return numberFormatters[key].format(Number(value) || 0);
}

function formatDateBR(dateString) {
    if (!dateString) {
        return '--';
    }
    const date = new Date(dateString);
    return Number.isNaN(date.getTime()) ? dateString : date.toLocaleDateString('pt-BR');
}

function escapeHtml(text) {
    const map = { '&': '&amp;', '<': '&lt;', '>': '&gt;', '"': '&quot;', "'": '&#039;' };
    return String(text ?? '').replace(/[&<>"']/g, m => map[m]);
}

async function abrirResumoFaturamento(pedidoId) {
    try {
        const response = await fetch(appUrl('/pedidos/resumo?id=' + pedidoId), {
            headers: { 'X-Requested-With': 'XMLHttpRequest' }
        });
        const result = await response.json();
        if (!result.success) {
            throw new Error(result.message || 'Não foi possível carregar o resumo');
        }

        const modalResult = await Swal.fire({
            title: '<i class="bi bi-card-checklist me-2"></i>Resumo do Pedido',
            html: montarResumoPedidoHtml(result.data),
            width: 900,
            showCancelButton: true,
            focusConfirm: false,
            confirmButtonText: '<i class="bi bi-credit-card me-1"></i> Pagamento',
            cancelButtonText: 'Fechar',
            customClass: { popup: 'swal-wide' }
        });

        if (modalResult.isConfirmed) {
            await abrirPagamentoPedido(result.data);
        }
    } catch (error) {
        Swal.fire({
            icon: 'error',
            title: 'Erro',
            text: error.message || 'Erro ao abrir resumo'
        });
    }
}

function montarResumoPedidoHtml(resumo) {
    const pedido = resumo.pedido || {};
    const cliente = resumo.cliente || {};
    const itens = Array.isArray(resumo.itens) ? resumo.itens : [];

    const itensHtml = itens.length
        ? `
            <div class="table-responsive">
                <table class="table table-sm mb-0">
                    <thead>
                        <tr>
                            <th>Produto</th>
                            <th class="text-center">Qtd</th>
                            <th class="text-end">Unit.</th>
                            <th class="text-end">Total</th>
                        </tr>
                    </thead>
                    <tbody>
                        ${itens.map(item => `
                            <tr>
                                <td>${escapeHtml(item.product_name || 'Produto')}</td>
                                <td class="text-center">${formatNumberBR(item.quantity ?? 0, 3)}</td>
                                <td class="text-end">${formatCurrencyBRL(item.unit_price ?? 0)}</td>
                                <td class="text-end fw-semibold">${formatCurrencyBRL(item.total_price ?? 0)}</td>
                            </tr>
                        `).join('')}
                    </tbody>
                </table>
            </div>
        `
        : '<p class="text-muted mb-0">Nenhum item cadastrado para este pedido.</p>';

    const totalGeralHtml = `
        <div class="text-end mt-3">
            <span class="text-muted me-2">Total Geral:</span>
            <span class="fs-5 fw-bold text-success">${formatCurrencyBRL(pedido.total)}</span>
        </div>
    `;

    return `
        <div class="resumo-pedido text-start">
            <div class="row g-3 mb-3">
                <div class="col-md-6">
                    <div class="border rounded p-3 bg-light h-100">
                        <h6 class="fw-bold text-primary mb-2"><i class="bi bi-receipt me-1"></i> Pedido</h6>
                        <p class="mb-1"><strong>Número:</strong> ${escapeHtml(pedido.sale_number || '--')}</p>
                        <p class="mb-1"><strong>Data:</strong> ${formatDateBR(pedido.sale_date)}</p>
                        <p class="mb-0"><strong>Total:</strong> ${formatCurrencyBRL(pedido.total)}</p>
                    </div>
                </div>
                <div class="col-md-6">
                    <div class="border rounded p-3 bg-light h-100">
                        <h6 class="fw-bold text-primary mb-2"><i class="bi bi-person me-1"></i> Cliente</h6>
                        <p class="mb-1"><strong>Nome:</strong> ${escapeHtml(cliente.nome || 'Não informado')}</p>
                        <p class="mb-1"><strong>Documento:</strong> ${escapeHtml(cliente.documento || '--')}</p>
                        <p class="mb-1"><strong>Email:</strong> ${escapeHtml(cliente.email || '--')}</p>
                        <p class="mb-0"><strong>Telefone:</strong> ${escapeHtml(cliente.telefone || '--')}</p>
                    </div>
                </div>
            </div>
            <div class="border rounded p-3">
                <h6 class="fw-bold text-primary mb-3"><i class="bi bi-box-seam me-1"></i> Itens do Pedido</h6>
                ${itensHtml}
                ${totalGeralHtml}
            </div>
        </div>
    `;
}

async function abrirPagamentoPedido(resumo) {
    const pedido = resumo.pedido || {};
    const metodos = Array.isArray(resumo.metodos_pagamento) ? resumo.metodos_pagamento : [];
    const pagamentoAtual = resumo.pagamento || {};

    if (!metodos.length) {
        await Swal.fire({
            icon: 'info',
            title: 'Sem métodos de pagamento',
            text: 'Cadastre um método ativo no menu Métodos de Pagamento.'
        });
        return;
    }

    const pagamentoHtml = gerarHtmlPagamento(pedido, metodos, pagamentoAtual);

    const resultado = await Swal.fire({
        title: '<i class="bi bi-credit-card me-2"></i>Pagamento',
        html: pagamentoHtml,
        width: 650,
        showCancelButton: true,
        focusConfirm: false,
        confirmButtonText: '<i class="bi bi-check2-circle me-1"></i> Salvar',
        cancelButtonText: 'Cancelar',
        didOpen: () => inicializarModalPagamento(pedido, pagamentoAtual),
        preConfirm: () => {
            const metodoSelect = document.getElementById('modal-metodo-pagamento');
            if (!metodoSelect || !metodoSelect.value) {
                Swal.showValidationMessage('Selecione um método de pagamento');
                return false;
            }

            const parcelas = parseInt(document.getElementById('modal-pagamento-parcelas').value || '1', 10);
            const termValue = document.getElementById('modal-pagamento-term').value || null;

            return {
                metodoId: parseInt(metodoSelect.value, 10),
                parcelas: parcelas > 0 ? parcelas : 1,
                paymentTermId: termValue ? parseInt(termValue, 10) : null
            };
        }
    });

    if (!resultado.isConfirmed || !resultado.value) {
        return;
    }

    await salvarPagamentoPedido({
        pedido_id: pedido.id,
        payment_method_id: resultado.value.metodoId,
        installments: resultado.value.parcelas,
        payment_term_id: resultado.value.paymentTermId
    });
}

function gerarHtmlPagamento(pedido, metodos, pagamentoAtual) {
    const options = metodos.map(metodo => `
        <option value="${metodo.id}">${escapeHtml(metodo.name || 'Método')}</option>
    `).join('');

    return `
        <div class="text-start">
            <div class="mb-3">
                <label class="form-label">Total do Pedido</label>
                <div class="fw-bold fs-4 text-success">${formatCurrencyBRL(pedido.total)}</div>
            </div>
            <div class="mb-3">
                <label for="modal-metodo-pagamento" class="form-label">Método de Pagamento</label>
                <select id="modal-metodo-pagamento" class="form-select">
                    <option value="">Selecione...</option>
                    ${options}
                </select>
                <div class="invalid-feedback d-none" id="modal-metodo-feedback">Selecione um método para continuar.</div>
            </div>
            <div class="mb-3">
                <button type="button" class="btn btn-outline-primary w-100" id="modal-btn-prazo">
                    <i class="bi bi-calendar-check me-1"></i> Escolher prazo
                </button>
            </div>
            <div id="modal-prazo-resumo" class="alert alert-info d-none"></div>
            <input type="hidden" id="modal-pagamento-parcelas" value="${Math.max(1, pagamentoAtual.parcelas || 1)}">
            <input type="hidden" id="modal-pagamento-term" value="${pagamentoAtual.payment_term_id || ''}">
        </div>
    `;
}

function inicializarModalPagamento(pedido, pagamentoAtual) {
    const metodoSelect = document.getElementById('modal-metodo-pagamento');
    const metodoFeedback = document.getElementById('modal-metodo-feedback');
    const prazoBtn = document.getElementById('modal-btn-prazo');
    const resumoPrazo = document.getElementById('modal-prazo-resumo');
    const parcelasField = document.getElementById('modal-pagamento-parcelas');
    const termField = document.getElementById('modal-pagamento-term');

    if (!metodoSelect || !prazoBtn || !resumoPrazo || !parcelasField || !termField) {
        return;
    }

    if (pagamentoAtual.metodo_atual) {
        metodoSelect.value = pagamentoAtual.metodo_atual;
    }

    const atualizarResumoPrazo = prazo => {
        if (!prazo) {
            resumoPrazo.classList.add('d-none');
            resumoPrazo.innerHTML = '';
            parcelasField.value = 1;
            termField.value = '';
            return;
        }
        resumoPrazo.classList.remove('d-none');
        resumoPrazo.innerHTML = `<strong>${prazo.parcelas}x</strong> • ${escapeHtml(prazo.descricao)}`;
        parcelasField.value = prazo.parcelas;
        termField.value = prazo.paymentTermId || '';
    };

    if (pagamentoAtual.payment_term_id) {
        atualizarResumoPrazo({
            parcelas: pagamentoAtual.parcelas || 1,
            descricao: 'Prazo pré-configurado',
            paymentTermId: pagamentoAtual.payment_term_id
        });
    }

    metodoSelect.addEventListener('change', () => {
        metodoSelect.classList.remove('is-invalid');
        metodoFeedback?.classList.add('d-none');
        atualizarResumoPrazo(null);
    });

    prazoBtn.addEventListener('click', async () => {
        if (!metodoSelect.value) {
            metodoSelect.classList.add('is-invalid');
            metodoFeedback?.classList.remove('d-none');
            return;
        }
        const prazoSelecionado = await selecionarPrazoParaPedido(metodoSelect.value, pedido.total);
        if (prazoSelecionado) {
            atualizarResumoPrazo(prazoSelecionado);

            const aplicarAgora = await Swal.fire({
                icon: 'question',
                title: 'Confirmar pagamento?',
                text: `Aplicar pagamento em ${prazoSelecionado.parcelas}x agora mesmo?`,
                showCancelButton: true,
                confirmButtonText: 'Sim, salvar',
                cancelButtonText: 'Ainda não'
            });

            if (aplicarAgora.isConfirmed) {
                const payload = {
                    pedido_id: pedido.id,
                    payment_method_id: parseInt(metodoSelect.value, 10),
                    installments: prazoSelecionado.parcelas,
                    payment_term_id: prazoSelecionado.paymentTermId
                };

                Swal.close(); // fecha o modal principal antes de salvar
                await salvarPagamentoPedido(payload);
                return;
            }
        }
    });
}

async function selecionarPrazoParaPedido(metodoId, totalPedido) {
    try {
        const response = await fetch(appUrl('/metodos-pagamento/buscar-prazos?metodo_id=' + metodoId), {
            headers: { 'X-Requested-With': 'XMLHttpRequest' }
        });
        const result = await response.json();
        if (!result.success || !result.data || !Array.isArray(result.data.prazos) || result.data.prazos.length === 0) {
            await Swal.fire({
                icon: 'info',
                title: 'Sem prazos configurados',
                text: 'Configure prazos para este método no menu Métodos de Pagamento.'
            });
            return null;
        }
        return await exibirSelecaoPrazo(result.data.prazos, totalPedido);
    } catch (error) {
        await Swal.fire({
            icon: 'error',
            title: 'Erro',
            text: 'Não foi possível carregar os prazos de pagamento.'
        });
        return null;
    }
}

function calcularDescricaoPrazo(parcelas, dias, total) {
    const valorParcela = total / parcelas;
    const valorBase = Math.floor(valorParcela * 100) / 100;
    const valorUltima = total - (valorBase * (parcelas - 1));

    if (parcelas === 1) {
        return `1x de ${formatCurrencyBRL(valorUltima)} (${dias} dias)`;
    }

    if (Math.abs(valorUltima - valorBase) < 0.01) {
        return `${parcelas}x de ${formatCurrencyBRL(valorBase)} (${dias} dias)`;
    }

    return `${parcelas - 1}x de ${formatCurrencyBRL(valorBase)} + 1x de ${formatCurrencyBRL(valorUltima)} (${dias} dias)`;
}

function exibirSelecaoPrazo(prazos, totalPedido) {
    const optionsHtml = prazos.map(prazo => {
        const diasBase = Math.max(1, parseInt(prazo.dias, 10) || 30);
        const parcelas = Math.max(1, Math.ceil(diasBase / 30));
        const descricao = calcularDescricaoPrazo(parcelas, diasBase, Number(totalPedido) || 0);
        const descricaoEncoded = encodeURIComponent(descricao);

        return `
            <label class="list-group-item d-flex align-items-start gap-3 prazo-opcao">
                <input class="form-check-input mt-1" type="radio" name="prazo-card"
                    value="${prazo.id}"
                    data-parcelas="${parcelas}"
                    data-dias="${diasBase}"
                    data-descricao="${descricaoEncoded}">
                <div>
                    <div class="fw-semibold">${parcelas}x - ${diasBase} dias</div>
                    <div class="text-muted small">${escapeHtml(descricao)}</div>
                </div>
            </label>
        `;
    }).join('');

    return Swal.fire({
        title: '<i class="bi bi-calendar-week me-2"></i>Selecione o prazo',
        html: `<div class="list-group">${optionsHtml}</div>`,
        width: 600,
        showCancelButton: true,
        focusConfirm: false,
        confirmButtonText: 'Aplicar',
        cancelButtonText: 'Cancelar',
        preConfirm: () => {
            const selecionado = document.querySelector('input[name="prazo-card"]:checked');
            if (!selecionado) {
                Swal.showValidationMessage('Escolha um prazo para continuar');
                return false;
            }
            return {
                paymentTermId: parseInt(selecionado.value, 10),
                parcelas: parseInt(selecionado.dataset.parcelas || '1', 10),
                dias: parseInt(selecionado.dataset.dias || '30', 10),
                descricao: decodeURIComponent(selecionado.dataset.descricao || '')
            };
        }
    }).then(result => (result.isConfirmed ? result.value : null));
}

async function salvarPagamentoPedido(payload) {
    try {
        const formData = new FormData();
        Object.entries(payload).forEach(([key, value]) => {
            if (value !== undefined && value !== null) {
                formData.append(key, value);
            }
        });

    const response = await fetch(appUrl('/pedidos/processar-pagamento'), {
            method: 'POST',
            headers: { 'X-Requested-With': 'XMLHttpRequest' },
        body: formData,
        credentials: 'same-origin'
        });
        const result = await response.json();

        if (!result.success) {
            throw new Error(result.message || 'Não foi possível atualizar o pagamento');
        }

        await Swal.fire({
            icon: 'success',
            title: 'Pagamento atualizado!',
            timer: 2000,
            showConfirmButton: false
        });

        window.location.reload();
    } catch (error) {
        Swal.fire({
            icon: 'error',
            title: 'Erro',
            text: error.message || 'Erro ao atualizar pagamento'
        });
    }
}

async function processarEnvaze(pedidoId) {
    try {
        const result = await Swal.fire({
            title: 'Houve quebra nesse pedido?',
            icon: 'question',
            showCancelButton: true,
            showDenyButton: true,
            confirmButtonText: 'Sim',
            denyButtonText: 'Não',
            cancelButtonText: 'Cancelar'
        });

        if (result.isDenied) {
            await finalizarEnvaze(pedidoId, false);
            return;
        }

        if (result.isConfirmed) {
            const dadosQuebra = await solicitarDadosQuebra();
            if (!dadosQuebra) {
                return;
            }
            await finalizarEnvaze(pedidoId, true, dadosQuebra.quantidade, dadosQuebra.validade);
        }
    } catch (error) {
        Swal.fire({
            icon: 'error',
            title: 'Erro',
            text: error.message || 'Não foi possível processar envaze'
        });
    }
}

async function solicitarDadosQuebra() {
    const anoAtual = new Date().getFullYear();
    const anosOptions = Array.from({ length: 11 }, (_, idx) => {
        const ano = anoAtual + idx;
        return `<option value="${ano}">${ano}</option>`;
    }).join('');

    return Swal.fire({
        title: 'Informar quebra',
        html: `
            <div class="text-start">
                <label class="form-label">Quantidade</label>
                <input type="number" id="envaze-quebra-quantidade" min="0" step="0.01" class="form-control" placeholder="0,00">
                <label class="form-label mt-3">Validade</label>
                <select id="envaze-quebra-validade" class="form-select">
                    <option value="">Selecione o ano</option>
                    ${anosOptions}
                </select>
            </div>
        `,
        focusConfirm: false,
        showCancelButton: true,
        confirmButtonText: 'Salvar',
        cancelButtonText: 'Cancelar',
        preConfirm: () => {
            const quantidadeInput = document.getElementById('envaze-quebra-quantidade');
            const validadeInput = document.getElementById('envaze-quebra-validade');
            const quantidade = parseFloat((quantidadeInput.value || '').replace(',', '.'));
            const validade = validadeInput.value || null;

            if (!quantidade || quantidade <= 0) {
                Swal.showValidationMessage('Informe a quantidade de quebra');
                return false;
            }

            return {
                quantidade,
                validade
            };
        }
    }).then(result => (result.isConfirmed ? result.value : null));
}

async function finalizarEnvaze(pedidoId, houveQuebra, quantidade = null, validade = null) {
    try {
        const formData = new FormData();
        formData.append('pedido_id', pedidoId);
        formData.append('teve_quebra', houveQuebra ? '1' : '0');
        if (houveQuebra) {
            formData.append('quantidade_quebra', quantidade);
            if (validade) {
                formData.append('validade_quebra', validade);
            }
        }

        const response = await fetch(appUrl('/pedidos/processar-envaze'), {
            method: 'POST',
            headers: { 'X-Requested-With': 'XMLHttpRequest' },
            body: formData,
            credentials: 'same-origin'
        });
        const result = await response.json();

        if (!result.success) {
            throw new Error(result.message || 'Não foi possível atualizar o envaze');
        }

        await Swal.fire({
            icon: 'success',
            title: 'Envaze atualizado!',
            timer: 1800,
            showConfirmButton: false
        });

        window.location.reload();
    } catch (error) {
        Swal.fire({
            icon: 'error',
            title: 'Erro',
            text: error.message || 'Erro ao atualizar envaze'
        });
    }
}

async function confirmarEntrega(pedidoId) {
    try {
        const result = await Swal.fire({
            title: 'Pedido entregue?',
            icon: 'question',
            showCancelButton: true,
            confirmButtonText: 'Sim',
            cancelButtonText: 'Não'
        });

        if (!result.isConfirmed) {
            return;
        }

        await atualizarLinhaTemporal(pedidoId, 'finalizado');

        await Swal.fire({
            icon: 'success',
            title: 'Pedido finalizado!',
            timer: 1500,
            showConfirmButton: false
        });

        window.location.reload();
    } catch (error) {
        Swal.fire({
            icon: 'error',
            title: 'Erro',
            text: error.message || 'Não foi possível finalizar o pedido'
        });
    }
}

async function atualizarLinhaTemporal(pedidoId, linhaTemporal) {
    const formData = new FormData();
    formData.append('pedido_id', pedidoId);
    formData.append('linha_temporal', linhaTemporal);

    const response = await fetch(appUrl('/pedidos/atualizar-linha-temporal'), {
        method: 'POST',
        headers: { 'X-Requested-With': 'XMLHttpRequest' },
        body: formData,
        credentials: 'same-origin'
    });
    const result = await response.json();

    if (!result.success) {
        throw new Error(result.message || 'Não foi possível atualizar a linha temporal');
    }
}

// Auto-refresh a cada 30 segundos (opcional)
// setInterval(function() {
//     if (document.visibilityState === 'visible') {
//         location.reload();
//     }
// }, 30000);
</script>

<?php
$content = ob_get_clean();
require __DIR__ . '/../layouts/app.php';
?>

