<!DOCTYPE html>
<html lang="pt-BR">

<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>PDV - Caixa</title>

    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.0/font/bootstrap-icons.css">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/sweetalert2@11/dist/sweetalert2.min.css">

    <!-- VERIFICAÇÃO CRÍTICA: Redireciona se URL estiver sem caminho base -->
    <script>
        (function () {
            console.log("==========================================");
            console.log("[CAIXA.PHP] Iniciando verificação de URL");
            console.log("[CAIXA.PHP] URL completa:", window.location.href);
            console.log("[CAIXA.PHP] Pathname:", window.location.pathname);
            console.log("[CAIXA.PHP] Host:", window.location.host);
            console.log("[CAIXA.PHP] Protocol:", window.location.protocol);

            const currentPath = window.location.pathname;
            const pathParts = currentPath.split('/').filter(p => p);

            console.log("[CAIXA.PHP] Path parts:", JSON.stringify(pathParts));
            console.log("[CAIXA.PHP] Primeiro segmento:", pathParts[0] || 'vazio');
            console.log("[CAIXA.PHP] Total de segmentos:", pathParts.length);

            // Se está em /pdv/caixa (primeiro segmento é 'pdv'), significa que falta /Systhema/
            if (pathParts.length >= 1 && pathParts[0] === 'pdv') {
                console.error("[CAIXA.PHP] ⚠️ PROBLEMA DETECTADO: URL sem base path!");
                console.error("[CAIXA.PHP] Path atual:", currentPath);
                console.error("[CAIXA.PHP] Path parts:", JSON.stringify(pathParts));

                const host = window.location.host;
                const protocol = window.location.protocol;
                const correctUrl = protocol + '//' + host + '/Systhema' + currentPath;

                console.error("[CAIXA.PHP] URL incorreta:", window.location.href);
                console.error("[CAIXA.PHP] URL corrigida:", correctUrl);
                console.error("[CAIXA.PHP] 🔄 REDIRECIONANDO AUTOMATICAMENTE...");

                window.location.replace(correctUrl);
                return;
            }

            console.log("[CAIXA.PHP] ✅ URL OK, continuando carregamento");
            console.log("==========================================");
        })();
    </script>

    <!-- Define appUrl e APP_BASE_PATH ANTES de qualquer script que possa usá-lo -->
    <script>
        (function () {
            // Detecta o diretório base dinamicamente baseado no caminho atual
            const currentPath = window.location.pathname;
            const pathParts = currentPath.split('/').filter(p => p);

            // Se estiver em um subdiretório (ex: /Systhema, /printjetdigital)
            let detectedBase = '';
            if (pathParts.length > 0 && pathParts[0] !== '') {
                detectedBase = '/' + pathParts[0];
            }

            const normalizedBase = detectedBase === '/' ? '' : detectedBase;
            window.APP_BASE_PATH = normalizedBase;

            window.appUrl = function (path = '/') {
                const safePath = typeof path === 'string' ? path : '';
                const normalizedPath = '/' + safePath.replace(/^\/+/, '');

                // Se não tem base path, retorna apenas o path normalizado
                if (!window.APP_BASE_PATH) {
                    return normalizedPath;
                }

                // Retorna base + path (ex: /Systhema/dashboard ou /printjetdigital/dashboard)
                return window.APP_BASE_PATH + normalizedPath;
            };

            window.url = window.appUrl;
        })();
    </script>

    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: #e9ecef;
            overflow: hidden;
            height: 100vh;
        }

        /* Header do PDV - Cor do Tema */
        .pdv-header {
            background: linear-gradient(135deg, #001122 0%, #003366 100%);
            color: white;
            padding: 15px 20px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            box-shadow: 0 2px 10px rgba(0, 17, 34, 0.3);
        }

        .pdv-header-left {
            display: flex;
            align-items: center;
            gap: 20px;
        }

        .pdv-header-title {
            font-size: 24px;
            font-weight: 700;
            margin: 0;
        }

        .pdv-header-info {
            font-size: 14px;
            opacity: 0.85;
        }

        .pdv-header-right {
            display: flex;
            gap: 10px;
            align-items: center;
        }

        .btn-header {
            padding: 8px 16px;
            border-radius: 8px;
            border: 2px solid rgba(255, 255, 255, 0.3);
            background: rgba(255, 255, 255, 0.1);
            color: white;
            font-weight: 600;
            transition: all 0.3s;
        }

        .btn-header:hover {
            background: rgba(255, 255, 255, 0.2);
            border-color: rgba(255, 255, 255, 0.5);
        }

        .btn-danger-outline {
            border-color: #dc3545 !important;
            color: #ff6b6b !important;
        }

        .btn-danger-outline:hover {
            background: rgba(220, 53, 69, 0.2) !important;
        }

        /* Modal de Funções */
        .funcoes-grid {
            display: grid;
            grid-template-columns: repeat(3, 1fr);
            gap: 12px;
            padding: 10px;
        }

        .btn-funcao {
            display: flex;
            flex-direction: column;
            align-items: center;
            justify-content: center;
            padding: 20px 10px;
            border: 2px solid #e0e0e0;
            border-radius: 12px;
            background: #f8f9fa;
            cursor: pointer;
            transition: all 0.2s ease;
            min-height: 90px;
        }

        .btn-funcao:hover {
            background: #e9ecef;
            border-color: #adb5bd;
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
        }

        .btn-funcao i {
            font-size: 28px;
            margin-bottom: 8px;
        }

        .btn-funcao span {
            font-size: 12px;
            font-weight: 600;
            color: #333;
            text-align: center;
        }

        .swal-funcoes-popup {
            width: 450px !important;
        }

        /* Modal Shipay */
        .swal-shipay-popup {
            width: 500px !important;
        }

        .shipay-grid {
            display: flex;
            flex-direction: column;
            gap: 12px;
            padding: 10px;
        }

        .btn-shipay-opcao {
            display: flex;
            align-items: center;
            padding: 15px 20px;
            border: 2px solid #e0e0e0;
            border-radius: 12px;
            background: #f8f9fa;
            cursor: pointer;
            transition: all 0.2s ease;
            text-align: left;
        }

        .btn-shipay-opcao:hover {
            background: #e9ecef;
            border-color: #adb5bd;
            transform: translateX(5px);
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
        }

        .btn-shipay-opcao i {
            font-size: 32px;
            margin-right: 15px;
            width: 40px;
            text-align: center;
        }

        .btn-shipay-opcao span {
            font-size: 14px;
            font-weight: 600;
            color: #333;
            display: block;
        }

        .btn-shipay-opcao small {
            font-size: 11px;
            color: #666;
            font-weight: normal;
        }

        /* Lista de PIX para estorno */
        .lista-pix-estorno {
            border: 1px solid #e0e0e0;
            border-radius: 8px;
            overflow: hidden;
        }

        .item-pix-estorno {
            padding: 12px 15px;
            border-bottom: 1px solid #e0e0e0;
            cursor: pointer;
            transition: all 0.2s ease;
            background: #fff;
        }

        .item-pix-estorno:last-child {
            border-bottom: none;
        }

        .item-pix-estorno:hover {
            background: #f8f9fa;
            transform: translateX(5px);
        }

        .item-pix-estorno:active {
            background: #e9ecef;
        }

        /* Conteúdo principal */
        .pdv-content {
            display: flex;
            height: calc(100vh - 70px);
        }

        /* Coluna da esquerda - Lista de produtos */
        .pdv-left {
            flex: 1;
            padding: 20px;
            overflow-y: auto;
            background: #f8f9fa;
            border-right: 2px solid #dee2e6;
        }

        .search-box {
            position: sticky;
            top: 0;
            background: #f8f9fa;
            padding-bottom: 15px;
            z-index: 10;
        }

        .search-input {
            width: 100%;
            height: 60px;
            border-radius: 12px;
            border: 3px solid #001122;
            padding: 0 20px;
            font-size: 18px;
            font-weight: 600;
            background: white;
        }

        .search-input:focus {
            outline: none;
            border-color: #001122;
            box-shadow: 0 0 0 0.3rem rgba(0, 17, 34, 0.25);
        }

        .produtos-lista {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(200px, 1fr));
            gap: 15px;
            margin-top: 20px;
        }

        .produto-card {
            background: white;
            border: 2px solid #dee2e6;
            border-radius: 12px;
            padding: 15px;
            cursor: pointer;
            transition: all 0.3s;
            text-align: center;
        }

        .produto-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 10px 25px rgba(0, 0, 0, 0.15);
            border-color: #001122;
        }

        .produto-card img {
            width: 80px;
            height: 80px;
            object-fit: contain;
            margin-bottom: 10px;
        }

        .produto-nome {
            font-weight: 600;
            font-size: 14px;
            margin-bottom: 5px;
            height: 40px;
            overflow: hidden;
            color: #212529;
        }

        .produto-preco {
            color: #212529;
            font-size: 20px;
            font-weight: 700;
        }

        /* Coluna da direita - Carrinho e totais */
        .pdv-right {
            width: 450px;
            background: #e9ecef;
            display: flex;
            flex-direction: column;
        }

        .carrinho-header {
            background: #001122;
            color: white;
            padding: 20px;
            border-bottom: 2px solid #003366;
        }

        .carrinho-header h3 {
            margin: 0;
            font-size: 20px;
            font-weight: 700;
        }

        .carrinho-itens {
            flex: 1;
            overflow-y: auto;
            padding: 15px;
        }

        .item-venda {
            background: white;
            border-radius: 10px;
            padding: 15px;
            margin-bottom: 10px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            box-shadow: 0 2px 5px rgba(0, 0, 0, 0.08);
            border: 1px solid #dee2e6;
        }

        .item-info {
            flex: 1;
        }

        .item-nome {
            font-weight: 600;
            font-size: 14px;
            margin-bottom: 5px;
            color: #212529;
        }

        .item-preco {
            color: #6c757d;
            font-size: 13px;
        }

        .item-qtd {
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .btn-qtd {
            width: 35px;
            height: 35px;
            border-radius: 8px;
            border: none;
            background: #001122;
            color: white;
            font-weight: 700;
            cursor: pointer;
            transition: background 0.2s;
        }

        .btn-qtd:hover {
            background: #003366;
        }

        .item-qtd-valor {
            font-weight: 700;
            font-size: 16px;
            min-width: 30px;
            text-align: center;
            color: #212529;
        }

        .item-total {
            font-size: 18px;
            font-weight: 700;
            color: #212529;
            min-width: 100px;
            text-align: right;
        }

        .btn-remover {
            width: 35px;
            height: 35px;
            border-radius: 8px;
            border: none;
            background: #6c757d;
            color: white;
            margin-left: 10px;
            cursor: pointer;
            transition: background 0.2s;
        }

        .btn-remover:hover {
            background: #001122;
        }

        .carrinho-totais {
            background: white;
            padding: 20px;
            border-top: 2px solid #dee2e6;
        }

        .total-linha {
            display: flex;
            justify-content: space-between;
            padding: 10px 0;
            font-size: 16px;
            color: #495057;
        }

        .total-linha.grande {
            font-size: 28px;
            font-weight: 700;
            color: #212529;
            border-top: 2px solid #dee2e6;
            margin-top: 10px;
            padding-top: 15px;
        }

        .btn-finalizar {
            width: 100%;
            height: 70px;
            font-size: 20px;
            font-weight: 700;
            border-radius: 12px;
            background: linear-gradient(135deg, #001122 0%, #003366 100%);
            border: none;
            color: white;
            margin-top: 15px;
            cursor: pointer;
            transition: all 0.3s;
        }

        .btn-finalizar:hover {
            transform: scale(1.02);
            box-shadow: 0 10px 25px rgba(33, 37, 41, 0.4);
        }

        .btn-finalizar:disabled {
            opacity: 0.5;
            cursor: not-allowed;
            transform: none;
            box-shadow: none;
        }

        .carrinho-vazio {
            text-align: center;
            padding: 60px 20px;
            color: #6c757d;
        }

        .carrinho-vazio i {
            font-size: 64px;
            opacity: 0.3;
            margin-bottom: 15px;
        }

        /* Responsivo */
        @media (max-width: 992px) {
            .pdv-content {
                flex-direction: column;
            }

            .pdv-right {
                width: 100%;
                height: 50vh;
            }

            .produtos-lista {
                grid-template-columns: repeat(auto-fill, minmax(150px, 1fr));
            }
        }

        /* Filtros de Grupos de Produtos */
        .filtros-grupos {
            display: flex;
            gap: 8px;
            margin-bottom: 15px;
            overflow-x: auto;
            padding-bottom: 5px;
            scrollbar-width: thin;
        }

        .filtros-grupos::-webkit-scrollbar {
            height: 4px;
        }

        .filtros-grupos::-webkit-scrollbar-thumb {
            background: #aaa;
            border-radius: 4px;
        }

        .btn-filtro-grupo {
            flex-shrink: 0;
            padding: 8px 16px;
            border-radius: 20px;
            font-size: 13px;
            font-weight: 500;
            display: flex;
            align-items: center;
            gap: 6px;
            border: 2px solid #ddd;
            background: #f8f9fa;
            color: #555;
            cursor: pointer;
            transition: all 0.2s ease;
            white-space: nowrap;
        }

        .btn-filtro-grupo:hover {
            background: #e9ecef;
            border-color: #adb5bd;
        }

        .btn-filtro-grupo.active {
            background: #333;
            border-color: #333;
            color: #fff;
        }

        .btn-filtro-grupo i {
            font-size: 14px;
        }

        /* Manter compatibilidade com btn-acao existente */
        .btn-acao {
            flex: 1;
            height: 45px;
            border-radius: 10px;
            font-weight: 600;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 8px;
            border: none;
            cursor: pointer;
            transition: all 0.2s;
        }

        .btn-acao.btn-primary {
            background: #001122;
            color: white;
        }

        .btn-acao.btn-primary:hover {
            background: #003366;
        }

        .btn-acao.btn-warning {
            background: #6c757d;
            color: white;
        }

        .btn-acao.btn-warning:hover {
            background: #5a6268;
        }

        .btn-acao.btn-danger {
            background: #868e96;
            color: white;
        }

        .btn-acao.btn-danger:hover {
            background: #6c757d;
        }

        /* Ícone do produto na cor do tema */
        .produto-card .bi-box-seam {
            color: #001122 !important;
        }
    </style>
</head>

<body>
    <!-- Header -->
    <div class="pdv-header">
        <div class="pdv-header-left">
            <div>
                <div class="pdv-header-title">
                    <i class="bi bi-shop me-2"></i>PDV - Caixa
                </div>
                <div class="pdv-header-info">
                    <i class="bi bi-person-circle me-1"></i><?= htmlspecialchars($operador) ?> |
                    <i class="bi bi-calendar3 ms-2 me-1"></i><span id="headerClock"></span>
                </div>
            </div>
        </div>

        <div class="pdv-header-right">
            <button class="btn btn-header" onclick="abrirFuncoes()">
                <i class="bi bi-grid-3x3-gap me-1"></i> Funções
            </button>
            <button class="btn btn-header btn-danger-outline" onclick="sairPDV()">
                <i class="bi bi-box-arrow-right me-1"></i> Sair
            </button>
        </div>
    </div>

    <!-- Conteúdo Principal -->
    <div class="pdv-content">
        <!-- Esquerda: Produtos -->
        <div class="pdv-left">
            <div class="search-box">
                <input type="text" class="search-input" id="searchProduto"
                    placeholder="🔍 Buscar produto por código, nome ou código de barras..." autocomplete="off">
            </div>

            <div class="filtros-grupos" id="filtrosGrupos">
                <button class="btn-filtro-grupo active" data-grupo="todos" onclick="filtrarPorGrupo('todos')">
                    <i class="bi bi-grid-3x3"></i> Todos
                </button>
                <!-- Grupos serão carregados via AJAX -->
            </div>

            <div class="produtos-lista" id="listaProdutos">
                <!-- Produtos serão carregados via AJAX -->
            </div>
        </div>

        <!-- Direita: Carrinho -->
        <div class="pdv-right">
            <div class="carrinho-header">
                <h3><i class="bi bi-cart3 me-2"></i>Itens da Venda</h3>
            </div>

            <div class="carrinho-itens" id="carrinhoItens">
                <div class="carrinho-vazio">
                    <i class="bi bi-cart-x"></i>
                    <p>Nenhum item adicionado</p>
                    <small>Busque e adicione produtos à venda</small>
                </div>
            </div>

            <div class="carrinho-totais">
                <div class="total-linha">
                    <span>Subtotal:</span>
                    <span id="subtotal">R$ 0,00</span>
                </div>
                <div class="total-linha">
                    <span>Desconto:</span>
                    <span id="desconto" style="color: #dc3545;">R$ 0,00</span>
                </div>
                <div class="total-linha grande">
                    <span>TOTAL:</span>
                    <span id="total">R$ 0,00</span>
                </div>

                <button class="btn btn-finalizar" id="btnFinalizar" onclick="finalizarVenda()" disabled>
                    <i class="bi bi-check-circle me-2"></i>
                    FINALIZAR VENDA (F9)
                </button>
            </div>
        </div>
    </div>

    <script src="https://code.jquery.com/jquery-3.7.0.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
    <script src="https://cdn.jsdelivr.net/npm/jquery-mask-plugin@1.14.16/dist/jquery.mask.min.js"></script>

    <script>
        // Dados da venda atual
        let carrinho = [];
        let descontoTotal = 0;
        let clienteSelecionado = null;
        let pagamentosSelecionados = [];

        // Relógio
        function updateHeaderClock() {
            const now = new Date();
            const hours = String(now.getHours()).padStart(2, '0');
            const minutes = String(now.getMinutes()).padStart(2, '0');
            const day = String(now.getDate()).padStart(2, '0');
            const month = String(now.getMonth() + 1).padStart(2, '0');

            $('#headerClock').text(`${day}/${month} ${hours}:${minutes}`);
        }
        updateHeaderClock();
        setInterval(updateHeaderClock, 60000);

        // Função helper para construir URLs com caminho base (GLOBAL E ROBUSTA)
        window.buildUrl = function (path) {
            // Garante que path começa com /
            path = '/' + path.replace(/^\/+/, '');

            // Detecta o caminho base da URL atual (SEMPRE funciona)
            const currentPath = window.location.pathname;
            const pathParts = currentPath.split('/').filter(p => p && p !== 'index.php');

            // Se estiver em /Systhema/pdv/caixa -> basePath = /Systhema
            let basePath = '';
            if (pathParts.length > 0) {
                // Pega o primeiro segmento como base path
                basePath = '/' + pathParts[0];
            }

            // Constrói a URL final
            const finalUrl = basePath + path;
            return finalUrl;
        };

        const buildUrl = window.buildUrl;

        // Grupo selecionado atualmente
        let grupoSelecionado = 'todos';

        // Carregar grupos de produtos
        function carregarGrupos() {
            console.log('[PDV] Carregando grupos...');
            $.ajax({
                url: buildUrl('/pdv/listar-grupos'),
                type: 'GET',
                dataType: 'json',
                success: function (response) {
                    console.log('[PDV] Resposta grupos:', response);
                    if (response.success && response.data && response.data.grupos) {
                        const grupos = response.data.grupos;
                        console.log('[PDV] Grupos encontrados:', grupos.length);

                        let html = `
                            <button class="btn-filtro-grupo active" data-grupo="todos" onclick="filtrarPorGrupo('todos')">
                                <i class="bi bi-grid-3x3"></i> Todos
                            </button>
                        `;

                        if (grupos.length > 0) {
                            grupos.forEach(grupo => {
                                html += `
                                    <button class="btn-filtro-grupo" data-grupo="${grupo.id}" onclick="filtrarPorGrupo(${grupo.id})">
                                        ${grupo.nome}
                                    </button>
                                `;
                            });
                        }

                        $('#filtrosGrupos').html(html);
                    } else {
                        console.warn('[PDV] Nenhum grupo retornado ou erro na resposta');
                    }
                },
                error: function (xhr) {
                    console.error('[PDV] Erro ao carregar grupos:', xhr.status, xhr.responseText);
                }
            });
        }

        // Filtrar produtos por grupo
        function filtrarPorGrupo(grupoId) {
            grupoSelecionado = grupoId;

            // Atualizar visual dos botões
            $('.btn-filtro-grupo').removeClass('active');
            $(`.btn-filtro-grupo[data-grupo="${grupoId}"]`).addClass('active');

            // Recarregar produtos com filtro
            const busca = $('#searchProduto').val() || '';
            carregarProdutos(busca, grupoId);
        }

        // Carregar produtos
        function carregarProdutos(busca = '', grupo = null) {
            $('#listaProdutos').html('<p class="text-center text-muted">Carregando produtos...</p>');

            const grupoFiltro = grupo !== null ? grupo : grupoSelecionado;

            $.ajax({
                url: buildUrl('/pdv/buscar-produtos'),
                type: 'GET',
                data: {
                    busca: busca,
                    grupo_id: grupoFiltro !== 'todos' ? grupoFiltro : ''
                },
                dataType: 'json',
                success: function (response) {
                    if (response.success && response.data.produtos) {
                        exibirProdutos(response.data.produtos);
                    } else {
                        $('#listaProdutos').html('<p class="text-center text-muted">Nenhum produto encontrado</p>');
                    }
                },
                error: function (xhr) {
                    console.error('Erro ao buscar produtos:', xhr.responseText);
                    let mensagemErro = 'Erro ao carregar produtos';
                    try {
                        const erro = JSON.parse(xhr.responseText);
                        mensagemErro = erro.message || erro.error || mensagemErro;
                        console.error('Erro detalhado:', erro);
                    } catch (e) { }
                    $('#listaProdutos').html(`<p class="text-center text-danger">${mensagemErro}</p>`);
                }
            });
        }

        // Exibir produtos no grid
        function exibirProdutos(produtos) {
            if (produtos.length === 0) {
                $('#listaProdutos').html('<p class="text-center text-muted">Nenhum produto encontrado</p>');
                return;
            }

            let html = '';
            produtos.forEach(produto => {
                const preco = parseFloat(produto.preco || 0);
                const estoque = parseInt(produto.estoque || 0);

                html += `
                    <div class="produto-card" onclick='adicionarProduto(${JSON.stringify(produto)})'>
                        <i class="bi bi-box-seam" style="font-size: 48px; color: #495057; margin-bottom: 10px;"></i>
                        <div class="produto-nome">${produto.name}</div>
                        <div class="produto-preco">R$ ${preco.toFixed(2).replace('.', ',')}</div>
                        <small class="text-muted">Estoque: ${estoque}</small>
                    </div>
                `;
            });

            $('#listaProdutos').html(html);
        }

        // Adicionar produto ao carrinho
        function adicionarProduto(produto) {
            const preco = parseFloat(produto.preco || 0);

            if (preco <= 0) {
                Swal.fire('Erro', 'Produto sem preço cadastrado', 'error');
                return;
            }

            const itemExistente = carrinho.find(item => item.id === produto.id);

            if (itemExistente) {
                itemExistente.quantidade++;
                itemExistente.total = itemExistente.quantidade * itemExistente.preco;
            } else {
                carrinho.push({
                    id: produto.id,
                    nome: produto.name,
                    preco: preco,
                    quantidade: 1,
                    total: preco,
                    // Campos fiscais (vêm do buscarProdutos com JOINs de impostos/impostos_configuracoes)
                    ncm: produto.ncm || null,
                    cest: produto.cest || null,
                    cfop: produto.cfop || null,
                    origem: produto.origem || null,
                    cst_icms: produto.cst_icms || null,
                    aliquota_icms: produto.aliquota_icms || 0,
                    base_calculo_icms: null,  // Calculado no processarVenda
                    cst_pis: produto.cst_pis || null,
                    aliquota_pis: produto.aliquota_pis || 0,
                    base_calculo_pis: null,   // Calculado no processarVenda
                    cst_cofins: produto.cst_cofins || null,
                    aliquota_cofins: produto.aliquota_cofins || 0,
                    base_calculo_cofins: null,  // Calculado no processarVenda
                    cst_ipi: produto.cst_ipi || null,
                    aliquota_ipi: produto.aliquota_ipi || 0,
                    origem_st: null  // Opcional
                });
            }

            atualizarCarrinho();
            $('#searchProduto').val('').focus();

            // Feedback visual
            const Toast = Swal.mixin({
                toast: true,
                position: 'top-end',
                showConfirmButton: false,
                timer: 1000
            });
            Toast.fire({
                icon: 'success',
                title: 'Produto adicionado!'
            });
        }

        // Atualizar exibição do carrinho
        function atualizarCarrinho() {
            const $container = $('#carrinhoItens');

            if (carrinho.length === 0) {
                $container.html(`
                    <div class="carrinho-vazio">
                        <i class="bi bi-cart-x"></i>
                        <p>Nenhum item adicionado</p>
                    </div>
                `);
                $('#btnFinalizar').prop('disabled', true);
                atualizarTotais();
                return;
            }

            let html = '';
            carrinho.forEach((item, index) => {
                html += `
                    <div class="item-venda">
                        <div class="item-info">
                            <div class="item-nome">${item.nome}</div>
                            <div class="item-preco">R$ ${item.preco.toFixed(2).replace('.', ',')}</div>
                        </div>
                        <div class="item-qtd">
                            <button class="btn-qtd" onclick="alterarQuantidade(${index}, -1)">-</button>
                            <span class="item-qtd-valor">${item.quantidade}</span>
                            <button class="btn-qtd" onclick="alterarQuantidade(${index}, 1)">+</button>
                        </div>
                        <div class="item-total">R$ ${item.total.toFixed(2).replace('.', ',')}</div>
                        <button class="btn-remover" onclick="removerItem(${index})">
                            <i class="bi bi-trash"></i>
                        </button>
                    </div>
                `;
            });

            $container.html(html);
            $('#btnFinalizar').prop('disabled', false);
            atualizarTotais();
        }

        // Alterar quantidade
        function alterarQuantidade(index, delta) {
            carrinho[index].quantidade += delta;

            if (carrinho[index].quantidade <= 0) {
                removerItem(index);
                return;
            }

            carrinho[index].total = carrinho[index].quantidade * carrinho[index].preco;
            atualizarCarrinho();
        }

        // Remover item
        function removerItem(index) {
            carrinho.splice(index, 1);
            atualizarCarrinho();
        }

        // Atualizar totais
        function atualizarTotais() {
            const subtotal = carrinho.reduce((sum, item) => sum + item.total, 0);
            const total = subtotal - descontoTotal;

            $('#subtotal').text('R$ ' + subtotal.toFixed(2).replace('.', ','));
            $('#desconto').text('R$ ' + descontoTotal.toFixed(2).replace('.', ','));
            $('#total').text('R$ ' + total.toFixed(2).replace('.', ','));
        }

        // Buscar cliente
        function buscarCliente() {
            Swal.fire({
                title: 'Buscar Cliente',
                html: '<input id="swal-cpf" class="swal2-input" placeholder="CPF/CNPJ ou Nome">',
                showCancelButton: true,
                confirmButtonText: 'Buscar',
                cancelButtonText: 'Cancelar',
                preConfirm: () => {
                    return $('#swal-cpf').val();
                }
            }).then((result) => {
                if (result.isConfirmed) {
                    // TODO: Implementar busca de cliente
                    Swal.fire('Info', 'Busca de cliente em desenvolvimento', 'info');
                }
            });
        }

        // Aplicar desconto
        function aplicarDesconto() {
            Swal.fire({
                title: 'Aplicar Desconto',
                html: `
                    <div class="mb-3">
                        <label>Tipo de Desconto:</label>
                        <select id="swal-tipo-desconto" class="swal2-input">
                            <option value="percentual">Percentual (%)</option>
                            <option value="valor">Valor (R$)</option>
                        </select>
                    </div>
                    <input id="swal-valor-desconto" class="swal2-input" type="number" placeholder="Valor" step="0.01">
                `,
                showCancelButton: true,
                confirmButtonText: 'Aplicar',
                cancelButtonText: 'Cancelar',
                preConfirm: () => {
                    const tipo = $('#swal-tipo-desconto').val();
                    const valor = parseFloat($('#swal-valor-desconto').val() || 0);
                    return {
                        tipo,
                        valor
                    };
                }
            }).then((result) => {
                if (result.isConfirmed) {
                    const {
                        tipo,
                        valor
                    } = result.value;
                    const subtotal = carrinho.reduce((sum, item) => sum + item.total, 0);

                    if (tipo === 'percentual') {
                        descontoTotal = (subtotal * valor) / 100;
                    } else {
                        descontoTotal = valor;
                    }

                    atualizarTotais();
                }
            });
        }

        // Cancelar venda
        function cancelarVenda() {
            if (carrinho.length === 0) return;

            Swal.fire({
                title: 'Cancelar Venda?',
                text: 'Todos os itens serão removidos do carrinho',
                icon: 'warning',
                showCancelButton: true,
                confirmButtonText: 'Sim, cancelar',
                cancelButtonText: 'Não',
                confirmButtonColor: '#dc3545'
            }).then((result) => {
                if (result.isConfirmed) {
                    carrinho = [];
                    descontoTotal = 0;
                    clienteSelecionado = null;
                    atualizarCarrinho();
                    Swal.fire('Cancelado!', 'Venda cancelada', 'success');
                }
            });
        }

        // Finalizar venda
        function finalizarVenda() {
            if (carrinho.length === 0) return;

            const subtotal = carrinho.reduce((sum, item) => sum + item.total, 0);
            const total = subtotal - descontoTotal;

            // Montar HTML dos métodos de pagamento
            const metodosPagamento = <?= json_encode($metodosPagamento ?? []) ?>;
            let metodosHtml = '<option value="">Selecione...</option>';
            metodosPagamento.forEach(m => {
                metodosHtml += `<option value="${m.id}" data-type="${m.type || ''}" data-shipay="${m.use_shipay || 0}">${m.name}</option>`;
            });

            // Variáveis de estado do modal
            let pixGerado = null;
            let pixPagamentoConfirmado = false;
            let ultimoValorPix = total;
            let pollingInterval = null;
            let pollingCount = 0;
            const MAX_POLLING = 120; // 120 * 3s = 6 minutos de timeout
            pagamentosSelecionados = [];
            let saldoRestante = total;

            Swal.fire({
                title: '<i class="bi bi-credit-card me-2"></i> Finalizar Venda',
                html: `
                    <div class="text-start">
                        <div class="alert alert-info mb-3">
                            <div class="d-flex justify-content-between align-items-center">
                                <span><strong>Saldo a receber:</strong></span>
                                <span class="h5 mb-0" id="saldo-pendente">R$ ${total.toFixed(2).replace('.', ',')}</span>
                            </div>
                            <div id="lista-pagamentos-adicionados" class="mt-2 small text-muted">Nenhum pagamento adicionado</div>
                        </div>

                        <div class="alert alert-success mb-3">
                            <div class="d-flex justify-content-between align-items-center">
                                <span><strong>Total a Pagar:</strong></span>
                                <span class="h4 mb-0 text-success">R$ ${total.toFixed(2).replace('.', ',')}</span>
                            </div>
                        </div>

                        <div class="mb-3">
                            <label class="form-label fw-bold">Forma de Pagamento *</label>
                            <select id="swal-metodo" class="form-select form-select-lg" onchange="onMetodoPagamentoChange()">
                                ${metodosHtml}
                            </select>
                        </div>

                        <div class="mb-3" id="div-valor-pagamento">
                            <label class="form-label fw-bold">Valor deste pagamento</label>
                            <input type="number" id="swal-valor-pagamento" class="form-control form-control-lg"
                                   step="0.01" placeholder="0,00" value="${total.toFixed(2)}" oninput="onValorPagamentoChange()">
                        </div>

                        <div class="mb-3" id="div-valor-recebido" style="display: none;">
                            <label class="form-label fw-bold">Valor Recebido</label>
                            <input type="number" id="swal-valor-recebido" class="form-control form-control-lg"
                                   step="0.01" placeholder="0,00" oninput="calcularTroco()">
                        </div>

                        <div class="mb-3" id="div-troco" style="display: none;">
                            <div class="alert alert-warning">
                                <div class="d-flex justify-content-between align-items-center">
                                    <span><strong>Troco:</strong></span>
                                    <span class="h5 mb-0" id="valor-troco">R$ 0,00</span>
                                </div>
                            </div>
                        </div>

                        <div class="mb-3" id="div-parcelas" style="display: none;">
                            <label class="form-label fw-bold">Parcelas</label>
                            <select id="swal-parcelas" class="form-select">
                                <option value="1">1x de R$ ${total.toFixed(2).replace('.', ',')}</option>
                                <option value="2">2x de R$ ${(total / 2).toFixed(2).replace('.', ',')}</option>
                                <option value="3">3x de R$ ${(total / 3).toFixed(2).replace('.', ',')}</option>
                                <option value="4">4x de R$ ${(total / 4).toFixed(2).replace('.', ',')}</option>
                                <option value="5">5x de R$ ${(total / 5).toFixed(2).replace('.', ',')}</option>
                                <option value="6">6x de R$ ${(total / 6).toFixed(2).replace('.', ',')}</option>
                            </select>
                        </div>

                        <div id="div-qrcode-pix" style="display: none;">
                            <div id="pix-loading" class="text-center py-3">
                                <div class="spinner-border text-success" role="status" style="width: 3rem; height: 3rem;"></div>
                                <p class="text-muted mt-2">Gerando QR Code PIX...</p>
                            </div>
                            <div id="pix-content" style="display: none;">
                                <div class="text-center">
                                    <div class="alert alert-info mb-3">
                                        <i class="bi bi-qr-code-scan me-2"></i>
                                        <strong>Escaneie o QR Code para pagar</strong>
                                    </div>
                                    <img id="pix-qrcode-img" src="" alt="QR Code PIX" style="max-width: 220px; border: 3px solid #198754; border-radius: 12px; padding: 10px; background: white;">
                                    <div class="mt-3">
                                        <button type="button" class="btn btn-outline-secondary btn-sm" onclick="copiarPixCopiaECola()">
                                            <i class="bi bi-clipboard me-1"></i> Copiar PIX Copia e Cola
                                        </button>
                                    </div>
                                    <textarea id="pix-copia-cola" class="form-control mt-2" rows="2" readonly style="font-size: 11px; display: none;"></textarea>
                                </div>
                            </div>
                            <div id="pix-error" style="display: none;">
                                <div class="alert alert-danger text-center">
                                    <i class="bi bi-exclamation-triangle me-2"></i>
                                    <span id="pix-error-msg">Erro ao gerar QR Code</span>
                                </div>
                            </div>
                        </div>
                    </div>
                `,
                showCancelButton: true,
                confirmButtonText: '<i class="bi bi-check-circle me-2"></i> Finalizar venda',
                cancelButtonText: 'Cancelar',
                confirmButtonColor: '#28a745',
                width: 520,
                didOpen: () => {
                    // Definir funções globais para o modal
                    $('#div-troco').hide();
                    $('#div-valor-recebido').hide();
                    $('#valor-troco').text('R$ 0,00');

                    window.formatarMoeda = function (valor) {
                        return 'R$ ' + Number(valor || 0).toFixed(2).replace('.', ',');
                    };

                    window.atualizarResumoPagamentos = function () {
                        $('#saldo-pendente').text(formatarMoeda(saldoRestante));

                        if (!pagamentosSelecionados.length) {
                            $('#lista-pagamentos-adicionados').text('Nenhum pagamento adicionado');
                            return;
                        }

                        const linhas = pagamentosSelecionados.map((p, idx) => {
                            const metodo = metodosPagamento.find(m => m.id == p.metodo_id);
                            const nome = metodo ? metodo.name : 'Método';
                            const prefixo = `${idx + 1}. ${nome}`;
                            const trocoTxt = p.troco && p.troco > 0 ? ` (troco ${formatarMoeda(p.troco)})` : '';
                            return `${prefixo}: ${formatarMoeda(p.valor)}${trocoTxt}`;
                        });

                        $('#lista-pagamentos-adicionados').html(linhas.join('<br>'));
                    };

                    window.registrarPagamento = function (pagamento) {
                        pagamentosSelecionados.push(pagamento);
                        saldoRestante = Math.max(0, saldoRestante - pagamento.valor);
                        atualizarResumoPagamentos();

                        $('#swal-valor-pagamento').val(saldoRestante.toFixed(2));
                        $('#div-troco').hide();
                        $('#valor-troco').text(formatarMoeda(0));
                        pixGerado = null;
                        pixPagamentoConfirmado = false;

                        if (saldoRestante <= 0.009) {
                            $('#btn-adicionar-pagamento').prop('disabled', true);
                        } else {
                            $('#btn-adicionar-pagamento').prop('disabled', false);
                        }
                    };

                    window.adicionarPagamentoAtual = function () {
                        const metodoId = $('#swal-metodo').val();
                        const metodo = metodosPagamento.find(m => m.id == metodoId);
                        if (!metodo) {
                            Swal.showValidationMessage('Selecione uma forma de pagamento');
                            return;
                        }

                        if (saldoRestante <= 0) {
                            Swal.showValidationMessage('Total já recebido');
                            return;
                        }

                        const tipo = (metodo.type || metodo.name || '').toLowerCase();
                        const valorInformado = parseFloat($('#swal-valor-pagamento').val() || 0);

                        if (!valorInformado || valorInformado <= 0) {
                            Swal.showValidationMessage('Informe o valor deste pagamento');
                            return;
                        }

                        let valorConsiderado = Math.min(valorInformado, saldoRestante);
                        let valorRecebido = valorConsiderado;
                        let troco = 0;

                        if (tipo === 'dinheiro' || tipo.includes('especie')) {
                            const recebidoInput = $('#swal-valor-recebido').val();
                            valorRecebido = recebidoInput === '' ? valorConsiderado : parseFloat(recebidoInput || 0);
                            if (valorRecebido < valorConsiderado) {
                                Swal.showValidationMessage('Valor recebido insuficiente');
                                return;
                            }
                            troco = Math.max(0, valorRecebido - valorConsiderado);
                        } else if (valorInformado > saldoRestante + 0.01) {
                            Swal.showValidationMessage('Valor excede o saldo pendente');
                            return;
                        } else {
                            // Métodos não-dinheiro recebem exatamente o valor considerado, sem troco
                            valorRecebido = valorConsiderado;
                            troco = 0;
                            $('#div-troco').hide();
                        }

                        const pagamento = {
                            metodo_id: metodoId,
                            parcelas: $('#swal-parcelas').val() || 1,
                            valor: valorConsiderado,
                            valor_recebido: valorRecebido,
                            troco: troco,
                            tipo
                        };

                        const useShipay = parseInt(metodo.use_shipay) === 1;
                        if (tipo === 'pix' && useShipay) {
                            if (!pixGerado || !pixGerado.order_id) {
                                Swal.showValidationMessage('Aguarde gerar/confirmar o PIX');
                                return;
                            }
                            if (!pixPagamentoConfirmado) {
                                Swal.showValidationMessage('Aguarde a confirmação do PIX');
                                return;
                            }

                            pagamento.pix_order_id = pixGerado.order_id || null;
                            pagamento.pix_order_ref = pixGerado.order_ref || null;
                            pagamento.pix_qrcode = pixGerado.qrcode || pixGerado.qr_code || null;
                            pagamento.pix_payload = pixGerado.qrcode_text || pixGerado.pix_copia_cola || null;
                        }

                        registrarPagamento(pagamento);
                    };

                    window.onMetodoPagamentoChange = function () {
                        const metodoId = $('#swal-metodo').val();
                        const metodo = metodosPagamento.find(m => m.id == metodoId);

                        // Se tinha PIX gerado e vai trocar de método, cancelar o PIX anterior
                        if (pixGerado && pixGerado.order_id) {
                            console.log('[PDV] Cancelando PIX anterior ao trocar método:', pixGerado.order_id);
                            cancelarPixPendente(pixGerado.order_id);
                        }

                        // Reset
                        $('#div-valor-recebido').hide();
                        $('#div-troco').hide();
                        $('#div-parcelas').hide();
                        $('#div-qrcode-pix').hide();
                        $('#pix-loading').show();
                        $('#pix-content').hide();
                        $('#pix-error').hide();
                        pixGerado = null;
                        pixPagamentoConfirmado = false;
                        $('#swal-valor-pagamento').val(saldoRestante.toFixed(2));
                        $('#swal-valor-recebido').val('');
                        $('#div-troco').hide();
                        $('#valor-troco').text('R$ 0,00');

                        if (!metodo) return;

                        const tipo = (metodo.type || metodo.name || '').toLowerCase();
                        const useShipay = parseInt(metodo.use_shipay) === 1;

                        // PIX com Shipay - gera QR Code automaticamente
                        if (tipo === 'pix' && useShipay) {
                            $('#div-qrcode-pix').show();
                            gerarQrCodePix();
                            return;
                        }

                        // Dinheiro - mostra campo de valor recebido e troco
                        if (tipo === 'dinheiro' || tipo.includes('especie')) {
                            $('#div-valor-recebido').show();
                            $('#swal-valor-recebido').val('').focus();
                        } else {
                            // Outros métodos: esconder campo de recebido e usar valor do pagamento
                            $('#div-valor-recebido').hide();
                            $('#div-troco').hide();
                            $('#swal-valor-recebido').val($('#swal-valor-pagamento').val() || saldoRestante.toFixed(2));
                        }

                        // Cartão de crédito - mostra parcelas
                        if (tipo === 'cartao_credito' || tipo.includes('crédito') || tipo.includes('credito')) {
                            $('#div-parcelas').show();
                        }
                    };

                    window.onValorPagamentoChange = function () {
                        const metodoId = $('#swal-metodo').val();
                        const metodo = metodosPagamento.find(m => m.id == metodoId);
                        const tipo = (metodo ? (metodo.type || metodo.name || '') : '').toLowerCase();
                        const valorInformado = parseFloat($('#swal-valor-pagamento').val() || 0);

                        if (tipo === 'dinheiro' || tipo.includes('especie')) {
                            $('#div-valor-recebido').show();
                            $('#swal-valor-recebido').val(valorInformado ? valorInformado.toFixed(2) : '');
                            calcularTroco();
                        } else {
                            // Para métodos não-dinheiro, igualar valor recebido ao valor do pagamento e ocultar troco
                            $('#div-valor-recebido').hide();
                            $('#div-troco').hide();
                            $('#swal-valor-recebido').val(valorInformado ? valorInformado.toFixed(2) : saldoRestante.toFixed(2));
                        }
                    };

                    window.gerarQrCodePix = function () {
                        const valorPix = parseFloat($('#swal-valor-pagamento').val() || saldoRestante);
                        ultimoValorPix = valorPix;
                        console.log('[PDV] Gerando PIX - Valor:', valorPix, 'Cliente:', clienteSelecionado);

                        // Limpar polling anterior se existir
                        if (pollingInterval) {
                            clearInterval(pollingInterval);
                            pollingInterval = null;
                        }
                        pollingCount = 0;

                        $.ajax({
                            url: buildUrl('/pdv/gerar-pix-shipay'),
                            type: 'POST',
                            data: {
                                total: valorPix.toFixed(2),
                                cliente_id: clienteSelecionado ? clienteSelecionado.id : ''
                            },
                            dataType: 'json',
                            success: function (response) {
                                $('#pix-loading').hide();
                                if (response.success && response.data) {
                                    pixGerado = response.data;

                                    // Exibir QR Code
                                    if (response.data.qrcode) {
                                        $('#pix-qrcode-img').attr('src', response.data.qrcode);
                                    } else if (response.data.qrcode_text) {
                                        // Gerar QR Code via biblioteca se não vier imagem
                                        $('#pix-qrcode-img').attr('src', 'https://api.qrserver.com/v1/create-qr-code/?size=200x200&data=' + encodeURIComponent(response.data.qrcode_text));
                                    }

                                    if (response.data.qrcode_text) {
                                        $('#pix-copia-cola').val(response.data.qrcode_text);
                                    }

                                    $('#pix-content').show();

                                    // Iniciar polling para verificar pagamento
                                    iniciarPollingPagamento(response.data.order_id);
                                } else {
                                    $('#pix-error-msg').text(response.message || 'Erro ao gerar QR Code');
                                    $('#pix-error').show();
                                }
                            },
                            error: function (xhr) {
                                console.error('[PDV] Erro ao gerar PIX:', xhr.status, xhr.responseText);
                                $('#pix-loading').hide();
                                let msg = 'Erro ao conectar com Shipay';
                                try {
                                    const err = JSON.parse(xhr.responseText);
                                    msg = err.message || msg;
                                    console.error('[PDV] Mensagem de erro:', msg);
                                } catch (e) {
                                    console.error('[PDV] Resposta não é JSON:', xhr.responseText);
                                }
                                $('#pix-error-msg').text(msg);
                                $('#pix-error').show();
                            }
                        });
                    };

                    // Polling para verificar status do pagamento
                    window.iniciarPollingPagamento = function (orderId) {
                        console.log('[PDV] Iniciando polling para order:', orderId);

                        // Adicionar indicador de aguardando pagamento
                        $('#pix-content').append(`
                            <div id="pix-aguardando" class="mt-3 text-center">
                                <div class="spinner-border spinner-border-sm text-success me-2" role="status"></div>
                                <span class="text-muted">Aguardando pagamento...</span>
                                <div class="small text-muted mt-1" id="pix-timer">Verificando a cada 3 segundos</div>
                            </div>
                        `);

                        pollingInterval = setInterval(function () {
                            pollingCount++;

                            if (pollingCount > MAX_POLLING) {
                                clearInterval(pollingInterval);
                                $('#pix-aguardando').html('<span class="text-warning"><i class="bi bi-clock"></i> Tempo esgotado. Clique em "Confirmar Pagamento" se já pagou.</span>');
                                return;
                            }

                            $.ajax({
                                url: buildUrl('/pdv/consultar-status-pix'),
                                type: 'POST',
                                data: { order_id: orderId },
                                dataType: 'json',
                                success: function (response) {
                                    console.log('[PDV] Status PIX:', response);

                                    if (response.success && response.data) {
                                        const status = response.data.status;
                                        const pago = response.data.pago;

                                        if (pago) {
                                            // PAGAMENTO CONFIRMADO!
                                            clearInterval(pollingInterval);
                                            console.log('[PDV] ✅ PAGAMENTO CONFIRMADO!');

                                            const metodoIdSalvo = $('#swal-metodo').val();
                                            const pixOrderIdSalvo = pixGerado ? pixGerado.order_id : orderId;
                                            const pixOrderRefSalvo = pixGerado ? pixGerado.order_ref : null;

                                            pixPagamentoConfirmado = true;

                                            const valorPagamentoPix = Math.min(ultimoValorPix, saldoRestante);

                                            registrarPagamento({
                                                metodo_id: metodoIdSalvo,
                                                parcelas: 1,
                                                valor: valorPagamentoPix,
                                                valor_recebido: valorPagamentoPix,
                                                troco: 0,
                                                tipo: 'pix',
                                                pix_order_id: pixOrderIdSalvo,
                                                pix_order_ref: pixOrderRefSalvo,
                                                pix_qrcode: pixGerado ? (pixGerado.qrcode || pixGerado.qr_code) : null,
                                                pix_payload: pixGerado ? (pixGerado.qrcode_text || pixGerado.pix_copia_cola) : null
                                            });

                                            // Verificar saldo após registrar (registrarPagamento atualiza saldoRestante)
                                            // Aguardar um pouco para garantir que a atualização do DOM aconteceu
                                            setTimeout(() => {
                                                if (saldoRestante <= 0.009) {
                                                    // Fechar modal e processar venda
                                                    Swal.close();

                                                    // Processar venda automaticamente
                                                    setTimeout(() => {
                                                        processarVenda({ pagamentos: pagamentosSelecionados });
                                                    }, 500);
                                                } else {
                                                    // Ainda falta pagamento - mostrar notificação mais visível
                                                    Swal.fire({
                                                        icon: 'success',
                                                        title: 'Pagamento PIX Confirmado!',
                                                        html: `
                                                            <div class="text-center">
                                                                <p class="h5 mb-3">✅ O pagamento PIX foi confirmado com sucesso!</p>
                                                                <p class="mb-0">Saldo restante: <strong>${formatarMoeda(saldoRestante)}</strong></p>
                                                                <p class="text-muted mt-2">Adicione mais pagamentos se necessário.</p>
                                                            </div>
                                                        `,
                                                        confirmButtonText: 'OK',
                                                        confirmButtonColor: '#0d6efd',
                                                        timer: 5000,
                                                        timerProgressBar: true
                                                    });
                                                }
                                            }, 100);

                                        } else if (status === 'cancelado' || status === 'expirado') {
                                            // Pagamento cancelado ou expirado
                                            clearInterval(pollingInterval);
                                            $('#pix-aguardando').html(`<span class="text-danger"><i class="bi bi-x-circle"></i> Pagamento ${status}</span>`);
                                        } else {
                                            // Ainda pendente
                                            $('#pix-timer').text(`Verificando... (${pollingCount})`);
                                        }
                                    }
                                },
                                error: function (xhr) {
                                    console.error('[PDV] Erro ao consultar status:', xhr.responseText);
                                }
                            });
                        }, 3000); // Verificar a cada 3 segundos
                    };

                    window.copiarPixCopiaECola = function () {
                        const texto = $('#pix-copia-cola').val();
                        if (texto) {
                            navigator.clipboard.writeText(texto).then(() => {
                                const Toast = Swal.mixin({ toast: true, position: 'top-end', showConfirmButton: false, timer: 2000 });
                                Toast.fire({ icon: 'success', title: 'PIX copiado!' });
                            });
                        }
                    };

                    // Função para cancelar PIX pendente (não bloqueia a UI)
                    window.cancelarPixPendente = function (orderId) {
                        if (!orderId) return;

                        console.log('[PDV] Cancelando PIX pendente:', orderId);

                        // Enviar cancelamento em background (não espera resposta)
                        $.ajax({
                            url: buildUrl('/pdv/cancelar-pix-shipay'),
                            type: 'POST',
                            data: { order_id: orderId },
                            dataType: 'json',
                            timeout: 5000 // 5 segundos de timeout
                        }).done(function (response) {
                            console.log('[PDV] PIX cancelado com sucesso:', response);
                        }).fail(function (xhr, status, error) {
                            console.warn('[PDV] Erro ao cancelar PIX (ignorado):', error);
                        });
                    };

                    window.calcularTroco = function () {
                        const metodoId = $('#swal-metodo').val();
                        const metodo = metodosPagamento.find(m => m.id == metodoId);
                        const tipo = (metodo ? (metodo.type || metodo.name || '') : '').toLowerCase();
                        if (!(tipo === 'dinheiro' || tipo.includes('especie'))) {
                            $('#div-troco').hide();
                            return;
                        }
                        const valorRecebido = parseFloat($('#swal-valor-recebido').val() || 0);
                        const valorPagamento = parseFloat($('#swal-valor-pagamento').val() || 0);
                        const referencia = Math.min(valorPagamento, saldoRestante || total);
                        const troco = valorRecebido - referencia;

                        if (troco >= 0) {
                            $('#div-troco').show();
                            $('#valor-troco').text('R$ ' + troco.toFixed(2).replace('.', ','));
                            $('#valor-troco').removeClass('text-danger').addClass('text-success');
                        } else {
                            $('#div-troco').show();
                            $('#valor-troco').text('Falta R$ ' + Math.abs(troco).toFixed(2).replace('.', ','));
                            $('#valor-troco').removeClass('text-success').addClass('text-danger');
                        }
                    };

                    atualizarResumoPagamentos();
                },
                preConfirm: () => {
                    // Se ainda falta valor, tenta adicionar o pagamento atual automaticamente
                    if (saldoRestante > 0.009) {
                        adicionarPagamentoAtual();
                    }

                    if (saldoRestante > 0.009) {
                        Swal.showValidationMessage('Ainda faltam ' + formatarMoeda(saldoRestante));
                        return false;
                    }

                    if (!pagamentosSelecionados.length) {
                        Swal.showValidationMessage('Adicione ao menos um pagamento');
                        return false;
                    }

                    return { pagamentos: pagamentosSelecionados };
                }
            }).then((result) => {
                // Parar polling quando modal fechar
                if (pollingInterval) {
                    clearInterval(pollingInterval);
                    pollingInterval = null;
                }

                if (result.isConfirmed) {
                    processarVenda({ pagamentos: pagamentosSelecionados });
                } else {
                    // Modal foi fechado/cancelado - cancelar PIX pendente se existir
                    if (pixGerado && pixGerado.order_id) {
                        console.log('[PDV] Modal fechado sem confirmar - cancelando PIX:', pixGerado.order_id);
                        cancelarPixPendente(pixGerado.order_id);
                    }
                }
            });
        }

        // Processar venda no servidor
        function processarVenda(pagamento) {
            const subtotal = carrinho.reduce((sum, item) => sum + item.total, 0);
            const total = subtotal - descontoTotal;

            const pagamentos = (pagamento && Array.isArray(pagamento.pagamentos)) ? pagamento.pagamentos :
                (pagamentosSelecionados.length ? pagamentosSelecionados : (pagamento ? [pagamento] : []));

            if (!pagamentos.length) {
                Swal.fire('Erro!', 'Adicione pelo menos um pagamento', 'error');
                return;
            }

            Swal.fire({
                title: 'Processando...',
                html: '<div class="spinner-border text-primary" role="status"></div><p class="mt-3">Finalizando venda...</p>',
                allowOutsideClick: false,
                showConfirmButton: false
            });

            const totalRecebido = pagamentos.reduce((sum, p) => sum + parseFloat(p.valor_recebido || p.valor || 0), 0);
            const primeiroPagamento = pagamentos[0] || {};
            const primeiroPix = pagamentos.find(p => p.pix_order_id) || {};

            // Garantir que valores null/undefined não sejam enviados como string
            const pixOrderId = primeiroPix.pix_order_id || '';
            const pixOrderRef = primeiroPix.pix_order_ref || '';
            const pixQrcode = primeiroPix.pix_qrcode || '';
            const pixPayload = primeiroPix.pix_payload || '';

            const dadosVenda = {
                itens: carrinho,
                subtotal: subtotal,
                desconto: descontoTotal,
                total: total,
                cliente_id: clienteSelecionado ? clienteSelecionado.id : '',
                metodo_pagamento_id: primeiroPagamento.metodo_id || pagamento.metodo_id,
                parcelas: primeiroPagamento.parcelas || pagamento.parcelas || 1,
                valor_recebido: totalRecebido,
                pagamentos: JSON.stringify(pagamentos),
                pix_order_id: pixOrderId,
                pix_order_ref: pixOrderRef,
                pix_qrcode: pixQrcode,
                pix_payload: pixPayload
            };
            console.log('[PDV] Enviando para /pdv/processar-venda:', dadosVenda);
            console.log('[PDV] pix_order_id sendo enviado:', pixOrderId, 'tipo:', typeof pixOrderId);

            $.ajax({
                url: buildUrl('/pdv/processar-venda'),
                type: 'POST',
                data: dadosVenda,
                dataType: 'json',
                success: function (response) {
                    if (response.success) {
                        const venda = response.data.venda || {};
                        const troco = parseFloat(response.data?.troco ?? (totalRecebido - total));
                        const temPix = primeiroPix.pix_order_id || response.data.qrcode;
                        const jaProcessada = response.data.ja_processada || false;

                        // Se venda já foi processada pelo webhook, limpar carrinho imediatamente
                        if (jaProcessada) {
                            carrinho = [];
                            descontoTotal = 0;
                            clienteSelecionado = null;
                            pagamentosSelecionados = [];
                            atualizarCarrinho();
                            $('#searchProduto').focus();
                        }

                        // Modal de sucesso - Pergunta tipo de documento
                        Swal.fire({
                            icon: 'success',
                            title: jaProcessada ? 'Venda já foi finalizada!' : 'Venda Finalizada!',
                            html: `
                                <div class="text-center">
                                    ${jaProcessada ? `
                                        <div class="alert alert-info mb-3">
                                            <i class="bi bi-info-circle me-2"></i>
                                            <strong>Esta venda já foi processada automaticamente pelo sistema.</strong>
                                        </div>
                                    ` : ''}
                                    <div class="alert alert-success mb-3">
                                        <h4 class="mb-0">Venda #${venda.sale_number || venda.id}</h4>
                                    </div>
                                    <p class="h3 text-success mb-3">R$ ${total.toFixed(2).replace('.', ',')}</p>
                                    ${troco > 0 ? `<p class="h5 text-warning mb-3">Troco: R$ ${troco.toFixed(2).replace('.', ',')}</p>` : ''}
                                    <hr>
                                    <p class="mb-2"><strong>Imprimir documento:</strong></p>
                                </div>
                            `,
                            showDenyButton: true,
                            showCancelButton: true,
                            confirmButtonText: '<i class="bi bi-file-earmark-text me-1"></i> NFCe',
                            confirmButtonColor: '#0d6efd',
                            denyButtonText: '<i class="bi bi-receipt me-1"></i> Cupom',
                            denyButtonColor: '#6c757d',
                            cancelButtonText: 'Sem impressão',
                            cancelButtonColor: '#dc3545',
                            allowOutsideClick: false
                        }).then((result) => {
                            if (result.isConfirmed) {
                                // Emissão NFC-e: enviar venda_id para buscar dados do banco
                                Swal.fire({
                                    title: 'Emitindo NFC-e...',
                                    html: '<div class="spinner-border text-primary" role="status"></div><p class="mt-3">Aguarde, processando dados da venda...</p>',
                                    allowOutsideClick: false,
                                    showConfirmButton: false
                                });

                                // Enviar venda_id para o backend buscar todos os dados
                                const payload = {
                                    venda_id: venda.id
                                };

                                // Enviar para rota proxy do PDV
                                return fetch(buildUrl('/pdv/emitir-nfce'), {
                                    method: 'POST',
                                    headers: { 'Content-Type': 'application/json' },
                                    body: JSON.stringify(payload)
                                })
                                    .then(r => r.json())
                                    .then(resp => {
                                        if (resp.success || (resp.data && resp.data.success)) {
                                            const data = resp.data || resp;
                                            const pdf = data.pdf_path || data.pdf || resp.pdf_path || resp.pdf;
                                            const chaveAcesso = data.chave_acesso || resp.chave_acesso || '';

                                            let html = `
                                                <div class="text-center">
                                                    <div class="alert alert-success mb-3">
                                                        <h5 class="mb-1">✅ NFC-e Emitida com Sucesso!</h5>
                                                        <p class="mb-0"><strong>Número:</strong> ${data.numero || resp.numero || 'N/A'}</p>
                                                        ${data.protocolo || resp.protocolo ? `<p class="mb-0"><strong>Protocolo:</strong> ${data.protocolo || resp.protocolo}</p>` : ''}
                                                    </div>
                                                    ${chaveAcesso ? `
                                                        <div class="alert alert-info mb-3 text-start">
                                                            <small><strong>Chave de Acesso:</strong><br>
                                                            <code style="font-size: 0.85em; word-break: break-all;">${chaveAcesso}</code></small>
                                                        </div>
                                                    ` : ''}
                                                </div>
                                            `;

                                            Swal.fire({
                                                icon: 'success',
                                                title: 'NFC-e Emitida',
                                                html: html,
                                                showCancelButton: true,
                                                confirmButtonText: pdf ? '<i class="bi bi-file-pdf me-1"></i> Abrir PDF' : 'OK',
                                                cancelButtonText: 'Fechar',
                                                width: '600px'
                                            }).then(btn => {
                                                if (btn.isConfirmed && pdf) {
                                                    // Construir URL completa do PDF
                                                    const pdfUrl = pdf.startsWith('http') ? pdf : buildUrl(pdf);
                                                    window.open(pdfUrl, '_blank');
                                                }
                                            });
                                        } else {
                                            const errorMsg = resp.error || resp.message || JSON.stringify(resp);
                                            throw new Error(errorMsg);
                                        }
                                    })
                                    .catch(err => {
                                        console.error('[NFCe] Erro:', err);
                                        Swal.fire('Erro NFCe', err.message || String(err), 'error');
                                    });
                            } else if (result.isDenied) {
                                // Imprimir Cupom
                                window.open(buildUrl('/pdv/imprimir-cupom?id=' + (venda.id || '')), '_blank');
                            }

                            // Limpar carrinho para nova venda automaticamente
                            carrinho = [];
                            descontoTotal = 0;
                            clienteSelecionado = null;
                            pagamentosSelecionados = [];
                            atualizarCarrinho();
                            $('#searchProduto').focus();
                        });
                    } else {
                        Swal.fire('Erro!', response.message || 'Erro ao processar venda', 'error');
                    }
                },
                error: function (xhr) {
                    console.error('Erro:', xhr);
                    let msg = 'Erro ao processar venda';
                    try {
                        const err = JSON.parse(xhr.responseText);
                        msg = err.message || msg;
                    } catch (e) { }
                    Swal.fire('Erro!', msg, 'error');
                }
            });
        }

        // Abrir modal de Funções
        function abrirFuncoes() {
            Swal.fire({
                title: '<i class="bi bi-grid-3x3-gap me-2"></i> Funções',
                html: `
                    <div class="funcoes-grid">
                        <button type="button" class="btn-funcao" onclick="Swal.close(); buscarCliente();">
                            <i class="bi bi-person-plus text-primary"></i>
                            <span>Cliente (F2)</span>
                        </button>
                        <button type="button" class="btn-funcao" onclick="Swal.close(); aplicarDesconto();">
                            <i class="bi bi-percent text-warning"></i>
                            <span>Desconto (F3)</span>
                        </button>
                        <button type="button" class="btn-funcao" onclick="Swal.close(); cancelarVendaAtual();">
                            <i class="bi bi-x-circle text-danger"></i>
                            <span>Cancelar (F4)</span>
                        </button>
                        <button type="button" class="btn-funcao" onclick="Swal.close(); abrirSangria();">
                            <i class="bi bi-dash-circle text-danger"></i>
                            <span>Sangria</span>
                        </button>
                        <button type="button" class="btn-funcao" onclick="Swal.close(); abrirSuprimento();">
                            <i class="bi bi-plus-circle text-success"></i>
                            <span>Suprimento</span>
                        </button>
                        <button type="button" class="btn-funcao" onclick="Swal.close(); abrirPagamentos();">
                            <i class="bi bi-cash-stack text-primary"></i>
                            <span>Pagamentos</span>
                        </button>
                        <button type="button" class="btn-funcao" onclick="Swal.close(); abrirShipay();">
                            <i class="bi bi-qr-code" style="color: #6f42c1;"></i>
                            <span>Shipay</span>
                        </button>
                        <button type="button" class="btn-funcao" onclick="Swal.close(); salvarOrcamento();">
                            <i class="bi bi-file-earmark-text text-warning"></i>
                            <span>Salvar Orçamento</span>
                        </button>
                        <button type="button" class="btn-funcao" onclick="Swal.close(); abrirVendaSalva();">
                            <i class="bi bi-folder2-open text-info"></i>
                            <span>Abrir Venda</span>
                        </button>
                        <button type="button" class="btn-funcao" onclick="Swal.close(); abrirAjustes();">
                            <i class="bi bi-gear text-secondary"></i>
                            <span>Ajustes</span>
                        </button>
                        <button type="button" class="btn-funcao" onclick="Swal.close(); fecharCaixa();">
                            <i class="bi bi-lock text-dark"></i>
                            <span>Fechar Caixa</span>
                        </button>
                    </div>
                `,
                showConfirmButton: false,
                showCloseButton: true,
                customClass: {
                    popup: 'swal-funcoes-popup'
                }
            });
        }

        // Sangria
        function abrirSangria() {
            Swal.fire({
                title: '<i class="bi bi-dash-circle text-danger me-2"></i> Sangria de Caixa',
                html: `
                    <div class="text-start">
                        <div class="mb-3">
                            <label class="form-label fw-bold">Valor da Sangria *</label>
                            <input type="number" id="swal-sangria-valor" class="form-control form-control-lg"
                                   step="0.01" min="0.01" placeholder="0,00" autofocus>
                        </div>
                        <div class="mb-3">
                            <label class="form-label fw-bold">Motivo</label>
                            <textarea id="swal-sangria-motivo" class="form-control" rows="2"
                                      placeholder="Ex: Retirada para pagamento de fornecedor"></textarea>
                        </div>
                    </div>
                `,
                showCancelButton: true,
                confirmButtonText: '<i class="bi bi-check-lg me-1"></i> Confirmar',
                confirmButtonColor: '#dc3545',
                cancelButtonText: 'Cancelar',
                preConfirm: () => {
                    const valor = parseFloat($('#swal-sangria-valor').val()) || 0;
                    if (valor <= 0) {
                        Swal.showValidationMessage('Informe um valor válido');
                        return false;
                    }

                    Swal.showLoading();

                    return $.ajax({
                        url: buildUrl('/pdv/registrar-sangria'),
                        type: 'POST',
                        data: {
                            valor: valor,
                            observacoes: $('#swal-sangria-motivo').val()
                        },
                        dataType: 'json'
                    }).then(response => {
                        if (!response.success) {
                            throw new Error(response.message || 'Erro ao registrar sangria');
                        }
                        return response;
                    }).catch(error => {
                        Swal.showValidationMessage(error.responseJSON?.message || error.message || 'Erro ao registrar sangria');
                    });
                }
            }).then((result) => {
                if (result.isConfirmed && result.value) {
                    const data = result.value.data || {};
                    const motivo = $('#swal-sangria-motivo').val() || 'Sangria de caixa';
                    Swal.fire({
                        icon: 'success',
                        title: 'Sangria Registrada!',
                        html: `
                            <div class="text-center">
                                <p class="h4 text-danger mb-0">- R$ ${parseFloat(data.valor || 0).toFixed(2).replace('.', ',')}</p>
                            </div>
                        `
                    }).then(() => {
                        imprimirComprovanteSangria(data.valor, motivo);
                    });
                }
            });
        }

        // Imprimir comprovante de sangria
        function imprimirComprovanteSangria(valor, motivo) {
            const dataAtual = new Date().toLocaleString('pt-BR');
            const operador = '<?= htmlspecialchars($operador ?? 'Operador') ?>';
            const caixa = 'Caixa PDV';

            const comprovante = `
                <!DOCTYPE html>
                <html>
                <head>
                    <meta charset="UTF-8">
                    <title>Comprovante de Sangria</title>
                    <style>
                        * { margin: 0; padding: 0; box-sizing: border-box; }
                        body {
                            font-family: 'Courier New', monospace;
                            font-size: 12px;
                            width: 80mm;
                            padding: 5mm;
                            background: #fff;
                        }
                        .center { text-align: center; }
                        .bold { font-weight: bold; }
                        .linha { border-top: 1px dashed #000; margin: 8px 0; }
                        .titulo { font-size: 16px; font-weight: bold; margin-bottom: 10px; }
                        .valor { font-size: 20px; font-weight: bold; margin: 15px 0; }
                        .info { margin: 5px 0; }
                        .assinatura { margin-top: 30px; border-top: 1px solid #000; padding-top: 5px; width: 80%; margin-left: auto; margin-right: auto; }
                        @media print {
                            body { width: 80mm; }
                            @page { margin: 0; size: 80mm auto; }
                        }
                    </style>
                </head>
                <body>
                    <div class="center">
                        <div class="titulo">COMPROVANTE DE SANGRIA</div>
                    </div>
                    <div class="linha"></div>
                    <div class="info"><strong>Data/Hora:</strong> ${dataAtual}</div>
                    <div class="info"><strong>Operador:</strong> ${operador}</div>
                    <div class="info"><strong>Caixa:</strong> ${caixa}</div>
                    <div class="linha"></div>
                    <div class="center valor">
                        R$ ${parseFloat(valor || 0).toFixed(2).replace('.', ',')}
                    </div>
                    <div class="linha"></div>
                    <div class="info"><strong>Motivo:</strong></div>
                    <div class="info">${motivo || 'Sangria de caixa'}</div>
                    <div class="linha"></div>
                    <div class="assinatura center">
                        <div>Assinatura do Responsável</div>
                    </div>
                    <div style="margin-top: 20px;" class="center">
                        <small>Via do Caixa</small>
                    </div>
                </body>
                </html>
            `;

            const janela = window.open('', '_blank', 'width=400,height=600');
            janela.document.write(comprovante);
            janela.document.close();
            janela.focus();
            setTimeout(() => {
                janela.print();
            }, 300);
        }

        // Suprimento
        function abrirSuprimento() {
            Swal.fire({
                title: '<i class="bi bi-plus-circle text-success me-2"></i> Suprimento de Caixa',
                html: `
                    <div class="text-start">
                        <div class="mb-3">
                            <label class="form-label fw-bold">Valor do Suprimento *</label>
                            <input type="number" id="swal-suprimento-valor" class="form-control form-control-lg"
                                   step="0.01" min="0.01" placeholder="0,00" autofocus>
                        </div>
                        <div class="mb-3">
                            <label class="form-label fw-bold">Motivo</label>
                            <textarea id="swal-suprimento-motivo" class="form-control" rows="2"
                                      placeholder="Ex: Troco inicial"></textarea>
                        </div>
                    </div>
                `,
                showCancelButton: true,
                confirmButtonText: '<i class="bi bi-check-lg me-1"></i> Confirmar',
                confirmButtonColor: '#28a745',
                cancelButtonText: 'Cancelar',
                preConfirm: () => {
                    const valor = parseFloat($('#swal-suprimento-valor').val()) || 0;
                    if (valor <= 0) {
                        Swal.showValidationMessage('Informe um valor válido');
                        return false;
                    }

                    Swal.showLoading();

                    return $.ajax({
                        url: buildUrl('/pdv/registrar-suprimento'),
                        type: 'POST',
                        data: {
                            valor: valor,
                            observacoes: $('#swal-suprimento-motivo').val()
                        },
                        dataType: 'json'
                    }).then(response => {
                        if (!response.success) {
                            throw new Error(response.message || 'Erro ao registrar suprimento');
                        }
                        return response;
                    }).catch(error => {
                        Swal.showValidationMessage(error.responseJSON?.message || error.message || 'Erro ao registrar suprimento');
                    });
                }
            }).then((result) => {
                if (result.isConfirmed && result.value) {
                    const data = result.value.data || {};
                    const motivo = $('#swal-suprimento-motivo').val() || 'Suprimento de caixa';
                    Swal.fire({
                        icon: 'success',
                        title: 'Suprimento Registrado!',
                        html: `
                            <div class="text-center">
                                <p class="h4 text-success mb-0">+ R$ ${parseFloat(data.valor || 0).toFixed(2).replace('.', ',')}</p>
                            </div>
                        `
                    }).then(() => {
                        imprimirComprovanteSuprimento(data.valor, motivo);
                    });
                }
            });
        }

        // Imprimir comprovante de suprimento
        function imprimirComprovanteSuprimento(valor, motivo) {
            const dataAtual = new Date().toLocaleString('pt-BR');
            const operador = '<?= htmlspecialchars($operador ?? 'Operador') ?>';
            const caixa = 'Caixa PDV';

            const comprovante = `
                <!DOCTYPE html>
                <html>
                <head>
                    <meta charset="UTF-8">
                    <title>Comprovante de Suprimento</title>
                    <style>
                        * { margin: 0; padding: 0; box-sizing: border-box; }
                        body {
                            font-family: 'Courier New', monospace;
                            font-size: 12px;
                            width: 80mm;
                            padding: 5mm;
                            background: #fff;
                        }
                        .center { text-align: center; }
                        .bold { font-weight: bold; }
                        .linha { border-top: 1px dashed #000; margin: 8px 0; }
                        .titulo { font-size: 16px; font-weight: bold; margin-bottom: 10px; }
                        .valor { font-size: 20px; font-weight: bold; margin: 15px 0; }
                        .info { margin: 5px 0; }
                        .assinatura { margin-top: 30px; border-top: 1px solid #000; padding-top: 5px; width: 80%; margin-left: auto; margin-right: auto; }
                        @media print {
                            body { width: 80mm; }
                            @page { margin: 0; size: 80mm auto; }
                        }
                    </style>
                </head>
                <body>
                    <div class="center">
                        <div class="titulo">COMPROVANTE DE SUPRIMENTO</div>
                    </div>
                    <div class="linha"></div>
                    <div class="info"><strong>Data/Hora:</strong> ${dataAtual}</div>
                    <div class="info"><strong>Operador:</strong> ${operador}</div>
                    <div class="info"><strong>Caixa:</strong> ${caixa}</div>
                    <div class="linha"></div>
                    <div class="center valor">
                        R$ ${parseFloat(valor || 0).toFixed(2).replace('.', ',')}
                    </div>
                    <div class="linha"></div>
                    <div class="info"><strong>Motivo:</strong></div>
                    <div class="info">${motivo || 'Suprimento de caixa'}</div>
                    <div class="linha"></div>
                    <div class="assinatura center">
                        <div>Assinatura do Responsável</div>
                    </div>
                    <div style="margin-top: 20px;" class="center">
                        <small>Via do Caixa</small>
                    </div>
                </body>
                </html>
            `;

            const janela = window.open('', '_blank', 'width=400,height=600');
            janela.document.write(comprovante);
            janela.document.close();
            janela.focus();
            setTimeout(() => {
                janela.print();
            }, 300);
        }

        // Pagamentos de despesas
        function abrirPagamentos() {
            Swal.fire({
                title: '<i class="bi bi-cash-stack text-primary me-2"></i> Pagamento de Despesa',
                html: `
                    <div class="text-start">
                        <div class="mb-3">
                            <label class="form-label fw-bold">Valor do Pagamento *</label>
                            <input type="number" id="swal-pagamento-valor" class="form-control form-control-lg"
                                   step="0.01" min="0.01" placeholder="0,00" autofocus>
                        </div>
                        <div class="mb-3">
                            <label class="form-label fw-bold">Descrição da Despesa *</label>
                            <input type="text" id="swal-pagamento-descricao" class="form-control"
                                   placeholder="Ex: Compra de material de escritório">
                        </div>
                        <div class="mb-3">
                            <label class="form-label fw-bold">Fornecedor/Beneficiário</label>
                            <input type="text" id="swal-pagamento-fornecedor" class="form-control"
                                   placeholder="Ex: Papelaria Central">
                        </div>
                    </div>
                `,
                showCancelButton: true,
                confirmButtonText: '<i class="bi bi-check-lg me-1"></i> Confirmar Pagamento',
                confirmButtonColor: '#0d6efd',
                cancelButtonText: 'Cancelar',
                preConfirm: () => {
                    const valor = parseFloat($('#swal-pagamento-valor').val()) || 0;
                    const descricao = $('#swal-pagamento-descricao').val().trim();

                    if (valor <= 0) {
                        Swal.showValidationMessage('Informe um valor válido');
                        return false;
                    }
                    if (!descricao) {
                        Swal.showValidationMessage('Informe a descrição da despesa');
                        return false;
                    }

                    Swal.showLoading();

                    return $.ajax({
                        url: buildUrl('/pdv/registrar-pagamento'),
                        type: 'POST',
                        data: {
                            valor: valor,
                            descricao: descricao,
                            fornecedor: $('#swal-pagamento-fornecedor').val()
                        },
                        dataType: 'json'
                    }).then(response => {
                        if (!response.success) {
                            throw new Error(response.message || 'Erro ao registrar pagamento');
                        }
                        return response;
                    }).catch(error => {
                        Swal.showValidationMessage(error.responseJSON?.message || error.message || 'Erro ao registrar pagamento');
                    });
                }
            }).then((result) => {
                if (result.isConfirmed && result.value) {
                    const data = result.value.data || {};
                    const descricao = $('#swal-pagamento-descricao').val() || 'Pagamento de despesa';
                    const fornecedor = $('#swal-pagamento-fornecedor').val() || '';
                    Swal.fire({
                        icon: 'success',
                        title: 'Pagamento Registrado!',
                        html: `
                            <div class="text-center">
                                <p class="h4 text-primary mb-0">- R$ ${parseFloat(data.valor || 0).toFixed(2).replace('.', ',')}</p>
                            </div>
                        `
                    }).then(() => {
                        imprimirComprovantePagamento(data.valor, descricao, fornecedor);
                    });
                }
            });
        }

        // Imprimir comprovante de pagamento
        function imprimirComprovantePagamento(valor, descricao, fornecedor) {
            const dataAtual = new Date().toLocaleString('pt-BR');
            const operador = '<?= htmlspecialchars($operador ?? 'Operador') ?>';
            const caixa = 'Caixa PDV';

            const comprovante = `
                <!DOCTYPE html>
                <html>
                <head>
                    <meta charset="UTF-8">
                    <title>Comprovante de Pagamento</title>
                    <style>
                        * { margin: 0; padding: 0; box-sizing: border-box; }
                        body {
                            font-family: 'Courier New', monospace;
                            font-size: 12px;
                            width: 80mm;
                            padding: 5mm;
                            background: #fff;
                        }
                        .center { text-align: center; }
                        .bold { font-weight: bold; }
                        .linha { border-top: 1px dashed #000; margin: 8px 0; }
                        .titulo { font-size: 16px; font-weight: bold; margin-bottom: 10px; }
                        .valor { font-size: 20px; font-weight: bold; margin: 15px 0; }
                        .info { margin: 5px 0; }
                        .assinatura { margin-top: 30px; border-top: 1px solid #000; padding-top: 5px; width: 80%; margin-left: auto; margin-right: auto; }
                        @media print {
                            body { width: 80mm; }
                            @page { margin: 0; size: 80mm auto; }
                        }
                    </style>
                </head>
                <body>
                    <div class="center">
                        <div class="titulo">COMPROVANTE DE PAGAMENTO</div>
                    </div>
                    <div class="linha"></div>
                    <div class="info"><strong>Data/Hora:</strong> ${dataAtual}</div>
                    <div class="info"><strong>Operador:</strong> ${operador}</div>
                    <div class="info"><strong>Caixa:</strong> ${caixa}</div>
                    <div class="linha"></div>
                    <div class="center valor">
                        R$ ${parseFloat(valor || 0).toFixed(2).replace('.', ',')}
                    </div>
                    <div class="linha"></div>
                    <div class="info"><strong>Descrição:</strong></div>
                    <div class="info">${descricao || 'Pagamento de despesa'}</div>
                    ${fornecedor ? `<div class="info"><strong>Fornecedor:</strong> ${fornecedor}</div>` : ''}
                    <div class="linha"></div>
                    <div class="assinatura center">
                        <div>Assinatura do Responsável</div>
                    </div>
                    <div style="margin-top: 20px;" class="center">
                        <small>Via do Caixa</small>
                    </div>
                </body>
                </html>
            `;

            const janela = window.open('', '_blank', 'width=400,height=600');
            janela.document.write(comprovante);
            janela.document.close();
            janela.focus();
            setTimeout(() => {
                janela.print();
            }, 300);
        }

        // Cancelar Venda Atual
        function cancelarVendaAtual() {
            if (carrinho.length === 0) {
                Swal.fire('Atenção', 'Não há itens no carrinho para cancelar.', 'warning');
                return;
            }

            Swal.fire({
                title: 'Cancelar Venda?',
                text: 'Todos os itens do carrinho serão removidos.',
                icon: 'warning',
                showCancelButton: true,
                confirmButtonText: 'Sim, Cancelar',
                confirmButtonColor: '#dc3545',
                cancelButtonText: 'Não'
            }).then((result) => {
                if (result.isConfirmed) {
                    carrinho = [];
                    descontoTotal = 0;
                    clienteSelecionado = null;
                    atualizarCarrinho();
                    Swal.fire('Venda Cancelada', 'O carrinho foi limpo.', 'success');
                }
            });
        }

        // Shipay - Menu de opções
        function abrirShipay() {
            Swal.fire({
                title: '<i class="bi bi-qr-code me-2"></i> Shipay',
                html: `
                    <div class="shipay-grid">
                        <button type="button" class="btn-shipay-opcao" onclick="Swal.close(); cancelarPixShipay();">
                            <i class="bi bi-x-circle text-warning"></i>
                            <span>Cancelar PIX</span>
                            <small>Cancelar cobrança pendente</small>
                        </button>
                        <button type="button" class="btn-shipay-opcao" onclick="Swal.close(); estornarPixShipay();">
                            <i class="bi bi-arrow-counterclockwise text-danger"></i>
                            <span>Estornar PIX</span>
                            <small>Devolver pagamento recebido</small>
                        </button>
                        <button type="button" class="btn-shipay-opcao" onclick="Swal.close(); consultarPixShipay();">
                            <i class="bi bi-search text-info"></i>
                            <span>Consultar PIX</span>
                            <small>Verificar status de cobrança</small>
                        </button>
                    </div>
                `,
                showConfirmButton: false,
                showCloseButton: true,
                customClass: {
                    popup: 'swal-shipay-popup'
                }
            });
        }

        // Cancelar PIX Shipay
        function cancelarPixShipay() {
            // Primeiro, buscar lista de cobranças pendentes que podem ser canceladas
            Swal.fire({
                title: '<i class="bi bi-x-circle text-warning me-2"></i> Cancelar PIX',
                html: `
                    <div class="text-center py-3">
                        <div class="spinner-border text-warning" role="status"></div>
                        <p class="mt-2 text-muted">Carregando cobranças pendentes...</p>
                    </div>
                `,
                showConfirmButton: false,
                showCloseButton: true,
                didOpen: () => {
                    // Buscar lista de cobranças pendentes
                    $.ajax({
                        url: buildUrl('/pdv/listar-pix-cancelaveis'),
                        type: 'GET',
                        dataType: 'json',
                        success: function (response) {
                            if (response.success && response.data && response.data.cobrancas) {
                                const cobrancas = response.data.cobrancas;

                                if (cobrancas.length === 0) {
                                    Swal.update({
                                        html: `
                                            <div class="text-center py-3">
                                                <i class="bi bi-inbox text-muted" style="font-size: 48px;"></i>
                                                <p class="mt-3 text-muted">Nenhuma cobrança PIX pendente encontrada nos últimos 7 dias.</p>
                                                <div class="mt-3">
                                                    <button type="button" class="btn btn-sm btn-outline-secondary" onclick="cancelarPixManual()">
                                                        <i class="bi bi-keyboard me-1"></i> Informar Order ID manualmente
                                                    </button>
                                                </div>
                                            </div>
                                        `
                                    });
                                    return;
                                }

                                let listaHtml = `
                                    <div class="text-start">
                                        <p class="mb-3 small text-muted">Selecione a cobrança PIX para cancelar:</p>
                                        <div class="lista-pix-estorno" style="max-height: 350px; overflow-y: auto;">
                                `;

                                cobrancas.forEach(cob => {
                                    const dataFormatada = new Date(cob.created_at).toLocaleString('pt-BR');
                                    const valorFormatado = parseFloat(cob.valor_parcela).toLocaleString('pt-BR', { style: 'currency', currency: 'BRL' });

                                    listaHtml += `
                                        <div class="item-pix-estorno item-pix-cancelar" data-order-id="${cob.shipay_order_id}" data-valor="${cob.valor_parcela}" data-venda="${cob.sale_number}">
                                            <div class="d-flex justify-content-between align-items-center">
                                                <div>
                                                    <strong class="text-primary">${cob.sale_number}</strong>
                                                    <br>
                                                    <small class="text-muted">${cob.cliente_nome || 'Cliente não identificado'}</small>
                                                    <br>
                                                    <small class="text-muted">${dataFormatada}</small>
                                                </div>
                                                <div class="text-end">
                                                    <span class="h5 text-warning mb-0">${valorFormatado}</span>
                                                    <br>
                                                    <span class="badge bg-warning text-dark">Pendente</span>
                                                </div>
                                            </div>
                                        </div>
                                    `;
                                });

                                listaHtml += `
                                        </div>
                                        <div class="alert alert-warning small mt-3">
                                            <i class="bi bi-exclamation-triangle me-1"></i>
                                            <strong>Atenção:</strong> A cobrança será cancelada e o QR Code não poderá mais ser pago.
                                        </div>
                                        <div class="text-center mt-2">
                                            <button type="button" class="btn btn-sm btn-outline-secondary" onclick="cancelarPixManual()">
                                                <i class="bi bi-keyboard me-1"></i> Informar Order ID manualmente
                                            </button>
                                        </div>
                                    </div>
                                `;

                                Swal.update({ html: listaHtml });

                                // Adicionar evento de clique nos itens
                                $('.item-pix-cancelar').on('click', function () {
                                    const orderId = $(this).data('order-id');
                                    const valor = $(this).data('valor');
                                    const venda = $(this).data('venda');

                                    confirmarCancelamentoPix(orderId, valor, venda);
                                });
                            } else {
                                Swal.update({
                                    html: `<div class="text-center text-danger py-3">
                                        <i class="bi bi-x-circle" style="font-size: 48px;"></i>
                                        <p class="mt-2">${response.message || 'Erro ao carregar cobranças'}</p>
                                    </div>`
                                });
                            }
                        },
                        error: function () {
                            Swal.update({
                                html: `<div class="text-center text-danger py-3">
                                    <i class="bi bi-x-circle" style="font-size: 48px;"></i>
                                    <p class="mt-2">Erro ao conectar com o servidor</p>
                                </div>`
                            });
                        }
                    });
                }
            });
        }

        // Cancelar PIX informando Order ID manualmente
        function cancelarPixManual() {
            Swal.fire({
                title: '<i class="bi bi-x-circle text-warning me-2"></i> Cancelar PIX',
                html: `
                    <div class="text-start">
                        <p class="mb-3">Informe o ID da cobrança PIX que deseja cancelar:</p>
                        <div class="mb-3">
                            <label class="form-label fw-bold">Order ID (Shipay) *</label>
                            <input type="text" id="swal-cancelar-order-id" class="form-control"
                                   placeholder="Ex: 99cf532c-48fe-4dd3-ac17-e0a741671997" autofocus>
                        </div>
                        <div class="alert alert-warning small">
                            <i class="bi bi-exclamation-triangle me-1"></i>
                            <strong>Atenção:</strong> Só é possível cancelar cobranças com status <strong>pendente</strong>.
                        </div>
                    </div>
                `,
                showCancelButton: true,
                confirmButtonText: '<i class="bi bi-x-circle me-1"></i> Cancelar PIX',
                confirmButtonColor: '#ffc107',
                cancelButtonText: 'Voltar',
                preConfirm: () => {
                    const orderId = $('#swal-cancelar-order-id').val().trim();
                    if (!orderId) {
                        Swal.showValidationMessage('Informe o Order ID');
                        return false;
                    }

                    Swal.showLoading();

                    return $.ajax({
                        url: buildUrl('/pdv/cancelar-pix-shipay'),
                        type: 'POST',
                        data: { order_id: orderId },
                        dataType: 'json'
                    }).then(response => {
                        if (!response.success) {
                            throw new Error(response.message || 'Erro ao cancelar');
                        }
                        return response;
                    }).catch(error => {
                        let msg = error.responseJSON?.message || error.message || 'Erro ao cancelar PIX';
                        Swal.showValidationMessage(msg);
                    });
                }
            }).then((result) => {
                if (result.isConfirmed && result.value) {
                    Swal.fire({
                        icon: 'success',
                        title: 'PIX Cancelado!',
                        text: 'A cobrança foi cancelada com sucesso.',
                        confirmButtonColor: '#28a745'
                    });
                }
            });
        }

        // Confirmar e executar cancelamento
        function confirmarCancelamentoPix(orderId, valor, venda) {
            Swal.fire({
                title: 'Confirmar Cancelamento?',
                html: `
                    <div class="text-center">
                        <p>Você está prestes a cancelar:</p>
                        <div class="alert alert-warning">
                            <strong>Venda: ${venda}</strong><br>
                            <span class="h4">R$ ${parseFloat(valor).toFixed(2).replace('.', ',')}</span>
                        </div>
                        <div class="mb-3 text-start">
                            <label class="form-label small fw-bold">Motivo do cancelamento (opcional):</label>
                            <textarea id="swal-cancelar-motivo" class="form-control form-control-sm" rows="2"
                                      placeholder="Ex: Cliente desistiu da compra"></textarea>
                        </div>
                    </div>
                `,
                icon: 'warning',
                showCancelButton: true,
                confirmButtonText: '<i class="bi bi-x-circle me-1"></i> Confirmar Cancelamento',
                confirmButtonColor: '#ffc107',
                cancelButtonText: 'Voltar',
                preConfirm: () => {
                    const motivo = $('#swal-cancelar-motivo').val();

                    Swal.showLoading();

                    return $.ajax({
                        url: buildUrl('/pdv/cancelar-pix-shipay'),
                        type: 'POST',
                        data: {
                            order_id: orderId,
                            motivo: motivo
                        },
                        dataType: 'json'
                    }).then(response => {
                        if (!response.success) {
                            throw new Error(response.message || 'Erro ao cancelar');
                        }
                        return response;
                    }).catch(error => {
                        let msg = error.responseJSON?.message || error.message || 'Erro ao cancelar PIX';
                        Swal.showValidationMessage(msg);
                    });
                }
            }).then((result) => {
                if (result.isConfirmed && result.value) {
                    Swal.fire({
                        icon: 'success',
                        title: 'PIX Cancelado!',
                        html: `<p>A cobrança de <strong>R$ ${parseFloat(valor).toFixed(2).replace('.', ',')}</strong> foi cancelada.</p>`,
                        confirmButtonColor: '#28a745'
                    });
                }
            });
        }

        // Estornar PIX Shipay
        function estornarPixShipay() {
            // Primeiro, buscar lista de PIX que podem ser estornados
            Swal.fire({
                title: '<i class="bi bi-arrow-counterclockwise text-danger me-2"></i> Estornar PIX',
                html: `
                    <div class="text-center py-3">
                        <div class="spinner-border text-primary" role="status"></div>
                        <p class="mt-2 text-muted">Carregando pagamentos PIX...</p>
                    </div>
                `,
                showConfirmButton: false,
                showCloseButton: true,
                didOpen: () => {
                    // Buscar lista de PIX
                    $.ajax({
                        url: buildUrl('/pdv/listar-pix-estornaveis'),
                        type: 'GET',
                        dataType: 'json',
                        success: function (response) {
                            if (response.success && response.data && response.data.pagamentos) {
                                const pagamentos = response.data.pagamentos;

                                if (pagamentos.length === 0) {
                                    Swal.update({
                                        html: `
                                            <div class="text-center py-3">
                                                <i class="bi bi-inbox text-muted" style="font-size: 48px;"></i>
                                                <p class="mt-3 text-muted">Nenhum pagamento PIX encontrado nos últimos 30 dias.</p>
                                            </div>
                                        `
                                    });
                                    return;
                                }

                                let listaHtml = `
                                    <div class="text-start">
                                        <p class="mb-3 small text-muted">Selecione o pagamento PIX para estornar:</p>
                                        <div class="lista-pix-estorno" style="max-height: 350px; overflow-y: auto;">
                                `;

                                pagamentos.forEach(pag => {
                                    const dataFormatada = new Date(pag.data_pagamento).toLocaleString('pt-BR');
                                    const valorFormatado = parseFloat(pag.valor_pago).toLocaleString('pt-BR', { style: 'currency', currency: 'BRL' });

                                    listaHtml += `
                                        <div class="item-pix-estorno" data-order-id="${pag.shipay_order_id}" data-valor="${pag.valor_pago}" data-venda="${pag.sale_number}">
                                            <div class="d-flex justify-content-between align-items-center">
                                                <div>
                                                    <strong class="text-primary">${pag.sale_number}</strong>
                                                    <br>
                                                    <small class="text-muted">${pag.cliente_nome || 'Cliente não identificado'}</small>
                                                    <br>
                                                    <small class="text-muted">${dataFormatada}</small>
                                                </div>
                                                <div class="text-end">
                                                    <span class="h5 text-success mb-0">${valorFormatado}</span>
                                                    <br>
                                                    <span class="badge bg-success">Pago</span>
                                                </div>
                                            </div>
                                        </div>
                                    `;
                                });

                                listaHtml += `
                                        </div>
                                        <div class="alert alert-danger small mt-3">
                                            <i class="bi bi-exclamation-octagon me-1"></i>
                                            <strong>Atenção:</strong> O valor será devolvido ao pagador.
                                        </div>
                                    </div>
                                `;

                                Swal.update({ html: listaHtml });

                                // Adicionar evento de clique nos itens
                                $('.item-pix-estorno').on('click', function () {
                                    const orderId = $(this).data('order-id');
                                    const valor = $(this).data('valor');
                                    const venda = $(this).data('venda');

                                    confirmarEstornoPix(orderId, valor, venda);
                                });
                            } else {
                                Swal.update({
                                    html: `<div class="text-center text-danger py-3">
                                        <i class="bi bi-x-circle" style="font-size: 48px;"></i>
                                        <p class="mt-2">${response.message || 'Erro ao carregar pagamentos'}</p>
                                    </div>`
                                });
                            }
                        },
                        error: function () {
                            Swal.update({
                                html: `<div class="text-center text-danger py-3">
                                    <i class="bi bi-x-circle" style="font-size: 48px;"></i>
                                    <p class="mt-2">Erro ao conectar com o servidor</p>
                                </div>`
                            });
                        }
                    });
                }
            });
        }

        // Confirmar e executar estorno
        function confirmarEstornoPix(orderId, valor, venda) {
            Swal.fire({
                title: 'Confirmar Estorno?',
                html: `
                    <div class="text-center">
                        <p>Você está prestes a estornar:</p>
                        <div class="alert alert-warning">
                            <strong>Venda: ${venda}</strong><br>
                            <span class="h4">R$ ${parseFloat(valor).toFixed(2).replace('.', ',')}</span>
                        </div>
                        <div class="mb-3 text-start">
                            <label class="form-label small fw-bold">Motivo do estorno (opcional):</label>
                            <textarea id="swal-estorno-motivo" class="form-control form-control-sm" rows="2"
                                      placeholder="Ex: Devolução solicitada pelo cliente"></textarea>
                        </div>
                    </div>
                `,
                icon: 'warning',
                showCancelButton: true,
                confirmButtonText: '<i class="bi bi-arrow-counterclockwise me-1"></i> Confirmar Estorno',
                confirmButtonColor: '#dc3545',
                cancelButtonText: 'Cancelar',
                preConfirm: () => {
                    const motivo = $('#swal-estorno-motivo').val();

                    Swal.showLoading();

                    return $.ajax({
                        url: buildUrl('/pdv/estornar-pix-shipay'),
                        type: 'POST',
                        data: {
                            order_id: orderId,
                            valor: valor,
                            motivo: motivo
                        },
                        dataType: 'json'
                    }).then(response => {
                        if (!response.success) {
                            throw new Error(response.message || 'Erro ao estornar');
                        }
                        return response;
                    }).catch(error => {
                        let msg = error.responseJSON?.message || error.message || 'Erro ao estornar PIX';
                        Swal.showValidationMessage(msg);
                    });
                }
            }).then((result) => {
                if (result.isConfirmed && result.value) {
                    Swal.fire({
                        icon: 'success',
                        title: 'Estorno Realizado!',
                        html: `<p>O valor de <strong>R$ ${parseFloat(valor).toFixed(2).replace('.', ',')}</strong> foi estornado.</p>`,
                        confirmButtonColor: '#28a745'
                    });
                }
            });
        }

        // Consultar PIX Shipay - Lista todas as cobranças
        function consultarPixShipay() {
            Swal.fire({
                title: '<i class="bi bi-qr-code text-info me-2"></i> Cobranças PIX',
                html: `
                    <div class="text-center py-3">
                        <div class="spinner-border text-info" role="status"></div>
                        <p class="mt-2 text-muted">Carregando cobranças...</p>
                    </div>
                `,
                showConfirmButton: false,
                showCloseButton: true,
                width: 650,
                didOpen: () => {
                    $.ajax({
                        url: buildUrl('/pdv/listar-cobrancas-shipay'),
                        type: 'GET',
                        dataType: 'json',
                        success: function (response) {
                            if (response.success && response.data && response.data.cobrancas) {
                                const cobrancas = response.data.cobrancas;

                                if (cobrancas.length === 0) {
                                    Swal.update({
                                        html: `
                                            <div class="text-center py-4">
                                                <i class="bi bi-inbox text-muted" style="font-size: 64px;"></i>
                                                <p class="mt-3 text-muted">Nenhuma cobrança PIX encontrada.</p>
                                                <div class="mt-3">
                                                    <button type="button" class="btn btn-sm btn-outline-info" onclick="consultarPixManual()">
                                                        <i class="bi bi-keyboard me-1"></i> Consultar por Order ID
                                                    </button>
                                                </div>
                                            </div>
                                        `
                                    });
                                    return;
                                }

                                let listaHtml = `
                                    <div class="text-start mb-3">
                                        <div class="input-group input-group-sm">
                                            <span class="input-group-text"><i class="bi bi-search"></i></span>
                                            <input type="text" class="form-control" id="filtroCobrancasPdv" placeholder="Filtrar por venda ou cliente...">
                                        </div>
                                    </div>
                                    <div class="small text-muted mb-2 d-flex justify-content-between align-items-center">
                                        <span><span class="badge bg-secondary">${cobrancas.length}</span> cobrança(s)</span>
                                        <button type="button" class="btn btn-sm btn-outline-info" onclick="consultarPixManual()">
                                            <i class="bi bi-keyboard me-1"></i> Consultar por ID
                                        </button>
                                    </div>
                                    <div class="lista-pix-estorno" style="max-height: 400px; overflow-y: auto;">
                                `;

                                cobrancas.forEach(cob => {
                                    const dataFormatada = cob.data_pagamento
                                        ? new Date(cob.data_pagamento).toLocaleString('pt-BR')
                                        : (cob.created_at ? new Date(cob.created_at).toLocaleString('pt-BR') : '-');
                                    const valorFormatado = parseFloat(cob.valor_pago || cob.valor_parcela || 0).toLocaleString('pt-BR', { style: 'currency', currency: 'BRL' });

                                    // Status com cores
                                    let statusBadge = 'bg-secondary';
                                    let statusText = cob.status || 'pendente';
                                    if (['pago', 'paid', 'approved'].includes(statusText.toLowerCase())) {
                                        statusBadge = 'bg-success';
                                        statusText = 'Pago';
                                    } else if (['pendente', 'pending'].includes(statusText.toLowerCase())) {
                                        statusBadge = 'bg-warning text-dark';
                                        statusText = 'Pendente';
                                    } else if (['cancelado', 'cancelled', 'expired', 'expirado'].includes(statusText.toLowerCase())) {
                                        statusBadge = 'bg-danger';
                                        statusText = 'Cancelado';
                                    } else if (['estornado', 'refunded'].includes(statusText.toLowerCase())) {
                                        statusBadge = 'bg-info';
                                        statusText = 'Estornado';
                                    }

                                    // Estorno info
                                    let estornoInfo = '';
                                    if (cob.refund_status && cob.refund_status !== 'none') {
                                        const refundBadge = cob.refund_status === 'approved' ? 'bg-info' : 'bg-warning text-dark';
                                        const refundText = cob.refund_status === 'approved' ? 'Estornado' : 'Estorno Pendente';
                                        estornoInfo = `<span class="badge ${refundBadge} ms-1">${refundText}</span>`;
                                    }

                                    listaHtml += `
                                        <div class="item-pix-estorno" onclick="verDetalheCobrancaPdv('${cob.shipay_order_id}')"
                                             data-busca="${(cob.sale_number || '').toLowerCase()} ${(cob.cliente_nome || '').toLowerCase()}">
                                            <div class="d-flex justify-content-between align-items-center">
                                                <div>
                                                    <strong class="text-primary">${cob.sale_number || '#' + cob.venda_id}</strong>
                                                    ${estornoInfo}
                                                    <br>
                                                    <small class="text-muted">
                                                        <i class="bi bi-person me-1"></i>${cob.cliente_nome || 'Cliente PDV'}
                                                    </small>
                                                    <br>
                                                    <small class="text-muted">
                                                        <i class="bi bi-clock me-1"></i>${dataFormatada}
                                                    </small>
                                                </div>
                                                <div class="text-end">
                                                    <span class="h6 mb-0">${valorFormatado}</span>
                                                    <br>
                                                    <span class="badge ${statusBadge}">${statusText}</span>
                                                </div>
                                            </div>
                                        </div>
                                    `;
                                });

                                listaHtml += '</div>';

                                Swal.update({ html: listaHtml });

                                // Filtro de busca
                                $('#filtroCobrancasPdv').on('input', function () {
                                    const termo = $(this).val().toLowerCase();
                                    $('.item-pix-estorno').each(function () {
                                        const busca = $(this).data('busca') || '';
                                        $(this).toggle(busca.includes(termo));
                                    });
                                });
                            } else {
                                Swal.update({
                                    html: `<div class="text-center text-danger py-3">
                                        <i class="bi bi-x-circle" style="font-size: 48px;"></i>
                                        <p class="mt-2">${response.message || 'Erro ao carregar cobranças'}</p>
                                    </div>`
                                });
                            }
                        },
                        error: function (xhr) {
                            const erro = xhr.responseJSON?.message || 'Erro ao conectar com o servidor';
                            Swal.update({
                                html: `<div class="text-center text-danger py-3">
                                    <i class="bi bi-x-circle" style="font-size: 48px;"></i>
                                    <p class="mt-2">${erro}</p>
                                </div>`
                            });
                        }
                    });
                }
            });
        }

        // Ver detalhes de uma cobrança específica
        function verDetalheCobrancaPdv(orderId) {
            Swal.close();

            Swal.fire({
                title: '<i class="bi bi-search text-info me-2"></i> Consultando...',
                html: '<div class="text-center py-3"><div class="spinner-border text-info"></div></div>',
                showConfirmButton: false,
                allowOutsideClick: false
            });

            $.ajax({
                url: buildUrl('/pdv/consultar-status-pix'),
                type: 'POST',
                data: { order_id: orderId },
                dataType: 'json'
            }).then(response => {
                if (!response.success) {
                    throw new Error(response.message || 'Erro ao consultar');
                }

                const data = response.data || {};
                const statusMap = {
                    'pendente': { icon: 'bi-clock', color: 'warning', text: 'Pendente' },
                    'pago': { icon: 'bi-check-circle', color: 'success', text: 'Pago' },
                    'aprovado': { icon: 'bi-check-circle', color: 'success', text: 'Aprovado' },
                    'cancelado': { icon: 'bi-x-circle', color: 'danger', text: 'Cancelado' },
                    'expirado': { icon: 'bi-clock-history', color: 'secondary', text: 'Expirado' },
                    'estornado': { icon: 'bi-arrow-counterclockwise', color: 'info', text: 'Estornado' }
                };
                const statusInfo = statusMap[data.status] || { icon: 'bi-question-circle', color: 'secondary', text: data.status };

                Swal.fire({
                    title: 'Status da Cobrança',
                    html: `
                        <div class="text-center mb-3">
                            <i class="bi ${statusInfo.icon} text-${statusInfo.color}" style="font-size: 48px;"></i>
                            <h4 class="text-${statusInfo.color} mt-2">${statusInfo.text}</h4>
                        </div>
                        <div class="text-start">
                            <table class="table table-sm">
                                <tr><td><strong>Order ID:</strong></td><td class="text-break small">${data.order_id || orderId}</td></tr>
                                <tr><td><strong>Status Shipay:</strong></td><td>${data.status_original || '-'}</td></tr>
                            </table>
                        </div>
                    `,
                    confirmButtonColor: '#17a2b8',
                    showCancelButton: true,
                    cancelButtonText: '<i class="bi bi-arrow-left me-1"></i> Voltar',
                    confirmButtonText: 'OK'
                }).then((result) => {
                    if (!result.isConfirmed) {
                        consultarPixShipay();
                    }
                });
            }).catch(error => {
                let msg = error.responseJSON?.message || error.message || 'Erro ao consultar PIX';
                Swal.fire({
                    icon: 'error',
                    title: 'Erro',
                    text: msg
                });
            });
        }

        // Consultar PIX manualmente por Order ID
        function consultarPixManual() {
            Swal.fire({
                title: '<i class="bi bi-search text-info me-2"></i> Consultar PIX',
                html: `
                    <div class="text-start">
                        <div class="mb-3">
                            <label class="form-label fw-bold">Order ID (Shipay) *</label>
                            <input type="text" id="swal-consultar-order-id" class="form-control"
                                   placeholder="Ex: 99cf532c-48fe-4dd3-ac17-e0a741671997" autofocus>
                        </div>
                    </div>
                `,
                showCancelButton: true,
                confirmButtonText: '<i class="bi bi-search me-1"></i> Consultar',
                confirmButtonColor: '#17a2b8',
                cancelButtonText: 'Voltar',
                preConfirm: () => {
                    const orderId = $('#swal-consultar-order-id').val().trim();
                    if (!orderId) {
                        Swal.showValidationMessage('Informe o Order ID');
                        return false;
                    }
                    return orderId;
                }
            }).then((result) => {
                if (result.isConfirmed && result.value) {
                    verDetalheCobrancaPdv(result.value);
                }
            });
        }

        // Salvar Orçamento
        function salvarOrcamento() {
            if (carrinho.length === 0) {
                Swal.fire('Atenção', 'Adicione itens ao carrinho antes de salvar o orçamento.', 'warning');
                return;
            }

            Swal.fire({
                title: '<i class="bi bi-file-earmark-text text-warning me-2"></i> Salvar Orçamento',
                html: `
                    <div class="text-start">
                        <div class="mb-3">
                            <label class="form-label fw-bold">Nome/Identificação</label>
                            <input type="text" id="swal-orcamento-nome" class="form-control"
                                   placeholder="Ex: Orçamento João Silva" autofocus>
                        </div>
                        <div class="alert alert-info small">
                            <i class="bi bi-info-circle me-1"></i> O orçamento poderá ser recuperado em "Abrir Venda".
                        </div>
                    </div>
                `,
                showCancelButton: true,
                confirmButtonText: '<i class="bi bi-save me-1"></i> Salvar',
                confirmButtonColor: '#ffc107',
                cancelButtonText: 'Cancelar'
            }).then((result) => {
                if (result.isConfirmed) {
                    // TODO: Implementar salvamento de orçamento
                    Swal.fire('Orçamento Salvo!', 'O orçamento foi salvo com sucesso.', 'success');
                    carrinho = [];
                    descontoTotal = 0;
                    atualizarCarrinho();
                }
            });
        }

        // Abrir Venda/Orçamento salvo
        function abrirVendaSalva() {
            Swal.fire({
                title: '<i class="bi bi-folder2-open text-info me-2"></i> Abrir Venda',
                html: `
                    <div class="text-center py-3">
                        <div class="spinner-border text-info" role="status"></div>
                        <p class="mt-2 text-muted">Carregando orçamentos e vendas pendentes...</p>
                    </div>
                `,
                showConfirmButton: false,
                showCloseButton: true,
                width: 650,
                didOpen: () => {
                    $.ajax({
                        url: buildUrl('/pdv/listar-vendas-pendentes'),
                        type: 'GET',
                        dataType: 'json',
                        success: function (response) {
                            if (response.success && response.data) {
                                const vendas = response.data.vendas || [];

                                if (vendas.length === 0) {
                                    Swal.update({
                                        html: `
                                            <div class="text-center py-4">
                                                <i class="bi bi-inbox text-muted" style="font-size: 64px;"></i>
                                                <p class="mt-3 text-muted">Nenhum orçamento ou venda pendente encontrado.</p>
                                            </div>
                                        `
                                    });
                                    return;
                                }

                                let listaHtml = `
                                    <div class="text-start mb-3">
                                        <div class="input-group input-group-sm">
                                            <span class="input-group-text"><i class="bi bi-search"></i></span>
                                            <input type="text" class="form-control" id="filtroVendasPendentes" placeholder="Filtrar por número ou cliente...">
                                        </div>
                                    </div>
                                    <div class="small text-muted mb-2">
                                        <span class="badge bg-secondary">${vendas.length}</span> registro(s) encontrado(s)
                                    </div>
                                    <div class="lista-vendas-pendentes" style="max-height: 400px; overflow-y: auto;">
                                `;

                                vendas.forEach(venda => {
                                    const dataFormatada = venda.sale_date
                                        ? new Date(venda.sale_date).toLocaleDateString('pt-BR')
                                        : '-';
                                    const valorFormatado = parseFloat(venda.total || 0).toLocaleString('pt-BR', { style: 'currency', currency: 'BRL' });

                                    // Status com cores
                                    let statusBadge = 'bg-secondary';
                                    let statusText = venda.status || 'pendente';
                                    if (statusText === 'orcamento') {
                                        statusBadge = 'bg-warning text-dark';
                                        statusText = 'Orçamento';
                                    } else if (statusText === 'pendente') {
                                        statusBadge = 'bg-info';
                                        statusText = 'Pendente';
                                    } else if (statusText === 'aberto') {
                                        statusBadge = 'bg-primary';
                                        statusText = 'Em Aberto';
                                    }

                                    listaHtml += `
                                        <div class="item-venda-pendente" onclick="carregarVendaPendente(${venda.id})"
                                             data-busca="${(venda.sale_number || '').toLowerCase()} ${(venda.cliente_nome || '').toLowerCase()}"
                                             style="padding: 12px 15px; border-bottom: 1px solid #eee; cursor: pointer; transition: all 0.2s ease;">
                                            <div class="d-flex justify-content-between align-items-center">
                                                <div>
                                                    <strong class="text-primary">${venda.sale_number || '#' + venda.id}</strong>
                                                    <span class="badge ${statusBadge} ms-2">${statusText}</span>
                                                    <br>
                                                    <small class="text-muted">
                                                        <i class="bi bi-person me-1"></i>${venda.cliente_nome || 'Cliente não informado'}
                                                    </small>
                                                    <br>
                                                    <small class="text-muted">
                                                        <i class="bi bi-calendar me-1"></i>${dataFormatada}
                                                        <span class="ms-2"><i class="bi bi-box me-1"></i>${venda.total_itens || 0} item(ns)</span>
                                                    </small>
                                                </div>
                                                <div class="text-end">
                                                    <span class="h5 mb-0">${valorFormatado}</span>
                                                </div>
                                            </div>
                                        </div>
                                    `;
                                });

                                listaHtml += '</div>';

                                Swal.update({ html: listaHtml });

                                // Hover effect
                                $('.item-venda-pendente').hover(
                                    function () { $(this).css('background-color', '#f0f8ff'); },
                                    function () { $(this).css('background-color', ''); }
                                );

                                // Filtro de busca
                                $('#filtroVendasPendentes').on('input', function () {
                                    const termo = $(this).val().toLowerCase();
                                    $('.item-venda-pendente').each(function () {
                                        const busca = $(this).data('busca') || '';
                                        $(this).toggle(busca.includes(termo));
                                    });
                                });
                            } else {
                                Swal.update({
                                    html: `<div class="text-center text-danger py-3">
                                        <i class="bi bi-x-circle" style="font-size: 48px;"></i>
                                        <p class="mt-2">${response.message || 'Erro ao carregar vendas'}</p>
                                    </div>`
                                });
                            }
                        },
                        error: function (xhr) {
                            const erro = xhr.responseJSON?.message || 'Erro ao conectar com o servidor';
                            Swal.update({
                                html: `<div class="text-center text-danger py-3">
                                    <i class="bi bi-x-circle" style="font-size: 48px;"></i>
                                    <p class="mt-2">${erro}</p>
                                </div>`
                            });
                        }
                    });
                }
            });
        }

        // Carregar venda pendente no carrinho
        function carregarVendaPendente(vendaId) {
            Swal.fire({
                title: 'Carregando...',
                html: '<div class="spinner-border text-info"></div>',
                showConfirmButton: false,
                allowOutsideClick: false
            });

            $.ajax({
                url: buildUrl('/pdv/carregar-venda'),
                type: 'GET',
                data: { id: vendaId },
                dataType: 'json',
                success: function (response) {
                    if (response.success && response.data) {
                        const venda = response.data.venda;
                        const itens = response.data.itens || [];

                        // Limpar carrinho atual
                        carrinho = [];
                        descontoTotal = 0;
                        clienteSelecionado = null;

                        // Carregar cliente se existir
                        if (venda.customer_id && venda.cliente_nome) {
                            clienteSelecionado = {
                                id: venda.customer_id,
                                nome: venda.cliente_nome
                            };
                        }

                        // Carregar itens
                        itens.forEach(item => {
                            carrinho.push({
                                id: item.product_id,
                                nome: item.product_name,
                                preco: parseFloat(item.unit_price),
                                quantidade: parseFloat(item.quantity),
                                total: parseFloat(item.total_price) + parseFloat(item.discount || 0)
                            });
                        });

                        // Carregar desconto
                        descontoTotal = parseFloat(venda.discount || 0);

                        // Salvar ID da venda para atualizar depois
                        window.vendaEmEdicao = vendaId;

                        // Atualizar interface
                        atualizarCarrinho();
                        atualizarClienteSelecionado();

                        Swal.fire({
                            icon: 'success',
                            title: 'Venda Carregada!',
                            text: `${venda.sale_number || 'Venda #' + vendaId} foi carregada no carrinho.`,
                            timer: 2000,
                            showConfirmButton: false
                        });
                    } else {
                        Swal.fire({
                            icon: 'error',
                            title: 'Erro',
                            text: response.message || 'Erro ao carregar venda'
                        });
                    }
                },
                error: function (xhr) {
                    Swal.fire({
                        icon: 'error',
                        title: 'Erro',
                        text: xhr.responseJSON?.message || 'Erro ao carregar venda'
                    });
                }
            });
        }

        // Atualizar exibição do cliente selecionado
        function atualizarClienteSelecionado() {
            if (clienteSelecionado) {
                $('#cliente-selecionado').html(`
                    <span class="badge bg-success me-2">
                        <i class="bi bi-person-check me-1"></i>${clienteSelecionado.nome}
                    </span>
                    <button type="button" class="btn btn-sm btn-outline-danger" onclick="removerCliente()">
                        <i class="bi bi-x"></i>
                    </button>
                `);
            } else {
                $('#cliente-selecionado').html('');
            }
        }

        // Remover cliente selecionado
        function removerCliente() {
            clienteSelecionado = null;
            atualizarClienteSelecionado();
        }

        // Ajustes do PDV
        function abrirAjustes() {
            // Carregar configurações salvas
            const configAjustes = JSON.parse(localStorage.getItem('pdv_ajustes') || '{}');
            const tabelaPrecoId = configAjustes.tabela_preco_id || '';
            const limiteSangria = configAjustes.limite_sangria || '';

            // Gerar opções de tabela de preço
            let opcoesTabela = '<option value="">Selecione uma tabela...</option>';
            <?php if (isset($tabelasPreco) && !empty($tabelasPreco)): ?>
                <?php foreach ($tabelasPreco as $tabela): ?>
                    opcoesTabela += '<option value="<?= $tabela['id'] ?>" ' +
                        (tabelaPrecoId == '<?= $tabela['id'] ?>' ? 'selected' : '') +
                        '><?= htmlspecialchars($tabela['nome']) ?></option>';
                <?php endforeach; ?>
            <?php endif; ?>

            Swal.fire({
                title: '<i class="bi bi-gear me-2"></i> Ajustes',
                html: `
                    <div class="text-start">
                        <div class="mb-3">
                            <label for="ajuste_tabela_preco" class="form-label">
                                <i class="bi bi-table me-1"></i> Tabela de Preço
                            </label>
                            <select class="form-select" id="ajuste_tabela_preco">
                                ${opcoesTabela}
                            </select>
                            <small class="text-muted">Tabela de preço que será usada para todos os produtos</small>
                        </div>
                        <div class="mb-3">
                            <label for="ajuste_limite_sangria" class="form-label">
                                <i class="bi bi-cash-coin me-1"></i> Limite Para Sangria (R$)
                            </label>
                            <input type="number" class="form-control" id="ajuste_limite_sangria"
                                step="0.01" min="0" placeholder="0.00" value="${limiteSangria || ''}">
                            <small class="text-muted">Valor máximo permitido para sangria</small>
                        </div>
                    </div>
                `,
                icon: null,
                showCancelButton: true,
                confirmButtonText: '<i class="bi bi-check-circle me-1"></i> Salvar',
                cancelButtonText: 'Cancelar',
                confirmButtonColor: '#28a745',
                width: 500,
                didOpen: () => {
                    // Focar no primeiro campo
                    document.getElementById('ajuste_tabela_preco').focus();
                },
                preConfirm: () => {
                    const tabelaPreco = document.getElementById('ajuste_tabela_preco').value;
                    const limiteSangria = parseFloat(document.getElementById('ajuste_limite_sangria').value) || 0;

                    return {
                        tabela_preco_id: tabelaPreco || null,
                        limite_sangria: limiteSangria
                    };
                }
            }).then((result) => {
                if (result.isConfirmed && result.value) {
                    // Salvar configurações no localStorage
                    localStorage.setItem('pdv_ajustes', JSON.stringify(result.value));

                    Swal.fire({
                        icon: 'success',
                        title: 'Configurações salvas!',
                        text: 'As alterações foram aplicadas com sucesso.',
                        timer: 2000,
                        showConfirmButton: false
                    });
                }
            });
        }

        // Função para obter tabela de preço configurada
        function getTabelaPrecoConfigurada() {
            const config = JSON.parse(localStorage.getItem('pdv_ajustes') || '{}');
            return config.tabela_preco_id || null;
        }

        // Função para obter limite de sangria configurado
        function getLimiteSangriaConfigurado() {
            const config = JSON.parse(localStorage.getItem('pdv_ajustes') || '{}');
            return parseFloat(config.limite_sangria) || 0;
        }

        // Fechar caixa
        function fecharCaixa() {
            const metodosPagamento = <?= json_encode($metodosPagamento ?? []) ?>;

            // Filtrar apenas métodos que usam PDV (se usar_pdv existir)
            const metodosPDV = metodosPagamento.filter(m => m.usar_pdv !== undefined ? m.usar_pdv == 1 : true);

            if (metodosPDV.length === 0) {
                Swal.fire({
                    icon: 'warning',
                    title: 'Atenção',
                    text: 'Nenhum método de pagamento configurado para PDV. Configure métodos de pagamento com "Usar no PDV" ativado.'
                });
                return;
            }

            // Criar HTML com duas colunas
            let htmlMetodos = '';
            metodosPDV.forEach(metodo => {
                htmlMetodos += `
                    <div class="row align-items-center mb-2 metodo-fechamento-item" style="padding: 8px; border-bottom: 1px solid #e9ecef;">
                        <div class="col-6">
                            <label class="form-label mb-0" style="font-weight: 600; color: #495057;">
                                <i class="bi bi-credit-card me-2"></i>${metodo.name || 'Método'}
                            </label>
                        </div>
                        <div class="col-6">
                            <div class="input-group">
                                <span class="input-group-text">R$</span>
                                <input type="text"
                                       class="form-control valor-metodo-fechamento"
                                       data-metodo-id="${metodo.id}"
                                       data-metodo-name="${metodo.name || ''}"
                                       placeholder="0,00"
                                       value="0,00"
                                       style="text-align: right;">
                            </div>
                        </div>
                    </div>
                `;
            });

            const html = `
                <div class="fechamento-caixa-container">
                    <div class="alert alert-info mb-3">
                        <i class="bi bi-info-circle me-2"></i>
                        <strong>Informe os valores recebidos por método de pagamento</strong>
                    </div>
                    <div class="fechamento-lista-metodos" style="max-height: 400px; overflow-y: auto; border: 1px solid #dee2e6; border-radius: 8px; padding: 10px;">
                        ${htmlMetodos}
                    </div>
                    <div class="mt-3 pt-3 border-top">
                        <div class="row">
                            <div class="col-6">
                                <strong>Total Informado:</strong>
                            </div>
                            <div class="col-6 text-end">
                                <strong id="total-fechamento" style="font-size: 1.2rem; color: #28a745;">R$ 0,00</strong>
                            </div>
                        </div>
                    </div>
                    <div class="mt-3">
                        <label class="form-label">
                            <i class="bi bi-chat-left-text me-2"></i> Observações (opcional)
                        </label>
                        <textarea class="form-control" id="observacoes-fechamento" rows="2" placeholder="Observações sobre o fechamento..."></textarea>
                    </div>
                </div>
            `;

            Swal.fire({
                title: '<i class="bi bi-lock me-2"></i> Fechar Caixa',
                html: html,
                width: '700px',
                showCancelButton: true,
                confirmButtonText: '<i class="bi bi-lock me-1"></i> Fechar Caixa',
                cancelButtonText: 'Cancelar',
                confirmButtonColor: '#dc3545',
                didOpen: () => {
                    // Aplicar máscara de moeda nos inputs (se jQuery Mask estiver disponível)
                    if (typeof $.fn.mask !== 'undefined') {
                        $('.valor-metodo-fechamento').mask('#.##0,00', {
                            reverse: true
                        });

                        // Calcular total quando valores mudarem (com jQuery Mask)
                        $('.valor-metodo-fechamento').on('input change', function() {
                            calcularTotalFechamento();
                        });
                    } else {
                        // Fallback: formatação manual de moeda
                        const formatarMoedaManual = function(e) {
                            const input = e.target;
                            let valor = input.value.replace(/\D/g, '');

                            if (valor === '') {
                                input.value = '0,00';
                            } else {
                                const numero = parseFloat(valor) / 100;
                                input.value = numero.toFixed(2).replace('.', ',').replace(/\B(?=(\d{3})+(?!\d))/g, '.');
                            }
                            calcularTotalFechamento();
                        };

                        $('.valor-metodo-fechamento').on('input', formatarMoedaManual);

                        // Formatar valores iniciais
                        $('.valor-metodo-fechamento').each(function() {
                            this.value = '0,00';
                        });
                    }

                    // Focar no primeiro input
                    setTimeout(() => {
                        const firstInput = $('.valor-metodo-fechamento').first();
                        firstInput.focus();
                        if (typeof $.fn.mask === 'undefined') {
                            firstInput.select();
                        }
                    }, 150);
                },
                preConfirm: () => {
                    // Coletar valores
                    const valoresMetodos = {};
                    let total = 0;
                    let temValor = false;

                    $('.valor-metodo-fechamento').each(function() {
                        const metodoId = $(this).data('metodo-id');
                        const valorStr = $(this).val().replace(/\./g, '').replace(',', '.');
                        const valor = parseFloat(valorStr) || 0;

                        if (valor > 0) {
                            temValor = true;
                        }

                        valoresMetodos[metodoId] = valor;
                        total += valor;
                    });

                    if (!temValor) {
                        Swal.showValidationMessage('Informe pelo menos um valor para fechamento');
                        return false;
                    }

                    const observacoes = $('#observacoes-fechamento').val().trim();

                    return {
                        valores_metodos: valoresMetodos,
                        observacoes: observacoes,
                        total: total
                    };
                }
            }).then((result) => {
                if (result.isConfirmed && result.value) {
                    // Mostrar loading
                    Swal.fire({
                        title: 'Fechando caixa...',
                        allowOutsideClick: false,
                        didOpen: () => {
                            Swal.showLoading();
                        }
                    });

                    // Enviar dados para o servidor
                    $.ajax({
                        url: buildUrl('/pdv/fechar-caixa'),
                        type: 'POST',
                        data: {
                            valores_metodos: JSON.stringify(result.value.valores_metodos),
                            observacoes: result.value.observacoes
                        },
                        dataType: 'json',
                        success: function(response) {
                            if (response.success) {
                                const sessaoId = response.data?.sessao_id || response.sessao_id;

                                if (sessaoId) {
                                    const printUrl = buildUrl('/pdv/imprimir-fechamento-caixa?id=' + sessaoId);

                                    const modalHtml = `
                                        <div style="width: 100%; height: 600px; overflow: auto;">
                                            <iframe id="printFrameFechamento" src="${printUrl}" style="width: 100%; height: 100%; border: none;"></iframe>
                                        </div>
                                    `;

                                    Swal.fire({
                                        html: modalHtml,
                                        width: '600px',
                                        showConfirmButton: false,
                                        showCloseButton: true,
                                        allowOutsideClick: false,
                                        didOpen: () => {
                                            const iframe = document.getElementById('printFrameFechamento');
                                            if (iframe) {
                                                iframe.onload = function() {
                                                    setTimeout(() => {
                                                        iframe.contentWindow.print();

                                                        // Listener para mensagem do iframe
                                                        const messageHandler = (event) => {
                                                            if (event.data === 'print-complete') {
                                                                window.removeEventListener('message', messageHandler);
                                                                Swal.close();
                                                                window.location.href = buildUrl(response.data.redirect || '/pdv/login');
                                                            }
                                                        };
                                                        window.addEventListener('message', messageHandler);

                                                        // Fallback para fechar o modal se a mensagem não for recebida
                                                        setTimeout(() => {
                                                            window.removeEventListener('message', messageHandler);
                                                            Swal.close();
                                                            window.location.href = buildUrl(response.data.redirect || '/pdv/login');
                                                        }, 5000); // 5 segundos de fallback
                                                    }, 500);
                                                };
                                            }
                                        }
                                    });
                                } else {
                                    // Se não tiver sessao_id, mostrar resumo e redirecionar
                                    Swal.fire({
                                        icon: 'success',
                                        title: 'Caixa Fechado!',
                                        html: (function() {
                                            const formatar = (v) => Number(v || 0).toFixed(2).replace('.', ',').replace(/\B(?=(\d{3})+(?!\d))/g, '.');
                                            return `
                                                <div class="text-start">
                                                    <p><strong>Resumo do Fechamento:</strong></p>
                                                    <ul class="text-start">
                                                        <li>Valor de Abertura: R$ ${formatar(response.data?.dados?.valor_abertura || 0)}</li>
                                                        <li>Total de Vendas: R$ ${formatar(response.data?.dados?.total_vendas || 0)}</li>
                                                        <li>Total Informado: R$ ${formatar(response.data?.dados?.valor_fechamento || 0)}</li>
                                                        <li>Diferença: <strong style="color: ${(response.data?.dados?.diferenca || 0) >= 0 ? '#28a745' : '#dc3545'}">R$ ${formatar(response.data?.dados?.diferenca || 0)}</strong></li>
                                                    </ul>
                                                </div>
                                            `;
                                        })(),
                                        confirmButtonText: 'OK',
                                        confirmButtonColor: '#28a745'
                                    }).then(() => {
                                        window.location.href = buildUrl(response.data?.redirect || '/pdv/login');
                                    });
                                }
                            } else {
                                Swal.fire({
                                    icon: 'error',
                                    title: 'Erro!',
                                    text: response.message
                                });
                            }
                        },
                        error: function(xhr, status, error) {
                            console.error('Erro ao fechar caixa:', {xhr, status, error});

                            let mensagemErro = 'Erro ao conectar com o servidor';
                            if (xhr.responseJSON && xhr.responseJSON.message) {
                                mensagemErro = xhr.responseJSON.message;
                            } else if (xhr.responseText) {
                                try {
                                    const response = JSON.parse(xhr.responseText);
                                    if (response.message) {
                                        mensagemErro = response.message;
                                    }
                                } catch (e) {
                                    mensagemErro = xhr.responseText.substring(0, 200);
                                }
                            }

                            Swal.fire({
                                icon: 'error',
                                title: 'Erro!',
                                text: mensagemErro
                            });
                        }
                    });
                }
            });
        }

        // Função para calcular total do fechamento
        function calcularTotalFechamento() {
            let total = 0;
            $('.valor-metodo-fechamento').each(function() {
                const valorStr = $(this).val().replace(/\./g, '').replace(',', '.');
                const valor = parseFloat(valorStr) || 0;
                total += valor;
            });

            // Formatar moeda
            const totalFormatado = 'R$ ' + Number(total || 0).toFixed(2).replace('.', ',').replace(/\B(?=(\d{3})+(?!\d))/g, '.');
            $('#total-fechamento').text(totalFormatado);
        }

        // Função auxiliar para formatar moeda (se não existir globalmente)
        if (typeof window.formatarMoeda === 'undefined') {
            window.formatarMoeda = function(valor) {
                return Number(valor || 0).toFixed(2).replace('.', ',').replace(/\B(?=(\d{3})+(?!\d))/g, '.');
            };
        }

        // Sair do PDV
        function sairPDV() {
            Swal.fire({
                title: 'Sair do PDV?',
                text: 'Você será desconectado do sistema',
                icon: 'question',
                showCancelButton: true,
                confirmButtonText: 'Sair',
                cancelButtonText: 'Cancelar'
            }).then((result) => {
                if (result.isConfirmed) {
                    // Redireciona diretamente para o logout (mais simples e robusto)
                    window.location.href = buildUrl('/pdv/logout');
                }
            });
        }

        // Atalhos de teclado
        $(document).on('keydown', function (e) {
            // F2 - Cliente
            if (e.key === 'F2') {
                e.preventDefault();
                buscarCliente();
            }
            // F3 - Desconto
            if (e.key === 'F3') {
                e.preventDefault();
                aplicarDesconto();
            }
            // F4 - Cancelar
            if (e.key === 'F4') {
                e.preventDefault();
                cancelarVenda();
            }
            // F9 - Finalizar
            if (e.key === 'F9') {
                e.preventDefault();
                finalizarVenda();
            }
        });

        // Busca ao digitar
        let timeoutBusca;
        $('#searchProduto').on('input', function () {
            clearTimeout(timeoutBusca);
            const busca = $(this).val();

            timeoutBusca = setTimeout(() => {
                carregarProdutos(busca);
            }, 300); // Aguarda 300ms após parar de digitar
        });

        // Busca ao pressionar Enter
        $('#searchProduto').on('keypress', function (e) {
            if (e.key === 'Enter') {
                e.preventDefault();
                const busca = $(this).val();
                carregarProdutos(busca);
            }
        });

        // Inicializar
        $(document).ready(function () {
            carregarGrupos();
            carregarProdutos();
            $('#searchProduto').focus();
        });
    </script>
</body>

</html>

</html>