<!DOCTYPE html>
<html lang="pt-BR">

<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Abertura de Caixa</title>

    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.0/font/bootstrap-icons.css">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/sweetalert2@11/dist/sweetalert2.min.css">

    <!-- Define appUrl e APP_BASE_PATH ANTES de qualquer script que possa usá-lo -->
    <script>
        (function () {
            // Detecta o diretório base dinamicamente baseado no caminho atual
            const currentPath = window.location.pathname;
            const pathParts = currentPath.split('/').filter(p => p);

            // Se estiver em um subdiretório (ex: /Systhema, /printjetdigital)
            let detectedBase = '';
            if (pathParts.length > 0 && pathParts[0] !== '') {
                detectedBase = '/' + pathParts[0];
            }

            const normalizedBase = detectedBase === '/' ? '' : detectedBase;
            window.APP_BASE_PATH = normalizedBase;

            window.appUrl = function (path = '/') {
                const safePath = typeof path === 'string' ? path : '';
                const normalizedPath = '/' + safePath.replace(/^\/+/, '');

                // Se não tem base path, retorna apenas o path normalizado
                if (!window.APP_BASE_PATH) {
                    return normalizedPath;
                }

                // Retorna base + path (ex: /Systhema/dashboard ou /printjetdigital/dashboard)
                return window.APP_BASE_PATH + normalizedPath;
            };

            window.url = window.appUrl;
        })();
    </script>

    <style>
        body {
            background: linear-gradient(135deg, #28a745 0%, #20c997 100%);
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
        }

        .abertura-container {
            width: 100%;
            max-width: 550px;
            padding: 20px;
        }

        .abertura-card {
            background: white;
            border-radius: 20px;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
            animation: slideIn 0.5s ease-out;
        }

        @keyframes slideIn {
            from {
                opacity: 0;
                transform: translateY(-50px);
            }

            to {
                opacity: 1;
                transform: translateY(0);
            }
        }

        .card-header {
            background: linear-gradient(135deg, #28a745 0%, #20c997 100%);
            color: white;
            padding: 40px 30px;
            text-align: center;
            border-radius: 20px 20px 0 0;
        }

        .card-header i {
            font-size: 72px;
            margin-bottom: 15px;
        }

        .card-body {
            padding: 40px 30px;
        }

        .form-label {
            font-weight: 600;
            color: #495057;
            margin-bottom: 10px;
        }

        .form-control {
            height: 60px;
            font-size: 18px;
            border-radius: 10px;
            border: 2px solid #e0e0e0;
        }

        .form-control:focus {
            border-color: #28a745;
            box-shadow: 0 0 0 0.2rem rgba(40, 167, 69, 0.25);
        }

        textarea.form-control {
            height: 100px;
            resize: none;
        }

        .input-group-text {
            border-radius: 10px 0 0 10px;
            border: 2px solid #e0e0e0;
            border-right: none;
            background: #f8f9fa;
            font-weight: 600;
        }

        .input-group .form-control {
            border-left: none;
            border-radius: 0 10px 10px 0;
        }

        .btn-abrir {
            width: 100%;
            height: 60px;
            font-size: 18px;
            font-weight: 600;
            border-radius: 10px;
            background: linear-gradient(135deg, #28a745 0%, #20c997 100%);
            border: none;
            color: white;
            margin-top: 20px;
        }

        .btn-abrir:hover {
            transform: translateY(-2px);
            box-shadow: 0 10px 20px rgba(40, 167, 69, 0.3);
        }

        .btn-voltar {
            width: 100%;
            height: 50px;
            border-radius: 10px;
            margin-top: 10px;
        }

        .info-box {
            background: #f8f9fa;
            border-left: 4px solid #28a745;
            padding: 15px;
            border-radius: 5px;
            margin-bottom: 20px;
        }
    </style>
</head>

<body>
    <div class="abertura-container">
        <div class="abertura-card">
            <div class="card-header">
                <i class="bi bi-cash-coin"></i>
                <h3 class="mb-2">Abertura de Caixa</h3>
                <p class="mb-0">Informe o valor inicial</p>
            </div>

            <div class="card-body">
                <div class="info-box">
                    <i class="bi bi-info-circle me-2"></i>
                    <strong>Operador:</strong> <?= htmlspecialchars($operador ?? 'N/A') ?><br>
                    <i class="bi bi-calendar3 me-2"></i>
                    <strong>Data:</strong> <?= date('d/m/Y H:i') ?>
                </div>

                <form id="formAbertura" method="POST" action="">
                    <div class="mb-3">
                        <label class="form-label">
                            <i class="bi bi-currency-dollar"></i> Valor de Abertura *
                        </label>
                        <div class="input-group">
                            <span class="input-group-text">R$</span>
                            <input type="text" class="form-control" id="valor_abertura" name="valor_abertura"
                                value="0,00" required autofocus>
                        </div>
                        <small class="text-muted">Valor em dinheiro no caixa</small>
                    </div>

                    <div class="mb-3">
                        <label class="form-label">
                            <i class="bi bi-chat-left-text"></i> Observações
                        </label>
                        <textarea class="form-control" name="observacoes"
                            placeholder="Observações sobre a abertura (opcional)"></textarea>
                    </div>

                    <button type="submit" class="btn btn-abrir">
                        <i class="bi bi-unlock me-2"></i>
                        Abrir Caixa
                    </button>

                    <button type="button" class="btn btn-outline-secondary btn-voltar" id="btnVoltar">
                        <i class="bi bi-arrow-left me-2"></i>
                        Voltar
                    </button>
                </form>
            </div>
        </div>
    </div>

    <script src="https://code.jquery.com/jquery-3.7.0.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
    <script src="https://cdn.jsdelivr.net/npm/jquery-mask-plugin@1.14.16/dist/jquery.mask.min.js"></script>

    <script>
        // Função helper para construir URLs com caminho base (GLOBAL E ROBUSTA)
        window.buildUrl = function (path) {
            // Garante que path começa com /
            path = '/' + path.replace(/^\/+/, '');

            // Detecta o caminho base da URL atual (SEMPRE funciona)
            const currentPath = window.location.pathname;
            const pathParts = currentPath.split('/').filter(p => p && p !== 'index.php');

            // Se estiver em /Systhema/pdv/abertura-caixa -> basePath = /Systhema
            let basePath = '';
            if (pathParts.length > 0) {
                // Pega o primeiro segmento como base path
                basePath = '/' + pathParts[0];
            }

            // Constrói a URL final
            const finalUrl = basePath + path;
            return finalUrl;
        };

        const buildUrl = window.buildUrl;

        // Limpar parâmetros da URL ao carregar a página (evitar loop de redirecionamento)
        if (window.location.search) {
            const urlWithoutParams = window.location.pathname;
            window.history.replaceState({}, document.title, urlWithoutParams);
        }

        // Botão voltar
        $('#btnVoltar').on('click', function () {
            window.location.href = buildUrl('/pdv/logout');
        });

        // Máscara de dinheiro
        $('#valor_abertura').mask('#.##0,00', {
            reverse: true
        });

        // Submit
        $('#formAbertura').on('submit', function (e) {
            e.preventDefault();
            e.stopPropagation();

            const $btn = $(this).find('.btn-abrir');
            const btnText = $btn.html();

            $btn.prop('disabled', true).html('<span class="spinner-border spinner-border-sm me-2"></span>Abrindo...');

            // Converter valor para formato numérico
            const valorInput = $('#valor_abertura').val();
            const valorNumerico = parseFloat(valorInput.replace(/\./g, '').replace(',', '.')) || 0;

            // Validar valor
            if (valorNumerico < 0) {
                Swal.fire({
                    icon: 'error',
                    title: 'Erro!',
                    text: 'O valor de abertura deve ser maior ou igual a zero'
                });
                $btn.prop('disabled', false).html(btnText);
                return false;
            }

            const formData = {
                valor_abertura: valorNumerico,
                observacoes: $('[name="observacoes"]').val() || ''
            };

            $.ajax({
                url: buildUrl('/pdv/abrir-caixa'),
                type: 'POST',
                data: formData,
                dataType: 'json',
                success: function (response) {
                    if (response.success) {
                        const sessaoId = response.data?.sessao_id;

                        // Abrir impressão de abertura de caixa em modal
                        if (sessaoId) {
                            const printUrl = buildUrl('/pdv/imprimir-abertura-caixa?id=' + sessaoId);

                            // Criar iframe para o conteúdo de impressão
                            const modalHtml = `
                                <div style="width: 100%; height: 600px; overflow: auto;">
                                    <iframe id="printFrame" src="${printUrl}" style="width: 100%; height: 100%; border: none;"></iframe>
                                </div>
                            `;

                            Swal.fire({
                                html: modalHtml,
                                width: '600px',
                                showConfirmButton: false,
                                showCloseButton: true,
                                allowOutsideClick: false,
                                didOpen: () => {
                                    // Listener para mensagem do iframe quando impressão for concluída
                                    const handlePrintComplete = function (event) {
                                        if (event.data === 'print-complete') {
                                            window.removeEventListener('message', handlePrintComplete);
                                            // Fechar modal após a impressão
                                            if (Swal.isVisible()) {
                                                Swal.close();
                                                // Redirecionar para o caixa
                                                redirectToCaixa(response);
                                            }
                                        }
                                    };
                                    window.addEventListener('message', handlePrintComplete);

                                    // Aguardar o iframe carregar
                                    const iframe = document.getElementById('printFrame');
                                    if (iframe) {
                                        iframe.onload = function () {
                                            // A impressão será disparada automaticamente pelo script dentro do iframe
                                            // O iframe enviará uma mensagem quando a impressão for concluída
                                        };
                                    }

                                    // Fallback: se a mensagem não chegar em 5 segundos, fechar mesmo assim
                                    setTimeout(() => {
                                        window.removeEventListener('message', handlePrintComplete);
                                        if (Swal.isVisible()) {
                                            Swal.close();
                                            redirectToCaixa(response);
                                        }
                                    }, 5000);
                                }
                            });
                            return; // Não continuar com o fluxo normal aqui
                        }

                        Swal.fire({
                            icon: 'success',
                            title: 'Caixa Aberto!',
                            text: 'Iniciando operação do caixa...',
                            timer: 1500,
                            showConfirmButton: false
                        }).then(() => {
                            redirectToCaixa(response);
                        });
                    } else {
                        Swal.fire({
                            icon: 'error',
                            title: 'Erro!',
                            text: response.message
                        });
                        $btn.prop('disabled', false).html(btnText);
                    }
                },
                error: function (xhr, status, error) {
                    console.error('[ABERTURA-CAIXA] Erro na requisição:', { xhr, status, error });
                    let errorMessage = 'Erro ao conectar com o servidor';

                    if (xhr.responseJSON && xhr.responseJSON.message) {
                        errorMessage = xhr.responseJSON.message;
                    } else if (xhr.status === 0) {
                        errorMessage = 'Erro de conexão. Verifique sua internet.';
                    } else if (xhr.status === 404) {
                        errorMessage = 'Rota não encontrada. Atualize a página.';
                    } else if (xhr.status >= 500) {
                        errorMessage = 'Erro no servidor. Tente novamente.';
                    }

                    Swal.fire({
                        icon: 'error',
                        title: 'Erro!',
                        text: errorMessage
                    });
                    $btn.prop('disabled', false).html(btnText);
                }
            });
        });

        // Função para redirecionar para o caixa
        function redirectToCaixa(response) {
            console.log("==========================================");
            console.log("[ABERTURA-CAIXA.PHP] Processando redirecionamento após abertura");

            const redirectPath = response.data?.redirect || '/pdv/caixa';
            console.log("[ABERTURA-CAIXA.PHP] Redirect path:", redirectPath);

            // SEMPRE construir URL completa com caminho base
            const currentUrl = window.location.href;
            const urlObj = new URL(currentUrl);
            const currentPath = urlObj.pathname;
            const pathParts = currentPath.split('/').filter(p => p && p !== 'index.php');

            console.log("[ABERTURA-CAIXA.PHP] URL atual:", currentUrl);
            console.log("[ABERTURA-CAIXA.PHP] Pathname:", currentPath);
            console.log("[ABERTURA-CAIXA.PHP] Path parts:", JSON.stringify(pathParts));

            // Pega o primeiro segmento como base path (ex: /sanctus ou /Systhema)
            let basePath = '';
            if (pathParts.length > 0) {
                basePath = '/' + pathParts[0];
            }

            console.log("[ABERTURA-CAIXA.PHP] Base path:", basePath);

            // Construir URL final garantindo o caminho base (sem query strings)
            const finalUrl = urlObj.origin + basePath + redirectPath;
            console.log("[ABERTURA-CAIXA.PHP] URL final:", finalUrl);
            console.log("[ABERTURA-CAIXA.PHP] 🔄 REDIRECIONANDO...");
            console.log("==========================================");

            // Forçar redirecionamento completo (substituir a página atual)
            window.location.replace(finalUrl);
        }
    </script>
</body>

</html>