<?php
// Garantir que a função url() esteja disponível
if (!function_exists('url')) {
    require_once __DIR__ . '/../../src/Helpers/GlobalHelpers.php';
}

// Disponibilizar modulosAtivos globalmente ANTES de definir a função
if (isset($modulosAtivos)) {
    $GLOBALS['__modulosAtivos'] = $modulosAtivos;
}

// Disponibilizar permissoesPerfil e user globalmente ANTES de definir a função
if (isset($permissoesPerfil)) {
    $GLOBALS['__permissoesPerfil'] = $permissoesPerfil;
}
if (isset($user)) {
    $GLOBALS['user'] = $user;
}

// Função helper para verificar se um módulo está ativo
function moduloAtivo($nomeInterface)
{
    // Pega a variável do escopo global
    $modulosAtivos = $GLOBALS['__modulosAtivos'] ?? null;

    // Por padrão, mostra se não houver controle de módulos ou se a variável não existir
    if ($modulosAtivos === null || !is_array($modulosAtivos)) {
        return true;
    }

    // Se o controle de módulos não está ativo, mostra tudo
    // IMPORTANTE: Se __controle for false, significa que não há controle de módulos ativo
    // então todos os módulos devem ser exibidos
    if (!isset($modulosAtivos['__controle']) || $modulosAtivos['__controle'] !== true) {
        return true;
    }

    // Se o controle está ativo, verifica se a lista existe
    // Se não existir ou não for array, não mostra nenhum módulo (controle ativo mas sem lista)
    if (!isset($modulosAtivos['lista']) || !is_array($modulosAtivos['lista'])) {
        return false;
    }

    // Verifica se o módulo está na lista de ativos (comparação estrita)
    // A lista contém apenas os nome_interface dos módulos que têm ativo = 'SIM', '1', 'TRUE', 'S' ou 'YES'
    $estaAtivo = in_array($nomeInterface, $modulosAtivos['lista'], true);

    return $estaAtivo;
}

// Função helper para verificar se o usuário tem permissão de visualização baseado no perfil de acesso
function temPermissaoVisualizar($module)
{
    $permissoesPerfil = $GLOBALS['__permissoesPerfil'] ?? null;

    // Tentar obter $user do escopo atual (extraído via extract) ou do global
    $user = null;
    if (isset($GLOBALS['user'])) {
        $user = $GLOBALS['user'];
    } elseif (function_exists('get_defined_vars')) {
        $vars = get_defined_vars();
        if (isset($vars['user'])) {
            $user = $vars['user'];
        }
    }

    // Se não há permissões configuradas, permite acesso
    if ($permissoesPerfil === null || !is_array($permissoesPerfil)) {
        return true;
    }

    // Se o usuário é admin (verificado no BaseController), permite tudo
    if (isset($permissoesPerfil['__is_admin']) && $permissoesPerfil['__is_admin'] === true) {
        return true;
    }

    // Se o usuário id = 1 e não tem permissões configuradas, permite acesso
    if ($user && isset($user['id']) && $user['id'] === 1 && empty($permissoesPerfil)) {
        return true;
    }

    // Verificar se há permissão específica para o módulo
    if (isset($permissoesPerfil[$module]) && is_array($permissoesPerfil[$module])) {
        return (bool) ($permissoesPerfil[$module]['can_view'] ?? false);
    }

    // Se não há registro de permissão para o módulo, nega acesso (exceto para id = 1)
    if ($user && isset($user['id']) && $user['id'] === 1) {
        return true;
    }

    return false;
}
?>
<!DOCTYPE html>
<html lang="pt-BR" data-theme="dark">

<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <meta http-equiv="X-UA-Compatible" content="ie=edge">
    <?php
    use App\Helpers\SkinHelper;
    $skinDisplayName = SkinHelper::getDisplayName();
    $skinFavicon = SkinHelper::getFavicon();
    $skinCssVariables = SkinHelper::getCssVariables();
    ?>
    <title><?= $pageTitle ?? 'Systhema ERP' ?> | <?= htmlspecialchars($skinDisplayName) ?></title>

    <!-- Favicon -->
    <link rel="icon" type="image/png" href="<?= $skinFavicon ?>">

    <!-- Bootstrap CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">

    <!-- Bootstrap Icons -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.0/font/bootstrap-icons.css" rel="stylesheet">

    <!-- Skin CSS Variables (carregar primeiro para ser base) -->
    <link href="<?= $skinCssVariables ?>" rel="stylesheet">

    <!-- Custom CSS (carregar depois para sobrescrever se necessário) -->
    <link href="<?= asset('/public/assets/css/app.css') ?>" rel="stylesheet">

    <!-- jQuery (carregado no HEAD para evitar erros) -->
    <script src="https://code.jquery.com/jquery-3.7.0.min.js"></script>

    <!-- SweetAlert2 -->
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>

    <!-- Define appUrl e APP_BASE_PATH no HEAD para estar disponível em todos os scripts -->
    <!-- IMPORTANTE: Cada sistema detecta seu próprio diretório automaticamente -->
    <!-- /printjetdigital, /Systhema, etc - cada um funciona isoladamente -->
    <script>
        (function () {
            // Detecta o diretório base dinamicamente baseado no caminho atual
            // Nunca redireciona entre sistemas diferentes
            const phpBasePath = <?= json_encode(rtrim(url('/'), '/')) ?>;
            let detectedBase = phpBasePath;

            // Fallback: detecta pelo caminho da página atual se PHP não retornou
            if (!detectedBase || detectedBase === '/') {
                const currentPath = window.location.pathname;
                // Remove o nome do arquivo se houver
                const pathParts = currentPath.split('/').filter(p => p);
                // Se estiver em um subdiretório (ex: /Systhema, /printjetdigital)
                if (pathParts.length > 0 && pathParts[0] !== '') {
                    detectedBase = '/' + pathParts[0];
                } else {
                    detectedBase = '';
                }
            }

            const normalizedBase = detectedBase === '/' ? '' : detectedBase;
            window.APP_BASE_PATH = normalizedBase;

            window.appUrl = function (path = '/') {
                const safePath = typeof path === 'string' ? path : '';
                const normalizedPath = '/' + safePath.replace(/^\/+/, '');

                // Se não tem base path, retorna apenas o path normalizado
                if (!window.APP_BASE_PATH) {
                    return normalizedPath;
                }

                // Retorna base + path (ex: /Systhema/dashboard ou /printjetdigital/dashboard)
                return window.APP_BASE_PATH + normalizedPath;
            };

            window.url = window.appUrl;
        })();
    </script>

    <!-- Interceptor Global AJAX - Detecta sessão expirada (401) -->
    <script>
        $(document).ajaxError(function (event, jqxhr, settings, thrownError) {
            if (jqxhr.status === 401) {
                Swal.fire({
                    icon: 'warning',
                    title: 'Sessão Expirada',
                    text: 'Sua sessão expirou. Você será redirecionado para o login.',
                    timer: 3000,
                    showConfirmButton: false,
                    allowOutsideClick: false
                }).then(() => {
                    window.location.href = appUrl('/login');
                });
            }
        });
    </script>

    <style>
        /* Variáveis CSS são definidas pelo arquivo CSS da skin ativa */
        /* As variáveis --primary-color, --dark-sidebar, etc. vêm do CSS da skin */

        body {
            background-color: var(--dark-bg);
            color: var(--text-primary);
            font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', sans-serif;
        }

        .sidebar {
            position: fixed;
            top: 0;
            left: 0;
            height: 100vh;
            width: 260px;
            background-color: var(--sidebar-bg, var(--dark-sidebar));
            border-right: 1px solid var(--border-color);
            overflow-y: auto;
            z-index: 1000;
            box-shadow: 2px 0 10px rgba(0, 0, 0, 0.05);
            transition: width 0.3s ease, transform 0.3s ease;
        }

        .sidebar.collapsed {
            width: 70px;
        }

        .sidebar.collapsed .sidebar-logo {
            font-size: 1.2rem;
            justify-content: center;
        }

        .sidebar.collapsed .sidebar-logo span {
            display: none;
        }

        .sidebar.collapsed .sidebar-item {
            justify-content: center;
            padding: 0.75rem;
            position: relative;
        }

        .sidebar.collapsed .sidebar-item span {
            display: none;
        }

        /* Esconder texto após ícone quando sidebar estiver recolhido */
        .sidebar.collapsed .sidebar-item i {
            margin-right: 0;
        }

        .sidebar.collapsed .sidebar-item::after {
            content: attr(title);
            position: absolute;
            left: 100%;
            margin-left: 10px;
            background: var(--dark-sidebar);
            color: #fff;
            padding: 0.5rem 0.75rem;
            border-radius: 4px;
            white-space: nowrap;
            opacity: 0;
            pointer-events: none;
            transition: opacity 0.3s;
            z-index: 1000;
            box-shadow: 0 2px 8px rgba(0, 0, 0, 0.2);
        }

        .sidebar.collapsed .sidebar-item:hover::after {
            opacity: 1;
        }

        .sidebar.collapsed .menu-group-header {
            justify-content: center;
            padding: 0.75rem;
        }

        .sidebar.collapsed .menu-group-header span {
            display: none;
        }

        .sidebar.collapsed .menu-group-header i.toggle-icon {
            display: none;
        }

        .sidebar.collapsed .menu-group-items {
            display: none;
        }

        .sidebar-header {
            padding: 1.5rem;
            border-bottom: 1px solid rgba(255, 255, 255, 0.15);
            background: rgba(0, 0, 0, 0.1);
            display: flex;
            justify-content: center;
            align-items: center;
        }

        .sidebar-logo {
            font-size: 1.5rem;
            font-weight: 700;
            color: var(--sidebar-text, #FFFFFF);
            text-decoration: none;
            display: flex;
            justify-content: center;
            align-items: center;
            width: 100%;
        }

        .sidebar-logo-img {
            max-width: 180px;
            max-height: 60px;
            width: auto;
            height: auto;
            object-fit: contain;
            display: block;
            margin: 0 auto;
        }

        .sidebar.collapsed .sidebar-logo-img {
            max-width: 40px;
            max-height: 40px;
        }

        .sidebar-menu {
            padding: 1rem 0;
        }

        .sidebar-item {
            padding: 0.75rem 1.5rem;
            color: rgba(255, 255, 255, 0.8);
            text-decoration: none;
            display: flex;
            align-items: center;
            transition: all 0.2s;
            font-size: 0.9rem;
        }

        .sidebar-item:hover {
            background-color: rgba(255, 255, 255, 0.1);
            color: #FFFFFF;
        }

        .sidebar-item.active {
            background-color: var(--primary-color);
            color: #fff;
            border-left: 3px solid var(--sidebar-active-border, var(--secondary-color));
        }

        /* Item PDV - usar cor do tema */
        .sidebar-item.pdv-item {
            background: linear-gradient(135deg, var(--primary-color) 0%, var(--primary-color-light) 100%);
            color: white;
            margin: 10px;
            border-radius: 8px;
            border-left: none;
        }

        .sidebar-item.pdv-item:hover {
            background: linear-gradient(135deg, var(--primary-color-light) 0%, var(--primary-color-lighter) 100%);
        }

        .sidebar-item i {
            margin-right: 0.75rem;
            font-size: 1.1rem;
        }

        .menu-group-header {
            padding: 0.75rem 1.5rem;
            color: rgba(255, 255, 255, 0.5);
            font-size: 0.75rem;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 1px;
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: space-between;
            transition: all 0.2s;
            margin-top: 1rem;
        }

        .menu-group-header:hover {
            color: rgba(255, 255, 255, 0.8);
        }

        .menu-group-header i.toggle-icon {
            font-size: 0.8rem;
            transition: transform 0.3s;
        }

        .menu-group-header.collapsed i.toggle-icon {
            transform: rotate(-90deg);
        }

        .menu-group-items {
            max-height: 1000px;
            overflow: hidden;
            transition: max-height 0.3s ease-in-out, opacity 0.3s;
            opacity: 1;
        }

        .menu-group-items.collapsed {
            max-height: 0;
            opacity: 0;
        }

        .main-content {
            margin-left: 260px;
            padding: 2rem;
            min-height: 100vh;
            background-color: var(--dark-bg);
            transition: margin-left 0.3s ease;
        }

        .main-content.sidebar-collapsed {
            margin-left: 70px;
        }

        .topbar {
            background-color: #FFFFFF;
            border-bottom: 1px solid var(--border-color);
            padding: 0.75rem 1.5rem;
            display: flex;
            justify-content: space-between;
            align-items: center;
            box-shadow: 0 1px 2px rgba(0, 0, 0, 0.02);
        }

        .content-wrapper {
            padding: 1.5rem;
            background-color: var(--dark-bg);
        }

        /* Botão hambúrguer para mobile */
        .sidebar-toggle {
            display: none;
            background: none;
            border: none;
            color: var(--text-primary);
            font-size: 1.5rem;
            cursor: pointer;
            padding: 0.5rem;
        }

        /* Botão para recolher sidebar no desktop */
        .sidebar-collapse-btn {
            display: block;
            background: none;
            border: none;
            color: var(--text-primary);
            font-size: 1.25rem;
            cursor: pointer;
            padding: 0.5rem;
            transition: transform 0.3s ease;
        }

        .sidebar-collapse-btn:hover {
            color: var(--primary-color);
        }

        .sidebar-collapse-btn.collapsed {
            transform: rotate(180deg);
        }

        @media (max-width: 991.98px) {
            .sidebar-collapse-btn {
                display: none;
            }
        }

        /* Responsividade Mobile */
        @media (max-width: 991.98px) {
            .sidebar {
                transform: translateX(-100%);
                transition: transform 0.3s ease;
                z-index: 1050;
            }

            .sidebar.show {
                transform: translateX(0);
            }

            .sidebar-overlay {
                display: none;
                position: fixed;
                top: 0;
                left: 0;
                right: 0;
                bottom: 0;
                background-color: rgba(0, 0, 0, 0.5);
                z-index: 1040;
            }

            .sidebar-overlay.show {
                display: block;
            }

            .main-content {
                margin-left: 0;
            }

            .sidebar-toggle {
                display: block;
            }

            .topbar {
                padding: 1rem;
                flex-wrap: wrap;
            }

            .content-wrapper {
                padding: 1rem;
            }

            .card {
                margin-bottom: 1rem;
            }
        }

        @media (max-width: 575.98px) {
            .topbar {
                padding: 0.75rem;
            }

            .content-wrapper {
                padding: 0.75rem;
            }

            .sidebar-header {
                padding: 1rem;
            }

            .sidebar-item {
                padding: 0.625rem 1rem;
                font-size: 0.85rem;
            }

            .btn {
                font-size: 0.875rem;
                padding: 0.375rem 0.75rem;
            }
        }

        .card {
            background-color: var(--dark-card);
            border: 1px solid var(--border-color);
            border-radius: 6px;
            margin-bottom: 1.5rem;
            box-shadow: 0 1px 3px rgba(0, 0, 0, 0.05);
        }

        .card-header {
            background-color: transparent;
            border-bottom: 1px solid var(--border-color);
            padding: 0.75rem 1rem;
            font-weight: 500;
            color: var(--text-primary);
            font-size: 0.85rem;
        }

        .btn-primary {
            background-color: var(--primary-color);
            border-color: var(--primary-color);
        }

        .btn-primary:hover {
            background-color: var(--primary-color-light);
            border-color: var(--primary-color-light);
        }

        /* Badges mais discretos */
        .badge {
            font-weight: 500;
            padding: 0.35em 0.65em;
            font-size: 0.75em;
        }

        /* Botões outline mais suaves */
        .btn-outline-primary {
            border-color: var(--primary-color);
            color: var(--primary-color);
        }

        .btn-outline-primary:hover {
            background-color: var(--primary-color);
            border-color: var(--primary-color);
            color: #fff;
        }

        .table {
            color: var(--text-primary);
        }

        .table thead th {
            border-color: var(--border-color);
            background-color: transparent;
            font-weight: 500;
            font-size: 0.8rem;
            text-transform: uppercase;
            letter-spacing: 0.3px;
            color: #6c757d;
            padding: 0.75rem;
        }

        .table td {
            border-color: var(--border-color);
        }

        .form-control,
        .form-select {
            background-color: #FFFFFF;
            border-color: var(--border-color);
            color: var(--text-primary);
        }

        .form-control:focus,
        .form-select:focus {
            background-color: #FFFFFF;
            border-color: var(--primary-color);
            color: var(--text-primary);
            box-shadow: 0 0 0 0.2rem rgba(0, 18, 35, 0.08);
        }

        .form-label {
            color: var(--text-primary);
            font-weight: 500;
            font-size: 0.9rem;
        }

        /* Estilos Clean - Interface Minimalista */
        h1,
        h2,
        h3,
        h4,
        h5,
        h6 {
            font-weight: 600;
            margin-bottom: 0.75rem;
        }

        h2 {
            font-size: 1.5rem;
            color: var(--text-primary);
        }

        /* Reduzir ícones grandes em títulos */
        h2 i,
        h1 i {
            font-size: 1.25rem;
            margin-right: 0.5rem;
        }

        /* Subtítulos mais discretos */
        .text-muted {
            font-size: 0.875rem;
            color: #6c757d !important;
        }

        /* Reduzir espaçamento de rows */
        .row.mb-4 {
            margin-bottom: 1rem !important;
        }

        /* Cards mais limpos */
        .card-body {
            padding: 1rem;
        }

        .card-body.p-0 {
            padding: 0;
        }

        /* Botões mais discretos */
        .btn {
            font-size: 0.875rem;
            padding: 0.5rem 1rem;
            font-weight: 500;
        }

        .btn-lg {
            font-size: 0.9375rem;
            padding: 0.625rem 1.25rem;
        }

        /* Badges menores e mais discretos */
        .badge {
            font-size: 0.7rem;
            padding: 0.25em 0.5em;
            font-weight: 500;
        }

        /* Tabelas mais limpas */
        .table {
            font-size: 0.875rem;
        }

        .table td {
            padding: 0.75rem;
            vertical-align: middle;
        }

        /* Inputs mais limpos */
        .form-control,
        .form-select {
            font-size: 0.875rem;
            padding: 0.5rem 0.75rem;
        }

        /* Remover sombras excessivas */
        .card {
            box-shadow: 0 1px 2px rgba(0, 0, 0, 0.03) !important;
        }

        /* Espaçamento reduzido entre elementos */
        .mb-3 {
            margin-bottom: 0.75rem !important;
        }

        .mb-4 {
            margin-bottom: 1rem !important;
        }

        .mt-4 {
            margin-top: 1rem !important;
        }

        /* Tabs mais discretas */
        .nav-tabs {
            border-bottom: 1px solid var(--border-color);
        }

        .nav-tabs .nav-link {
            font-size: 0.875rem;
            padding: 0.5rem 1rem;
            color: #6c757d;
        }

        .nav-tabs .nav-link.active {
            color: var(--primary-color);
            font-weight: 500;
        }

        /* Botões de Acesso Rápido - Design Minimalista */
        .quick-access-buttons {
            flex: 1;
            justify-content: center;
            padding: 0 1rem;
            gap: 0.5rem;
        }

        .quick-btn-modern {
            display: inline-flex;
            align-items: center;
            justify-content: center;
            padding: 0.5rem;
            border-radius: 6px;
            background: transparent;
            color: #6c757d;
            text-decoration: none;
            transition: all 0.2s ease;
            border: none;
            font-size: 1rem;
            font-weight: 400;
            position: relative;
            overflow: hidden;
            box-shadow: none;
            min-width: 40px;
            height: 40px;
        }

        .quick-btn-modern:hover {
            background: #f8f9fa;
            color: var(--primary-color);
        }

        .quick-btn-modern::before {
            content: '';
            position: absolute;
            top: 50%;
            left: 50%;
            width: 0;
            height: 0;
            border-radius: 50%;
            background: radial-gradient(circle, var(--primary-color) 0%, transparent 70%);
            transform: translate(-50%, -50%);
            transition: width 0.5s cubic-bezier(0.34, 1.56, 0.64, 1), height 0.5s cubic-bezier(0.34, 1.56, 0.64, 1);
            opacity: 0.15;
        }

        .quick-btn-modern:hover::before {
            width: 120%;
            height: 120%;
        }

        .quick-btn-modern::after {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            border-radius: 10px;
            background: linear-gradient(135deg, rgba(255, 255, 255, 0.1) 0%, transparent 50%);
            opacity: 0;
            transition: opacity 0.3s ease;
        }

        .quick-btn-modern:hover::after {
            opacity: 1;
        }

        .quick-btn-modern:hover {
            background: linear-gradient(135deg, var(--primary-color) 0%, var(--primary-color-light) 100%);
            color: #ffffff;
            border-color: var(--primary-color);
            transform: translateY(-3px) scale(1.05);
            box-shadow: 0 8px 20px rgba(0, 17, 34, 0.4), 0 0 0 2px rgba(0, 17, 34, 0.1);
        }

        .quick-btn-modern:active {
            transform: translateY(-1px) scale(1.02);
            box-shadow: 0 4px 12px rgba(0, 17, 34, 0.3);
        }

        /* Efeito de brilho no hover */
        .quick-btn-modern:hover .quick-btn-icon {
            filter: drop-shadow(0 0 8px rgba(255, 255, 255, 0.5));
        }

        .quick-btn-icon {
            display: flex;
            align-items: center;
            justify-content: center;
            width: 24px;
            height: 24px;
            font-size: 1.25rem;
            transition: transform 0.3s cubic-bezier(0.4, 0, 0.2, 1);
            position: relative;
            z-index: 1;
        }

        .quick-btn-modern:hover .quick-btn-icon {
            transform: scale(1.2) rotate(5deg);
        }

        .quick-btn-label {
            display: none;
        }

        /* Responsividade para botões rápidos */
        @media (max-width: 1199.98px) {
            .quick-access-buttons {
                display: none !important;
            }
        }

        @media (max-width: 1199.98px) {
            .quick-access-buttons {
                display: none !important;
            }
        }

        @media (max-width: 991.98px) {
            .topbar {
                padding: 0.75rem 1rem;
            }
        }

        /* Alerta de Contas Vencidas */
        .alert-contas-vencidas {
            background: linear-gradient(135deg, #fff3cd 0%, #ffeaa7 100%);
            border: 1px solid #ffc107;
            border-radius: 12px;
            padding: 0.75rem 1rem;
            margin-bottom: 1rem;
            box-shadow: 0 2px 8px rgba(255, 193, 7, 0.2);
            animation: slideDown 0.3s ease-out;
        }

        @keyframes slideDown {
            from {
                opacity: 0;
                transform: translateY(-10px);
            }

            to {
                opacity: 1;
                transform: translateY(0);
            }
        }

        .alert-contas-vencidas .alert-content {
            display: flex;
            align-items: center;
            gap: 0.75rem;
            color: #856404;
            font-weight: 500;
        }

        .alert-contas-vencidas .alert-icon {
            font-size: 1.25rem;
            color: #ff6b35;
        }

        .alert-contas-vencidas .alert-text {
            flex: 1;
        }

        .alert-contas-vencidas .alert-total {
            font-weight: 700;
            font-size: 1.1rem;
            color: #856404;
        }

        .alert-contas-vencidas a {
            color: #856404;
            text-decoration: underline;
            font-weight: 600;
        }

        .alert-contas-vencidas a:hover {
            color: #533f03;
        }
    </style>
</head>

<body>
    <!-- Overlay para fechar sidebar no mobile -->
    <div class="sidebar-overlay" id="sidebarOverlay"></div>

    <!-- Sidebar -->
    <aside class="sidebar" id="sidebar">
        <div class="sidebar-header">
            <a href="<?= url('/dashboard') ?>" class="sidebar-logo d-flex align-items-center justify-content-center">
                <?php
                // Prioridade: Logo da empresa > Logo da skin > Fallback
                $logoUrl = null;

                // 1. Tentar usar logo da empresa se existir
                if (!empty($companyLogo) && trim($companyLogo) !== '') {
                    $logoValue = trim($companyLogo);
                    $logoPath = null;

                    // Normalizar o caminho: garantir que comece com /storage/
                    if (strpos($logoValue, '/storage/') === 0) {
                        $logoPath = $logoValue;
                    } elseif (strpos($logoValue, 'storage/') === 0) {
                        $logoPath = '/' . $logoValue;
                    } elseif (strpos($logoValue, '/') === false && strpos($logoValue, '\\') === false) {
                        $logoPath = '/storage/uploads/logos/' . $logoValue;
                    } else {
                        $logoPath = $logoValue;
                    }

                    if ($logoPath) {
                        $logoUrl = asset($logoPath);
                    }
                }

                // 2. Se não tiver logo da empresa, usar logo da skin
                if (!$logoUrl) {
                    $logoUrl = SkinHelper::getLogo();
                }

                // 3. Fallback final (se a skin não tiver logo)
                if (!$logoUrl) {
                    $logoUrl = asset('/storage/uploads/logos/tupan.png');
                }
                ?>
                <img src="<?php echo htmlspecialchars($logoUrl); ?>" alt="Logo" class="sidebar-logo-img"
                    onerror="this.onerror=null; this.style.display='none'; this.nextElementSibling.style.display='flex';">
                <span style="display: none;"><i class="bi bi-graph-up-arrow"></i>
                    <span><?= htmlspecialchars($skinDisplayName) ?></span></span>
            </a>
        </div>

        <nav class="sidebar-menu">
            <?php if (temPermissaoVisualizar('dashboard')): ?>
                <a href="<?= url('/dashboard') ?>"
                    class="sidebar-item <?= isset($activeMenu) && $activeMenu === 'dashboard' ? 'active' : '' ?>"
                    title="Dashboard">
                    <i class="bi bi-speedometer2"></i> <span>Dashboard</span>
                </a>
            <?php endif; ?>

            <a href="<?= url('/tickets/abrir') ?>"
                class="sidebar-item <?= isset($activeMenu) && $activeMenu === 'abrir-ticket' ? 'active' : '' ?>"
                title="Abrir Ticket">
                <i class="bi bi-plus-circle"></i> <span>Abrir Ticket</span>
            </a>

            <a href="<?= url('/tickets/acompanhar') ?>"
                class="sidebar-item <?= isset($activeMenu) && $activeMenu === 'acompanhar-tickets' ? 'active' : '' ?>"
                title="Acompanhar Tickets">
                <i class="bi bi-eye"></i> <span>Acompanhar Tickets</span>
            </a>

            <!-- PDV - Frente de Loja (destaque) -->
            <?php if (function_exists('moduloFrenteLojaAtivo') && moduloFrenteLojaAtivo()): ?>
                <a href="<?= url('/pdv/login') ?>" class="sidebar-item pdv-item" target="_blank">
                    <i class="bi bi-shop"></i> Frente de Loja (PDV)
                </a>
            <?php endif; ?>

            <!-- Grupo: Registros -->
            <div class="menu-group-header collapsed" onclick="toggleMenuGroup(this)">
                <span>Registros</span>
                <i class="bi bi-chevron-down toggle-icon"></i>
            </div>
            <div class="menu-group-items collapsed">
                <?php if (moduloAtivo('modulo_pessoas') && temPermissaoVisualizar('pessoas')): ?>
                    <a href="<?= url('/pessoas') ?>"
                        class="sidebar-item <?= isset($activeMenu) && $activeMenu === 'pessoas' ? 'active' : '' ?>"
                        title="Pessoas">
                        <i class="bi bi-people"></i> <span>Pessoas</span>
                    </a>
                <?php endif; ?>
                <?php if (moduloAtivo('modulo_grupos') && temPermissaoVisualizar('grupos-pessoas')): ?>
                    <a href="<?= url('/grupos-pessoas') ?>"
                        class="sidebar-item <?= isset($activeMenu) && $activeMenu === 'grupos' ? 'active' : '' ?>"
                        title="Grupos">
                        <i class="bi bi-folder"></i> <span>Grupos</span>
                    </a>
                <?php endif; ?>
                <?php if (moduloAtivo('modulo_conta_bancaria') && temPermissaoVisualizar('contas')): ?>
                    <a href="<?= url('/contas') ?>"
                        class="sidebar-item <?= isset($activeMenu) && $activeMenu === 'Contas Bancárias' ? 'active' : '' ?>">
                        <i class="bi bi-folder"></i> Contas Bancárias
                    </a>
                <?php endif; ?>
                <?php if (moduloAtivo('modulo_metodos_pagamento') && temPermissaoVisualizar('metodos-pagamento')): ?>
                    <a href="<?= url('/metodos-pagamento') ?>" class="sidebar-item" <?= isset($activeMenu) && $activeMenu === 'Métodos de Pagamento' ? 'active' : '' ?>">
                        <i class="bi bi-credit-card"></i> Métodos de Pagamento
                    </a>
                <?php endif; ?>
                <?php if (moduloAtivo('modulo_itens') && temPermissaoVisualizar('itens')): ?>
                    <a href="<?= url('/itens') ?>" class="sidebar-item" <?= isset($activeMenu) && $activeMenu === 'Itens' ? 'active' : '' ?>">
                        <i class="bi bi-box-seam"></i> Itens
                    </a>
                <?php endif; ?>
                <?php if (moduloAtivo('modulo_plano_contas') && temPermissaoVisualizar('plano-contas')): ?>
                    <a href="<?= url('/plano-contas') ?>" class="sidebar-item" <?= isset($activeMenu) && $activeMenu === 'Plano de Contas' ? 'active' : '' ?>">
                        <i class="bi bi-diagram-3"></i> Plano de Contas
                    </a>
                <?php endif; ?>
                <?php if (moduloAtivo('modulo_centro_custos') && temPermissaoVisualizar('centro-custos')): ?>
                    <a href="<?= url('/centro-custos') ?>" class="sidebar-item" <?= isset($activeMenu) && $activeMenu === 'Centro de Custos' ? 'active' : '' ?>">
                        <i class="bi bi-building"></i> Centro de Custos
                    </a>
                <?php endif; ?>
                <?php if (moduloAtivo('modulo_locais_estoque') && temPermissaoVisualizar('gestor-estoque')): ?>
                    <a href="<?= url('/gestor-estoque') ?>" class="sidebar-item" <?= isset($activeMenu) && $activeMenu === 'Locais de Estoque' ? 'active' : '' ?>">
                        <i class="bi bi-building"></i> Locais de Estoque
                    </a>
                <?php endif; ?>
                <?php //if (moduloAtivo('modulo_impostos') && temPermissaoVisualizar('impostos')): ?>
                <a href="<?= url('/impostos') ?>" class="sidebar-item" <?= isset($activeMenu) && $activeMenu === 'Impostos' ? 'active' : '' ?>">
                    <i class="bi bi-percent"></i> Gestão de Impostos
                </a>
                <?php //endif; ?>
                <?php if (moduloAtivo('modulo_areas_comerciais') && temPermissaoVisualizar('areas')): ?>
                    <a href="<?= url('/areas') ?>"
                        class="sidebar-item <?= isset($activeMenu) && $activeMenu === 'areas' ? 'active' : '' ?>">
                        <i class="bi bi-map"></i> Áreas Comerciais
                    </a>
                <?php endif; ?>
            </div>

            <!-- Grupo: Operacional -->
            <div class="menu-group-header collapsed" onclick="toggleMenuGroup(this)">
                <span>Operacional</span>
                <i class="bi bi-chevron-down toggle-icon"></i>
            </div>
            <div class="menu-group-items collapsed">
                <?php if (moduloAtivo('modulo_pedido') && temPermissaoVisualizar('pedidos')): ?>
                    <a href="<?= url('/pedidos') ?>"
                        class="sidebar-item <?= isset($activeMenu) && $activeMenu === 'pedidos' ? 'active' : '' ?>">
                        <i class="bi bi-clipboard-check"></i> Pedidos
                    </a>
                <?php endif; ?>
                <?php if (moduloAtivo('modulo_compras') && temPermissaoVisualizar('compras')): ?>
                    <a href="<?= url('/compras') ?>"
                        class="sidebar-item <?= isset($activeMenu) && $activeMenu === 'compras' ? 'active' : '' ?>">
                        <i class="bi bi-cart-plus"></i> Compras
                    </a>
                <?php endif; ?>
                <?php /* Comentado temporariamente
<a href="<?= url('/orcamentos-agua') ?>"
class="sidebar-item <?= isset($activeMenu) && $activeMenu === 'orcamentos-agua' ? 'active' : '' ?>">
<i class="bi bi-droplet"></i> Orçamentos Água Mineral
</a>
*/ ?>
                <?php if (moduloAtivo('modulo_vendas') && temPermissaoVisualizar('vendas')): ?>
                    <a href="<?= url('/vendas') ?>"
                        class="sidebar-item <?= isset($activeMenu) && $activeMenu === 'vendas' ? 'active' : '' ?>">
                        <i class="bi bi-cart-check"></i> Vendas
                    </a>
                <?php endif; ?>
                <?php if (moduloAtivo('modulo_estoque') && temPermissaoVisualizar('estoque')): ?>
                    <a href="<?= url('/estoque') ?>" class="sidebar-item">
                        <i class="bi bi-boxes"></i> Gestão de Estoque
                    </a>
                <?php endif; ?>
                <?php if (moduloAtivo('modulo_ordem_servico') && temPermissaoVisualizar('ordem-servico')): ?>
                    <a href="<?= url('/ordem-servico') ?>"
                        class="sidebar-item <?= isset($activeMenu) && $activeMenu === 'ordem-servico' ? 'active' : '' ?>">
                        <i class="bi bi-tools"></i> Ordem de Serviço
                    </a>
                <?php endif; ?>
                <?php if (moduloAtivo('modulo_ordem_servico_pmoc') && temPermissaoVisualizar('ordem-servico-pmoc')): ?>
                    <a href="<?= url('/ordem-servico-pmoc') ?>"
                        class="sidebar-item <?= isset($activeMenu) && $activeMenu === 'ordem-servico-pmoc' ? 'active' : '' ?>">
                        <i class="bi bi-clipboard-check"></i> Ordem de Serviço PMOC
                    </a>
                <?php endif; ?>
                <?php if (moduloAtivo('modulo_contratos') && temPermissaoVisualizar('contratos')): ?>
                    <a href="<?= url('/contratos') ?>"
                        class="sidebar-item <?= isset($activeMenu) && $activeMenu === 'contratos' ? 'active' : '' ?>">
                        <i class="bi bi-file-earmark-text"></i> Contratos
                    </a>
                <?php endif; ?>
                <!-- <a href="<?= url('/agenda') ?>" class="sidebar-item">
                    <i class="bi bi-calendar-event"></i> Agenda
                </a> -->
            </div>

            <!-- Grupo: Financeiro -->
            <div class="menu-group-header collapsed" onclick="toggleMenuGroup(this)">
                <span>Financeiro</span>
                <i class="bi bi-chevron-down toggle-icon"></i>
            </div>
            <div class="menu-group-items collapsed">
                <?php if (moduloAtivo('modulo_entradas') && temPermissaoVisualizar('entradas')): ?>
                    <a href="<?= url('/entradas') ?>" class="sidebar-item">
                        <i class="bi bi-arrow-down-circle"></i> Entradas
                    </a>
                <?php endif; ?>
                <?php if (moduloAtivo('modulo_saidas') && temPermissaoVisualizar('saidas')): ?>
                    <a href="<?= url('/saidas') ?>" class="sidebar-item">
                        <i class="bi bi-arrow-up-circle"></i> Saídas
                    </a>
                <?php endif; ?>
                <?php if (moduloAtivo('modulo_fluxo_caixa') && temPermissaoVisualizar('fluxo-caixa')): ?>
                    <a href="<?= url('/fluxo-caixa') ?>" class="sidebar-item">
                        <i class="bi bi-cash-stack"></i> Fluxo de Caixa
                    </a>
                <?php endif; ?>
            </div>

            <!-- Grupo: Gestão -->
            <div class="menu-group-header collapsed" onclick="toggleMenuGroup(this)">
                <span>Gestão</span>
                <i class="bi bi-chevron-down toggle-icon"></i>
            </div>
            <div class="menu-group-items collapsed">
                <?php if (temPermissaoVisualizar('empresas')): ?>
                    <a href="<?= url('/empresas') ?>"
                        class="sidebar-item <?= isset($activeMenu) && $activeMenu === 'empresas' ? 'active' : '' ?>">
                        <i class="bi bi-building-gear"></i> Empresas
                    </a>
                <?php endif; ?>
                <a href="<?= url('/tabelas-preco') ?>"
                    class="sidebar-item <?= isset($activeMenu) && $activeMenu === 'tabelas-preco' ? 'active' : '' ?>">
                    <i class="bi bi-table"></i> Tabelas de Preço
                </a>
                <?php if (temPermissaoVisualizar('usuarios')): ?>
                    <a href="<?= url('/usuarios') ?>" class="sidebar-item">
                        <i class="bi bi-person-gear"></i> Usuários
                    </a>
                <?php endif; ?>
                <?php if (temPermissaoVisualizar('perfis-acesso')): ?>
                    <a href="<?= url('/perfis-acesso') ?>"
                        class="sidebar-item <?= isset($activeMenu) && $activeMenu === 'perfis-acesso' ? 'active' : '' ?>">
                        <i class="bi bi-shield-check"></i> Perfis de Acesso
                    </a>
                <?php endif; ?>
                <?php if (moduloAtivo('modulo_relatorios') && temPermissaoVisualizar('relatorios')): ?>
                    <a href="<?= url('/relatorios') ?>" class="sidebar-item">
                        <i class="bi bi-file-earmark-text"></i> Relatórios
                    </a>
                <?php endif; ?>
                <?php if (temPermissaoVisualizar('parametros')): ?>
                    <a href="<?= url('/parametros') ?>" class="sidebar-item">
                        <i class="bi bi-gear"></i> Parâmetros
                    </a>
                <?php endif; ?>
                <a href="<?= url('/gestao/backup') ?>"
                    class="sidebar-item <?= isset($activeMenu) && $activeMenu === 'backup' ? 'active' : '' ?>">
                    <i class="bi bi-database-check"></i> Backup
                </a>
            </div>

            <!-- Grupo: Fiscal -->
            <div class="menu-group-header collapsed" onclick="toggleMenuGroup(this)">
                <span>Fiscal</span>
                <i class="bi bi-chevron-down toggle-icon"></i>
            </div>
            <div class="menu-group-items collapsed">
                <?php if (moduloAtivo('modulo_arquivos_fiscais') && temPermissaoVisualizar('arquivos-fiscais')): ?>
                    <a href="<?= url('/arquivos-fiscais') ?>"
                        class="sidebar-item <?= isset($activeMenu) && $activeMenu === 'arquivos-fiscais' ? 'active' : '' ?>">
                        <i class="bi bi-file-earmark-text"></i> Arquivos Fiscais
                    </a>
                <?php endif; ?>
            </div>

            <!-- Grupo: Reservado ao Suporte -->
            <?php if (isset($user) && ($user['role'] === 'admin' || $user['role'] === 'suporte')): ?>
                <div class="menu-group-header collapsed" onclick="toggleMenuGroup(this)">
                    <span>Reservado ao Suporte</span>
                    <i class="bi bi-chevron-down toggle-icon"></i>
                </div>
                <div class="menu-group-items collapsed">
                    <a href="<?= url('/tickets') ?>"
                        class="sidebar-item <?= isset($activeMenu) && $activeMenu === 'tickets' ? 'active' : '' ?>">
                        <i class="bi bi-ticket-perforated"></i> Tickets
                    </a>
                    <a href="<?= url('/suporte/contrato') ?>"
                        class="sidebar-item <?= isset($activeMenu) && $activeMenu === 'suporte' ? 'active' : '' ?>">
                        <i class="bi bi-file-earmark-text"></i> Contrato
                    </a>
                    <a href="<?= url('/suporte/limpar-tabelas') ?>"
                        class="sidebar-item <?= isset($activeMenu) && $activeMenu === 'suporte' ? 'active' : '' ?>">
                        <i class="bi bi-trash"></i> Limpar Tabelas
                    </a>
                    <a href="<?= url('/suporte/importar-banco') ?>"
                        class="sidebar-item <?= isset($activeMenu) && $activeMenu === 'suporte' ? 'active' : '' ?>">
                        <i class="bi bi-upload"></i> Importar Banco de Dados
                    </a>
                    <a href="<?= url('/importacao') ?>"
                        class="sidebar-item <?= isset($activeMenu) && $activeMenu === 'importacao' ? 'active' : '' ?>">
                        <i class="bi bi-upload"></i> Importação
                    </a>
                    <a href="<?= url('/agenda-tecnica') ?>"
                        class="sidebar-item <?= isset($activeMenu) && $activeMenu === 'agenda-tecnica' ? 'active' : '' ?>">
                        <i class="bi bi-calendar-check"></i> Agenda Técnica
                    </a>
                </div>
            <?php endif; ?>
        </nav>
    </aside>

    <!-- Main Content -->
    <main class="main-content">
        <!-- Alerta de Contas Vencidas -->
        <?php if (!empty($contasVencidas['quantidade']) && $contasVencidas['quantidade'] > 0): ?>
            <div class="container-fluid px-4 pt-3">
                <div class="alert-contas-vencidas">
                    <div class="alert-content">
                        <i class="bi bi-exclamation-triangle-fill alert-icon"></i>
                        <div class="alert-text">
                            <strong>▲ Atenção!</strong> Você tem
                            <a href="<?= url('/entradas?status=vencido') ?>">
                                <?= $contasVencidas['quantidade'] ?> conta(s) a receber vencida(s)
                            </a>
                        </div>
                        <div class="alert-total">
                            Total em atraso: R$ <?= number_format($contasVencidas['total'], 2, ',', '.') ?>
                        </div>
                    </div>
                </div>
            </div>
        <?php endif; ?>

        <!-- Topbar -->
        <header class="topbar">
            <div class="d-flex align-items-center gap-2">
                <button class="sidebar-toggle" id="sidebarToggle" type="button" aria-label="Toggle sidebar">
                    <i class="bi bi-list"></i>
                </button>
                <button class="sidebar-collapse-btn" id="sidebarCollapseBtn" type="button" aria-label="Recolher sidebar"
                    title="Recolher/Expandir Menu">
                    <i class="bi bi-chevron-left"></i>
                </button>
                <h5 class="mb-0 d-none d-md-block"><?= $pageTitle ?? 'Dashboard' ?></h5>
            </div>

            <!-- Botões de Acesso Rápido -->
            <div class="quick-access-buttons d-none d-lg-flex align-items-center gap-2 ms-4">
                <?php if (moduloAtivo('modulo_pessoas')): ?>
                    <a href="<?= url('/pessoas') ?>" class="quick-btn-modern" title="Pessoas" data-bs-toggle="tooltip"
                        data-bs-placement="bottom">
                        <div class="quick-btn-icon">
                            <i class="bi bi-people"></i>
                        </div>
                        <span class="quick-btn-label">Pessoas</span>
                    </a>
                <?php endif; ?>
                <?php if (moduloAtivo('modulo_vendas')): ?>
                    <a href="<?= url('/vendas') ?>" class="quick-btn-modern" title="Vendas" data-bs-toggle="tooltip"
                        data-bs-placement="bottom">
                        <div class="quick-btn-icon">
                            <i class="bi bi-cart-check"></i>
                        </div>
                        <span class="quick-btn-label">Vendas</span>
                    </a>
                <?php endif; ?>
                <?php if (moduloAtivo('modulo_compras')): ?>
                    <a href="<?= url('/compras') ?>" class="quick-btn-modern" title="Compras" data-bs-toggle="tooltip"
                        data-bs-placement="bottom">
                        <div class="quick-btn-icon">
                            <i class="bi bi-cart-plus"></i>
                        </div>
                        <span class="quick-btn-label">Compras</span>
                    </a>
                <?php endif; ?>
                <?php if (moduloAtivo('modulo_financeiro')): ?>
                    <a href="<?= url('/entradas') ?>" class="quick-btn-modern" title="Contas a Receber"
                        data-bs-toggle="tooltip" data-bs-placement="bottom">
                        <div class="quick-btn-icon">
                            <i class="bi bi-wallet2"></i>
                        </div>
                        <span class="quick-btn-label">Receber</span>
                    </a>
                    <a href="<?= url('/saidas') ?>" class="quick-btn-modern" title="Contas a Pagar" data-bs-toggle="tooltip"
                        data-bs-placement="bottom">
                        <div class="quick-btn-icon">
                            <i class="bi bi-credit-card"></i>
                        </div>
                        <span class="quick-btn-label">Pagar</span>
                    </a>
                    <a href="<?= url('/fluxo-caixa') ?>" class="quick-btn-modern" title="Fluxo de Caixa"
                        data-bs-toggle="tooltip" data-bs-placement="bottom">
                        <div class="quick-btn-icon">
                            <i class="bi bi-cash-coin"></i>
                        </div>
                        <span class="quick-btn-label">Fluxo de Caixa</span>
                    </a>
                <?php endif; ?>
                <?php if (moduloAtivo('modulo_relatorios')): ?>
                    <a href="<?= url('/relatorios') ?>" class="quick-btn-modern" title="Relatórios" data-bs-toggle="tooltip"
                        data-bs-placement="bottom">
                        <div class="quick-btn-icon">
                            <i class="bi bi-file-earmark-bar-graph"></i>
                        </div>
                        <span class="quick-btn-label">Relatórios</span>
                    </a>
                <?php endif; ?>
                <?php if (moduloAtivo('modulo_estoque')): ?>
                    <a href="<?= url('/estoque') ?>" class="quick-btn-modern" title="Gestão de Estoque"
                        data-bs-toggle="tooltip" data-bs-placement="bottom">
                        <div class="quick-btn-icon">
                            <i class="bi bi-boxes"></i>
                        </div>
                        <span class="quick-btn-label">Estoque</span>
                    </a>
                <?php endif; ?>
                <?php if (moduloAtivo('modulo_ordem_servico')): ?>
                    <a href="<?= url('/ordem-servico') ?>" class="quick-btn-modern" title="Ordem de Serviço"
                        data-bs-toggle="tooltip" data-bs-placement="bottom">
                        <div class="quick-btn-icon">
                            <i class="bi bi-tools"></i>
                        </div>
                        <span class="quick-btn-label">Ordem de Serviço</span>
                    </a>
                <?php endif; ?>
            </div>

            <div class="d-flex align-items-center gap-2 gap-md-3 ms-auto">
                <div class="dropdown">
                    <button class="btn btn-outline-secondary dropdown-toggle" type="button" id="userMenu"
                        data-bs-toggle="dropdown">
                        <i class="bi bi-person-circle"></i> <?= htmlspecialchars($user['name'] ?? 'Usuário') ?>
                    </button>
                    <ul class="dropdown-menu dropdown-menu-end">
                        <!-- <li><a class="dropdown-item" href="<?= url('/parametros') ?>"><i class="bi bi-star"></i> Minha
                                Assinatura</a>
                        </li> -->
                        <!-- <li><a class="dropdown-item" href="<?= url('/empresas') ?>"><i class="bi bi-building"></i>
                                Minhas Empresas</a>
                        </li> -->
                        <li>
                            <hr class="dropdown-divider">
                        </li>
                        <li><a class="dropdown-item text-danger" href="<?= url('/logout') ?>"><i
                                    class="bi bi-box-arrow-right"></i>
                                Sair</a></li>
                    </ul>
                </div>
            </div>
        </header>

        <!-- Content -->
        <div class="content-wrapper">
            <?= $content ?? '' ?>
        </div>
    </main>

    <!-- Bootstrap JS -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>

    <!-- Custom JS -->
    <script>
        // Toggle menu group (expandir/recolher)
        function toggleMenuGroup(header) {
            const items = header.nextElementSibling;
            const icon = header.querySelector('.toggle-icon');

            header.classList.toggle('collapsed');
            items.classList.toggle('collapsed');
        }

        // Gerenciamento de menus - SEMPRE iniciam recolhidos
        document.addEventListener('DOMContentLoaded', function () {
            const headers = document.querySelectorAll('.menu-group-header');
            headers.forEach((header) => {
                const menuItems = header.nextElementSibling;

                // FORÇAR todos os menus a iniciar recolhidos (sem exceção)
                header.classList.add('collapsed');
                menuItems.classList.add('collapsed');
            });

            // Inicializar tooltips dos botões de acesso rápido
            const tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
            tooltipTriggerList.map(function (tooltipTriggerEl) {
                return new bootstrap.Tooltip(tooltipTriggerEl);
            });

            // Toggle sidebar no mobile
            const sidebarToggle = document.getElementById('sidebarToggle');
            const sidebar = document.getElementById('sidebar');
            const sidebarOverlay = document.getElementById('sidebarOverlay');

            if (sidebarToggle && sidebar && sidebarOverlay) {
                function toggleSidebar() {
                    sidebar.classList.toggle('show');
                    sidebarOverlay.classList.toggle('show');
                    document.body.style.overflow = sidebar.classList.contains('show') ? 'hidden' : '';
                }

                sidebarToggle.addEventListener('click', toggleSidebar);
                sidebarOverlay.addEventListener('click', toggleSidebar);

                // Fechar sidebar ao clicar em um link (mobile)
                const sidebarLinks = sidebar.querySelectorAll('.sidebar-item');
                sidebarLinks.forEach(link => {
                    link.addEventListener('click', function () {
                        if (window.innerWidth <= 991.98) {
                            sidebar.classList.remove('show');
                            sidebarOverlay.classList.remove('show');
                            document.body.style.overflow = '';
                        }
                    });
                });
            }

            // Botão para recolher sidebar no desktop
            const sidebarCollapseBtn = document.getElementById('sidebarCollapseBtn');
            const mainContent = document.querySelector('.main-content');

            if (sidebarCollapseBtn && sidebar && mainContent) {
                // Verificar estado salvo no localStorage
                const isCollapsed = localStorage.getItem('sidebarCollapsed') === 'true';

                if (isCollapsed) {
                    sidebar.classList.add('collapsed');
                    mainContent.classList.add('sidebar-collapsed');
                    sidebarCollapseBtn.classList.add('collapsed');
                }

                sidebarCollapseBtn.addEventListener('click', function () {
                    sidebar.classList.toggle('collapsed');
                    mainContent.classList.toggle('sidebar-collapsed');
                    sidebarCollapseBtn.classList.toggle('collapsed');

                    // Salvar estado no localStorage
                    const isNowCollapsed = sidebar.classList.contains('collapsed');
                    localStorage.setItem('sidebarCollapsed', isNowCollapsed ? 'true' : 'false');
                });
            }
        });
    </script>
    <script src="<?= url('/public/assets/js/app.js') ?>"></script>
</body>

</html>