<?php
ob_start();
?>

<div class="row mb-4">
    <div class="col-md-6">
        <h2><i class="bi bi-arrow-down-circle me-2"></i> Contas a Receber</h2>
        <p class="text-muted">Gerencie suas contas a receber</p>
    </div>
    <div class="col-md-6 text-end">
        <!-- Botões discretos apenas com ícones -->
        <button type="button" class="btn btn-outline-secondary me-2" onclick="abrirProvisionamento()"
            title="Provisionamento">
            <i class="bi bi-calendar-check"></i>
        </button>
        <button type="button" class="btn btn-outline-secondary me-2" onclick="abrirRecebimentoDiario()"
            title="Recebimento Diário">
            <i class="bi bi-calendar-day"></i>
        </button>
        <button type="button" class="btn btn-outline-secondary me-2" onclick="abrirSicacred()" title="Sicacred">
            <i class="bi bi-shield-check"></i>
        </button>
        <button type="button" class="btn btn-outline-purple me-2" onclick="consultarCobrancasShipay()"
            title="Cobranças Shipay">
            <i class="bi bi-qr-code"></i>
        </button>
        <a href="<?= url('/entradas/criar') ?>" class="btn btn-primary">
            <i class="bi bi-plus-circle me-2"></i> Nova Conta
        </a>
    </div>
</div>

<!-- Filtros -->
<div class="card mb-4">
    <div class="card-body">
        <form method="GET" class="row g-3">
            <div class="col-md-6">
                <input type="text" class="form-control" name="search" placeholder="Buscar por descrição, cliente..."
                    value="<?= htmlspecialchars($search ?? '') ?>">
            </div>
            <div class="col-md-3">
                <select class="form-select" name="status">
                    <option value="">Todos os Status</option>
                    <?php if (!empty($statusList)): ?>
                        <?php foreach ($statusList as $st):
                            $stId = $st['id'] ?? $st['codigo'] ?? '';
                            ?>
                            <option value="<?= htmlspecialchars($stId) ?>" <?= ($status ?? '') == $stId ? 'selected' : '' ?>>
                                <?= htmlspecialchars($st['nome'] ?? '') ?>
                            </option>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </select>
            </div>
            <div class="col-md-3">
                <button type="submit" class="btn btn-secondary w-100">
                    <i class="bi bi-search"></i> Buscar
                </button>
            </div>
        </form>
    </div>
</div>

<!-- Listagem -->
<div class="card">
    <div class="card-header d-flex justify-content-between align-items-center">
        <span><i class="bi bi-list-ul me-2"></i> Lista de Contas a Receber</span>
        <span class="badge bg-primary"><?= count($contas ?? []) ?> registro(s)</span>
    </div>
    <div class="card-body p-0">
        <?php if (!empty($contas)): ?>
            <div class="table-responsive">
                <table class="table table-hover mb-0">
                    <thead>
                        <tr>
                            <th width="60">ID</th>
                            <th>Descrição</th>
                            <th>Cliente</th>
                            <th>Valor</th>
                            <th>Recebido</th>
                            <th>Restante</th>
                            <th>Vencimento</th>
                            <th>Status</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($contas as $conta): ?>
                            <tr class="conta-row" data-id="<?= $conta['id'] ?>"
                                data-boleto-status="<?= $conta['boleto_status'] ?? '' ?>"
                                data-nosso-numero="<?= $conta['nosso_numero_boleto'] ?? '' ?>"
                                data-shipay-charge="<?= $conta['shipay_charge_id'] ?? '' ?>"
                                data-shipay-status="<?= $conta['shipay_status'] ?? '' ?>" style="cursor: pointer;">
                                <td><?= $conta['id'] ?></td>
                                <td><strong><?= htmlspecialchars($conta['description']) ?></strong></td>
                                <td><?= htmlspecialchars($conta['customer_name'] ?? '-') ?></td>
                                <td><?= formatCurrency($conta['amount'] ?? 0) ?></td>
                                <td><?= formatCurrency($conta['amount_received'] ?? 0) ?></td>
                                <td><?= formatCurrency($conta['amount_remaining'] ?? 0) ?></td>
                                <td><?= date('d/m/Y', strtotime($conta['due_date'])) ?></td>
                                <td>
                                    <?php
                                    if (!empty($conta['status_nome'])) {
                                        $statusCor = $conta['status_cor'] ?? '#6c757d';
                                        $statusIcone = $conta['status_icone'] ?? '';
                                        echo '<span class="badge" style="background-color: ' . htmlspecialchars($statusCor) . ';">';
                                        if ($statusIcone) {
                                            echo '<i class="bi bi-' . htmlspecialchars($statusIcone) . ' me-1"></i>';
                                        }
                                        echo htmlspecialchars($conta['status_nome']);
                                        echo '</span>';
                                    } else {
                                        echo '<span class="badge bg-secondary">-</span>';
                                    }
                                    ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php else: ?>
            <div class="text-center py-5">
                <i class="bi bi-arrow-down-circle" style="font-size: 4rem; color: var(--text-secondary);"></i>
                <p class="text-muted mt-3">Nenhuma conta a receber encontrada</p>
                <a href="<?= url('/entradas/criar') ?>" class="btn btn-primary mt-2">
                    <i class="bi bi-plus-circle me-2"></i> Cadastrar Primeira Conta
                </a>
            </div>
        <?php endif; ?>
    </div>
</div>

<style>
    /* Padronizar altura dos botões */
    .col-md-6.text-end .btn {
        height: 38px;
        display: inline-flex;
        align-items: center;
        justify-content: center;
        padding: 0.375rem 0.75rem;
    }

    .col-md-6.text-end .btn i {
        font-size: 1rem;
    }

    /* Botão Shipay Purple */
    .btn-outline-purple {
        color: #6f42c1;
        border-color: #6f42c1;
    }

    .btn-outline-purple:hover {
        color: #fff;
        background-color: #6f42c1;
        border-color: #6f42c1;
    }

    .conta-row {
        transition: all 0.2s ease;
    }

    .conta-row:hover {
        background-color: rgba(13, 110, 253, 0.05) !important;
        transform: scale(1.01);
        box-shadow: 0 2px 8px rgba(0, 0, 0, 0.1);
    }

    /* Lista de cobranças Shipay */
    .lista-cobrancas-shipay {
        max-height: 450px;
        overflow-y: auto;
    }

    .item-cobranca-shipay {
        padding: 12px 15px;
        border-bottom: 1px solid #eee;
        cursor: pointer;
        transition: all 0.2s ease;
    }

    .item-cobranca-shipay:hover {
        background-color: #f8f5ff;
    }

    .item-cobranca-shipay:last-child {
        border-bottom: none;
    }

    /* SweetAlert2 - Botões QUADRADOS Profissionais */
    .swal-html-container {
        margin: 0 !important;
        padding: 0 !important;
        overflow: hidden !important;
    }

    .acoes-grid {
        display: grid;
        grid-template-columns: repeat(4, 110px);
        gap: 8px;
        padding: 0;
        margin: 0;
        justify-content: center;
    }

    @media (max-width: 576px) {
        .acoes-grid {
            grid-template-columns: repeat(2, 110px);
        }
    }

    .btn-acao {
        width: 110px;
        height: 110px;
        border: 2px solid #e0e0e0;
        border-radius: 0;
        background-color: #ffffff;
        display: flex;
        flex-direction: column;
        align-items: center;
        justify-content: center;
        cursor: pointer;
        transition: all 0.2s ease;
        font-weight: 600;
        font-size: 10px;
        color: #2c3e50;
    }

    .btn-acao:hover {
        border-color: #2c3e50;
        background-color: #2c3e50;
        color: #ffffff;
        transform: translateY(-2px);
        box-shadow: 0 4px 8px rgba(0, 0, 0, 0.15);
    }

    .btn-acao i {
        font-size: 28px;
        margin-bottom: 6px;
    }

    .btn-acao span {
        display: block;
        margin-top: 3px;
        font-size: 9px;
        text-transform: uppercase;
        letter-spacing: 0.3px;
    }

    .btn-acao-delete {
        border-color: #dc3545;
        color: #dc3545;
    }

    .btn-acao-delete:hover {
        background-color: #dc3545;
        color: #ffffff;
        border-color: #dc3545;
    }

    .swal2-popup.swal-acoes-conta {
        border-radius: 0 !important;
        padding: 20px 15px !important;
        width: 500px !important;
        max-width: 500px !important;
    }

    .swal2-title {
        font-size: 16px !important;
        font-weight: 600 !important;
        margin-bottom: 20px !important;
        color: #2c3e50 !important;
        text-transform: uppercase;
        letter-spacing: 1px;
    }

    .swal2-close {
        border-radius: 0 !important;
        font-size: 28px !important;
        color: #999 !important;
    }

    .swal2-close:hover {
        color: #000 !important;
    }
</style>

<script>
    // Evento de click na linha da tabela
    document.addEventListener('DOMContentLoaded', function () {
        document.querySelectorAll('.conta-row').forEach(row => {
            row.addEventListener('click', function (e) {
                // Prevenir se clicar em um link ou botão dentro da linha
                if (e.target.closest('a, button')) return;

                const contaId = this.dataset.id;
                mostrarAcoesConta(contaId);
            });
        });
    });

    function mostrarAcoesConta(id) {
        // Obter dados do boleto da linha
        const row = document.querySelector(`.conta-row[data-id="${id}"]`);
        const boletoStatus = row?.dataset.boletoStatus || '';
        const nossoNumero = row?.dataset.nossoNumero || '';
        const temBoleto = nossoNumero || boletoStatus;

        // Dados Shipay
        const shipayCharge = row?.dataset.shipayCharge || '';
        const shipayStatus = row?.dataset.shipayStatus || '';
        const temShipay = shipayCharge || shipayStatus;

        // Definir textos e ícones dinamicamente - Banco Inter
        const boletoIcon = temBoleto ? 'bi-file-text' : 'bi-bank';
        const boletoText = temBoleto ? 'Consultar Boleto' : 'Gerar Boleto';
        const boletoTitle = temBoleto ? 'Consultar dados do boleto' : 'Gerar Boleto Banco Inter';

        // Definir textos e ícones dinamicamente - Shipay
        const shipayIcon = temShipay ? 'bi-qr-code' : 'bi-credit-card-2-front';
        const shipayText = temShipay ? 'Ver Pix/Boleto' : 'Shipay';
        const shipayTitle = temShipay ? 'Visualizar boleto Shipay' : 'Gerar Boleto Híbrido (Pix + Boleto)';

        Swal.fire({
            title: 'Ações',
            html: `
            <div class="acoes-grid">
                <button type="button" class="btn-acao" onclick="editarConta(${id})" title="Editar Conta">
                    <i class="bi bi-pencil"></i>
                    <span>Editar</span>
                </button>
                <button type="button" class="btn-acao" onclick="visualizarConta(${id})" title="Visualizar Detalhes">
                    <i class="bi bi-eye"></i>
                    <span>Ver</span>
                </button>
                <button type="button" class="btn-acao" onclick="receberConta(${id})" title="Marcar como Recebido">
                    <i class="bi bi-check-circle"></i>
                    <span>Receber</span>
                </button>
                <button type="button" class="btn-acao" onclick="gerarRecibo(${id})" title="Gerar Recibo">
                    <i class="bi bi-receipt"></i>
                    <span>Recibo</span>
                </button>
                <button type="button" class="btn-acao" onclick="gerarDuplicata(${id})" title="Gerar Duplicata">
                    <i class="bi bi-file-earmark-text"></i>
                    <span>Duplicata</span>
                </button>
                <!-- Botão Banco Inter comentado temporariamente
                <button type="button" class="btn-acao" onclick="gerarBoletoInter(${id})" title="${boletoTitle}" id="btn-boleto-${id}">
                    <i class="bi ${boletoIcon}"></i>
                    <span id="btn-boleto-text-${id}">${boletoText}</span>
                </button>
                -->
                <button type="button" class="btn-acao" onclick="gerarBoletoShipay(${id})" title="${shipayTitle}" id="btn-shipay-${id}" style="border-color: #6f42c1; color: #6f42c1;">
                    <i class="bi ${shipayIcon}"></i>
                    <span id="btn-shipay-text-${id}">${shipayText}</span>
                </button>
                <button type="button" class="btn-acao" onclick="gerenciarAnexos(${id})" title="Gerenciar Anexos">
                    <i class="bi bi-paperclip"></i>
                    <span>Anexos</span>
                </button>
                <button type="button" class="btn-acao btn-acao-delete" onclick="deletarConta(${id})" title="Excluir Conta">
                    <i class="bi bi-trash"></i>
                    <span>Excluir</span>
                </button>
            </div>
        `,
            showConfirmButton: false,
            showCloseButton: true,
            width: '500px',
            padding: '20px',
            customClass: {
                popup: 'swal-acoes-conta',
                htmlContainer: 'swal-html-container'
            }
        });
    }

    function editarConta(id) {
        Swal.close();
        window.location.href = appUrl('/entradas/editar?id=' + id);
    }

    function visualizarConta(id) {
        Swal.close();
        window.location.href = appUrl('/entradas/visualizar?id=' + id);
    }

    function receberConta(id) {
        Swal.close();
        Swal.fire({
            title: 'Marcar como Recebido?',
            text: 'Esta conta será marcada como totalmente recebida.',
            icon: 'question',
            showCancelButton: true,
            confirmButtonText: 'Sim, marcar como recebido',
            cancelButtonText: 'Cancelar',
            confirmButtonColor: '#28a745',
            cancelButtonColor: '#6c757d'
        }).then((result) => {
            if (result.isConfirmed) {
                $.ajax({
                    url: appUrl('/entradas/receber'),
                    type: 'POST',
                    data: { id: id },
                    dataType: 'json',
                    success: function (response) {
                        if (response.success) {
                            Swal.fire({
                                title: 'Sucesso!',
                                text: response.message,
                                icon: 'success',
                                confirmButtonText: 'OK'
                            }).then(() => {
                                location.reload();
                            });
                        } else {
                            Swal.fire({
                                title: 'Erro!',
                                text: response.message,
                                icon: 'error',
                                confirmButtonText: 'OK'
                            });
                        }
                    },
                    error: function () {
                        Swal.fire({
                            title: 'Erro!',
                            text: 'Erro ao marcar conta como recebida',
                            icon: 'error',
                            confirmButtonText: 'OK'
                        });
                    }
                });
            }
        });
    }

    function deletarConta(id) {
        Swal.close();
        Swal.fire({
            title: 'Tem certeza?',
            text: 'Esta ação não poderá ser desfeita!',
            icon: 'warning',
            showCancelButton: true,
            confirmButtonText: 'Sim, excluir',
            cancelButtonText: 'Cancelar',
            confirmButtonColor: '#dc3545',
            cancelButtonColor: '#6c757d'
        }).then((result) => {
            if (result.isConfirmed) {
                $.ajax({
                    url: appUrl('/entradas/deletar'),
                    type: 'POST',
                    data: { id: id },
                    dataType: 'json',
                    success: function (response) {
                        if (response.success) {
                            Swal.fire({
                                title: 'Excluído!',
                                text: response.message,
                                icon: 'success',
                                confirmButtonText: 'OK'
                            }).then(() => {
                                location.reload();
                            });
                        } else {
                            Swal.fire({
                                title: 'Erro!',
                                text: response.message,
                                icon: 'error',
                                confirmButtonText: 'OK'
                            });
                        }
                    },
                    error: function () {
                        Swal.fire({
                            title: 'Erro!',
                            text: 'Erro ao excluir conta a receber',
                            icon: 'error',
                            confirmButtonText: 'OK'
                        });
                    }
                });
            }
        });
    }

    function gerenciarAnexos(id) {
        Swal.close();

        // Carregar anexos primeiro
        $.ajax({
            url: appUrl('/entradas/anexos/listar'),
            type: 'GET',
            data: { conta_id: id },
            success: function (response) {
                const anexos = response.success ? (response.data?.anexos || []) : [];
                exibirModalAnexos(id, anexos);
            },
            error: function () {
                exibirModalAnexos(id, []);
            }
        });
    }

    function exibirModalAnexos(contaId, anexos) {
        let listaHtml = '';
        if (anexos.length === 0) {
            listaHtml = '<div class="text-center text-muted py-3"><i class="bi bi-inbox"></i> Nenhum anexo cadastrado</div>';
        } else {
            listaHtml = '<div class="table-responsive" style="max-height: 300px; overflow-y: auto;"><table class="table table-sm table-hover">';
            listaHtml += '<thead><tr><th>Arquivo</th><th>Tamanho</th><th width="100">Ações</th></tr></thead><tbody>';
            anexos.forEach(function (anexo) {
                const tamanho = formatarTamanhoAnexo(anexo.tamanho);
                const icone = obterIconeArquivoAnexo(anexo.nome_original);
                listaHtml += '<tr>';
                listaHtml += '<td><i class="bi ' + icone + ' me-2"></i>' + anexo.nome_original + '</td>';
                listaHtml += '<td>' + tamanho + '</td>';
                listaHtml += '<td>';
                listaHtml += '<a href="' + appUrl('/entradas/anexos/download?id=' + anexo.id) + '" class="btn btn-sm btn-outline-primary me-1" title="Download"><i class="bi bi-download"></i></a>';
                listaHtml += '<button onclick="deletarAnexoModal(' + anexo.id + ', ' + contaId + ')" class="btn btn-sm btn-outline-danger" title="Excluir"><i class="bi bi-trash"></i></button>';
                listaHtml += '</td>';
                listaHtml += '</tr>';
            });
            listaHtml += '</tbody></table></div>';
        }

        Swal.fire({
            title: 'Gerenciar Anexos',
            html: `
            <div class="text-start">
                <form id="formUploadAnexoModal" enctype="multipart/form-data" class="mb-3">
                    <input type="hidden" name="conta_id" value="${contaId}">
                    <div class="mb-2">
                        <label class="form-label small">Selecionar Arquivo</label>
                        <input type="file" class="form-control form-control-sm" name="arquivo"
                            accept=".jpg,.jpeg,.png,.pdf" required>
                        <small class="text-muted">JPG, JPEG, PNG ou PDF (máx. 10MB)</small>
                    </div>
                    <div class="mb-2">
                        <label class="form-label small">Descrição (opcional)</label>
                        <input type="text" class="form-control form-control-sm" name="descricao"
                            placeholder="Ex: Nota fiscal, comprovante...">
                    </div>
                    <button type="submit" class="btn btn-primary btn-sm w-100">
                        <i class="bi bi-upload me-1"></i> Enviar Anexo
                    </button>
                </form>
                <hr>
                <h6 class="small">Anexos Cadastrados</h6>
                <div id="listaAnexosModal">${listaHtml}</div>
            </div>
        `,
            width: 600,
            showConfirmButton: false,
            showCloseButton: true,
            didOpen: () => {
                $('#formUploadAnexoModal').on('submit', function (e) {
                    e.preventDefault();
                    const formData = new FormData(this);
                    $.ajax({
                        url: appUrl('/entradas/anexos/upload'),
                        type: 'POST',
                        data: formData,
                        processData: false,
                        contentType: false,
                        success: function (response) {
                            if (response.success) {
                                Swal.fire({
                                    icon: 'success',
                                    title: 'Sucesso!',
                                    text: response.message,
                                    timer: 1500,
                                    showConfirmButton: false
                                });
                                gerenciarAnexos(contaId); // Recarregar
                            } else {
                                Swal.fire({
                                    icon: 'error',
                                    title: 'Erro',
                                    text: response.message
                                });
                            }
                        },
                        error: function () {
                            Swal.fire({
                                icon: 'error',
                                title: 'Erro',
                                text: 'Erro ao fazer upload do anexo'
                            });
                        }
                    });
                });
            }
        });
    }

    function deletarAnexoModal(anexoId, contaId) {
        Swal.fire({
            title: 'Confirmar Exclusão',
            text: 'Tem certeza que deseja excluir este anexo?',
            icon: 'warning',
            showCancelButton: true,
            confirmButtonText: 'Sim, excluir',
            cancelButtonText: 'Cancelar',
            confirmButtonColor: '#dc3545'
        }).then((result) => {
            if (result.isConfirmed) {
                $.ajax({
                    url: appUrl('/entradas/anexos/deletar'),
                    type: 'POST',
                    data: { anexo_id: anexoId },
                    success: function (response) {
                        if (response.success) {
                            gerenciarAnexos(contaId); // Recarregar
                        } else {
                            Swal.fire({
                                icon: 'error',
                                title: 'Erro',
                                text: response.message
                            });
                        }
                    },
                    error: function () {
                        Swal.fire({
                            icon: 'error',
                            title: 'Erro',
                            text: 'Erro ao excluir anexo'
                        });
                    }
                });
            }
        });
    }

    function formatarTamanhoAnexo(bytes) {
        if (bytes === 0) return '0 Bytes';
        const k = 1024;
        const sizes = ['Bytes', 'KB', 'MB', 'GB'];
        const i = Math.floor(Math.log(bytes) / Math.log(k));
        return Math.round(bytes / Math.pow(k, i) * 100) / 100 + ' ' + sizes[i];
    }

    function obterIconeArquivoAnexo(nomeArquivo) {
        const extensao = nomeArquivo.split('.').pop().toLowerCase();
        const icones = {
            'pdf': 'bi-file-pdf',
            'jpg': 'bi-file-image',
            'jpeg': 'bi-file-image',
            'png': 'bi-file-image'
        };
        return icones[extensao] || 'bi-file';
    }

    function gerarRecibo(id) {
        Swal.close();
        window.open(appUrl('/entradas/recibo?id=' + id), '_blank');
    }

    function gerarDuplicata(id) {
        Swal.close();
        window.open(appUrl('/entradas/duplicata?id=' + id), '_blank');
    }

    function gerarBoletoInter(id) {
        Swal.close();

        // Verificar se já tem boleto
        const row = $(`.conta-row[data-id="${id}"]`);
        const boletoStatus = row.data('boleto-status');
        const nossoNumero = row.data('nosso-numero');
        const temBoleto = nossoNumero || boletoStatus;

        const titulo = temBoleto ? 'Consultar Boleto Banco Inter' : 'Gerar Boleto Banco Inter?';
        const descricao = temBoleto
            ? '<p>Consultando dados do boleto no Banco Inter...</p>'
            : '<p>Será gerado um boleto via API do Banco Inter para esta conta.</p>';
        const btnText = temBoleto ? '<i class="bi bi-file-text me-2"></i> Consultar' : '<i class="bi bi-bank me-2"></i> Gerar Boleto';
        const btnIcon = temBoleto ? 'info' : 'question';

        Swal.fire({
            title: titulo,
            html: `
            ${descricao}
            ${!temBoleto ? `
            <div class="alert alert-info mt-3 text-start">
                <small>
                    <i class="bi bi-info-circle me-1"></i>
                    <strong>Atenção:</strong> Certifique-se de que:
                    <ul class="mb-0 mt-2">
                        <li>Cliente tem CPF/CNPJ e email cadastrados</li>
                        <li>Conta bancária do Inter está configurada</li>
                        <li>Client ID e Secret estão corretos</li>
                    </ul>
                </small>
            </div>
            ` : ''}
        `,
            icon: btnIcon,
            showCancelButton: true,
            confirmButtonText: btnText,
            cancelButtonText: 'Cancelar',
            showLoaderOnConfirm: true,
            preConfirm: () => {
                return $.ajax({
                    url: appUrl('/entradas/gerar-boleto-inter'),
                    type: 'POST',
                    data: { id: id },
                    dataType: 'json'
                }).then(response => {
                    if (!response.success) {
                        throw new Error(response.message || 'Erro desconhecido');
                    }
                    return response;
                }).catch(error => {
                    // Melhor tratamento de erro
                    let errorMessage = 'Erro desconhecido';

                    if (error.responseJSON) {
                        errorMessage = error.responseJSON.message || JSON.stringify(error.responseJSON);
                    } else if (error.responseText) {
                        errorMessage = error.responseText;
                    } else if (error.message) {
                        errorMessage = error.message;
                    } else if (error.statusText) {
                        errorMessage = error.statusText;
                    }

                    console.error('Erro completo:', error);
                    Swal.showValidationMessage(`Erro: ${errorMessage}`);
                });
            },
            allowOutsideClick: () => !Swal.isLoading()
        }).then((result) => {
            if (result.isConfirmed && result.value) {
                // Acessar dados corretos (pode estar em result.value.data ou direto em result.value)
                const responseData = result.value.data || result.value;

                // Validar se a resposta tem o objeto boleto
                if (!responseData.boleto) {
                    Swal.fire({
                        icon: 'error',
                        title: 'Erro',
                        text: 'Resposta inválida do servidor. Dados do boleto não encontrados.'
                    });
                    return;
                }

                const boleto = responseData.boleto;
                const isPendente = responseData.pendente || false;

                // Verificar se é um boleto pendente ou com dados vazios
                const dadosVazios = !boleto.nosso_numero && !boleto.linha_digitavel && !boleto.codigo_barras;

                if (isPendente || boleto.status === 'EMITIDO_PENDENTE' || dadosVazios) {
                    Swal.fire({
                        icon: 'info',
                        title: 'Boleto Criado - Aguardando Dados',
                        html: `
                        <div class="text-start">
                            <div class="alert alert-success mb-3">
                                <i class="bi bi-check-circle me-2"></i>
                                <strong>Boleto criado com sucesso no Banco Inter!</strong>
                            </div>
                            <p>⏳ Os dados do boleto estão sendo processados pelo banco e estarão disponíveis em breve.</p>
                            <hr>
                            <p><strong>📋 Próximos passos:</strong></p>
                            <ol>
                                <li><strong>Aguarde 2-5 minutos</strong> (processamento do banco)</li>
                                <li><strong>Clique em "Boleto Inter" novamente</strong> para consultar os dados</li>
                                <li>Ou acesse diretamente o <strong>painel do Banco Inter</strong></li>
                            </ol>
                            <div class="alert alert-info mt-3">
                                <small><i class="bi bi-info-circle me-1"></i> Isso é normal na primeira emissão. A próxima consulta trará todos os dados: linha digitável, QR Code PIX, PDF, etc.</small>
                            </div>
                        </div>
                    `,
                        confirmButtonText: 'Entendi',
                        width: 600
                    }).then(() => {
                        location.reload();
                    });
                    return;
                }


                // Verificar se é um boleto já existente
                const isExisting = (result.value.message || responseData.message || '').includes('já foi emitido');

                Swal.fire({
                    icon: 'success',
                    title: isExisting ? 'Boleto Localizado!' : 'Boleto Gerado!',
                    html: `
                    ${isExisting ? '<div class="alert alert-info mb-3"><small><i class="bi bi-info-circle me-1"></i> Este boleto já foi emitido anteriormente.</small></div>' : ''}
                    <div class="text-start">
                        ${boleto.nosso_numero ? `<p><strong>Nosso Número:</strong> ${boleto.nosso_numero}</p>` : ''}
                        ${boleto.linha_digitavel ? `
                            <p><strong>Linha Digitável:</strong><br>
                               <code class="small">${boleto.linha_digitavel}</code>
                               <button class="btn btn-sm btn-outline-secondary ms-2" onclick="navigator.clipboard.writeText('${boleto.linha_digitavel}')">
                                   <i class="bi bi-clipboard"></i>
                               </button>
                            </p>
                        ` : ''}
                        ${boleto.qrcode ? `
                            <div class="mb-3">
                                ${boleto.qrcode.startsWith('data:image') ? `
                                    <div class="text-center mb-3">
                                        <img src="${boleto.qrcode}" alt="QR Code PIX" style="max-width: 200px; border: 2px solid #198754; border-radius: 10px; padding: 5px; background: white;">
                                        <p class="small text-muted mt-1 mb-0">Escaneie com o app do banco</p>
                                    </div>
                                ` : ''}
                                <p class="mb-1"><strong>PIX Copia e Cola:</strong></p>
                                <textarea class="form-control small" rows="2" readonly id="pixQrCode">${boleto.qrcode_text || boleto.qrcode}</textarea>
                                <button class="btn btn-sm btn-success mt-1" onclick="navigator.clipboard.writeText(document.getElementById('pixQrCode').value); this.innerHTML='<i class=\\'bi bi-check\\'></i> Copiado!'">
                                    <i class="bi bi-clipboard"></i> Copiar PIX
                                </button>
                            </div>
                        ` : ''}
                        ${boleto.pdf_url ? `
                            <hr class="my-3">
                            <p class="mb-2"><strong><i class="bi bi-file-pdf me-1"></i> PDF do Boleto:</strong></p>
                            <div class="mb-3 d-flex gap-2">
                                <a href="${boleto.pdf_url}" target="_blank" class="btn btn-sm btn-primary" download="boleto_${boleto.nosso_numero}.pdf">
                                    <i class="bi bi-download me-1"></i> Baixar PDF
                                </a>
                                <button type="button" class="btn btn-sm btn-success" onclick="enviarBoletoPorEmail(${id})">
                                    <i class="bi bi-envelope me-1"></i> Enviar por Email
                                </button>
                            </div>
                            <div style="border: 1px solid #ddd; border-radius: 5px; overflow: hidden; max-height: 600px;">
                                <iframe src="${boleto.pdf_url}"
                                        style="width: 100%; height: 600px; border: none;"
                                        type="application/pdf">
                                    <p>Seu navegador não suporta visualização de PDF.
                                       <a href="${boleto.pdf_url}" target="_blank">Clique aqui para baixar</a>
                                    </p>
                                </iframe>
                            </div>
                        ` : ''}
                        ${boleto.emitido_em ? `<p class="mt-3 text-muted"><small>Emitido em: ${boleto.emitido_em}</small></p>` : ''}
                    </div>
                `,
                    confirmButtonText: 'OK',
                    width: 900
                });
            }
        });
    }

    /**
     * Gerar Boletos em Lote para contas do Banco Inter
     */
    function gerarBoletosEmLote() {
        Swal.fire({
            title: 'Gerar Boletos em Lote?',
            html: `
            <div class="text-start">
                <p>Esta ação irá gerar boletos do <strong>Banco Inter</strong> para todas as contas a receber que:</p>
                <ul>
                    <li>Têm conta bancária do Banco Inter selecionada</li>
                    <li>Ainda não têm boleto emitido</li>
                </ul>
                <div class="alert alert-warning mt-3">
                    <small><i class="bi bi-exclamation-triangle me-1"></i> Este processo pode demorar alguns minutos dependendo da quantidade de contas.</small>
                </div>
            </div>
        `,
            icon: 'question',
            showCancelButton: true,
            confirmButtonText: '<i class="bi bi-bank me-2"></i> Processar Lote',
            cancelButtonText: 'Cancelar',
            showLoaderOnConfirm: true,
            allowOutsideClick: () => !Swal.isLoading(),
            preConfirm: () => {
                return $.ajax({
                    url: appUrl('/entradas/gerar-boletos-lote'),
                    type: 'POST',
                    dataType: 'json'
                }).then(response => {
                    if (!response.success) {
                        throw new Error(response.message);
                    }
                    return response;
                }).catch(error => {
                    let errorMessage = 'Erro desconhecido';

                    if (error.responseJSON && error.responseJSON.message) {
                        errorMessage = error.responseJSON.message;
                    } else if (error.message) {
                        errorMessage = error.message;
                    } else if (error.statusText) {
                        errorMessage = error.statusText;
                    }

                    console.error('Erro completo:', error);
                    Swal.showValidationMessage(`Erro: ${errorMessage}`);
                });
            }
        }).then((result) => {
            if (result.isConfirmed && result.value) {
                const data = result.value.data || result.value;

                // Construir HTML com resultados
                let resultadoHtml = `
                <div class="text-start">
                    <div class="row mb-3">
                        <div class="col-3"><strong>Total:</strong></div>
                        <div class="col-9">${data.total || 0} contas</div>
                    </div>
                    <div class="row mb-3">
                        <div class="col-3"><strong>✅ Sucesso:</strong></div>
                        <div class="col-9 text-success">${data.sucesso || 0}</div>
                    </div>
                    <div class="row mb-3">
                        <div class="col-3"><strong>⏳ Pendentes:</strong></div>
                        <div class="col-9 text-warning">${data.pendentes || 0}</div>
                    </div>
                    <div class="row mb-3">
                        <div class="col-3"><strong>❌ Erros:</strong></div>
                        <div class="col-9 text-danger">${data.erros || 0}</div>
                    </div>
            `;

                if (data.detalhes && data.detalhes.length > 0) {
                    resultadoHtml += `
                    <hr>
                    <h6>Detalhes:</h6>
                    <div class="table-responsive" style="max-height: 300px; overflow-y: auto;">
                        <table class="table table-sm">
                            <thead>
                                <tr>
                                    <th>ID</th>
                                    <th>Status</th>
                                    <th>Mensagem</th>
                                </tr>
                            </thead>
                            <tbody>
                `;

                    data.detalhes.forEach(item => {
                        let badgeClass = 'secondary';
                        if (item.status === 'sucesso') badgeClass = 'success';
                        else if (item.status === 'pendente') badgeClass = 'warning';
                        else if (item.status === 'erro') badgeClass = 'danger';

                        resultadoHtml += `
                        <tr>
                            <td>#${item.id}</td>
                            <td><span class="badge bg-${badgeClass}">${item.status}</span></td>
                            <td><small>${item.mensagem}</small></td>
                        </tr>
                    `;
                    });

                    resultadoHtml += `
                            </tbody>
                        </table>
                    </div>
                `;
                }

                resultadoHtml += '</div>';

                Swal.fire({
                    icon: data.total > 0 ? 'success' : 'info',
                    title: 'Processamento Concluído!',
                    html: resultadoHtml,
                    confirmButtonText: 'OK',
                    width: 700
                }).then(() => {
                    location.reload();
                });
            }
        });
    }

    /**
     * Gerar/Consultar Boleto Híbrido via Shipay (Pix + Boleto)
     */
    function gerarBoletoShipay(id) {
        Swal.close();

        // Verificar se já tem boleto Shipay
        const row = $(`.conta-row[data-id="${id}"]`);
        const shipayCharge = row.data('shipay-charge');
        const shipayStatus = row.data('shipay-status');
        const temShipay = shipayCharge || shipayStatus;

        // Se já tem cobrança, vai direto para consulta
        if (temShipay) {
            processarShipay(id, 'consultar');
            return;
        }

        // Mostrar opções de tipo de cobrança
        Swal.fire({
            title: '<i class="bi bi-credit-card-2-front me-2" style="color: #6f42c1;"></i> Shipay',
            html: `
            <p class="text-muted mb-4">Selecione o tipo de cobrança:</p>
            <div class="d-flex gap-3 justify-content-center">
                <button type="button" class="btn-shipay-tipo" onclick="processarShipay(${id}, 'pix')">
                    <i class="bi bi-qr-code" style="font-size: 48px; color: #198754;"></i>
                    <strong class="d-block mt-2">PIX Instantâneo</strong>
                    <small class="text-muted">QR Code para pagamento imediato</small>
                </button>
                <button type="button" class="btn-shipay-tipo" onclick="processarShipay(${id}, 'boleto')">
                    <i class="bi bi-upc-scan" style="font-size: 48px; color: #0d6efd;"></i>
                    <strong class="d-block mt-2">Boleto</strong>
                    <small class="text-muted">Boleto bancário tradicional</small>
                </button>
            </div>
            <div class="alert alert-warning mt-4 text-start small">
                <i class="bi bi-exclamation-triangle me-1"></i>
                <strong>Ambiente de Testes:</strong> No sandbox, ambos geram QR Code PIX (simulação).
                Em produção, o boleto requer wallet de banco configurado no painel Shipay.
            </div>
            <div class="alert alert-light mt-2 text-start small">
                <i class="bi bi-info-circle me-1"></i>
                <strong>Dica:</strong> Certifique-se que o cliente tem CPF/CNPJ cadastrado.
            </div>
        `,
            showConfirmButton: false,
            showCloseButton: true,
            width: 550,
            customClass: {
                popup: 'swal-shipay-opcoes'
            }
        });
    }

    /**
     * Processar geração de cobrança Shipay (PIX ou Boleto)
     */
    function processarShipay(id, tipo) {
        Swal.close();

        const isPix = tipo === 'pix';
        const isConsulta = tipo === 'consultar';

        const titulo = isConsulta ? 'Cobrança Shipay' : (isPix ? 'Gerar PIX Instantâneo' : 'Gerar Boleto');
        const icone = isConsulta ? 'bi-qr-code' : (isPix ? 'bi-qr-code' : 'bi-upc-scan');
        const cor = isPix ? '#198754' : '#0d6efd';

        Swal.fire({
            title: `<i class="bi ${icone} me-2" style="color: ${cor};"></i> ${titulo}`,
            html: `
            <div class="text-center py-3">
                <div class="spinner-border" style="color: ${cor};" role="status"></div>
                <p class="mt-2 text-muted">${isConsulta ? 'Consultando cobrança...' : 'Gerando cobrança...'}</p>
            </div>
        `,
            showConfirmButton: false,
            showCloseButton: true,
            allowOutsideClick: false,
            didOpen: () => {
                $.ajax({
                    url: appUrl('/entradas/gerar-boleto-shipay'),
                    type: 'POST',
                    data: { id: id, tipo: tipo },
                    dataType: 'json'
                }).then(response => {
                    if (!response || !response.success) {
                        const errorMsg = response?.message || response?.error || 'Erro desconhecido ao gerar boleto Shipay';
                        throw new Error(errorMsg);
                    }
                    exibirResultadoShipay(id, response, tipo);
                }).catch(error => {
                    let errorMessage = 'Erro desconhecido ao gerar boleto Shipay';

                    // Verificar diferentes formatos de resposta de erro
                    // Prioridade: responseJSON.message > responseJSON.error > responseText (parsed) > error.message
                    if (error.responseJSON) {
                        errorMessage = error.responseJSON.message || error.responseJSON.error || 'Erro ao gerar boleto Shipay';
                    } else if (error.responseText) {
                        try {
                            const parsed = JSON.parse(error.responseText);
                            errorMessage = parsed.message || parsed.error || errorMessage;
                        } catch (e) {
                            // Se não for JSON válido, usar o texto da resposta (limitado a 200 caracteres)
                            if (error.responseText && error.responseText.length < 200) {
                                errorMessage = error.responseText;
                            }
                        }
                    } else if (error.message) {
                        errorMessage = error.message;
                    }

                    // Log detalhado para debug
                    console.error('Erro Shipay:', {
                        error: error,
                        status: error.status,
                        statusText: error.statusText,
                        responseJSON: error.responseJSON,
                        responseText: error.responseText,
                        message: errorMessage
                    });

                    // Exibir mensagem de erro mais detalhada
                    let errorTitle = 'Erro ao Gerar Boleto';
                    let errorText = errorMessage;

                    // Se a mensagem for muito genérica, tentar extrair mais detalhes
                    if (errorMessage.includes('Erro ao salvar dados') || errorMessage.includes('logs do servidor')) {
                        errorTitle = 'Erro ao Salvar Boleto';
                        errorText = 'O boleto foi gerado com sucesso, mas houve um erro ao salvar os dados no banco. ' +
                            'Verifique os logs do servidor ou entre em contato com o suporte. ' +
                            'O boleto pode ter sido gerado mesmo assim.';
                    }

                    Swal.fire({
                        icon: 'error',
                        title: errorTitle,
                        text: errorText,
                        width: '600px',
                        confirmButtonText: 'OK'
                    });
                });
            }
        });
    }

    /**
     * Exibir resultado da cobrança Shipay
     */
    function exibirResultadoShipay(id, result, tipo) {
        const responseData = result.data || result;
        const boleto = responseData.boleto;
        const isExisting = responseData.existente || false;

        if (!boleto) {
            Swal.fire({
                icon: 'error',
                title: 'Erro',
                text: 'Dados da cobrança não retornados'
            });
            return;
        }

        const isPix = tipo === 'pix';
        const cor = isPix ? '#198754' : '#0d6efd';

        // Montar HTML com dados do boleto
        let htmlContent = '';

        if (isExisting) {
            htmlContent += '<div class="alert alert-info mb-3"><small><i class="bi bi-info-circle me-1"></i> Cobrança já emitida anteriormente.</small></div>';
        }

        htmlContent += '<div class="text-start">';

        // Status (com ID para atualização dinâmica)
        if (boleto.status) {
            const statusBadge = boleto.status === 'paid' || boleto.status === 'pago'
                ? 'bg-success'
                : boleto.status === 'cancelled' || boleto.status === 'cancelado'
                    ? 'bg-danger'
                    : 'bg-warning';
            htmlContent += `<p><strong>Status:</strong> <span class="badge ${statusBadge}" id="shipay-status-badge-${id}">${boleto.status}</span></p>`;
        }

        // Adicionar indicador de aguardando pagamento se status for pendente
        const statusLower = (boleto.status || '').toLowerCase();
        const isPendente = statusLower === 'pending' || statusLower === 'pendingv' || statusLower === 'pendente';
        if (isPendente) {
            htmlContent += `
            <div id="shipay-aguardando-${id}" class="mt-3 text-center">
                <div class="spinner-border spinner-border-sm text-success me-2" role="status"></div>
                <span class="text-muted">Aguardando pagamento...</span>
                <div class="small text-muted mt-1" id="shipay-timer-${id}">Verificando a cada 3 segundos</div>
            </div>
            `;
        }

        // Linha Digitável e Código de Barras (para boleto híbrido)
        if (boleto.linha_digitavel) {
            htmlContent += `
            <div class="alert alert-primary mb-3">
                <p class="mb-2"><strong><i class="bi bi-upc-scan me-1"></i> Boleto Bancário</strong></p>
                <p class="mb-1"><strong>Linha Digitável:</strong></p>
                <div class="input-group mb-2">
                    <input type="text" class="form-control form-control-sm" value="${boleto.linha_digitavel}" readonly id="linhaDigitavelShipay">
                    <button class="btn btn-outline-secondary btn-sm" type="button" onclick="navigator.clipboard.writeText(document.getElementById('linhaDigitavelShipay').value); this.innerHTML='<i class=\\'bi bi-check\\'></i> Copiado!'">
                        <i class="bi bi-clipboard"></i>
                    </button>
                </div>
                ${boleto.codigo_barras ? `
                    <p class="mb-1 mt-3"><strong>Código de Barras:</strong></p>
                    <div class="input-group">
                        <input type="text" class="form-control form-control-sm" value="${boleto.codigo_barras}" readonly id="codigoBarrasShipay">
                        <button class="btn btn-outline-secondary btn-sm" type="button" onclick="navigator.clipboard.writeText(document.getElementById('codigoBarrasShipay').value); this.innerHTML='<i class=\\'bi bi-check\\'></i> Copiado!'">
                            <i class="bi bi-clipboard"></i>
                        </button>
                    </div>
                ` : ''}
            </div>
        `;
        }

        // QR Code Pix
        if (boleto.qrcode) {
            const isBase64Image = boleto.qrcode.startsWith('data:image');
            const pixCopiaECola = boleto.qrcode_text || (!isBase64Image ? boleto.qrcode : '');

            htmlContent += `
            <div class="alert alert-success mb-3">
                <p class="mb-2 text-center"><strong>⚡ PIX - Pague Instantâneo</strong></p>

                ${isBase64Image ? `
                    <div class="text-center mb-3">
                        <img src="${boleto.qrcode}" alt="QR Code PIX" style="max-width: 220px; border: 3px solid #198754; border-radius: 12px; padding: 8px; background: white; box-shadow: 0 4px 12px rgba(25,135,84,0.3);">
                        <p class="small text-muted mt-2 mb-0"><i class="bi bi-phone me-1"></i> Escaneie o QR Code com o app do seu banco</p>
                    </div>
                ` : ''}

                ${pixCopiaECola ? `
                    <hr>
                    <p class="mb-2"><strong><i class="bi bi-copy me-1"></i> Pix Copia e Cola:</strong></p>
                    <textarea class="form-control form-control-sm" rows="3" readonly id="pixShipayQrCode" style="font-size: 10px;">${pixCopiaECola}</textarea>
                    <button class="btn btn-success btn-sm mt-2 w-100" onclick="navigator.clipboard.writeText(document.getElementById('pixShipayQrCode').value); this.innerHTML='<i class=\\'bi bi-check\\'></i> Copiado!'">
                        <i class="bi bi-clipboard"></i> Copiar código Pix
                    </button>
                ` : ''}
            </div>
        `;
        }

        // Botão de Segunda Via
        if (isExisting && boleto.status !== 'paid' && boleto.status !== 'pago') {
            htmlContent += `
            <div class="alert alert-warning mb-3">
                <p class="mb-2"><small><i class="bi bi-info-circle me-1"></i> Precisa de uma segunda via?</small></p>
                <button type="button" class="btn btn-sm btn-warning" onclick="reemitirBoletoShipay(${id})">
                    <i class="bi bi-arrow-repeat me-1"></i> Gerar Segunda Via
                </button>
            </div>
        `;
        }

        // PDF do Boleto
        if (boleto.pdf_url) {
            htmlContent += `
            <hr class="my-3">
            <p class="mb-2"><strong><i class="bi bi-file-pdf me-1"></i> PDF:</strong></p>
            <div class="mb-3 d-flex gap-2 flex-wrap">
                <a href="${boleto.pdf_url}" target="_blank" class="btn btn-sm btn-primary">
                    <i class="bi bi-download me-1"></i> Baixar PDF
                </a>
                <button type="button" class="btn btn-sm btn-success" onclick="enviarBoletoShipayPorEmail(${id})">
                    <i class="bi bi-envelope me-1"></i> Enviar por Email
                </button>
            </div>
            <div style="border: 1px solid #ddd; border-radius: 5px; overflow: hidden;">
                <iframe src="${boleto.pdf_url}" style="width: 100%; height: 500px; border: none;"></iframe>
            </div>
        `;
        }

        htmlContent += '</div>';

        Swal.fire({
            icon: 'success',
            title: isExisting ? 'Cobrança Shipay' : (isPix ? 'PIX Gerado!' : 'Boleto Gerado!'),
            html: htmlContent,
            confirmButtonText: 'OK',
            confirmButtonColor: cor,
            width: 800,
            didOpen: () => {
                // Iniciar polling se status for pendente
                if (isPendente && boleto.charge_id) {
                    iniciarPollingBoletoShipay(id, boleto.charge_id);
                }
            }
        }).then(() => {
            // Parar polling ao fechar o modal
            if (window.shipayPollingInterval) {
                clearInterval(window.shipayPollingInterval);
                window.shipayPollingInterval = null;
            }
            if (!isExisting) {
                location.reload();
            }
        });
    }

    // Função para iniciar polling do status do boleto Shipay
    function iniciarPollingBoletoShipay(contaId, chargeId) {
        let pollingCount = 0;
        const MAX_POLLING = 100; // 100 tentativas = ~5 minutos (3s cada)

        console.log('[Entradas] Iniciando polling para boleto Shipay - Conta ID:', contaId, 'Charge ID:', chargeId);

        window.shipayPollingInterval = setInterval(function () {
            pollingCount++;

            if (pollingCount > MAX_POLLING) {
                clearInterval(window.shipayPollingInterval);
                const timerEl = document.getElementById(`shipay-timer-${contaId}`);
                const aguardandoEl = document.getElementById(`shipay-aguardando-${contaId}`);
                if (timerEl) timerEl.textContent = 'Tempo esgotado. Recarregue a página para verificar o status.';
                if (aguardandoEl) {
                    aguardandoEl.innerHTML = '<span class="text-warning"><i class="bi bi-clock"></i> Tempo esgotado. Recarregue a página se já pagou.</span>';
                }
                return;
            }

            $.ajax({
                url: appUrl('/entradas/consultar-boleto-shipay'),
                type: 'GET',
                data: { id: contaId },
                dataType: 'json',
                success: function (response) {
                    console.log('[Entradas] Status boleto Shipay:', response);

                    if (response.success && response.data && response.data.boleto) {
                        const status = (response.data.boleto.status || '').toLowerCase();
                        const statusPago = status === 'paid' || status === 'pago' || status === 'approved' || status === 'paid_out';

                        // Atualizar badge de status
                        const badgeEl = document.getElementById(`shipay-status-badge-${contaId}`);
                        if (badgeEl) {
                            const statusText = response.data.boleto.status || 'pending';
                            const statusBadge = statusPago
                                ? 'bg-success'
                                : status === 'cancelled' || status === 'cancelado'
                                    ? 'bg-danger'
                                    : 'bg-warning';
                            badgeEl.className = `badge ${statusBadge}`;
                            badgeEl.textContent = statusText;
                        }

                        if (statusPago) {
                            // PAGAMENTO CONFIRMADO!
                            clearInterval(window.shipayPollingInterval);
                            window.shipayPollingInterval = null;
                            console.log('[Entradas] ✅ PAGAMENTO CONFIRMADO!');

                            const aguardandoEl = document.getElementById(`shipay-aguardando-${contaId}`);
                            if (aguardandoEl) {
                                aguardandoEl.innerHTML = '<span class="text-success"><i class="bi bi-check-circle"></i> Pagamento confirmado!</span>';
                            }

                            // Atualizar status na tabela principal também
                            setTimeout(() => {
                                location.reload();
                            }, 2000);

                        } else if (status === 'cancelled' || status === 'cancelado' || status === 'expirado' || status === 'expired') {
                            // Pagamento cancelado ou expirado
                            clearInterval(window.shipayPollingInterval);
                            window.shipayPollingInterval = null;
                            const aguardandoEl = document.getElementById(`shipay-aguardando-${contaId}`);
                            if (aguardandoEl) {
                                aguardandoEl.innerHTML = `<span class="text-danger"><i class="bi bi-x-circle"></i> Pagamento ${status}</span>`;
                            }
                        } else {
                            // Ainda pendente - atualizar contador
                            const timerEl = document.getElementById(`shipay-timer-${contaId}`);
                            if (timerEl) {
                                timerEl.textContent = `Verificando... (${pollingCount}/${MAX_POLLING})`;
                            }
                        }
                    }
                },
                error: function (xhr) {
                    console.error('[Entradas] Erro ao consultar status boleto Shipay:', xhr.status, xhr.responseText);
                    const timerEl = document.getElementById(`shipay-timer-${contaId}`);
                    if (timerEl) {
                        timerEl.textContent = `Erro ao consultar (${pollingCount}/${MAX_POLLING}) - Tentando novamente...`;
                    }
                    // Não parar o polling em caso de erro, continuar tentando
                    // Mas se for erro 400 ou 404 repetido, pode ser problema de endpoint
                    if (xhr.status === 400 && pollingCount > 5) {
                        console.warn('[Entradas] Múltiplos erros 400 - pode ser problema de endpoint');
                    }
                }
            });
        }, 3000); // Verificar a cada 3 segundos
    }

    // Estilo para os botões de tipo Shipay
    const styleShipay = document.createElement('style');
    styleShipay.textContent = `
    .btn-shipay-tipo {
        width: 200px;
        padding: 25px 15px;
        border: 2px solid #e0e0e0;
        border-radius: 12px;
        background: #fff;
        cursor: pointer;
        transition: all 0.3s ease;
        text-align: center;
    }
    .btn-shipay-tipo:hover {
        border-color: #6f42c1;
        background: #f8f5ff;
        transform: translateY(-3px);
        box-shadow: 0 8px 20px rgba(111, 66, 193, 0.15);
    }
    .btn-shipay-tipo strong {
        color: #2c3e50;
    }
    .swal-shipay-opcoes {
        border-radius: 16px !important;
    }
`;
    document.head.appendChild(styleShipay);

    /**
     * Enviar boleto Shipay por email
     */
    function enviarBoletoShipayPorEmail(contaId) {
        $.ajax({
            url: appUrl('/entradas/visualizar'),
            method: 'GET',
            data: { id: contaId },
            success: function (response) {
                const email = response.data?.pessoa_email || response.pessoa_email || '';
                Swal.fire({
                    title: 'Enviar Boleto Shipay por Email',
                    html: `
                    <div class="text-start">
                        <label for="emailDestinoShipay" class="form-label">Email do destinatário:</label>
                        <input type="email" id="emailDestinoShipay" class="form-control" value="${email}" placeholder="exemplo@email.com">
                        <small class="text-muted">O boleto e código Pix serão enviados por email</small>
                    </div>
                `,
                    showCancelButton: true,
                    confirmButtonText: '<i class="bi bi-envelope me-1"></i> Enviar',
                    confirmButtonColor: '#6f42c1',
                    cancelButtonText: 'Cancelar',
                    preConfirm: () => {
                        const email = document.getElementById('emailDestinoShipay').value;
                        if (!email || !/^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(email)) {
                            Swal.showValidationMessage('Por favor, informe um email válido');
                            return false;
                        }
                        return email;
                    }
                }).then((result) => {
                    if (result.isConfirmed) {
                        Swal.fire({
                            title: 'Enviando...',
                            text: 'Aguarde enquanto enviamos o email',
                            allowOutsideClick: false,
                            didOpen: () => Swal.showLoading()
                        });

                        $.ajax({
                            url: appUrl('/entradas/enviar-boleto-shipay-email'),
                            method: 'POST',
                            data: { conta_id: contaId, email: result.value },
                            success: function (response) {
                                Swal.fire({
                                    icon: 'success',
                                    title: 'Email Enviado!',
                                    text: response.message || 'Boleto enviado com sucesso!',
                                    confirmButtonColor: '#6f42c1'
                                });
                            },
                            error: function (xhr) {
                                const erro = xhr.responseJSON || {};
                                Swal.fire({
                                    icon: 'error',
                                    title: 'Erro ao Enviar',
                                    text: erro.message || 'Não foi possível enviar o email'
                                });
                            }
                        });
                    }
                });
            },
            error: function () {
                Swal.fire({
                    icon: 'error',
                    title: 'Erro',
                    text: 'Não foi possível carregar os dados da conta'
                });
            }
        });
    }

    /**
     * Reemitir boleto Shipay (Segunda Via)
     */
    function reemitirBoletoShipay(id) {
        Swal.fire({
            title: 'Gerar Segunda Via?',
            html: `
            <div class="text-start">
                <p>Isso irá:</p>
                <ul>
                    <li><strong>Cancelar</strong> o boleto atual</li>
                    <li><strong>Gerar</strong> um novo boleto com os mesmos dados</li>
                </ul>
                <div class="alert alert-warning mt-3">
                    <small><i class="bi bi-exclamation-triangle me-1"></i> O código Pix e linha digitável anteriores serão <strong>invalidados</strong>.</small>
                </div>
            </div>
        `,
            icon: 'question',
            showCancelButton: true,
            confirmButtonText: '<i class="bi bi-arrow-repeat me-1"></i> Gerar Segunda Via',
            confirmButtonColor: '#ffc107',
            cancelButtonText: 'Cancelar',
            showLoaderOnConfirm: true,
            preConfirm: () => {
                return $.ajax({
                    url: appUrl('/entradas/reemitir-boleto-shipay'),
                    type: 'POST',
                    data: { id: id },
                    dataType: 'json'
                }).then(response => {
                    if (!response.success) {
                        throw new Error(response.message || 'Erro desconhecido');
                    }
                    return response;
                }).catch(error => {
                    let errorMessage = error.responseJSON?.message || error.message || 'Erro desconhecido';
                    Swal.showValidationMessage(`Erro: ${errorMessage}`);
                });
            },
            allowOutsideClick: () => !Swal.isLoading()
        }).then((result) => {
            if (result.isConfirmed && result.value) {
                Swal.fire({
                    icon: 'success',
                    title: 'Segunda Via Gerada!',
                    text: 'O novo boleto foi gerado com sucesso.',
                    confirmButtonColor: '#6f42c1'
                }).then(() => {
                    // Reabrir modal com novos dados
                    gerarBoletoShipay(id);
                });
            }
        });
    }

    function enviarBoletoPorEmail(contaId) {
        console.log('🔥 FUNÇÃO enviarBoletoPorEmail CHAMADA! ID:', contaId);

        // Buscar os dados da conta para pegar o email do cliente
        console.log('📡 Fazendo requisição AJAX para:', appUrl('/entradas/visualizar?id=' + contaId));
        $.ajax({
            url: appUrl('/entradas/visualizar'),
            method: 'GET',
            data: { id: contaId },
            success: function (conta) {
                console.log('✅ Dados da conta recebidos:', conta);
                const email = conta.data?.pessoa_email || conta.pessoa_email || '';
                Swal.fire({
                    title: 'Enviar Boleto por Email',
                    html: `
                    <div class="text-start">
                        <label for="emailDestino" class="form-label">Email do destinatário:</label>
                        <input type="email" id="emailDestino" class="form-control"
                               value="${email}"
                               placeholder="exemplo@email.com">
                        <small class="text-muted">O boleto será enviado em anexo (PDF)</small>
                    </div>
                `,
                    showCancelButton: true,
                    confirmButtonText: 'Enviar',
                    cancelButtonText: 'Cancelar',
                    confirmButtonColor: '#28a745',
                    preConfirm: () => {
                        const email = document.getElementById('emailDestino').value;
                        if (!email) {
                            Swal.showValidationMessage('Por favor, informe um email válido');
                            return false;
                        }
                        // Validação básica de email
                        const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
                        if (!emailRegex.test(email)) {
                            Swal.showValidationMessage('Email inválido');
                            return false;
                        }
                        return email;
                    }
                }).then((result) => {
                    if (result.isConfirmed) {
                        console.log('📧 Enviando email para:', result.value);
                        // Enviar o email
                        $.ajax({
                            url: appUrl('/entradas/enviar-boleto-email'),
                            method: 'POST',
                            data: {
                                conta_id: contaId,
                                email: result.value
                            },
                            success: function (response) {
                                console.log('✅ Email enviado com sucesso:', response);
                                Swal.fire({
                                    icon: 'success',
                                    title: 'Email Enviado!',
                                    text: response.message || 'O boleto foi enviado com sucesso para ' + result.value,
                                    confirmButtonColor: '#28a745'
                                });
                            },
                            error: function (xhr) {
                                console.error('❌ Erro ao enviar email:', xhr);
                                const erro = xhr.responseJSON || {};
                                Swal.fire({
                                    icon: 'error',
                                    title: 'Erro ao Enviar',
                                    text: erro.message || 'Não foi possível enviar o email. Tente novamente.'
                                });
                            }
                        });
                    }
                });
            },
            error: function (xhr) {
                console.error('❌ Erro ao carregar dados da conta:', xhr);
                Swal.fire({
                    icon: 'error',
                    title: 'Erro',
                    text: 'Não foi possível carregar os dados da conta'
                });
            }
        });
    }

    /**
     * Consultar todas as cobranças Shipay emitidas
     */
    function consultarCobrancasShipay() {
        Swal.fire({
            title: '<i class="bi bi-qr-code me-2" style="color: #6f42c1;"></i> Cobranças Shipay',
            html: `
            <div class="text-center py-3">
                <div class="spinner-border text-purple" style="color: #6f42c1;" role="status"></div>
                <p class="mt-2 text-muted">Carregando cobranças...</p>
            </div>
        `,
            showConfirmButton: false,
            showCloseButton: true,
            width: 700,
            didOpen: () => {
                $.ajax({
                    url: appUrl('/entradas/listar-cobrancas-shipay'),
                    type: 'GET',
                    dataType: 'json',
                    success: function (response) {
                        if (response.success && response.data && response.data.cobrancas) {
                            const cobrancas = response.data.cobrancas;

                            if (cobrancas.length === 0) {
                                Swal.update({
                                    html: `
                                    <div class="text-center py-4">
                                        <i class="bi bi-inbox text-muted" style="font-size: 64px;"></i>
                                        <p class="mt-3 text-muted">Nenhuma cobrança Shipay encontrada.</p>
                                        <p class="small text-muted">Gere cobranças PIX ou Boleto para vê-las aqui.</p>
                                    </div>
                                `
                                });
                                return;
                            }

                            let listaHtml = `
                            <div class="text-start mb-3">
                                <div class="input-group input-group-sm">
                                    <span class="input-group-text"><i class="bi bi-search"></i></span>
                                    <input type="text" class="form-control" id="filtroCobrancas" placeholder="Filtrar por descrição ou cliente...">
                                </div>
                            </div>
                            <div class="small text-muted mb-2">
                                <span class="badge bg-secondary">${cobrancas.length}</span> cobrança(s) encontrada(s)
                            </div>
                            <div class="lista-cobrancas-shipay border rounded">
                        `;

                            cobrancas.forEach(cob => {
                                const dataFormatada = cob.due_date ? new Date(cob.due_date).toLocaleDateString('pt-BR') : '-';
                                const valorFormatado = parseFloat(cob.amount || 0).toLocaleString('pt-BR', { style: 'currency', currency: 'BRL' });

                                // Status com cores
                                let statusBadge = 'bg-secondary';
                                let statusText = cob.shipay_status || 'pending';
                                if (['paid', 'pago', 'approved'].includes(statusText.toLowerCase())) {
                                    statusBadge = 'bg-success';
                                    statusText = 'Pago';
                                } else if (['pending', 'pendente'].includes(statusText.toLowerCase())) {
                                    statusBadge = 'bg-warning text-dark';
                                    statusText = 'Pendente';
                                } else if (['cancelled', 'cancelado', 'expired', 'expirado'].includes(statusText.toLowerCase())) {
                                    statusBadge = 'bg-danger';
                                    statusText = statusText.charAt(0).toUpperCase() + statusText.slice(1);
                                } else if (['refunded', 'estornado'].includes(statusText.toLowerCase())) {
                                    statusBadge = 'bg-info';
                                    statusText = 'Estornado';
                                }

                                listaHtml += `
                                <div class="item-cobranca-shipay" onclick="verDetalheCobranca(${cob.id})" data-busca="${(cob.description || '').toLowerCase()} ${(cob.customer_name || '').toLowerCase()}">
                                    <div class="d-flex justify-content-between align-items-center">
                                        <div>
                                            <strong class="text-primary">#${cob.id}</strong>
                                            <span class="ms-2">${cob.description || 'Sem descrição'}</span>
                                            <br>
                                            <small class="text-muted">
                                                <i class="bi bi-person me-1"></i>${cob.customer_name || 'Cliente não informado'}
                                            </small>
                                            <br>
                                            <small class="text-muted">
                                                <i class="bi bi-calendar me-1"></i>Venc: ${dataFormatada}
                                            </small>
                                        </div>
                                        <div class="text-end">
                                            <span class="h6 mb-0">${valorFormatado}</span>
                                            <br>
                                            <span class="badge ${statusBadge}">${statusText}</span>
                                        </div>
                                    </div>
                                </div>
                            `;
                            });

                            listaHtml += '</div>';

                            Swal.update({ html: listaHtml });

                            // Adicionar filtro de busca
                            $('#filtroCobrancas').on('input', function () {
                                const termo = $(this).val().toLowerCase();
                                $('.item-cobranca-shipay').each(function () {
                                    const busca = $(this).data('busca');
                                    $(this).toggle(busca.includes(termo));
                                });
                            });
                        } else {
                            Swal.update({
                                html: `<div class="text-center text-danger py-3">
                                <i class="bi bi-x-circle" style="font-size: 48px;"></i>
                                <p class="mt-2">${response.message || 'Erro ao carregar cobranças'}</p>
                            </div>`
                            });
                        }
                    },
                    error: function (xhr) {
                        const erro = xhr.responseJSON?.message || 'Erro ao conectar com o servidor';
                        Swal.update({
                            html: `<div class="text-center text-danger py-3">
                            <i class="bi bi-x-circle" style="font-size: 48px;"></i>
                            <p class="mt-2">${erro}</p>
                        </div>`
                        });
                    }
                });
            }
        });
    }

    /**
     * Ver detalhes de uma cobrança específica
     */
    function verDetalheCobranca(contaId) {
        Swal.close();
        // Reutiliza a função existente de visualizar boleto Shipay
        gerarBoletoShipay(contaId);
    }

    /**
     * Abrir modal de Provisionamento
     */
    function abrirProvisionamento() {
        Swal.fire({
            title: 'Provisionamento',
            text: 'Funcionalidade em desenvolvimento',
            icon: 'info',
            confirmButtonText: 'OK'
        });
    }

    /**
     * Abrir modal de Sicacred
     */
    function abrirSicacred() {
        Swal.fire({
            title: '<i class="bi bi-shield-check me-2"></i> Sicacred',
            html: `
            <div class="text-start">
                <div class="alert alert-info mb-3">
                    <i class="bi bi-info-circle me-2"></i>
                    <strong>Integração Sicacred</strong><br>
                    <small>Envie informações de clientes, compras e documentos para análise de crédito.</small>
                </div>

                <div class="mb-3">
                    <label class="form-label">Data Inicial</label>
                    <input type="date" id="sicacredDataInicial" class="form-control" required autocomplete="off" data-lpignore="true">
                </div>

                <div class="mb-3">
                    <label class="form-label">Data Final</label>
                    <input type="date" id="sicacredDataFinal" class="form-control" required autocomplete="off" data-lpignore="true">
                </div>

                <div class="mb-3">
                    <label class="form-label">Email do Responsável (opcional)</label>
                    <input type="email" id="sicacredEmail" class="form-control" placeholder="email@responsavel.com.br" autocomplete="off" data-lpignore="true">
                    <small class="text-muted">Emails separados por ponto e vírgula (;) se houver mais de um</small>
                </div>

                <div class="d-flex gap-2">
                    <button type="button" class="btn btn-outline-primary btn-sm" onclick="sicacredConsultarDatas()">
                        <i class="bi bi-calendar-check me-1"></i> Consultar Datas
                    </button>
                    <button type="button" class="btn btn-primary btn-sm" onclick="sicacredEnviarDados()">
                        <i class="bi bi-send me-1"></i> Enviar Dados
                    </button>
                </div>
            </div>
        `,
            showConfirmButton: false,
            showCloseButton: true,
            width: 600,
            didOpen: () => {
                // Definir data final como hoje
                const hoje = new Date().toISOString().split('T')[0];
                document.getElementById('sicacredDataFinal').value = hoje;
            }
        });
    }

    /**
     * Consultar datas sugeridas do Sicacred
     */
    function sicacredConsultarDatas() {
        Swal.fire({
            title: 'Consultando...',
            html: '<div class="spinner-border text-primary" role="status"></div>',
            showConfirmButton: false,
            allowOutsideClick: false,
            didOpen: () => {
                $.ajax({
                    url: appUrl('/entradas/sicacred/consultar-datas'),
                    type: 'GET',
                    dataType: 'json',
                    success: function (response) {
                        if (response.success && response.data) {
                            const data = response.data;
                            const diasAtraso = data.dias_atraso || 0;
                            const alertClass = diasAtraso >= 7 ? 'alert-warning' : 'alert-info';

                            Swal.fire({
                                title: 'Datas Consultadas',
                                html: `
                                <div class="text-start">
                                    <div class="alert ${alertClass} mb-3">
                                        <i class="bi bi-${diasAtraso >= 7 ? 'exclamation-triangle' : 'info-circle'} me-2"></i>
                                        <strong>${data.mensagem || 'Consulta realizada com sucesso'}</strong>
                                    </div>

                                    <div class="row mb-2">
                                        <div class="col-6"><strong>Data Sugerida:</strong></div>
                                        <div class="col-6">${data.data_sugerida || '-'}</div>
                                    </div>

                                    <div class="row mb-2">
                                        <div class="col-6"><strong>Último Envio:</strong></div>
                                        <div class="col-6">${data.data_ultimo_envio || '-'}</div>
                                    </div>

                                    <div class="row mb-3">
                                        <div class="col-6"><strong>Dias de Atraso:</strong></div>
                                        <div class="col-6">
                                            <span class="badge ${diasAtraso >= 7 ? 'bg-warning' : 'bg-info'}">
                                                ${diasAtraso} dias
                                            </span>
                                        </div>
                                    </div>

                                    ${data.data_sugerida ? `
                                        <button type="button" class="btn btn-sm btn-primary w-100" onclick="preencherDataSugerida('${data.data_sugerida}')">
                                            <i class="bi bi-calendar-check me-1"></i> Usar Data Sugerida
                                        </button>
                                    ` : ''}
                                </div>
                            `,
                                confirmButtonText: 'OK'
                            });
                        } else {
                            Swal.fire({
                                icon: 'error',
                                title: 'Erro',
                                text: response.message || 'Erro ao consultar datas'
                            });
                        }
                    },
                    error: function (xhr) {
                        const erro = xhr.responseJSON?.message || 'Erro ao conectar com o servidor';
                        Swal.fire({
                            icon: 'error',
                            title: 'Erro',
                            text: erro
                        });
                    }
                });
            }
        });
    }

    /**
     * Preencher data sugerida no formulário
     */
    function preencherDataSugerida(dataSugerida) {
        Swal.close();
        abrirSicacred();
        setTimeout(() => {
            document.getElementById('sicacredDataInicial').value = dataSugerida;
        }, 300);
    }

    /**
     * Enviar dados para o Sicacred
     */
    function sicacredEnviarDados() {
        const dataInicial = document.getElementById('sicacredDataInicial').value;
        const dataFinal = document.getElementById('sicacredDataFinal').value;
        const email = document.getElementById('sicacredEmail').value;

        if (!dataInicial || !dataFinal) {
            Swal.fire({
                icon: 'warning',
                title: 'Atenção',
                text: 'Preencha a data inicial e data final'
            });
            return;
        }

        Swal.fire({
            title: 'Enviando dados...',
            html: `
            <div class="text-center py-3">
                <div class="spinner-border text-primary" role="status"></div>
                <p class="mt-2 text-muted">Enviando informações para o Sicacred...</p>
                <small class="text-muted">Este processo pode levar alguns minutos</small>
            </div>
        `,
            showConfirmButton: false,
            allowOutsideClick: false,
            didOpen: () => {
                $.ajax({
                    url: appUrl('/entradas/sicacred/enviar-dados'),
                    type: 'POST',
                    data: {
                        data_inicial: dataInicial,
                        data_final: dataFinal,
                        email_responsavel: email
                    },
                    dataType: 'json',
                    success: function (response) {
                        if (response.success) {
                            Swal.fire({
                                icon: 'success',
                                title: 'Sucesso!',
                                html: `
                                <div class="text-start">
                                    <p><strong>Dados enviados com sucesso!</strong></p>
                                    <p class="mb-0">${response.data?.mensagem || 'Envio realizado com sucesso'}</p>
                                    <p class="mt-2 small text-muted">Código: ${response.data?.codigo || 'N/A'}</p>
                                </div>
                            `,
                                confirmButtonText: 'OK'
                            });
                        } else {
                            Swal.fire({
                                icon: 'error',
                                title: 'Erro',
                                text: response.message || 'Erro ao enviar dados'
                            });
                        }
                    },
                    error: function (xhr) {
                        const erro = xhr.responseJSON?.message || 'Erro ao conectar com o servidor';
                        Swal.fire({
                            icon: 'error',
                            title: 'Erro',
                            html: `
                            <div class="text-start">
                                <p><strong>Erro ao enviar dados:</strong></p>
                                <p class="mb-0">${erro}</p>
                            </div>
                        `
                        });
                    }
                });
            }
        });
    }

    /**
     * Abrir modal de Recebimento Diário
     */
    function abrirRecebimentoDiario() {
        Swal.fire({
            title: '<i class="bi bi-calendar-day me-2"></i> Recebimento Diário',
            html: `
            <div class="text-center py-3">
                <div class="spinner-border text-primary" role="status"></div>
                <p class="mt-2 text-muted">Carregando títulos do dia...</p>
            </div>
        `,
            showConfirmButton: false,
            showCloseButton: true,
            width: 900,
            didOpen: () => {
                $.ajax({
                    url: appUrl('/entradas/recebimento-diario'),
                    type: 'GET',
                    dataType: 'json',
                    success: function (response) {
                        if (response.success && response.data) {
                            const data = response.data;
                            const totalizadores = data.totalizadores || {};

                            let html = `
                            <div class="text-start">
                                <div class="row mb-3">
                                    <div class="col-md-4">
                                        <div class="card border-primary">
                                            <div class="card-body text-center">
                                                <h5 class="text-primary mb-1">${totalizadores.quantidade_total || 0}</h5>
                                                <small class="text-muted">Total de Títulos</small>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="col-md-4">
                                        <div class="card border-success">
                                            <div class="card-body text-center">
                                                <h5 class="text-success mb-1">${totalizadores.quantidade_pagos || 0}</h5>
                                                <small class="text-muted">Títulos Pagos</small>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="col-md-4">
                                        <div class="card border-warning">
                                            <div class="card-body text-center">
                                                <h5 class="text-warning mb-1">${totalizadores.quantidade_nao_pagos || 0}</h5>
                                                <small class="text-muted">Títulos Pendentes</small>
                                            </div>
                                        </div>
                                    </div>
                                </div>

                                <div class="row mb-3">
                                    <div class="col-md-4">
                                        <div class="card bg-light">
                                            <div class="card-body text-center">
                                                <h6 class="mb-1">${formatCurrency(totalizadores.total_geral || 0)}</h6>
                                                <small class="text-muted">Valor Total</small>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="col-md-4">
                                        <div class="card bg-light">
                                            <div class="card-body text-center">
                                                <h6 class="text-success mb-1">${formatCurrency(totalizadores.total_pagos || 0)}</h6>
                                                <small class="text-muted">Total Recebido</small>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="col-md-4">
                                        <div class="card bg-light">
                                            <div class="card-body text-center">
                                                <h6 class="text-warning mb-1">${formatCurrency(totalizadores.total_nao_pagos || 0)}</h6>
                                                <small class="text-muted">Total Pendente</small>
                                            </div>
                                        </div>
                                    </div>
                                </div>

                                <hr>

                                <h6 class="mb-2"><i class="bi bi-check-circle text-success me-2"></i> Títulos Pagos (${data.pagos?.length || 0})</h6>
                                <div class="table-responsive mb-4" style="max-height: 250px; overflow-y: auto;">
                                    <table class="table table-sm table-hover">
                                        <thead>
                                            <tr>
                                                <th>ID</th>
                                                <th>Descrição</th>
                                                <th>Cliente</th>
                                                <th>Valor</th>
                                                <th>Status</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                        `;

                            if (data.pagos && data.pagos.length > 0) {
                                data.pagos.forEach(titulo => {
                                    const statusCor = titulo.status_cor || '#6c757d';
                                    html += `
                                    <tr>
                                        <td>#${titulo.id}</td>
                                        <td>${titulo.description || '-'}</td>
                                        <td>${titulo.customer_name || '-'}</td>
                                        <td>${formatCurrency(titulo.amount || 0)}</td>
                                        <td><span class="badge" style="background-color: ${statusCor};">${titulo.status_nome || '-'}</span></td>
                                    </tr>
                                `;
                                });
                            } else {
                                html += '<tr><td colspan="5" class="text-center text-muted">Nenhum título pago encontrado</td></tr>';
                            }

                            html += `
                                        </tbody>
                                    </table>
                                </div>

                                <h6 class="mb-2"><i class="bi bi-clock-history text-warning me-2"></i> Títulos Pendentes (${data.nao_pagos?.length || 0})</h6>
                                <div class="table-responsive" style="max-height: 250px; overflow-y: auto;">
                                    <table class="table table-sm table-hover">
                                        <thead>
                                            <tr>
                                                <th>ID</th>
                                                <th>Descrição</th>
                                                <th>Cliente</th>
                                                <th>Valor</th>
                                                <th>Recebido</th>
                                                <th>Pendente</th>
                                                <th>Status</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                        `;

                            if (data.nao_pagos && data.nao_pagos.length > 0) {
                                data.nao_pagos.forEach(titulo => {
                                    const valor = parseFloat(titulo.amount || 0);
                                    const recebido = parseFloat(titulo.amount_received || 0);
                                    const pendente = valor - recebido;
                                    const statusCor = titulo.status_cor || '#6c757d';
                                    html += `
                                    <tr>
                                        <td>#${titulo.id}</td>
                                        <td>${titulo.description || '-'}</td>
                                        <td>${titulo.customer_name || '-'}</td>
                                        <td>${formatCurrency(valor)}</td>
                                        <td>${formatCurrency(recebido)}</td>
                                        <td><strong class="text-warning">${formatCurrency(pendente)}</strong></td>
                                        <td><span class="badge" style="background-color: ${statusCor};">${titulo.status_nome || '-'}</span></td>
                                    </tr>
                                `;
                                });
                            } else {
                                html += '<tr><td colspan="7" class="text-center text-muted">Nenhum título pendente encontrado</td></tr>';
                            }

                            html += `
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        `;

                            Swal.update({
                                html: html,
                                showConfirmButton: true,
                                confirmButtonText: 'Fechar'
                            });
                        } else {
                            Swal.update({
                                html: `<div class="text-center text-danger py-3">
                                <i class="bi bi-x-circle" style="font-size: 48px;"></i>
                                <p class="mt-2">${response.message || 'Erro ao carregar dados'}</p>
                            </div>`,
                                showConfirmButton: true
                            });
                        }
                    },
                    error: function (xhr) {
                        const erro = xhr.responseJSON?.message || 'Erro ao conectar com o servidor';
                        Swal.update({
                            html: `<div class="text-center text-danger py-3">
                            <i class="bi bi-x-circle" style="font-size: 48px;"></i>
                            <p class="mt-2">${erro}</p>
                        </div>`,
                            showConfirmButton: true
                        });
                    }
                });
            }
        });
    }

    function formatCurrency(value) {
        return parseFloat(value || 0).toLocaleString('pt-BR', {
            style: 'currency',
            currency: 'BRL'
        });
    }
</script>

<?php
$content = ob_get_clean();
$activeMenu = 'entradas';
include ROOT_PATH . '/views/layouts/app.php';
?>