<?php
ob_start();
?>

<div class="row mb-4">
    <div class="col-md-6">
        <h2><i class="bi bi-cart-plus me-2"></i> Compras</h2>
        <p class="text-muted">Gerencie suas compras e fornecedores</p>
    </div>
    <div class="col-md-6 text-end">
        <button type="button" class="btn btn-info" onclick="abrirMonitorNFe()">
            <i class="bi bi-receipt-cutoff me-2"></i> Monitor NF-e
        </button>
        <a href="<?= url('/compras/criar') ?>" class="btn btn-primary">
            <i class="bi bi-plus-circle me-2"></i> Nova Compra
        </a>
    </div>
</div>

<!-- Filtros -->
<div class="card mb-4">
    <div class="card-body">
        <form method="GET" class="row g-3">
            <div class="col-md-4">
                <input type="text" class="form-control" name="search" placeholder="Buscar por número, fornecedor..."
                    value="<?= htmlspecialchars($search ?? '') ?>">
            </div>
            <div class="col-md-2">
                <select class="form-select" name="status">
                    <option value="">Todos os Status</option>
                    <option value="orcamento" <?= ($status ?? '') === 'orcamento' ? 'selected' : '' ?>>Orçamento
                    </option>
                    <option value="pedido" <?= ($status ?? '') === 'pedido' ? 'selected' : '' ?>>Pedido</option>
                    <option value="recebido" <?= ($status ?? '') === 'recebido' ? 'selected' : '' ?>>Recebido</option>
                    <option value="cancelado" <?= ($status ?? '') === 'cancelado' ? 'selected' : '' ?>>Cancelado
                    </option>
                </select>
            </div>
            <?php if (!empty($todasEmpresas)): ?>
                <div class="col-md-3">
                    <select class="form-select" name="empresa">
                        <option value="">Todas as Empresas</option>
                        <?php foreach ($todasEmpresas as $emp): ?>
                            <option value="<?= $emp['id'] ?>" <?= (isset($empresa) && $empresa == $emp['id']) ? 'selected' : '' ?>>
                                <?= htmlspecialchars($emp['nome_fantasia'] ?: $emp['razao_social']) ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
            <?php endif; ?>
            <div class="col-md-<?= !empty($todasEmpresas) ? '3' : '6' ?>">
                <button type="submit" class="btn btn-secondary w-100">
                    <i class="bi bi-search"></i> Buscar
                </button>
            </div>
        </form>
    </div>
</div>

<!-- Listagem -->
<div class="card">
    <div class="card-header d-flex justify-content-between align-items-center">
        <span><i class="bi bi-list-ul me-2"></i> Lista de Compras</span>
        <span class="badge bg-primary"><?= count($compras ?? []) ?> registro(s)</span>
    </div>
    <div class="card-body p-0">
        <?php if (!empty($compras)): ?>
            <div class="table-responsive">
                <table class="table table-hover mb-0">
                    <thead>
                        <tr>
                            <th width="60">ID</th>
                            <th>Número</th>
                            <th>Fornecedor</th>
                            <th>Data</th>
                            <th>Total</th>
                            <th>Status</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($compras as $compra): ?>
                            <tr style="cursor: pointer;" onclick="abrirOpcoesCompra(event)"
                                data-compra-id="<?= $compra['id'] ?>"
                                data-compra-numero="<?= htmlspecialchars($compra['numero'] ?? $compra['id']) ?>"
                                data-compra-fornecedor="<?= htmlspecialchars($compra['supplier_name'] ?? '-') ?>"
                                data-compra-total="<?= number_format($compra['total'] ?? 0, 2, ',', '.') ?>"
                                data-compra-status="<?= htmlspecialchars($compra['status']) ?>">
                                <td><?= $compra['id'] ?></td>
                                <td><strong><?= htmlspecialchars($compra['numero'] ?? '-') ?></strong></td>
                                <td><?= htmlspecialchars($compra['supplier_name'] ?? '-') ?></td>
                                <td><?= date('d/m/Y', strtotime($compra['data_compra'])) ?></td>
                                <td>R$ <?= number_format($compra['total'] ?? 0, 2, ',', '.') ?></td>
                                <td>
                                    <span class="badge bg-secondary">
                                        <?= htmlspecialchars($compra['status'] ?? '-') ?>
                                    </span>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php else: ?>
            <div class="text-center py-5">
                <i class="bi bi-cart-plus" style="font-size: 4rem; color: var(--text-secondary);"></i>
                <p class="text-muted mt-3">Nenhuma compra encontrada</p>
                <a href="<?= url('/compras/criar') ?>" class="btn btn-primary mt-2">
                    <i class="bi bi-plus-circle me-2"></i> Cadastrar Primeira Compra
                </a>
            </div>
        <?php endif; ?>
    </div>
</div>

<style>
    /* Modal de Opções da Compra */
    .swal-compra-opcoes {
        width: 600px !important;
        max-width: 90vw;
        overflow: visible !important;
    }

    .swal-compra-container {
        overflow: visible !important;
    }

    .swal-compra-opcoes .swal2-html-container {
        overflow: visible !important;
        padding: 0 !important;
        margin: 0 !important;
    }

    .compra-header {
        background: linear-gradient(135deg, #16a085 0%, #27ae60 100%);
        color: white;
        padding: 1.5rem;
        margin: 0;
        position: relative;
    }

    .compra-header-top {
        display: flex;
        justify-content: space-between;
        align-items: center;
        margin-bottom: 0.5rem;
    }

    .compra-numero {
        font-size: 1.5rem;
        font-weight: bold;
        margin: 0;
    }

    .compra-total-badge {
        background: rgba(255, 255, 255, 0.2);
        padding: 0.5rem 1rem;
        border-radius: 8px;
        font-size: 1.25rem;
        font-weight: bold;
        padding-right: 3rem;
    }

    .compra-fornecedor-row {
        display: flex;
        justify-content: space-between;
        align-items: center;
        font-size: 1rem;
        opacity: 0.95;
    }

    .swal2-close {
        color: white !important;
        font-size: 2rem !important;
        top: 1rem !important;
        right: 1rem !important;
        background: transparent !important;
        z-index: 10;
    }

    .swal2-close:hover {
        color: rgba(255, 255, 255, 0.8) !important;
    }

    .compra-opcoes-grid {
        display: grid;
        grid-template-columns: repeat(2, 1fr);
        gap: 1rem;
        padding: 1.5rem;
    }

    .compra-opcao-card {
        display: flex;
        flex-direction: column;
        align-items: center;
        justify-content: center;
        padding: 1.5rem;
        border: 2px solid #e0e0e0;
        border-radius: 12px;
        background: white;
        cursor: pointer;
        transition: all 0.3s ease;
        gap: 0.75rem;
        text-decoration: none;
        color: inherit;
    }

    .compra-opcao-card:hover {
        transform: translateY(-4px);
        box-shadow: 0 8px 16px rgba(0, 0, 0, 0.1);
        text-decoration: none;
    }

    .compra-opcao-card i {
        font-size: 2.5rem;
        margin-bottom: 0.5rem;
    }

    .compra-opcao-card span {
        font-weight: 600;
        font-size: 0.95rem;
        text-align: center;
    }

    /* Cores das opções */
    .compra-opcao-primary {
        border-color: #007bff;
        color: #007bff;
    }

    .compra-opcao-primary:hover {
        background: #007bff;
        color: white;
    }

    .compra-opcao-success {
        border-color: #28a745;
        color: #28a745;
    }

    .compra-opcao-success:hover {
        background: #28a745;
        color: white;
    }

    .compra-opcao-info {
        border-color: #17a2b8;
        color: #17a2b8;
    }

    .compra-opcao-info:hover {
        background: #17a2b8;
        color: white;
    }

    .compra-opcao-warning {
        border-color: #ffc107;
        color: #856404;
    }

    .compra-opcao-warning:hover {
        background: #ffc107;
        color: #856404;
    }

    .compra-opcao-danger {
        border-color: #dc3545;
        color: #dc3545;
    }

    .compra-opcao-danger:hover {
        background: #dc3545;
        color: white;
    }
</style>

<script>
    function abrirOpcoesCompra(event) {
        const row = event.currentTarget;
        const compraId = row.dataset.compraId;
        const compraNumero = row.dataset.compraNumero;
        const compraFornecedor = row.dataset.compraFornecedor;
        const compraTotal = row.dataset.compraTotal;
        const compraStatus = row.dataset.compraStatus;

        // Montar HTML do header customizado
        const headerHtml = `
        <div class="compra-header">
            <div class="compra-header-top">
                <h3 class="compra-numero">Compra: ${compraNumero}</h3>
                <div class="compra-total-badge">R$ ${compraTotal}</div>
            </div>
            <div class="compra-fornecedor-row">
                <span><i class="bi bi-building me-2"></i>${compraFornecedor}</span>
            </div>
        </div>
    `;

        // Montar HTML das opções
        const opcoesHtml = `
        <div class="compra-opcoes-grid">
            <button type="button" class="compra-opcao-card compra-opcao-primary" onclick="window.location.href=appUrl('/compras/visualizar?id=${compraId}')">
                <i class="bi bi-eye"></i>
                <span>Visualizar</span>
            </button>
            <button type="button" class="compra-opcao-card compra-opcao-success" onclick="window.open(appUrl('/compras/imprimir?id=${compraId}'), '_blank')">
                <i class="bi bi-printer"></i>
                <span>Imprimir Pedido</span>
            </button>
            <button type="button" class="compra-opcao-card compra-opcao-info" onclick="enviarEmailCompra(${compraId}, '${compraNumero}')">
                <i class="bi bi-envelope"></i>
                <span>Enviar por Email</span>
            </button>
            <button type="button" class="compra-opcao-card compra-opcao-warning" onclick="window.location.href=appUrl('/compras/editar?id=${compraId}')">
                <i class="bi bi-pencil"></i>
                <span>Editar</span>
            </button>
            <button type="button" class="compra-opcao-card compra-opcao-danger" onclick="deletarCompra(${compraId})">
                <i class="bi bi-trash"></i>
                <span>Excluir</span>
            </button>
        </div>
    `;

        Swal.fire({
            html: headerHtml + opcoesHtml,
            showConfirmButton: false,
            showCloseButton: true,
            padding: '0',
            customClass: {
                popup: 'swal-compra-opcoes',
                htmlContainer: 'swal-compra-container'
            }
        });
    }

    async function enviarEmailCompra(compraId, compraNumero) {
        Swal.close();

        const {
            value: formValues
        } = await Swal.fire({
            title: 'Enviar Compra por E-mail',
            html: `
            <div class="text-start">
                <div class="mb-3">
                    <label class="form-label">E-mail do Destinatário *</label>
                    <input type="email" id="swal-email" class="form-control" required>
                </div>
                <div class="mb-3">
                    <label class="form-label">Assunto</label>
                    <input type="text" id="swal-assunto" class="form-control" value="Pedido de Compra #${compraNumero}">
                </div>
                <div class="mb-3">
                    <label class="form-label">Mensagem</label>
                    <textarea id="swal-mensagem" class="form-control" rows="4" placeholder="Mensagem opcional..."></textarea>
                </div>
                <div class="mb-3 form-check">
                    <input type="checkbox" id="swal-anexar-pdf" class="form-check-input" checked>
                    <label class="form-check-label" for="swal-anexar-pdf">Anexar PDF do pedido</label>
                </div>
            </div>
        `,
            focusConfirm: false,
            showCancelButton: true,
            confirmButtonText: '<i class="bi bi-send me-2"></i> Enviar',
            cancelButtonText: 'Cancelar',
            preConfirm: () => {
                const email = document.getElementById('swal-email').value;
                const assunto = document.getElementById('swal-assunto').value;
                const mensagem = document.getElementById('swal-mensagem').value;
                const anexarPdf = document.getElementById('swal-anexar-pdf').checked;

                if (!email) {
                    Swal.showValidationMessage('Por favor, informe o e-mail do destinatário');
                    return false;
                }

                return {
                    email,
                    assunto,
                    mensagem,
                    anexarPdf
                };
            }
        });

        if (formValues) {
            // Mostrar loading
            Swal.fire({
                title: 'Enviando...',
                text: 'Por favor, aguarde',
                allowOutsideClick: false,
                allowEscapeKey: false,
                didOpen: () => {
                    Swal.showLoading();
                }
            });

            // Enviar e-mail
            $.ajax({
                url: appUrl('/compras/enviar-email'),
                type: 'POST',
                data: {
                    compra_id: compraId,
                    email: formValues.email,
                    assunto: formValues.assunto,
                    mensagem: formValues.mensagem,
                    anexar_pdf: formValues.anexarPdf
                },
                dataType: 'json',
                success: function (response) {
                    if (response.success) {
                        Swal.fire({
                            icon: 'success',
                            title: 'Sucesso!',
                            text: response.message
                        });
                    } else {
                        Swal.fire({
                            icon: 'error',
                            title: 'Erro',
                            text: response.message
                        });
                    }
                },
                error: function () {
                    Swal.fire({
                        icon: 'error',
                        title: 'Erro',
                        text: 'Erro ao enviar e-mail. Tente novamente.'
                    });
                }
            });
        }
    }

    async function deletarCompra(id) {
        Swal.close();

        const confirmed = await confirmDelete('Tem certeza que deseja excluir esta compra?');
        if (confirmed) {
            $.ajax({
                url: appUrl('/compras/deletar'),
                type: 'POST',
                data: {
                    id: id
                },
                dataType: 'json',
                success: function (response) {
                    if (response.success) {
                        Toast.success(response.message);
                        setTimeout(() => location.reload(), 1500);
                    } else {
                        Toast.error(response.message);
                    }
                }
            });
        }
    }

    // Monitor de NF-e (Manifesto do Destinatário)
    async function abrirMonitorNFe() {
        // Abrir modal SEM VALIDAR EMPRESA - usar empresa_id vazio ou null
        // O backend vai buscar empresas e usar a primeira disponível
        consultarMonitorNFe(null);
    }

    // Função para consultar monitor NF-e para uma empresa específica
    function consultarMonitorNFe(empresaId) {
        // Mostrar loading
        Swal.fire({
            title: 'Carregando...',
            html: 'Consultando NF-e na SEFAZ...',
            allowOutsideClick: false,
            allowEscapeKey: false,
            didOpen: () => {
                Swal.showLoading();
            }
        });

        // Buscar NF-e pendentes - se empresaId for null, backend usa primeira empresa disponível
        const params = empresaId ? { empresa_id: empresaId } : {};

        $.ajax({
            url: appUrl('/compras/monitor-nfe'),
            type: 'GET',
            data: params,
            dataType: 'json',
            success: function (response) {
                console.log('Response Monitor NF-e:', response);
                if (response.success) {
                    mostrarModalMonitorNFe(response.data);
                } else {
                    // Mesmo com erro, tentar abrir modal vazio
                    mostrarModalMonitorNFe({
                        nfes: [],
                        empresa: { id: null, nome: 'Erro', cnpj: '' },
                        empresas: [],
                        aviso: response.message || 'Erro ao consultar NF-e'
                    });
                }
            },
            error: function (xhr) {
                const response = xhr.responseJSON || {};
                // Mesmo com erro, tentar abrir modal vazio
                mostrarModalMonitorNFe({
                    nfes: [],
                    empresa: { id: null, nome: 'Erro', cnpj: '' },
                    empresas: [],
                    aviso: response.message || 'Erro ao consultar SEFAZ'
                });
            }
        });
    }

    function mostrarModalMonitorNFe(data) {
        console.log('Data recebida no modal:', data);
        console.log('Empresa:', data.empresa);
        console.log('CNPJ:', data.empresa?.cnpj);
        console.log('Nome:', data.empresa?.nome);

        const nfes = data.nfes || [];
        const empresa = data.empresa || {};
        const empresas = data.empresas || [];
        const aviso = data.aviso || '';

        // Criar opções do seletor de empresas - sempre mostrar se houver empresas
        let seletorEmpresasHtml = '';
        if (empresas.length > 0) {
            let opcoesEmpresas = '<option value="">Selecione uma empresa...</option>';
            empresas.forEach(emp => {
                const nome = emp.razao_social || emp.nome_fantasia || 'Empresa';
                const cnpj = emp.cnpj ? emp.cnpj.replace(/^(\d{2})(\d{3})(\d{3})(\d{4})(\d{2})$/, '$1.$2.$3/$4-$5') : '';
                const selected = (emp.id == empresa.id) ? 'selected' : '';
                opcoesEmpresas += `<option value="${emp.id}" ${selected}>${nome}${cnpj ? ' - ' + cnpj : ''}</option>`;
            });

            seletorEmpresasHtml = `
            <div style="margin-top: 1rem; padding: 0.75rem; background: rgba(255,255,255,0.1); border-radius: 4px;">
                <label style="display: block; margin-bottom: 0.5rem; font-size: 0.9rem; opacity: 0.9; font-weight: 500;">Empresa:</label>
                <select id="seletor_empresa_monitor" class="form-select" style="background: white; color: #333; width: 100%;">
                    ${opcoesEmpresas}
                </select>
            </div>
        `;
        }

        // Montar header - organizado simetricamente
        const headerHtml = `
        <div style="background: linear-gradient(135deg, #17a2b8 0%, #138496 100%); color: white; padding: 1.5rem; margin: -1rem -1rem 0 -1rem; border-radius: 8px 8px 0 0; position: relative;">
            <div style="display: flex; justify-content: space-between; align-items: flex-start; margin-bottom: 0.5rem;">
                <div style="flex: 1;">
                    <h3 style="margin: 0; font-size: 1.5rem; font-weight: 600;">
                        <i class="bi bi-receipt-cutoff me-2"></i>
                        Monitor de NF-e
                    </h3>
                </div>
                <button onclick="importarNFeManual()" style="background: rgba(255,255,255,0.2); border: 1px solid rgba(255,255,255,0.3); color: white; padding: 0.5rem 0.75rem; border-radius: 4px; font-size: 0.85rem; cursor: pointer; transition: all 0.2s; white-space: nowrap;" onmouseover="this.style.background='rgba(255,255,255,0.3)'" onmouseout="this.style.background='rgba(255,255,255,0.2)'">
                    <i class="bi bi-upload me-1"></i> Importar Manual
                </button>
            </div>
            <p style="margin: 0; opacity: 0.9; font-size: 0.95rem; padding-bottom: 0.5rem;">
                CNPJ: ${empresa.cnpj || 'Não configurado'} | ${empresa.nome || 'Empresa'}
            </p>
            ${seletorEmpresasHtml}
        </div>
    `;

        // Aviso de certificado ou limite SEFAZ
        let avisoHtml = '';
        if (aviso) {
            // Verificar se é mensagem de limite SEFAZ (não mostrar link de certificado)
            const isLimiteSefaz = aviso.includes('SEFAZ limita') || aviso.includes('próxima consulta');

            avisoHtml = `
            <div style="background: #fff3cd; border: 1px solid #ffc107; padding: 1rem; margin: 1rem; border-radius: 8px; color: #856404;">
                <i class="bi bi-exclamation-triangle me-2"></i>
                <strong>Atenção:</strong> ${aviso}
                ${!isLimiteSefaz ? `<br><small>Configure o certificado digital em <a href="<?= url('/empresas') ?>">Gestão de Empresas</a></small>` : ''}
            </div>
        `;
        }

        // Montar tabela de NF-e
        let tabelaHtml = '';

        if (nfes.length === 0) {
            tabelaHtml = `
            <div style="text-align: center; padding: 3rem; color: #6c757d;">
                <i class="bi bi-inbox" style="font-size: 4rem; opacity: 0.3;"></i>
                <p style="margin-top: 1rem; font-size: 1.1rem;">Nenhuma NF-e pendente</p>
                <small>Não há notas fiscais aguardando manifestação</small>
            </div>
        `;
        } else {
            tabelaHtml = `
            <div style="max-height: 500px; overflow-y: auto; margin-top: 1rem;">
                <table class="table table-hover">
                    <thead style="position: sticky; top: 0; background: white; z-index: 1;">
                        <tr>
                            <th width="50">#</th>
                            <th>Fornecedor</th>
                            <th>Chave</th>
                            <th>Valor</th>
                            <th>Data</th>
                            <th>Status</th>
                            <th class="text-center" width="150">Ações</th>
                        </tr>
                    </thead>
                    <tbody>
        `;

            nfes.forEach((nfe, index) => {
                const statusBadge = {
                    'pendente': '<span class="badge bg-warning">Pendente</span>',
                    'confirmada': '<span class="badge bg-success">Confirmada</span>',
                    'desconhecida': '<span class="badge bg-secondary">Desconhecida</span>',
                    'nao_realizada': '<span class="badge bg-danger">Não Realizada</span>'
                };

                // Formatar CNPJ se existir
                let cnpjFormatado = nfe.cnpj_fornecedor || '-';
                if (cnpjFormatado !== '-' && cnpjFormatado.length === 14) {
                    cnpjFormatado = cnpjFormatado.replace(/^(\d{2})(\d{3})(\d{3})(\d{4})(\d{2})$/, '$1.$2.$3/$4-$5');
                }

                // Formatar chave NFe (mostrar últimos 8 dígitos)
                let chaveExibida = '-';
                if (nfe.chave_nfe) {
                    if (nfe.chave_nfe.length > 20) {
                        chaveExibida = nfe.chave_nfe.substring(0, 8) + '...' + nfe.chave_nfe.substring(nfe.chave_nfe.length - 8);
                    } else {
                        chaveExibida = nfe.chave_nfe;
                    }
                }

                // Montar ações - apenas ícones
                let acoesHtml = '-';
                if (nfe.status === 'pendente' || !nfe.status) {
                    acoesHtml = `
                    <div class="btn-group btn-group-sm" role="group">
                        <button class="btn btn-info btn-sm" onclick="manifestarNFe('${nfe.chave_nfe || ''}', 'ciencia')" title="Ciência da Operação (libera XML completo)">
                            <i class="bi bi-eye"></i>
                        </button>
                        <button class="btn btn-success btn-sm" onclick="manifestarNFe('${nfe.chave_nfe || ''}', 'confirmar')" title="Confirmar Operação">
                            <i class="bi bi-check-circle"></i>
                        </button>
                        <button class="btn btn-secondary btn-sm" onclick="manifestarNFe('${nfe.chave_nfe || ''}', 'desconhecimento')" title="Desconhecimento">
                            <i class="bi bi-question-circle"></i>
                        </button>
                        <button class="btn btn-danger btn-sm" onclick="manifestarNFe('${nfe.chave_nfe || ''}', 'nao_realizada')" title="Operação Não Realizada">
                            <i class="bi bi-x-circle"></i>
                        </button>
                    </div>
                `;
                } else if (nfe.status === 'confirmada') {
                    // Se já foi importada, não mostrar botão de importar
                    if (nfe.ja_importada) {
                        acoesHtml = `
                        <div class="btn-group btn-group-sm" role="group">
                            <button class="btn btn-danger btn-sm" onclick="downloadNFePdf('${nfe.chave_nfe || ''}')" title="Download PDF (DANFE)">
                                <i class="bi bi-file-pdf"></i>
                            </button>
                            <button class="btn btn-secondary btn-sm" onclick="downloadNFeXml('${nfe.chave_nfe || ''}')" title="Download XML">
                                <i class="bi bi-file-code"></i>
                            </button>
                        </div>
                    `;
                    } else {
                        acoesHtml = `
                        <div class="btn-group btn-group-sm" role="group">
                            <button class="btn btn-info btn-sm" onclick="importarNFeManual('${nfe.chave_nfe || ''}')" title="Importar NFe">
                                <i class="bi bi-download"></i>
                            </button>
                            <button class="btn btn-danger btn-sm" onclick="downloadNFePdf('${nfe.chave_nfe || ''}')" title="Download PDF (DANFE)">
                                <i class="bi bi-file-pdf"></i>
                            </button>
                            <button class="btn btn-secondary btn-sm" onclick="downloadNFeXml('${nfe.chave_nfe || ''}')" title="Download XML">
                                <i class="bi bi-file-code"></i>
                            </button>
                        </div>
                    `;
                    }
                }

                tabelaHtml += `
                <tr>
                    <td>${index + 1}</td>
                    <td>
                        <strong>${nfe.razao_social || nfe.razao_social_fornecedor || '-'}</strong><br>
                        <small class="text-muted">CNPJ: ${cnpjFormatado}</small>
                    </td>
                    <td><small style="font-family: monospace;" title="${nfe.chave_nfe || ''}">${chaveExibida}</small></td>
                    <td><strong>R$ ${parseFloat(nfe.valor || nfe.valor_total || 0).toFixed(2).replace('.', ',')}</strong></td>
                    <td>${nfe.data_emissao || '-'}</td>
                    <td>${statusBadge[nfe.status || 'pendente'] || '<span class="badge bg-secondary">-</span>'}</td>
                    <td class="text-center">${acoesHtml}</td>
                </tr>
            `;
            });

            tabelaHtml += `
                    </tbody>
                </table>
            </div>
        `;
        }

        // Montar rodapé com estatísticas
        const rodapeHtml = `
        <div style="background: #f8f9fa; padding: 1rem; margin: 1rem -1rem -1rem -1rem; border-radius: 0 0 8px 8px; border-top: 1px solid #dee2e6;">
            <div class="row text-center">
                <div class="col">
                    <strong style="font-size: 1.5rem; color: #17a2b8;">${nfes.length}</strong>
                    <p style="margin: 0; font-size: 0.85rem; color: #6c757d;">Total de NF-e</p>
                </div>
                <div class="col">
                    <strong style="font-size: 1.5rem; color: #ffc107;">${nfes.filter(n => n.status === 'pendente').length}</strong>
                    <p style="margin: 0; font-size: 0.85rem; color: #6c757d;">Pendentes</p>
                </div>
                <div class="col">
                    <strong style="font-size: 1.5rem; color: #28a745;">${nfes.filter(n => n.status === 'confirmada').length}</strong>
                    <p style="margin: 0; font-size: 0.85rem; color: #6c757d;">Confirmadas</p>
                </div>
            </div>
        </div>
    `;

        Swal.fire({
            html: headerHtml + avisoHtml + tabelaHtml + rodapeHtml,
            width: '95%',
            heightAuto: false,
            showConfirmButton: false,
            showCloseButton: true,
            customClass: {
                popup: 'swal-monitor-nfe'
            },
            didOpen: () => {
                // Ajustar altura após abrir o modal (+15px como solicitado)
                const popup = document.querySelector('.swal-monitor-nfe');
                if (popup) {
                    popup.style.height = 'calc(80vh + 15px)';
                    popup.style.maxHeight = 'calc(80vh + 15px)';
                }

                // Adicionar evento ao seletor de empresas para buscar notas automaticamente
                const seletor = document.getElementById('seletor_empresa_monitor');
                if (seletor) {
                    seletor.addEventListener('change', function () {
                        const novaEmpresaId = this.value;
                        if (novaEmpresaId) {
                            // Recarregar notas para a empresa selecionada
                            buscarNotasEmpresaSelecionada(novaEmpresaId);
                        }
                    });
                }
            }
        });
    }

    // Função para buscar notas de uma empresa selecionada sem fechar o modal
    function buscarNotasEmpresaSelecionada(empresaId) {
        // Mostrar loading dentro do modal atual
        Swal.fire({
            title: 'Carregando...',
            html: 'Consultando NF-e para a empresa selecionada...',
            allowOutsideClick: false,
            allowEscapeKey: false,
            didOpen: () => {
                Swal.showLoading();
            }
        });

        // Buscar NF-e para a empresa selecionada
        $.ajax({
            url: appUrl('/compras/monitor-nfe'),
            type: 'GET',
            data: { empresa_id: empresaId },
            dataType: 'json',
            success: function (response) {
                console.log('Response Monitor NF-e (empresa selecionada):', response);
                if (response.success) {
                    // Atualizar modal com novos dados
                    mostrarModalMonitorNFe(response.data);
                } else {
                    Swal.fire({
                        icon: 'error',
                        title: 'Erro',
                        text: response.message || 'Erro ao consultar NF-e'
                    });
                }
            },
            error: function (xhr) {
                const response = xhr.responseJSON || {};
                Swal.fire({
                    icon: 'error',
                    title: 'Erro',
                    text: response.message || 'Erro ao consultar SEFAZ'
                });
            }
        });
    }

    async function manifestarNFe(chaveNfe, tipoManifesto) {
        const tipoLabels = {
            'ciencia': 'Ciência da Operação',
            'confirmar': 'Confirmação da Operação',
            'desconhecimento': 'Desconhecimento da Operação',
            'nao_realizada': 'Operação Não Realizada'
        };

        const result = await Swal.fire({
            title: 'Confirmar Manifestação',
            html: `
            <p>Deseja manifestar <strong>${tipoLabels[tipoManifesto]}</strong> para esta NF-e?</p>
            <p><small>Chave: ${chaveNfe}</small></p>
        `,
            icon: 'question',
            showCancelButton: true,
            confirmButtonText: 'Sim, Manifestar',
            cancelButtonText: 'Cancelar',
            confirmButtonColor: '#28a745'
        });

        if (result.isConfirmed) {
            Swal.fire({
                title: 'Enviando...',
                text: 'Enviando manifestação para SEFAZ',
                allowOutsideClick: false,
                didOpen: () => {
                    Swal.showLoading();
                }
            });

            $.ajax({
                url: appUrl('/compras/manifestar-nfe'),
                type: 'POST',
                data: {
                    chave_nfe: chaveNfe,
                    tipo_manifesto: tipoManifesto
                },
                dataType: 'json',
                success: function (response) {
                    if (response.success) {
                        Swal.fire({
                            icon: 'success',
                            title: 'Sucesso!',
                            text: response.message,
                            timer: 2000
                        }).then(() => {
                            abrirMonitorNFe(); // Recarregar monitor
                        });
                    } else {
                        Swal.fire({
                            icon: 'error',
                            title: 'Erro',
                            text: response.message
                        });
                    }
                },
                error: function (xhr) {
                    const response = xhr.responseJSON || {};
                    Swal.fire({
                        icon: 'error',
                        title: 'Erro',
                        text: response.message || 'Erro ao manifestar NF-e'
                    });
                }
            });
        }
    }

    function downloadNFePdf(chaveNfe) {
        if (!chaveNfe) {
            Swal.fire({
                icon: 'error',
                title: 'Erro',
                text: 'Chave da NF-e não informada'
            });
            return;
        }

        // Abrir em nova aba para download
        window.open(appUrl('/compras/nfe/pdf?chave=' + encodeURIComponent(chaveNfe)), '_blank');
    }

    function downloadNFeXml(chaveNfe) {
        if (!chaveNfe) {
            Swal.fire({
                icon: 'error',
                title: 'Erro',
                text: 'Chave da NF-e não informada'
            });
            return;
        }

        // Abrir em nova aba para download
        window.open(appUrl('/compras/nfe/xml?chave=' + encodeURIComponent(chaveNfe)), '_blank');
    }

    async function importarNFeManual(chaveNfe = null) {
        // Normalizar chave removendo formatação
        if (chaveNfe) {
            chaveNfe = chaveNfe.replace(/[^0-9]/g, '');
            importarPorChave(chaveNfe);
            return;
        }

        const { value: metodo } = await Swal.fire({
            title: '<i class="bi bi-upload me-2"></i> Importação Manual de NF-e',
            html: `
            <div class="text-start">
                <p class="text-muted mb-3">Escolha o método de importação:</p>
                <div class="d-grid gap-2">
                    <button type="button" id="btn-por-chave" class="btn btn-outline-primary text-start" onclick="escolherMetodoImportacao('chave')" style="padding: 0.75rem;">
                        <i class="bi bi-key me-2"></i>
                        <strong>Por Chave de Acesso</strong><br>
                        <small class="text-muted">Digite a chave de 44 dígitos da NF-e</small>
                    </button>
                    <button type="button" id="btn-por-arquivo" class="btn btn-outline-success text-start" onclick="escolherMetodoImportacao('arquivo')" style="padding: 0.75rem;">
                        <i class="bi bi-file-earmark-code me-2"></i>
                        <strong>Por Arquivo XML</strong><br>
                        <small class="text-muted">Envie o arquivo XML da NF-e</small>
                    </button>
                </div>
            </div>
        `,
            showCancelButton: true,
            cancelButtonText: 'Cancelar',
            showConfirmButton: false,
            width: '500px'
        });
    }

    let metodoImportacaoEscolhido = null;

    function escolherMetodoImportacao(metodo) {
        metodoImportacaoEscolhido = metodo;
        Swal.close();

        if (metodo === 'chave') {
            importarPorChave();
        } else if (metodo === 'arquivo') {
            importarPorArquivo();
        }
    }

    async function importarPorChave(chaveNfePreenchida = null) {
        // Normalizar chave removendo formatação
        let chaveNfe = chaveNfePreenchida ? chaveNfePreenchida.replace(/[^0-9]/g, '') : null;

        // Se não foi passada, pedir ao usuário
        if (!chaveNfe) {
            const { value: chave } = await Swal.fire({
                title: '<i class="bi bi-key me-2"></i> Importar por Chave',
                html: `
                <div class="text-start">
                    <p class="text-muted mb-3">Digite a chave de acesso da NF-e (44 dígitos):</p>
                    <input type="text" id="chave-nfe-manual" class="swal2-input" placeholder="Ex: 35210312345678000123550010000001234567890123" maxlength="44" style="width: 100%; font-family: monospace;">
                    <p class="text-muted mt-2 mb-0" style="font-size: 0.85rem;">
                        <i class="bi bi-info-circle me-1"></i>
                        A chave deve conter exatamente 44 números
                    </p>
                </div>
            `,
                showCancelButton: true,
                confirmButtonText: '<i class="bi bi-download me-1"></i> Importar',
                cancelButtonText: 'Cancelar',
                confirmButtonColor: '#17a2b8',
                preConfirm: () => {
                    let chave = document.getElementById('chave-nfe-manual').value.trim();
                    // Remover formatação
                    chave = chave.replace(/[^0-9]/g, '');
                    if (!chave) {
                        Swal.showValidationMessage('Digite a chave da NF-e');
                        return false;
                    }
                    if (chave.length !== 44 || !/^\d+$/.test(chave)) {
                        Swal.showValidationMessage('A chave deve conter exatamente 44 números');
                        return false;
                    }
                    return chave;
                }
            });

            if (!chave) {
                return;
            }
            chaveNfe = chave;
        }

        if (chaveNfe) {
            const loadingSwal = Swal.fire({
                title: 'Importando...',
                text: 'Buscando NF-e no SEFAZ',
                allowOutsideClick: false,
                allowEscapeKey: false,
                showConfirmButton: false,
                didOpen: () => {
                    Swal.showLoading();
                }
            });

            $.ajax({
                url: appUrl('/compras/importar-nfe-manual'),
                type: 'POST',
                data: {
                    chave_nfe: chaveNfe
                },
                dataType: 'json',
                success: function (response) {
                    // Fechar o loading primeiro
                    Swal.close();

                    // Pequeno delay para garantir que o loading fechou
                    setTimeout(() => {
                        if (response.success) {
                            // Se o fornecedor existe e tem XML, mostrar animação e redirecionar para relacionamento
                            if (response.data && response.data.redirect_to_relacionamento && response.data.compra_id) {
                                Swal.fire({
                                    title: 'Gerando Lançamento',
                                    text: 'Aguarde...',
                                    allowOutsideClick: false,
                                    allowEscapeKey: false,
                                    showConfirmButton: false,
                                    didOpen: () => {
                                        Swal.showLoading();
                                    }
                                });

                                // Redirecionar para tela de relacionamento
                                setTimeout(() => {
                                    window.location.href = appUrl('/compras/relacionar-produtos-xml?compra_id=' + response.data.compra_id);
                                }, 500);
                                return;
                            }

                            // Verificar se o fornecedor não foi encontrado
                            if (response.data && response.data.fornecedor_nao_encontrado) {
                                Swal.fire({
                                    icon: 'question',
                                    title: 'Fornecedor não encontrado',
                                    html: `
                                        <div class="text-start">
                                            <p><strong>CNPJ:</strong> ${response.data.dados_emitente ? response.data.dados_emitente.cnpj : response.data.cnpj || 'N/A'}</p>
                                            <p><strong>Nome:</strong> ${response.data.dados_emitente ? response.data.dados_emitente.nome : response.data.nome || 'Não informado'}</p>
                                            <p class="mt-3">${response.data.mensagem}</p>
                                        </div>
                                    `,
                                    showCancelButton: true,
                                    confirmButtonText: '<i class="bi bi-person-plus me-1"></i> Sim, cadastrar',
                                    cancelButtonText: 'Cancelar',
                                    confirmButtonColor: '#28a745',
                                    cancelButtonColor: '#6c757d',
                                    allowOutsideClick: false,
                                    allowEscapeKey: true
                                }).then((result) => {
                                    if (result.isConfirmed) {
                                        // Cadastrar fornecedor automaticamente
                                        Swal.fire({
                                            title: 'Cadastrando...',
                                            text: 'Cadastrando fornecedor e criando compra',
                                            allowOutsideClick: false,
                                            allowEscapeKey: false,
                                            showConfirmButton: false,
                                            didOpen: () => {
                                                Swal.showLoading();
                                            }
                                        });

                                        $.ajax({
                                            url: appUrl('/compras/cadastrar-fornecedor-e-criar-compra'),
                                            type: 'POST',
                                            data: {
                                                cnpj: response.data.dados_emitente ? response.data.dados_emitente.cnpj : response.data.cnpj,
                                                dados_emitente: response.data.dados_emitente || {
                                                    cnpj: response.data.cnpj,
                                                    nome: response.data.nome
                                                },
                                                chave_nfe: chaveNfe
                                            },
                                            dataType: 'json',
                                            success: function (responseCadastro) {
                                                Swal.close();
                                                if (responseCadastro.success) {
                                                    Swal.fire({
                                                        icon: 'success',
                                                        title: 'Sucesso!',
                                                        text: responseCadastro.message,
                                                        timer: 2000,
                                                        showConfirmButton: false
                                                    }).then(() => {
                                                        abrirMonitorNFe();
                                                        if (responseCadastro.data && responseCadastro.data.compra_id) {
                                                            // Se tem XML, redirecionar para tela de relacionamento
                                                            if (responseCadastro.data.redirect_to_relacionamento) {
                                                                window.location.href = appUrl('/compras/relacionar-produtos-xml?compra_id=' + responseCadastro.data.compra_id);
                                                            } else {
                                                                window.location.href = appUrl('/compras/editar?id=' + responseCadastro.data.compra_id);
                                                            }
                                                        } else {
                                                            window.location.href = appUrl('/compras');
                                                        }
                                                    });
                                                } else {
                                                    Swal.fire({
                                                        icon: 'error',
                                                        title: 'Erro',
                                                        text: responseCadastro.message
                                                    });
                                                }
                                            },
                                            error: function (xhr) {
                                                Swal.close();
                                                const responseError = xhr.responseJSON || {};
                                                Swal.fire({
                                                    icon: 'error',
                                                    title: 'Erro',
                                                    text: responseError.message || 'Erro ao cadastrar fornecedor'
                                                });
                                            }
                                        });
                                    } else {
                                        // Fechar modal do monitor se estiver aberto
                                        Swal.close();
                                        abrirMonitorNFe();
                                    }
                                });
                            } else {
                                // Fornecedor encontrado ou compra criada
                                Swal.fire({
                                    icon: 'success',
                                    title: 'Importado!',
                                    text: response.message,
                                    timer: 2000,
                                    showConfirmButton: false
                                }).then(() => {
                                    abrirMonitorNFe();
                                    if (response.data && response.data.compra_id) {
                                        // Se tem XML, redirecionar para tela de relacionamento
                                        if (response.data.redirect_to_relacionamento) {
                                            window.location.href = appUrl('/compras/relacionar-produtos-xml?compra_id=' + response.data.compra_id);
                                        } else {
                                            window.location.href = appUrl('/compras/editar?id=' + response.data.compra_id);
                                        }
                                    } else {
                                        window.location.href = appUrl('/compras');
                                    }
                                });
                            }
                        } else {
                            Swal.fire({
                                icon: 'error',
                                title: 'Erro',
                                html: response.message
                            });
                        }
                    }, 100);
                },
                error: function (xhr) {
                    Swal.close();
                    const response = xhr.responseJSON || {};
                    console.error('Erro ao importar NF-e:', response);
                    console.error('Status:', xhr.status);
                    console.error('Response Text:', xhr.responseText);

                    let errorMsg = response.message || 'Erro ao importar NF-e';
                    if (response.errors && typeof response.errors === 'object') {
                        if (response.errors.exception) {
                            errorMsg += '<br><br><strong>Detalhes:</strong><br>' + response.errors.exception;
                        } else {
                            const errorList = Object.values(response.errors).flat();
                            if (errorList.length > 0) {
                                errorMsg += '<br><br><strong>Detalhes:</strong><br>' + errorList.join('<br>');
                            }
                        }
                    }

                    Swal.fire({
                        icon: 'error',
                        title: 'Erro ao importar NF-e',
                        html: errorMsg,
                        width: '600px'
                    });
                }
            });
        }
    }

    async function importarPorArquivo() {
        const { value: formData } = await Swal.fire({
            title: '<i class="bi bi-file-earmark-code me-2"></i> Importar por Arquivo XML',
            html: `
            <div class="text-start">
                <p class="text-muted mb-3">Selecione o arquivo XML da NF-e:</p>
              <input type="file" id="xml-file-upload" class="form-control" accept=".xml,application/xml,text/xml" style="width: 100%;">
                <p class="text-muted mt-2 mb-0" style="font-size: 0.85rem;">
                    <i class="bi bi-info-circle me-1"></i>
                    Formatos aceitos: XML (máx. 10MB)
                </p>
            </div>
        `,
            showCancelButton: true,
            confirmButtonText: '<i class="bi bi-upload me-1"></i> Importar',
            cancelButtonText: 'Cancelar',
            confirmButtonColor: '#28a745',
            preConfirm: () => {
                const fileInput = document.getElementById('xml-file-upload');
                const file = fileInput.files[0];

                if (!file) {
                    Swal.showValidationMessage('Selecione um arquivo XML');
                    return false;
                }

                if (file.size > 10 * 1024 * 1024) {
                    Swal.showValidationMessage('O arquivo é muito grande. Tamanho máximo: 10MB');
                    return false;
                }

                const formData = new FormData();
                formData.append('xml_file', file);
                return formData;
            }
        });

        if (formData) {
            Swal.fire({
                title: 'Importando...',
                text: 'Processando XML da NF-e',
                allowOutsideClick: false,
                didOpen: () => {
                    Swal.showLoading();
                }
            });

            $.ajax({
                url: appUrl('/compras/processar-xml'),
                type: 'POST',
                data: formData,
                processData: false,
                contentType: false,
                dataType: 'json',
                success: function (response) {
                    if (response.success) {
                        Swal.fire({
                            icon: 'success',
                            title: 'Importado!',
                            text: response.message,
                            confirmButtonText: 'Ver Compra',
                            showCancelButton: true,
                            cancelButtonText: 'Fechar'
                        }).then((result) => {
                            abrirMonitorNFe();
                            if (result.isConfirmed) {
                                window.location.href = appUrl(response.data?.redirect || '/compras');
                            }
                        });
                    } else {
                        Swal.fire({
                            icon: 'error',
                            title: 'Erro',
                            text: response.message || 'Erro ao processar XML'
                        });
                    }
                },
                error: function (xhr) {
                    const response = xhr.responseJSON || {};
                    Swal.fire({
                        icon: 'error',
                        title: 'Erro',
                        text: response.message || 'Erro ao processar XML'
                    });
                }
            });
        }
    }
</script>

<style>
    .swal-monitor-nfe {
        padding: 0 !important;
        width: 95% !important;
        max-width: 95% !important;
        height: calc(80vh + 15px) !important;
        max-height: calc(80vh + 15px) !important;
    }

    .swal-monitor-nfe .swal2-html-container {
        padding: 0 !important;
        margin: 0 !important;
        overflow: visible !important;
        max-height: calc(80vh - 185px);
    }
</style>

<?php
$content = ob_get_clean();
$activeMenu = 'compras';
include \ROOT_PATH . '/views/layouts/app.php';
?>