<?php

declare(strict_types=1);

namespace App\Services;

use Mpdf\Mpdf;
use Exception;

/**
 * Serviço para Geração de PDFs
 * Utiliza a biblioteca mPDF
 */
class PdfService
{
    private Mpdf $mpdf;

    /**
     * Inicializa o mPDF
     */
    public function __construct(array $config = [])
    {
        $defaultConfig = [
            'mode' => 'utf-8',
            'format' => 'A4',
            'default_font_size' => 10,
            'default_font' => 'dejavusans',
            'margin_left' => 15,
            'margin_right' => 15,
            'margin_top' => 16,
            'margin_bottom' => 16,
            'margin_header' => 9,
            'margin_footer' => 9,
            'orientation' => 'P', // P-Portrait, L-Landscape
            'tempDir' => \ROOT_PATH . '/storage/cache/mpdf'
        ];

        $config = array_merge($defaultConfig, $config);

        // Cria diretório de cache se não existir
        if (!is_dir($config['tempDir'])) {
            mkdir($config['tempDir'], 0755, true);
        }

        $this->mpdf = new Mpdf($config);
    }

    /**
     * Gera PDF a partir de HTML
     *
     * @param string $html Conteúdo HTML
     * @param string $filename Nome do arquivo
     * @param string $output Tipo de saída: I-Inline, D-Download, F-File, S-String
     * @return mixed
     */
    public function gerarPdf(string $html, string $filename = 'documento.pdf', string $output = 'I')
    {
        try {
            $this->mpdf->WriteHTML($html);
            return $this->mpdf->Output($filename, $output);
        } catch (Exception $e) {
            error_log("Erro ao gerar PDF: " . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Gera PDF de relatório de vendas
     */
    public function gerarRelatorioVendas(array $vendas, array $filtros = []): string
    {
        $html = $this->montarHtmlRelatorioVendas($vendas, $filtros);
        return $this->gerarPdf($html, 'relatorio-vendas.pdf', 'I');
    }

    /**
     * Monta HTML do relatório de vendas
     */
    private function montarHtmlRelatorioVendas(array $vendas, array $filtros): string
    {
        $periodoInicio = $filtros['data_inicio'] ?? '';
        $periodoFim = $filtros['data_fim'] ?? '';
        $periodo = '';

        if ($periodoInicio && $periodoFim) {
            $periodo = date('d/m/Y', strtotime($periodoInicio)) . ' a ' . date('d/m/Y', strtotime($periodoFim));
        }

        $totalGeral = array_sum(array_column($vendas, 'total'));

        $html = '
        <style>
            body { font-family: Arial, sans-serif; }
            h1 { color: #333; text-align: center; }
            .header { text-align: center; margin-bottom: 20px; }
            .periodo { text-align: center; color: #666; margin-bottom: 30px; }
            table { width: 100%; border-collapse: collapse; margin-top: 20px; }
            th { background-color: #3788d8; color: white; padding: 10px; text-align: left; }
            td { border-bottom: 1px solid #ddd; padding: 8px; }
            tr:hover { background-color: #f5f5f5; }
            .text-right { text-align: right; }
            .total { font-weight: bold; background-color: #f0f0f0; }
            .footer { margin-top: 30px; text-align: center; color: #666; font-size: 10px; }
        </style>

        <div class="header">
            <h1>Relatório de Vendas</h1>
            ' . ($periodo ? '<div class="periodo">Período: ' . $periodo . '</div>' : '') . '
        </div>

        <table>
            <thead>
                <tr>
                    <th>Número</th>
                    <th>Data</th>
                    <th>Cliente</th>
                    <th>Status</th>
                    <th class="text-right">Valor</th>
                </tr>
            </thead>
            <tbody>';

        foreach ($vendas as $venda) {
            $html .= '
                <tr>
                    <td>' . htmlspecialchars($venda['numero'] ?? '#' . $venda['id']) . '</td>
                    <td>' . date('d/m/Y', strtotime($venda['data_venda'])) . '</td>
                    <td>' . htmlspecialchars($venda['cliente_name'] ?? 'Não informado') . '</td>
                    <td>' . htmlspecialchars($venda['status']) . '</td>
                    <td class="text-right">R$ ' . number_format($venda['total'], 2, ',', '.') . '</td>
                </tr>';
        }

        $html .= '
            </tbody>
            <tfoot>
                <tr class="total">
                    <td colspan="4" class="text-right">Total:</td>
                    <td class="text-right">R$ ' . number_format($totalGeral, 2, ',', '.') . '</td>
                </tr>
            </tfoot>
        </table>

        <div class="footer">
            Gerado em: ' . date('d/m/Y H:i:s') . ' | Aurion ERP
        </div>';

        return $html;
    }

    /**
     * Gera PDF de boleto (exemplo)
     */
    public function gerarBoleto(array $dados): string
    {
        $html = $this->montarHtmlBoleto($dados);
        return $this->gerarPdf($html, 'boleto.pdf', 'I');
    }

    /**
     * Monta HTML do boleto
     */
    private function montarHtmlBoleto(array $dados): string
    {
        // HTML simplificado do boleto
        return '
        <style>
            .boleto { border: 2px solid #000; padding: 20px; }
            .boleto-header { font-size: 14px; font-weight: bold; }
            .linha-digitavel { font-size: 16px; margin: 20px 0; text-align: center; }
            .barcode { text-align: center; margin: 20px 0; }
        </style>

        <div class="boleto">
            <div class="boleto-header">Banco: ' . htmlspecialchars($dados['banco']) . '</div>
            <div class="linha-digitavel">' . htmlspecialchars($dados['linha_digitavel']) . '</div>
            <table>
                <tr>
                    <td>Beneficiário: ' . htmlspecialchars($dados['beneficiario']) . '</td>
                </tr>
                <tr>
                    <td>Pagador: ' . htmlspecialchars($dados['pagador']) . '</td>
                </tr>
                <tr>
                    <td>Valor: R$ ' . number_format($dados['valor'], 2, ',', '.') . '</td>
                </tr>
                <tr>
                    <td>Vencimento: ' . date('d/m/Y', strtotime($dados['vencimento'])) . '</td>
                </tr>
            </table>
        </div>';
    }

    /**
     * Adiciona cabeçalho customizado
     */
    public function setHeader(string $html): void
    {
        $this->mpdf->SetHTMLHeader($html);
    }

    /**
     * Adiciona rodapé customizado
     */
    public function setFooter(string $html): void
    {
        $this->mpdf->SetHTMLFooter($html);
    }

    /**
     * Adiciona marca d'água
     */
    public function setWatermark(string $text): void
    {
        $this->mpdf->SetWatermarkText($text);
        $this->mpdf->showWatermarkText = true;
    }
}