<?php

namespace App\Services;

use PDO;
use App\Core\Database;

class InadimplenciaService
{
    private $db;
    private $tenantDb;

    public function __construct()
    {
        $this->db = Database::getInstance();
    }

    /**
     * Define a conexão do tenant
     */
    public function setTenantConnection(PDO $tenantDb)
    {
        $this->tenantDb = $tenantDb;
    }

    /**
     * Verifica se um cliente está inadimplente em qualquer empresa
     */
    public function isClienteInadimplenteUniversal($clienteId, $diasBloqueio = 0)
    {
        // Buscar todas as empresas ativas
        $stmt = $this->db->prepare("
            SELECT id, subdomain
            FROM tenants
            WHERE status = 'active'
        ");
        $stmt->execute();
        $empresas = $stmt->fetchAll(PDO::FETCH_ASSOC);

        foreach ($empresas as $empresa) {
            try {
                // Conectar ao banco da empresa
                $tenantDb = $this->getTenantConnection($empresa['subdomain']);

                if ($tenantDb) {
                    // Verificar inadimplência nesta empresa
                    if ($this->isClienteInadimplenteNaEmpresa($tenantDb, $clienteId, $diasBloqueio)) {
                        return [
                            'inadimplente' => true,
                            'empresa' => $empresa['subdomain'],
                            'motivo' => 'Cliente inadimplente na empresa ' . $empresa['subdomain']
                        ];
                    }
                }
            } catch (Exception $e) {
                // Log do erro mas continua verificando outras empresas
                error_log("Erro ao verificar inadimplência na empresa {$empresa['subdomain']}: " . $e->getMessage());
            }
        }

        return ['inadimplente' => false];
    }

    /**
     * Verifica se um cliente está inadimplente em uma empresa específica
     */
    private function isClienteInadimplenteNaEmpresa(PDO $tenantDb, $clienteId, $diasBloqueio = 0)
    {
        $dataLimite = date('Y-m-d', strtotime("-{$diasBloqueio} days"));

        $stmt = $tenantDb->prepare("
            SELECT COUNT(*) as total
            FROM accounts_receivable
            WHERE client_id = ?
            AND due_date <= ?
            AND status IN ('pending', 'overdue')
            AND amount > paid_amount
        ");

        $stmt->execute([$clienteId, $dataLimite]);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);

        return $result['total'] > 0;
    }

    /**
     * Obtém conexão com o banco do tenant
     */
    private function getTenantConnection($subdomain)
    {
        try {
            $dbName = "Systhema_tenant_{$subdomain}";
            $dsn = "mysql:host=localhost;dbname={$dbName};charset=utf8mb4";
            return new PDO($dsn, 'root', '', [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC
            ]);
        } catch (Exception $e) {
            return null;
        }
    }

    /**
     * Bloqueia cliente universalmente (marca em todas as empresas)
     */
    public function bloquearClienteUniversal($clienteId, $motivo = 'Inadimplência')
    {
        $empresas = $this->getTodasEmpresas();
        $resultados = [];

        foreach ($empresas as $empresa) {
            try {
                $tenantDb = $this->getTenantConnection($empresa['subdomain']);

                if ($tenantDb) {
                    // Atualizar status do cliente para bloqueado
                    $stmt = $tenantDb->prepare("
                        UPDATE clients
                        SET status = 'blocked',
                            blocked_reason = ?,
                            blocked_at = NOW()
                        WHERE id = ?
                    ");

                    $stmt->execute([$motivo, $clienteId]);
                    $resultados[$empresa['subdomain']] = 'sucesso';
                }
            } catch (Exception $e) {
                $resultados[$empresa['subdomain']] = 'erro: ' . $e->getMessage();
            }
        }

        return $resultados;
    }

    /**
     * Desbloqueia cliente universalmente
     */
    public function desbloquearClienteUniversal($clienteId)
    {
        $empresas = $this->getTodasEmpresas();
        $resultados = [];

        foreach ($empresas as $empresa) {
            try {
                $tenantDb = $this->getTenantConnection($empresa['subdomain']);

                if ($tenantDb) {
                    $stmt = $tenantDb->prepare("
                        UPDATE clients
                        SET status = 'active',
                            blocked_reason = NULL,
                            blocked_at = NULL
                        WHERE id = ?
                    ");

                    $stmt->execute([$clienteId]);
                    $resultados[$empresa['subdomain']] = 'sucesso';
                }
            } catch (Exception $e) {
                $resultados[$empresa['subdomain']] = 'erro: ' . $e->getMessage();
            }
        }

        return $resultados;
    }

    /**
     * Obtém todas as empresas ativas
     */
    private function getTodasEmpresas()
    {
        $stmt = $this->db->prepare("
            SELECT id, subdomain
            FROM tenants
            WHERE status = 'active'
        ");
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    /**
     * Verifica se o bloqueio universal está ativado para uma empresa
     */
    public function isBloqueioUniversalAtivo($companyId)
    {
        $tenantDb = $this->tenantDb;

        $stmt = $tenantDb->prepare("
            SELECT param_value
            FROM system_parameters
            WHERE param_key = 'bloqueio_universal_inadimplente'
            AND company_id = ?
        ");

        $stmt->execute([$companyId]);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);

        return $result && $result['param_value'] == '1';
    }

    /**
     * Obtém os dias de bloqueio configurados para uma empresa
     */
    public function getDiasBloqueio($companyId)
    {
        $tenantDb = $this->tenantDb;

        $stmt = $tenantDb->prepare("
            SELECT param_value
            FROM system_parameters
            WHERE param_key = 'dias_bloqueio_inadimplente'
            AND company_id = ?
        ");

        $stmt->execute([$companyId]);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);

        return $result ? (int) $result['param_value'] : 0;
    }
}
