<?php

declare(strict_types=1);

namespace App\Middleware;

use App\Core\Request;
use App\Core\Response;
use App\Core\Session;
use App\Core\TenantManager;
use PDO;

/**
 * Middleware de Permissões
 * Verifica se o usuário tem permissão para acessar o módulo
 */
class PermissionMiddleware
{
    private Session $session;
    private TenantManager $tenantManager;
    private string $module;

    public function __construct(string $module = '')
    {
        $config = require \ROOT_PATH . '/config/app.php';
        $this->session = new Session($config['session']);
        $this->tenantManager = new TenantManager();
        $this->module = $module;
    }

    /**
     * Processa a requisição
     */
    public function handle(Request $request, Response $response): bool
    {
        if (!$this->session->isAuthenticated()) {
            $response->unauthorized('Você precisa estar autenticado');
            return false;
        }

        $user = $this->session->getUser();

        // Admin tem acesso a tudo
        if ($user['role'] === 'admin') {
            return true;
        }

        // Verifica permissões específicas
        if (!$this->hasPermission($user['id'], $this->module, $request->method())) {
            if ($request->isAjax() || $request->isJson()) {
                $response->forbidden('Você não tem permissão para acessar este recurso');
                return false;
            }

            $response->redirect('/dashboard');
            return false;
        }

        return true;
    }

    /**
     * Verifica se o usuário tem permissão
     */
    private function hasPermission(int $userId, string $module, string $method): bool
    {
        try {
            $db = $this->tenantManager->getTenantConnection();

            if (!$db) {
                return false;
            }

            $stmt = $db->prepare("
                SELECT
                    can_view, can_create, can_edit, can_delete
                FROM user_permissions
                WHERE user_id = :user_id AND module = :module
                LIMIT 1
            ");

            $stmt->execute([
                'user_id' => $userId,
                'module' => $module
            ]);

            $permission = $stmt->fetch();

            if (!$permission) {
                return false;
            }

            // Verifica permissão baseada no método HTTP
            return match($method) {
                'GET' => (bool) $permission['can_view'],
                'POST' => (bool) ($permission['can_create'] || $permission['can_edit']),
                'PUT', 'PATCH' => (bool) $permission['can_edit'],
                'DELETE' => (bool) $permission['can_delete'],
                default => false
            };

        } catch (\Exception $e) {
            error_log("Erro ao verificar permissão: " . $e->getMessage());
            return false;
        }
    }
}

