<?php

namespace App\Middleware;

use App\Core\Request;
use App\Core\Response;
use App\Services\InadimplenciaService;

class InadimplenciaMiddleware
{
    private $inadimplenciaService;

    public function __construct()
    {
        $this->inadimplenciaService = new InadimplenciaService();
    }

    public function handle(Request $request, Response $response, $next)
    {
        // Verificar se é uma operação que precisa de verificação de inadimplência
        if ($this->precisaVerificarInadimplencia($request)) {

            // Obter ID do cliente da requisição
            $clienteId = $this->getClienteIdFromRequest($request);

            if ($clienteId) {
                // Verificar se bloqueio universal está ativo
                $companyId = $_SESSION['company_id'] ?? null;

                if ($companyId && $this->inadimplenciaService->isBloqueioUniversalAtivo($companyId)) {
                    $diasBloqueio = $this->inadimplenciaService->getDiasBloqueio($companyId);

                    // Verificar inadimplência universal
                    $resultado = $this->inadimplenciaService->isClienteInadimplenteUniversal($clienteId, $diasBloqueio);

                    if ($resultado['inadimplente']) {
                        // Cliente inadimplente - bloquear operação
                        return $this->bloquearOperacao($response, $resultado['motivo']);
                    }
                }
            }
        }

        // Continuar com a próxima operação
        return $next($request, $response);
    }

    /**
     * Verifica se a operação precisa de verificação de inadimplência
     */
    private function precisaVerificarInadimplencia(Request $request)
    {
        $path = $request->path();
        $method = $request->method();

        // Operações que precisam de verificação
        $operacoesCriticas = [
            'POST' => [
                '/vendas',
                '/vendas/store',
                '/pedidos',
                '/pedidos/store',
                '/orcamentos',
                '/orcamentos/store'
            ],
            'PUT' => [
                '/vendas/',
                '/pedidos/',
                '/orcamentos/'
            ]
        ];

        foreach ($operacoesCriticas[$method] ?? [] as $operacao) {
            if (strpos($path, $operacao) === 0) {
                return true;
            }
        }

        return false;
    }

    /**
     * Extrai o ID do cliente da requisição
     */
    private function getClienteIdFromRequest(Request $request)
    {
        // Tentar obter do POST
        $clienteId = $request->input('client_id') ?? $request->input('cliente_id');

        if ($clienteId) {
            return $clienteId;
        }

        // Tentar obter da URL (para operações PUT)
        $path = $request->path();
        if (preg_match('/\/(\d+)\/edit$/', $path, $matches)) {
            // Para edição, precisaríamos buscar o cliente_id do registro
            // Isso seria implementado conforme necessário
        }

        return null;
    }

    /**
     * Bloqueia a operação e retorna resposta de erro
     */
    private function bloquearOperacao(Response $response, $motivo)
    {
        if ($response->isAjax()) {
            return $response->json([
                'success' => false,
                'message' => 'Operação bloqueada: ' . $motivo,
                'error_type' => 'inadimplencia_universal',
                'details' => $motivo
            ], 403);
        }

        // Para requisições não-AJAX, redirecionar com mensagem
        $_SESSION['error'] = 'Operação bloqueada: ' . $motivo;
        return $response->redirect('/dashboard');
    }
}
