<?php

namespace App\Integrations;

use Exception;

/**
 * Gerenciador de Subdomínios
 * Automatiza a criação de subdomínios em diferentes painéis de controle
 */
class SubdomainManager
{
    private string $panelType;
    private array $config;

    public function __construct()
    {
        // Detectar tipo de painel
        $this->panelType = $_ENV['HOSTING_PANEL'] ?? 'wildcard'; // wildcard, cpanel, plesk, directadmin

        $this->config = [
            'base_domain' => $_ENV['BASE_DOMAIN'] ?? 'Systhema.com.br',
            'document_root' => $_ENV['DOCUMENT_ROOT'] ?? '/public_html',
            'cpanel' => [
                'api_url' => $_ENV['CPANEL_API_URL'] ?? '',
                'api_user' => $_ENV['CPANEL_API_USER'] ?? '',
                'api_token' => $_ENV['CPANEL_API_TOKEN'] ?? '',
            ],
            'plesk' => [
                'api_url' => $_ENV['PLESK_API_URL'] ?? '',
                'api_key' => $_ENV['PLESK_API_KEY'] ?? '',
            ],
            'directadmin' => [
                'api_url' => $_ENV['DIRECTADMIN_API_URL'] ?? '',
                'api_user' => $_ENV['DIRECTADMIN_API_USER'] ?? '',
                'api_password' => $_ENV['DIRECTADMIN_API_PASSWORD'] ?? '',
            ],
        ];
    }

    /**
     * Cria um subdomínio automaticamente
     *
     * @param string $subdomain Subdomínio a ser criado
     * @return array ['success' => bool, 'message' => string, 'url' => string|null]
     */
    public function createSubdomain(string $subdomain): array
    {
        try {
            // Validação
            if (!preg_match('/^[a-z0-9-]+$/', $subdomain)) {
                return [
                    'success' => false,
                    'message' => 'Subdomínio inválido. Use apenas letras minúsculas, números e hífen.'
                ];
            }

            error_log("SubdomainManager - Criando subdomínio: {$subdomain}");

            // Escolhe método baseado no painel
            switch ($this->panelType) {
                case 'wildcard':
                    return $this->createViaWildcard($subdomain);

                case 'cpanel':
                    return $this->createViaCpanel($subdomain);

                case 'plesk':
                    return $this->createViaPlesk($subdomain);

                case 'directadmin':
                    return $this->createViaDirectAdmin($subdomain);

                case 'manual':
                    return $this->createViaManual($subdomain);

                default:
                    return [
                        'success' => false,
                        'message' => 'Tipo de painel não suportado: ' . $this->panelType
                    ];
            }

        } catch (Exception $e) {
            error_log("SubdomainManager - Erro: " . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Erro ao criar subdomínio: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Método 1: Via Wildcard DNS (RECOMENDADO - Mais simples)
     *
     * Configura um DNS wildcard (*.Systhema.com.br) apontando para o servidor
     * O Apache/VirtualHost já está configurado para aceitar qualquer subdomínio
     * Não precisa criar nada no servidor, apenas garantir que o DNS está OK
     */
    private function createViaWildcard(string $subdomain): array
    {
        $url = "https://{$subdomain}." . $this->config['base_domain'];

        error_log("SubdomainManager - Wildcard: Subdomínio {$subdomain} disponível em {$url}");

        return [
            'success' => true,
            'message' => 'Subdomínio criado via Wildcard DNS',
            'url' => $url,
            'method' => 'wildcard'
        ];
    }

    /**
     * Método 2: Via cPanel API
     */
    private function createViaCpanel(string $subdomain): array
    {
        if (
            empty($this->config['cpanel']['api_url']) ||
            empty($this->config['cpanel']['api_user']) ||
            empty($this->config['cpanel']['api_token'])
        ) {

            return [
                'success' => false,
                'message' => 'Configuração do cPanel não encontrada. Use wildcard ou configure as variáveis CPANEL_* no .env'
            ];
        }

        // Tentar com addsubdomain (sem underscore) primeiro
        $url = $this->config['cpanel']['api_url'] . '/execute/SubDomain/addsubdomain';

        $data = [
            'domain' => $subdomain,
            'rootdomain' => $this->config['base_domain'],
            'dir' => $this->config['document_root'] . '/' . $subdomain
        ];

        // Fazer a requisição com timeout maior
        $ch = curl_init($url . '?' . http_build_query($data));
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HTTPHEADER => [
                'Authorization: cpanel ' . $this->config['cpanel']['api_user'] . ':' . $this->config['cpanel']['api_token']
            ],
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_SSL_VERIFYHOST => false,
            CURLOPT_TIMEOUT => 30, // Aumentar timeout para 30 segundos
        ]);

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $curlError = curl_error($ch);
        curl_close($ch);

        // Se deu timeout, verificar se o subdomínio foi criado
        if ($curlError || $httpCode === 0) {
            error_log("SubdomainManager - Timeout/erro ao criar subdomínio, verificando se foi criado...");

            // Verificar se o subdomínio existe
            $checkUrl = $this->config['cpanel']['api_url'] . '/execute/DomainInfo/list_domains';
            $ch = curl_init($checkUrl);
            curl_setopt_array($ch, [
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_HTTPHEADER => [
                    'Authorization: cpanel ' . $this->config['cpanel']['api_user'] . ':' . $this->config['cpanel']['api_token']
                ],
                CURLOPT_SSL_VERIFYPEER => false,
                CURLOPT_SSL_VERIFYHOST => false,
                CURLOPT_TIMEOUT => 10,
            ]);

            $checkResponse = curl_exec($ch);
            $checkHttpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);

            if ($checkHttpCode === 200) {
                $checkData = json_decode($checkResponse, true);
                if (isset($checkData['data']['sub_domains'])) {
                    $fullSubdomain = $subdomain . '.' . $this->config['base_domain'];
                    if (in_array($fullSubdomain, $checkData['data']['sub_domains'])) {
                        return [
                            'success' => true,
                            'message' => 'Subdomínio criado via cPanel API (verificação pós-timeout)',
                            'url' => "https://{$fullSubdomain}",
                            'method' => 'cpanel'
                        ];
                    }
                }
            }

            return [
                'success' => false,
                'message' => 'Timeout ao criar subdomínio: ' . $curlError
            ];
        }

        if ($httpCode === 200) {
            $result = json_decode($response, true);

            if (isset($result['status']) && $result['status'] === 1) {
                return [
                    'success' => true,
                    'message' => 'Subdomínio criado via cPanel API',
                    'url' => "https://{$subdomain}." . $this->config['base_domain'],
                    'method' => 'cpanel'
                ];
            }
        }

        return [
            'success' => false,
            'message' => 'Erro ao criar subdomínio no cPanel: ' . ($response ?: 'Erro desconhecido')
        ];
    }

    /**
     * Método 3: Via Plesk API
     */
    private function createViaPlesk(string $subdomain): array
    {
        // Implementação do Plesk API
        // Similar ao cPanel, mas usando a API do Plesk

        return [
            'success' => false,
            'message' => 'Plesk API ainda não implementada'
        ];
    }

    /**
     * Método 4: Via DirectAdmin API
     */
    private function createViaDirectAdmin(string $subdomain): array
    {
        // Implementação do DirectAdmin API

        return [
            'success' => false,
            'message' => 'DirectAdmin API ainda não implementada'
        ];
    }

    /**
     * Método 5: Via Script Manual (SSH/Comando)
     *
     * Executa script shell para criar subdomínio
     */
    private function createViaManual(string $subdomain): array
    {
        $scriptPath = ROOT_PATH . '/scripts/create_subdomain.sh';

        if (!file_exists($scriptPath)) {
            return [
                'success' => false,
                'message' => 'Script de criação manual não encontrado'
            ];
        }

        // Executa script (requer permissões adequadas)
        $command = escapeshellcmd($scriptPath) . ' ' . escapeshellarg($subdomain);
        $output = shell_exec($command . ' 2>&1');

        if ($output && strpos($output, 'success') !== false) {
            return [
                'success' => true,
                'message' => 'Subdomínio criado via script manual',
                'url' => "https://{$subdomain}." . $this->config['base_domain'],
                'method' => 'manual'
            ];
        }

        return [
            'success' => false,
            'message' => 'Erro ao executar script: ' . ($output ?: 'Nenhuma saída')
        ];
    }

    /**
     * Verifica se um subdomínio já existe
     */
    public function subdomainExists(string $subdomain): bool
    {
        // Verifica se o DNS responde
        $host = $subdomain . '.' . $this->config['base_domain'];
        $ip = gethostbyname($host);

        return $ip !== $host; // Se retornou IP diferente do hostname, existe
    }

    /**
     * Remove um subdomínio
     */
    public function deleteSubdomain(string $subdomain): array
    {
        // Implementar remoção se necessário

        return [
            'success' => false,
            'message' => 'Remoção de subdomínio ainda não implementada'
        ];
    }
}

