<?php

namespace App\Services;

use NFePHP\CTe\Tools;
use NFePHP\CTe\Common\Standardize;
use NFePHP\Common\Certificate;
use NFePHP\DA\CTe\Dacte;

class CTEService
{
    /**
     * Log personalizado no projeto
     */
    private function log($message)
    {
        $logFile = __DIR__ . "/../../logs/cte_debug.log";
        $logDir = dirname($logFile);

        if (!is_dir($logDir)) {
            mkdir($logDir, 0755, true);
        }

        $timestamp = date('Y-m-d H:i:s');
        $logMessage = "[$timestamp] $message" . PHP_EOL;
        file_put_contents($logFile, $logMessage, FILE_APPEND | LOCK_EX);

        error_log($message);
    }

    /**
     * Criar estrutura de diretórios para CT-e
     */
    private function criarEstruturaDiretorios($cnpj)
    {
        $basePath = __DIR__ . "/../../arquivos/{$cnpj}/cte";
        $anoMes = date('Y_m');

        $diretorios = [
            "{$basePath}/xml/{$anoMes}/assinados",
            "{$basePath}/pdf/{$anoMes}/emitidos",
            "{$basePath}/eventos/cancelamento",
            "{$basePath}/eventos/carta_correcao"
        ];

        foreach ($diretorios as $dir) {
            if (!is_dir($dir)) {
                mkdir($dir, 0755, true);
            }
        }
    }

    /**
     * Emite CT-e (Conhecimento de Transporte Eletrônico)
     */
    public function emitir($dados)
    {
        try {
            $this->log("🚚 EMISSÃO DE CT-e (Modelo 57)");

            // Normalizar estrutura
            if (isset($dados['cte'])) {
                $this->log("🔧 Normalizando estrutura do JSON com objeto 'cte'");
                $dados['numero'] = $dados['cte']['numero'] ?? $dados['numero'] ?? '1';
                $dados['serie'] = $dados['cte']['serie'] ?? $dados['serie'] ?? '001';
                $dados['data_emissao'] = $dados['cte']['data_emissao'] ?? $dados['data_emissao'] ?? date('Y-m-d H:i:s');
            }

            $this->criarEstruturaDiretorios($dados['emitente']['cnpj']);

            // Gerar XML
            $xml = $this->gerarXML($dados);

            // Assinar
            $xmlAssinado = $this->assinarXML($xml, $dados['emitente']);

            // Enviar para SEFAZ
            $resultado = $this->enviarParaSEFAZ($xmlAssinado, $dados);

            if ($resultado['success']) {
                $xmlParaPDF = $resultado['xml_protocolado'] ?? $xmlAssinado;

                if ($resultado['xml_protocolado']) {
                    $xmlPath = $this->salvarXML($resultado['xml_protocolado'], $dados);
                } else {
                    $xmlPath = $this->salvarXML($xmlAssinado, $dados);
                }

                $pdfPath = $this->gerarPDF($dados, $xmlParaPDF, $resultado['protocolo'] ?? null);

                return [
                    'success' => true,
                    'numero' => $dados['numero'],
                    'serie' => $dados['serie'],
                    'chave_acesso' => $resultado['chave_acesso'],
                    'protocolo' => $resultado['protocolo'],
                    'cStat' => $resultado['cStat'] ?? null,
                    'xml_path' => $xmlPath,
                    'pdf_path' => $pdfPath,
                    'data_emissao' => $dados['data_emissao'],
                    'valor_total' => $dados['valores']['valor_total_servico'],
                    'message' => 'CT-e emitido com sucesso'
                ];
            } else {
                return [
                    'success' => false,
                    'chave_acesso' => $resultado['chave_acesso'] ?? null,
                    'cStat' => $resultado['cStat'] ?? null,
                    'numero' => $dados['numero'],
                    'serie' => $dados['serie'],
                    'error' => $resultado['error']
                ];
            }
        } catch (\Exception $e) {
            $this->log("❌ ERRO na emissão: " . $e->getMessage());
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }

    /**
     * Gera XML do CT-e
     */
    private function gerarXML($dados)
    {
        $this->log("🚀 GERANDO XML DO CT-e (MODELO 57)");

        $certificadoPath = __DIR__ . "/../../certificados/empresas/{$dados['emitente']['cnpj']}/{$dados['emitente']['cnpj']}.pfx";

        if (!file_exists($certificadoPath)) {
            throw new \Exception("Certificado não encontrado: {$certificadoPath}");
        }

        $configCertificado = require __DIR__ . '/../../config_certificado.php';
        $certificado = \App\Core\CertificateHelper::loadPfx($certificadoPath, $configCertificado['senha_certificado']);

        $config = [
            "atualizacao" => date('Y-m-d H:i:s'),
            "tpAmb" => 1,
            "razaosocial" => $dados['emitente']['nome'],
            "cnpj" => preg_replace('/[^0-9]/', '', $dados['emitente']['cnpj']),
            "siglaUF" => $dados['emitente']['estado'],
            "schemes" => "PL_CTe_300",
            "versao" => '3.00'
        ];

        $configJson = json_encode($config);
        $tools = new Tools($configJson, $certificado);

        $cte = new \NFePHP\CTe\Make();

        // TAG infCte
        $std = new \stdClass();
        $std->versao = '3.00';
        $cte->taginfCTe($std);

        // TAG ide
        $std = new \stdClass();
        $std->cUF = $this->getCodigoUF($dados['emitente']['estado']);
        $std->cCT = str_pad(rand(1, 99999999), 8, '0', STR_PAD_LEFT);
        $std->CFOP = $dados['cfop'];
        $std->natOp = $dados['natureza_operacao'] ?? 'Prestação de serviço de transporte';
        $std->mod = 57; // Modelo 57 = CT-e
        $std->serie = intval($dados['serie']);
        $std->nCT = intval($dados['numero']);
        $std->dhEmi = date('Y-m-d\TH:i:sP', strtotime($dados['data_emissao']));
        $std->tpImp = $dados['tipo_impressao'] ?? 1; // 1=Retrato, 2=Paisagem
        $std->tpEmis = 1; // 1=Normal
        $std->tpAmb = 1; // Produção
        $std->tpCTe = $dados['tipo_cte'] ?? 0; // 0=Normal, 1=Complemento, 2=Anulação, 3=Substituto
        $std->procEmi = 0;
        $std->verProc = '1.0';
        $std->cMunEnv = $this->getCodigoMunicipio($dados['emitente']['cidade'], $dados['emitente']['estado']);
        $std->xMunEnv = $dados['emitente']['cidade'];
        $std->UFEnv = $dados['emitente']['estado'];
        $std->modal = $dados['modal'] ?? '01'; // 01=Rodoviário
        $std->tpServ = $dados['tipo_servico'] ?? 0; // 0=Normal, 1=Subcontratação, 2=Redespacho, 3=Redespacho Intermediário, 4=Serviço vinculado a multimodal
        $std->cMunIni = $this->getCodigoMunicipio($dados['municipio_inicio'], $dados['uf_inicio']);
        $std->xMunIni = $dados['municipio_inicio'];
        $std->UFIni = $dados['uf_inicio'];
        $std->cMunFim = $this->getCodigoMunicipio($dados['municipio_fim'], $dados['uf_fim']);
        $std->xMunFim = $dados['municipio_fim'];
        $std->UFFim = $dados['uf_fim'];
        $std->retira = $dados['retira'] ?? 0; // 0=Sim, 1=Não
        $std->indIEToma = $dados['tomador']['indicador_ie'] ?? 9; // 1=Contribuinte, 2=Isento, 9=Não contribuinte

        // Tomador do serviço
        $toma = $dados['tomador']['tipo'] ?? 0; // 0=Remetente, 1=Expedidor, 2=Recebedor, 3=Destinatário, 4=Outros
        $std->toma = $toma;

        $cte->tagide($std);

        // TAG compl (complemento) - Opcional
        if (isset($dados['observacoes'])) {
            $std = new \stdClass();
            $std->xObs = $dados['observacoes'];
            $cte->tagcompl($std);
        }

        // TAG emit (Emitente)
        $std = new \stdClass();
        $std->CNPJ = preg_replace('/[^0-9]/', '', $dados['emitente']['cnpj']);
        $std->IE = preg_replace('/[^0-9]/', '', $dados['emitente']['inscricao_estadual']);
        $std->xNome = $dados['emitente']['nome'];
        $std->xFant = $dados['emitente']['nome_fantasia'] ?? $dados['emitente']['nome'];
        $cte->tagemit($std);

        // TAG enderEmit
        $std = new \stdClass();
        $std->xLgr = $dados['emitente']['endereco'];
        $std->nro = $dados['emitente']['numero'];
        $std->xCpl = $dados['emitente']['complemento'] ?? null;
        $std->xBairro = $dados['emitente']['bairro'];
        $std->cMun = $this->getCodigoMunicipio($dados['emitente']['cidade'], $dados['emitente']['estado']);
        $std->xMun = $dados['emitente']['cidade'];
        $std->CEP = preg_replace('/[^0-9]/', '', $dados['emitente']['cep']);
        $std->UF = $dados['emitente']['estado'];
        $std->fone = preg_replace('/[^0-9]/', '', $dados['emitente']['telefone'] ?? '');
        $cte->tagenderEmit($std);

        // TAG rem (Remetente)
        $rem = $dados['remetente'];
        $std = new \stdClass();

        if (isset($rem['cnpj'])) {
            $std->CNPJ = preg_replace('/[^0-9]/', '', $rem['cnpj']);
        } else {
            $std->CPF = preg_replace('/[^0-9]/', '', $rem['cpf']);
        }

        $std->IE = isset($rem['inscricao_estadual']) ? preg_replace('/[^0-9]/', '', $rem['inscricao_estadual']) : null;
        $std->xNome = $rem['nome'];
        $std->xFant = $rem['nome_fantasia'] ?? null;
        $std->fone = preg_replace('/[^0-9]/', '', $rem['telefone'] ?? '');
        $cte->tagrem($std);

        // TAG enderReme
        $std = new \stdClass();
        $std->xLgr = $rem['endereco'];
        $std->nro = $rem['numero'];
        $std->xCpl = $rem['complemento'] ?? null;
        $std->xBairro = $rem['bairro'];
        $std->cMun = $this->getCodigoMunicipio($rem['cidade'], $rem['estado']);
        $std->xMun = $rem['cidade'];
        $std->CEP = preg_replace('/[^0-9]/', '', $rem['cep']);
        $std->UF = $rem['estado'];
        $std->cPais = '1058';
        $std->xPais = 'BRASIL';
        $cte->tagenderReme($std);

        // TAG dest (Destinatário)
        $dest = $dados['destinatario'];
        $std = new \stdClass();

        if (isset($dest['cnpj'])) {
            $std->CNPJ = preg_replace('/[^0-9]/', '', $dest['cnpj']);
        } else {
            $std->CPF = preg_replace('/[^0-9]/', '', $dest['cpf']);
        }

        $std->IE = isset($dest['inscricao_estadual']) ? preg_replace('/[^0-9]/', '', $dest['inscricao_estadual']) : null;
        $std->xNome = $dest['nome'];
        $std->fone = preg_replace('/[^0-9]/', '', $dest['telefone'] ?? '');
        $cte->tagdest($std);

        // TAG enderDest
        $std = new \stdClass();
        $std->xLgr = $dest['endereco'];
        $std->nro = $dest['numero'];
        $std->xCpl = $dest['complemento'] ?? null;
        $std->xBairro = $dest['bairro'];
        $std->cMun = $this->getCodigoMunicipio($dest['cidade'], $dest['estado']);
        $std->xMun = $dest['cidade'];
        $std->CEP = preg_replace('/[^0-9]/', '', $dest['cep']);
        $std->UF = $dest['estado'];
        $std->cPais = '1058';
        $std->xPais = 'BRASIL';
        $cte->tagenderDest($std);

        // TAG vPrest (Valores da Prestação)
        $valores = $dados['valores'];
        $std = new \stdClass();
        $std->vTPrest = number_format($valores['valor_total_servico'], 2, '.', '');
        $std->vRec = number_format($valores['valor_receber'], 2, '.', '');
        $cte->tagvPrest($std);

        // TAG Comp (Componentes do valor)
        foreach ($valores['componentes'] as $comp) {
            $std = new \stdClass();
            $std->xNome = $comp['nome'];
            $std->vComp = number_format($comp['valor'], 2, '.', '');
            $cte->tagComp($std);
        }

        // TAG imp (Impostos)
        $std = new \stdClass();
        $std->vTotTrib = number_format($valores['valor_total_tributos'] ?? 0, 2, '.', '');
        $std->infAdFisco = $valores['info_adicional_fisco'] ?? null;
        $cte->tagimp($std);

        // TAG ICMS
        $icms = $dados['icms'];
        $std = new \stdClass();
        $std->CST = $icms['cst'];
        $std->vBC = number_format($icms['base_calculo'], 2, '.', '');
        $std->pICMS = number_format($icms['aliquota'], 2, '.', '');
        $std->vICMS = number_format($icms['valor'], 2, '.', '');

        $cte->tagICMS($std);

        // TAG infCarga (Informações da Carga)
        $carga = $dados['carga'];
        $std = new \stdClass();
        $std->vCarga = number_format($carga['valor_carga'], 2, '.', '');
        $std->proPred = $carga['produto_predominante'];
        $std->xOutCat = $carga['outras_caracteristicas'] ?? null;
        $cte->taginfCarga($std);

        // TAG infQ (Quantidades da carga)
        foreach ($carga['quantidades'] as $qtd) {
            $std = new \stdClass();
            $std->cUnid = $qtd['codigo_unidade']; // 00=M3, 01=KG, 02=TON, 03=UNIDADE, 04=LITROS, 05=MMBTU
            $std->tpMed = $qtd['tipo_medida'];
            $std->qCarga = number_format($qtd['quantidade'], 4, '.', '');
            $cte->taginfQ($std);
        }

        // TAG infModal (Modal Rodoviário)
        $std = new \stdClass();
        $std->versaoModal = '3.00';
        $cte->taginfModal($std);

        // TAG rodo (Rodoviário)
        $std = new \stdClass();
        $std->RNTRC = $dados['rodoviario']['rntrc'] ?? null;
        $cte->tagrodo($std);

        // TAG infNFe (Documentos vinculados)
        if (isset($dados['documentos'])) {
            foreach ($dados['documentos'] as $doc) {
                $std = new \stdClass();
                $std->chave = $doc['chave_nfe'];
                $std->PIN = $doc['pin'] ?? null;
                $cte->taginfNFe($std);
            }
        }

        // Gerar XML
        $xml = $cte->getXML();
        $this->log("✅ XML do CT-e gerado com sucesso");

        return $xml;
    }

    /**
     * Assinar XML
     */
    private function assinarXML($xml, $emitente)
    {
        $this->log("🔐 ASSINANDO XML DO CT-e");

        $certificadoPath = __DIR__ . "/../../certificados/empresas/{$emitente['cnpj']}/{$emitente['cnpj']}.pfx";
        $configCertificado = require __DIR__ . '/../../config_certificado.php';
        $certificado = \App\Core\CertificateHelper::loadPfx($certificadoPath, $configCertificado['senha_certificado']);

        $config = [
            "atualizacao" => date('Y-m-d H:i:s'),
            "tpAmb" => 1,
            "razaosocial" => $emitente['nome'],
            "cnpj" => preg_replace('/[^0-9]/', '', $emitente['cnpj']),
            "siglaUF" => $emitente['estado'],
            "schemes" => "PL_CTe_300",
            "versao" => '3.00'
        ];

        $configJson = json_encode($config);
        $tools = new Tools($configJson, $certificado);

        $xmlAssinado = $tools->signCTe($xml);
        $this->log("✅ XML assinado com sucesso");

        return $xmlAssinado;
    }

    /**
     * Enviar para SEFAZ
     */
    private function enviarParaSEFAZ($xml, $dados)
    {
        $this->log("📤 ENVIANDO CT-e PARA SEFAZ");

        $certificadoPath = __DIR__ . "/../../certificados/empresas/{$dados['emitente']['cnpj']}/{$dados['emitente']['cnpj']}.pfx";
        $configCertificado = require __DIR__ . '/../../config_certificado.php';
        $certificado = \App\Core\CertificateHelper::loadPfx($certificadoPath, $configCertificado['senha_certificado']);

        $config = [
            "atualizacao" => date('Y-m-d H:i:s'),
            "tpAmb" => 1,
            "razaosocial" => $dados['emitente']['nome'],
            "cnpj" => preg_replace('/[^0-9]/', '', $dados['emitente']['cnpj']),
            "siglaUF" => $dados['emitente']['estado'],
            "schemes" => "PL_CTe_300",
            "versao" => '3.00'
        ];

        $configJson = json_encode($config);
        $tools = new Tools($configJson, $certificado);

        // Enviar (CT-e usa modo síncrono)
        $response = $tools->sefazEnvia($xml);

        $st = new Standardize($response);
        $std = $st->toStd();

        if (isset($std->protCTe->infProt->cStat)) {
            $cStat = $std->protCTe->infProt->cStat;

            if (in_array($cStat, [100, 150])) {
                $protocoloNumero = $std->protCTe->infProt->nProt;
                $chaveAcesso = $std->protCTe->infProt->chCTe;

                $this->log("✅ CT-e autorizado! Protocolo: $protocoloNumero");

                $xmlProtocolado = $this->criarXMLProtocolado($xml, $std->protCTe);

                return [
                    'success' => true,
                    'chave_acesso' => $chaveAcesso,
                    'protocolo' => $protocoloNumero,
                    'cStat' => $cStat,
                    'xml_protocolado' => $xmlProtocolado
                ];
            } else {
                $motivo = $std->protCTe->infProt->xMotivo ?? 'Erro desconhecido';
                return [
                    'success' => false,
                    'chave_acesso' => $std->protCTe->infProt->chCTe ?? null,
                    'cStat' => $cStat,
                    'error' => "CTe rejeitado [$cStat]: $motivo"
                ];
            }
        } else {
            return [
                'success' => false,
                'cStat' => $std->cStat ?? null,
                'error' => "Rejeição: " . ($std->xMotivo ?? 'Erro desconhecido')
            ];
        }
    }

    /**
     * Criar XML protocolado
     */
    private function criarXMLProtocolado($xml, $protCTe)
    {
        $xmlProtocolado = '<?xml version="1.0" encoding="UTF-8"?>';
        $xmlProtocolado .= '<cteProc xmlns="http://www.portalfiscal.inf.br/cte" versao="3.00">';
        $xmlProtocolado .= $xml;
        $xmlProtocolado .= '<protCTe versao="3.00">';
        $xmlProtocolado .= '<infProt>';
        $xmlProtocolado .= '<tpAmb>' . $protCTe->infProt->tpAmb . '</tpAmb>';
        $xmlProtocolado .= '<verAplic>' . $protCTe->infProt->verAplic . '</verAplic>';
        $xmlProtocolado .= '<chCTe>' . $protCTe->infProt->chCTe . '</chCTe>';
        $xmlProtocolado .= '<dhRecbto>' . $protCTe->infProt->dhRecbto . '</dhRecbto>';
        $xmlProtocolado .= '<nProt>' . $protCTe->infProt->nProt . '</nProt>';
        $xmlProtocolado .= '<digVal>' . $protCTe->infProt->digVal . '</digVal>';
        $xmlProtocolado .= '<cStat>' . $protCTe->infProt->cStat . '</cStat>';
        $xmlProtocolado .= '<xMotivo>' . $protCTe->infProt->xMotivo . '</xMotivo>';
        $xmlProtocolado .= '</infProt>';
        $xmlProtocolado .= '</protCTe>';
        $xmlProtocolado .= '</cteProc>';

        return $xmlProtocolado;
    }

    /**
     * Salvar XML
     */
    private function salvarXML($xml, $dados)
    {
        $chaveAcesso = $this->extrairChaveAcessoDoXML($xml);
        $cnpj = preg_replace('/[^0-9]/', '', $dados['emitente']['cnpj']);
        $anoMes = date('Y_m');

        $xmlPath = __DIR__ . "/../../arquivos/{$cnpj}/cte/xml/{$anoMes}/assinados/{$chaveAcesso}.xml";
        file_put_contents($xmlPath, $xml);

        $this->log("✅ XML salvo: {$xmlPath}");

        return "/pontti_nfe/arquivos/{$cnpj}/cte/xml/{$anoMes}/assinados/{$chaveAcesso}.xml";
    }

    /**
     * Gerar PDF (DACTE)
     */
    private function gerarPDF($dados, $xml, $protocolo)
    {
        $this->log("🖨️ GERANDO DACTE");

        try {
            $chaveAcesso = $this->extrairChaveAcessoDoXML($xml);
            $cnpj = preg_replace('/[^0-9]/', '', $dados['emitente']['cnpj']);
            $anoMes = date('Y_m');

            $dacte = new Dacte($xml);
            $pdf = $dacte->render();

            $pdfPath = __DIR__ . "/../../arquivos/{$cnpj}/cte/pdf/{$anoMes}/emitidos/CTe_{$chaveAcesso}.pdf";
            file_put_contents($pdfPath, $pdf);

            $this->log("✅ DACTE gerado: {$pdfPath}");

            return "/pontti_nfe/arquivos/{$cnpj}/cte/pdf/{$anoMes}/emitidos/CTe_{$chaveAcesso}.pdf";
        } catch (\Exception $e) {
            $this->log("❌ ERRO ao gerar DACTE: " . $e->getMessage());
            return null;
        }
    }

    /**
     * Cancelar CT-e
     */
    public function cancelar($dados)
    {
        $this->log("❌ CANCELAMENTO DE CT-e");

        try {
            $certificadoPath = __DIR__ . "/../../certificados/empresas/{$dados['emitente']['cnpj']}/{$dados['emitente']['cnpj']}.pfx";
            $configCertificado = require __DIR__ . '/../../config_certificado.php';
            $certificadoConteudo = file_get_contents($certificadoPath);
            $certificado = \App\Core\CertificateHelper::loadPfx($certificadoPath, $configCertificado['senha_certificado']);

            $config = [
                "atualizacao" => date('Y-m-d H:i:s'),
                "tpAmb" => 1,
                "razaosocial" => $dados['emitente']['nome'],
                "cnpj" => preg_replace('/[^0-9]/', '', $dados['emitente']['cnpj']),
                "siglaUF" => $dados['emitente']['estado'],
                "schemes" => "PL_CTe_300",
                "versao" => '3.00'
            ];

            $configJson = json_encode($config);
            $tools = new Tools($configJson, $certificado);

            $evento = $tools->sefazCancela(
                $dados['chave_acesso'],
                $dados['justificativa'],
                $dados['protocolo']
            );

            $st = new Standardize($evento);
            $std = $st->toStd();

            if (isset($std->retEventoCTe->infEvento->cStat) && $std->retEventoCTe->infEvento->cStat == 135) {
                return [
                    'success' => true,
                    'chave_acesso' => $dados['chave_acesso'],
                    'protocolo_cancelamento' => $std->retEventoCTe->infEvento->nProt,
                    'message' => 'CT-e cancelado com sucesso'
                ];
            } else {
                return [
                    'success' => false,
                    'error' => $std->retEventoCTe->infEvento->xMotivo ?? 'Erro ao cancelar'
                ];
            }
        } catch (\Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }

    /**
     * Carta de Correção
     */
    public function cartaCorrecao($dados)
    {
        $this->log("📝 CARTA DE CORREÇÃO DO CT-e");

        try {
            $certificadoPath = __DIR__ . "/../../certificados/empresas/{$dados['emitente']['cnpj']}/{$dados['emitente']['cnpj']}.pfx";
            $configCertificado = require __DIR__ . '/../../config_certificado.php';
            $certificadoConteudo = file_get_contents($certificadoPath);
            $certificado = \App\Core\CertificateHelper::loadPfx($certificadoPath, $configCertificado['senha_certificado']);

            $config = [
                "atualizacao" => date('Y-m-d H:i:s'),
                "tpAmb" => 1,
                "razaosocial" => $dados['emitente']['nome'],
                "cnpj" => preg_replace('/[^0-9]/', '', $dados['emitente']['cnpj']),
                "siglaUF" => $dados['emitente']['estado'],
                "schemes" => "PL_CTe_300",
                "versao" => '3.00'
            ];

            $configJson = json_encode($config);
            $tools = new Tools($configJson, $certificado);

            $evento = $tools->sefazCartaCorrecao(
                $dados['chave_acesso'],
                $dados['correcao'],
                $dados['numero_sequencial'] ?? 1
            );

            $st = new Standardize($evento);
            $std = $st->toStd();

            if (isset($std->retEventoCTe->infEvento->cStat) && $std->retEventoCTe->infEvento->cStat == 135) {
                return [
                    'success' => true,
                    'chave_acesso' => $dados['chave_acesso'],
                    'protocolo_carta_correcao' => $std->retEventoCTe->infEvento->nProt,
                    'message' => 'Carta de correção registrada com sucesso'
                ];
            } else {
                return [
                    'success' => false,
                    'error' => $std->retEventoCTe->infEvento->xMotivo ?? 'Erro ao registrar carta de correção'
                ];
            }
        } catch (\Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }

    /**
     * Extrair chave de acesso do XML
     */
    private function extrairChaveAcessoDoXML($xml)
    {
        if (preg_match('/Id="CTe(\d{44})"/', $xml, $matches)) {
            return $matches[1];
        }
        if (preg_match('/<chCTe>(\d{44})<\/chCTe>/', $xml, $matches)) {
            return $matches[1];
        }
        return null;
    }

    /**
     * Obter código da UF
     */
    private function getCodigoUF($uf)
    {
        $ufs = [
            'AC' => 12, 'AL' => 27, 'AP' => 16, 'AM' => 13, 'BA' => 29,
            'CE' => 23, 'DF' => 53, 'ES' => 32, 'GO' => 52, 'MA' => 21,
            'MT' => 51, 'MS' => 50, 'MG' => 31, 'PA' => 15, 'PB' => 25,
            'PR' => 41, 'PE' => 26, 'PI' => 22, 'RJ' => 33, 'RN' => 24,
            'RS' => 43, 'RO' => 11, 'RR' => 14, 'SC' => 42, 'SP' => 35,
            'SE' => 28, 'TO' => 17
        ];

        return $ufs[$uf] ?? 0;
    }

    /**
     * Obter código do município (IBGE)
     */
    private function getCodigoMunicipio($cidade, $uf)
    {
        $municipios = [
            'Recife' => 2611606,
            'Paulista' => 2610707,
            'São Paulo' => 3550308,
            'Rio de Janeiro' => 3304557,
            'Caruaru' => 2604106,
            'Petrolina' => 2611101
        ];

        return $municipios[$cidade] ?? 9999999;
    }
}

