<?php

namespace App\Controllers;

use App\Core\BaseController;
use App\Services\NFeService;

class NFeController extends BaseController
{
    private $nfeService;

    public function __construct()
    {
        $this->nfeService = new NFeService();
    }

    public function emitir()
    {
        // Verificar se a biblioteca NFePHP está disponível
        if (!class_exists('\NFePHP\NFe\Tools')) {
            $this->sendError(500, 'Biblioteca NFePHP não encontrada. Sistema configurado APENAS para SEFAZ real.');
            return;
        }

        // Verificar se as extensões SOAP estão disponíveis
        if (!defined('SOAP_1_2') || !defined('SOAP_1_1')) {
            $this->sendError(500, 'Extensões SOAP não configuradas. Sistema configurado APENAS para SEFAZ real.');
            return;
        }

        $input = file_get_contents('php://input');
        error_log("Input recebido: " . $input);

        $dados = json_decode($input, true);
        error_log("Dados decodificados: " . json_encode($dados));

        if (!$dados) {
            // error_log("ERRO: Dados inválidos ou vazios");
            $this->sendError(400, 'Dados inválidos');
        }

        try {
            // Validar dados obrigatórios básicos
            $this->validarDadosBasicosNFe($dados);

            // Incrementar dados automaticamente
            $dados = $this->incrementarDadosNFe($dados);

            // Emitir NFe usando o serviço (sem banco de dados)
            $resultado = $this->nfeService->emitirSemBanco($dados);
            error_log("📊 Resultado do serviço NFeService: " . json_encode($resultado));

            if ($resultado['success']) {
                $response = [
                    'success' => true,
                    'chave_acesso' => $resultado['chave_acesso'],
                    'protocolo' => $resultado['protocolo'],
                    'cStat' => $resultado['cStat'],
                    'xml_path' => $resultado['xml_path'] ?? null,
                    'pdf_path' => $resultado['pdf_path'] ?? null,
                    'message' => 'NFe emitida com sucesso'
                ];

                // Adicionar warning se existir (modo de desenvolvimento)
                if (isset($resultado['warning'])) {
                    $response['warning'] = $resultado['warning'];
                }

                return $response;
            } else {
                $errorResponse = [
                    'success' => false,
                    'chave_acesso' => $resultado['chave_acesso'] ?? null,
                    'cStat' => $resultado['cStat'] ?? null,
                    'error' => $resultado['error']
                ];

                error_log("📊 Response de erro do Controller: " . json_encode($errorResponse));
                return $errorResponse;
            }
        } catch (\Exception $e) {
            http_response_code(500);
            return ['error' => $e->getMessage()];
        }
    }

    public function cancelar()
    {
        $input = file_get_contents('php://input');
        $dados = json_decode($input, true);

        if (!$dados) {
            $this->sendError(400, 'Dados inválidos');
            return;
        }

        try {
            // Validar dados obrigatórios
            $camposObrigatorios = ['chave_acesso', 'justificativa', 'empresa'];
            foreach ($camposObrigatorios as $campo) {
                if (!isset($dados[$campo]) || empty($dados[$campo])) {
                    $this->sendError(400, "Campo obrigatório não informado: {$campo}");
                    return;
                }
            }

            // Validar justificativa (mínimo 15 caracteres)
            if (strlen($dados['justificativa']) < 15) {
                $this->sendError(400, 'Justificativa deve ter pelo menos 15 caracteres');
                return;
            }

            // Cancelar NFe
            $resultado = $this->nfeService->cancelarNFe($dados);

            if ($resultado['success']) {
                return [
                    'success' => true,
                    'chave_acesso' => $resultado['chave_acesso'],
                    'protocolo_cancelamento' => $resultado['protocolo_cancelamento'],
                    'data_cancelamento' => $resultado['data_cancelamento'],
                    'message' => 'NFe cancelada com sucesso'
                ];
            } else {
                return [
                    'success' => false,
                    'error' => $resultado['error']
                ];
            }
        } catch (\Exception $e) {
            $this->sendError(500, 'Erro interno: ' . $e->getMessage());
            return;
        }
    }

    public function inutilizar()
    {
        $input = file_get_contents('php://input');
        error_log("Input inutilização recebido: " . $input);

        $dados = json_decode($input, true);
        error_log("Dados decodificados: " . json_encode($dados));

        if (!$dados) {
            $this->sendError(400, 'Dados inválidos');
            return;
        }

        try {
            // Validar dados obrigatórios
            $camposObrigatorios = ['empresa', 'serie', 'numero_inicial', 'numero_final', 'justificativa'];
            foreach ($camposObrigatorios as $campo) {
                if (!isset($dados[$campo]) || empty($dados[$campo])) {
                    $this->sendError(400, "Campo obrigatório não informado: {$campo}");
                    return;
                }
            }

            // Validar empresa
            $camposEmpresa = ['nome', 'cnpj', 'inscricao_estadual', 'estado'];
            foreach ($camposEmpresa as $campo) {
                if (!isset($dados['empresa'][$campo]) || empty($dados['empresa'][$campo])) {
                    $this->sendError(400, "Campo obrigatório da empresa não informado: {$campo}");
                    return;
                }
            }

            // Validar justificativa (mínimo 15 caracteres)
            if (strlen($dados['justificativa']) < 15) {
                $this->sendError(400, 'Justificativa deve ter pelo menos 15 caracteres');
                return;
            }

            // Validar que numero_inicial <= numero_final
            if (intval($dados['numero_inicial']) > intval($dados['numero_final'])) {
                $this->sendError(400, 'Número inicial não pode ser maior que número final');
                return;
            }

            // Inutilizar NFe
            $resultado = $this->nfeService->inutilizarNFe($dados);

            if ($resultado['success']) {
                return [
                    'success' => true,
                    'protocolo_inutilizacao' => $resultado['protocolo_inutilizacao'],
                    'numero_inicial' => $dados['numero_inicial'],
                    'numero_final' => $dados['numero_final'],
                    'serie' => $dados['serie'],
                    'cStat' => $resultado['cStat'] ?? null,
                    'message' => 'Numeração inutilizada com sucesso'
                ];
            } else {
                return [
                    'success' => false,
                    'cStat' => $resultado['cStat'] ?? null,
                    'error' => $resultado['error']
                ];
            }
        } catch (\Exception $e) {
            error_log("Erro ao inutilizar: " . $e->getMessage());
            $this->sendError(500, 'Erro interno: ' . $e->getMessage());
            return;
        }
    }

    public function cartaCorrecao()
    {
        $input = file_get_contents('php://input');
        error_log("Input carta de correção recebido: " . $input);

        $dados = json_decode($input, true);
        error_log("Dados decodificados: " . json_encode($dados));

        if (!$dados) {
            $this->sendError(400, 'Dados inválidos');
            return;
        }

        try {
            // Validar dados obrigatórios
            $camposObrigatorios = ['chave_acesso', 'correcao', 'empresa'];
            foreach ($camposObrigatorios as $campo) {
                if (!isset($dados[$campo]) || empty($dados[$campo])) {
                    $this->sendError(400, "Campo obrigatório não informado: {$campo}");
                    return;
                }
            }

            // Validar empresa
            $camposEmpresa = ['nome', 'cnpj', 'inscricao_estadual', 'estado'];
            foreach ($camposEmpresa as $campo) {
                if (!isset($dados['empresa'][$campo]) || empty($dados['empresa'][$campo])) {
                    $this->sendError(400, "Campo obrigatório da empresa não informado: {$campo}");
                    return;
                }
            }

            // Validar correção (mínimo 15 caracteres)
            if (strlen($dados['correcao']) < 15) {
                $this->sendError(400, 'Correção deve ter pelo menos 15 caracteres');
                return;
            }

            // Enviar carta de correção
            $resultado = $this->nfeService->cartaCorrecao($dados);

            if ($resultado['success']) {
                return [
                    'success' => true,
                    'chave_acesso' => $resultado['chave_acesso'],
                    'protocolo_cce' => $resultado['protocolo_cce'],
                    'numero_sequencial' => $resultado['numero_sequencial'],
                    'data_registro' => $resultado['data_registro'],
                    'cStat' => $resultado['cStat'] ?? null,
                    'message' => 'Carta de Correção registrada com sucesso'
                ];
            } else {
                return [
                    'success' => false,
                    'cStat' => $resultado['cStat'] ?? null,
                    'error' => $resultado['error']
                ];
            }
        } catch (\Exception $e) {
            error_log("Erro ao enviar carta de correção: " . $e->getMessage());
            $this->sendError(500, 'Erro interno: ' . $e->getMessage());
            return;
        }
    }


    /**
     * Validação básica para emissão sem banco de dados
     */
    private function validarDadosBasicosNFe($dados)
    {
        // Campos mínimos obrigatórios
        $camposObrigatorios = [
            'cliente'
        ];

        foreach ($camposObrigatorios as $campo) {
            if (!isset($dados[$campo]) || empty($dados[$campo])) {
                throw new \Exception("Campo obrigatório não informado: {$campo}");
            }
        }

        // Validar estrutura da empresa
        $camposEmpresa = ['nome', 'cnpj', 'inscricao_estadual', 'estado', 'cidade'];
        foreach ($camposEmpresa as $campo) {
            if (!isset($dados['empresa'][$campo]) || empty($dados['empresa'][$campo])) {
                throw new \Exception("Campo obrigatório da empresa não informado: {$campo}");
            }
        }

        // Validar estrutura do cliente
        $camposCliente = ['nome'];
        foreach ($camposCliente as $campo) {
            if (!isset($dados['cliente'][$campo]) || empty($dados['cliente'][$campo])) {
                throw new \Exception("Campo obrigatório do cliente não informado: {$campo}");
            }
        }

        // Validar que CPF ou CNPJ foi informado
        $temCPF = !empty($dados['cliente']['cpf'] ?? '');
        $temCNPJ = !empty($dados['cliente']['cnpj'] ?? '');
        if (!$temCPF && !$temCNPJ) {
            throw new \Exception("É obrigatório informar CPF (pessoa física) ou CNPJ (pessoa jurídica) do cliente");
        }

        // Validar pagamento se informado
        if (isset($dados['pagamento'])) {
            $camposPagamento = ['forma_pagamento', 'valor_pagamento'];
            foreach ($camposPagamento as $campo) {
                if (!isset($dados['pagamento'][$campo]) || empty($dados['pagamento'][$campo])) {
                    throw new \Exception("Campo obrigatório do pagamento não informado: {$campo}");
                }
            }
        }

        // Validar transportadora se informada
        if (isset($dados['transportadora'])) {
            $camposTransportadora = ['nome', 'cnpj'];
            foreach ($camposTransportadora as $campo) {
                if (!isset($dados['transportadora'][$campo]) || empty($dados['transportadora'][$campo])) {
                    throw new \Exception("Campo obrigatório da transportadora não informado: {$campo}");
                }
            }
        }

        // Validar itens
        if (!is_array($dados['itens']) || empty($dados['itens'])) {
            throw new \Exception('A NFe deve ter pelo menos um item');
        }

        // Validar alíquotas ICMS antes de processar
        foreach ($dados['itens'] as $index => $item) {
            if (!isset($item['aliquota_icms']) || empty($item['aliquota_icms'])) {
                $codigo = isset($item['codigo']) ? $item['codigo'] : 'N/A';
                throw new \Exception("Alíquota ICMS obrigatória não informada para o item {$index} (código: {$codigo})");
            }

            $aliquota = floatval($item['aliquota_icms']);
            if ($aliquota <= 0) {
                $codigo = isset($item['codigo']) ? $item['codigo'] : 'N/A';
                throw new \Exception("Alíquota ICMS do item {$index} (código: {$codigo}) deve ser maior que 0,00. Valor informado: {$item['aliquota_icms']}");
            }
        }

        foreach ($dados['itens'] as $index => $item) {
            $camposItem = [
                'codigo',
                'descricao',
                'ncm',
                'cfop',
                'unidade_comercial',
                'quantidade_comercial',
                'valor_unitario',
                'valor_total',
                'aliquota_icms'
            ];

            foreach ($camposItem as $campo) {
                if (!isset($item[$campo]) || empty($item[$campo])) {
                    throw new \Exception("Campo obrigatório do item {$index} não informado: {$campo}");
                }
            }

            // Validação específica para aliquota_icms
            if (isset($item['aliquota_icms'])) {
                $aliquota = floatval($item['aliquota_icms']);
                if ($aliquota <= 0) {
                    throw new \Exception("Alíquota ICMS do item {$index} deve ser maior que 0,00. Valor informado: {$item['aliquota_icms']}");
                }
            }
        }
    }

    /**
     * Incrementa automaticamente os dados necessários para a NFe
     */
    private function incrementarDadosNFe($dados)
    {
        // Gerar número da NFe se não informado
        if (!isset($dados['numero']) || empty($dados['numero'])) {
            $dados['numero'] = $this->gerarNumeroNFe();
        }

        // Gerar série se não informada
        if (!isset($dados['serie']) || empty($dados['serie'])) {
            $dados['serie'] = '001';
        }

        // Definir data de emissão se não informada
        if (!isset($dados['data_emissao']) || empty($dados['data_emissao'])) {
            $dados['data_emissao'] = date('Y-m-d');
        }

        // Definir data de saída se não informada (mesmo dia da emissão)
        if (!isset($dados['data_saida']) || empty($dados['data_saida'])) {
            $dados['data_saida'] = $dados['data_emissao'];
        }

        // Calcular valor total se não informado
        if (!isset($dados['valor_total']) || empty($dados['valor_total'])) {
            $dados['valor_total'] = $this->calcularValorTotal($dados['itens']);
        }

        // Definir valores padrão se não informados
        $dados['valor_desconto'] = $dados['valor_desconto'] ?? 0;
        $dados['valor_frete'] = $dados['valor_frete'] ?? 0;
        $dados['valor_seguro'] = $dados['valor_seguro'] ?? 0;
        $dados['valor_outros'] = $dados['valor_outros'] ?? 0;
        $dados['valor_icms'] = $dados['valor_icms'] ?? 0;
        $dados['valor_ipi'] = $dados['valor_ipi'] ?? 0;
        $dados['valor_pis'] = $dados['valor_pis'] ?? 0;
        $dados['valor_cofins'] = $dados['valor_cofins'] ?? 0;

        // Definir status inicial
        $dados['status'] = 'pendente';

        // Garantir ambiente de produção
        $dados['empresa']['ambiente_nfe'] = 'producao';

        // Incrementar dados de pagamento se não informados
        $dados = $this->incrementarDadosPagamento($dados);

        // Incrementar dados da transportadora se não informados
        $dados = $this->incrementarDadosTransportadora($dados);

        // Incrementar CSTs dos itens se não informadas
        $dados = $this->incrementarCSTsItens($dados);

        // Gerar chave de acesso
        $dados['chave_acesso'] = $this->gerarChaveAcesso($dados);

        return $dados;
    }

    /**
     * Incrementa automaticamente os dados de pagamento
     */
    private function incrementarDadosPagamento($dados)
    {
        // Se não foi informado pagamento, criar um padrão
        if (!isset($dados['pagamento']) || empty($dados['pagamento'])) {
            $dados['pagamento'] = [
                'forma_pagamento' => 'dinheiro',
                'valor_pagamento' => $dados['valor_total'],
                'data_vencimento' => $dados['data_emissao'],
                'data_pagamento' => $dados['data_emissao'],
                'tipo_integracao' => '1' // À vista
            ];
        } else {
            // Completar dados de pagamento se parcialmente informados
            $dados['pagamento']['forma_pagamento'] = $dados['pagamento']['forma_pagamento'] ?? 'dinheiro';
            $dados['pagamento']['valor_pagamento'] = $dados['pagamento']['valor_pagamento'] ?? $dados['valor_total'];
            $dados['pagamento']['data_vencimento'] = $dados['pagamento']['data_vencimento'] ?? $dados['data_emissao'];
            $dados['pagamento']['data_pagamento'] = $dados['pagamento']['data_pagamento'] ?? $dados['data_emissao'];
            $dados['pagamento']['tipo_integracao'] = $dados['pagamento']['tipo_integracao'] ?? '1';

            // Campos opcionais com valores padrão
            $dados['pagamento']['troco'] = $dados['pagamento']['troco'] ?? 0;
            $dados['pagamento']['observacoes'] = $dados['pagamento']['observacoes'] ?? '';
        }

        return $dados;
    }

    /**
     * Incrementa automaticamente os dados da transportadora
     */
    private function incrementarDadosTransportadora($dados)
    {
        // Se não foi informada transportadora, não fazer nada
        if (!isset($dados['transportadora']) || empty($dados['transportadora'])) {
            return $dados;
        }

        // Completar dados da transportadora se parcialmente informados
        $dados['transportadora']['nome'] = $dados['transportadora']['nome'] ?? '';
        $dados['transportadora']['cnpj'] = $dados['transportadora']['cnpj'] ?? '';
        $dados['transportadora']['inscricao_estadual'] = $dados['transportadora']['inscricao_estadual'] ?? '';
        $dados['transportadora']['endereco'] = $dados['transportadora']['endereco'] ?? '';
        $dados['transportadora']['cidade'] = $dados['transportadora']['cidade'] ?? '';
        $dados['transportadora']['estado'] = $dados['transportadora']['estado'] ?? '';
        $dados['transportadora']['cep'] = $dados['transportadora']['cep'] ?? '';
        $dados['transportadora']['email'] = $dados['transportadora']['email'] ?? '';
        $dados['transportadora']['telefone'] = $dados['transportadora']['telefone'] ?? '';

        // Dados específicos do transporte
        $dados['transportadora']['modalidade_frete'] = $dados['transportadora']['modalidade_frete'] ?? '0'; // 0=Por conta do emitente
        $dados['transportadora']['placa_veiculo'] = $dados['transportadora']['placa_veiculo'] ?? '';
        $dados['transportadora']['uf_veiculo'] = $dados['transportadora']['uf_veiculo'] ?? '';
        $dados['transportadora']['rntc'] = $dados['transportadora']['rntc'] ?? ''; // RNTRC
        $dados['transportadora']['quantidade_volumes'] = $dados['transportadora']['quantidade_volumes'] ?? 1;
        $dados['transportadora']['especie_volumes'] = $dados['transportadora']['especie_volumes'] ?? 'VOLUMES';
        $dados['transportadora']['marca_volumes'] = $dados['transportadora']['marca_volumes'] ?? '';
        $dados['transportadora']['numeracao_volumes'] = $dados['transportadora']['numeracao_volumes'] ?? '';
        $dados['transportadora']['peso_liquido'] = $dados['transportadora']['peso_liquido'] ?? 0;
        $dados['transportadora']['peso_bruto'] = $dados['transportadora']['peso_bruto'] ?? 0;

        // Dados específicos do frete
        $dados['transportadora']['valor_frete'] = $dados['transportadora']['valor_frete'] ?? 0;
        $dados['transportadora']['valor_seguro'] = $dados['transportadora']['valor_seguro'] ?? 0;
        $dados['transportadora']['valor_outros'] = $dados['transportadora']['valor_outros'] ?? 0;
        $dados['transportadora']['valor_despesa_acessorias'] = $dados['transportadora']['valor_despesa_acessorias'] ?? 0;
        $dados['transportadora']['base_calculo_icms'] = $dados['transportadora']['base_calculo_icms'] ?? 0;
        $dados['transportadora']['aliquota_icms'] = $dados['transportadora']['aliquota_icms'] ?? 0;
        $dados['transportadora']['valor_icms'] = $dados['transportadora']['valor_icms'] ?? 0;
        $dados['transportadora']['cfop'] = $dados['transportadora']['cfop'] ?? '5353'; // CFOP para transporte
        $dados['transportadora']['codigo_municipio'] = $dados['transportadora']['codigo_municipio'] ?? '';

        return $dados;
    }

    /**
     * Incrementa automaticamente as CSTs dos itens
     */
    private function incrementarCSTsItens($dados)
    {
        foreach ($dados['itens'] as $index => $item) {
            // CST ICMS
            $dados['itens'][$index]['cst_icms'] = $item['cst_icms'] ?? '00'; // Tributada integralmente
            $dados['itens'][$index]['base_calculo_icms'] = $item['base_calculo_icms'] ?? $item['valor_total'];
            // aliquota_icms é obrigatório - não define valor padrão
            $dados['itens'][$index]['aliquota_icms'] = $item['aliquota_icms'];
            $dados['itens'][$index]['valor_icms'] = $item['valor_icms'] ?? 0;

            // CST IPI
            $dados['itens'][$index]['cst_ipi'] = $item['cst_ipi'] ?? '00'; // Entrada com recuperação de crédito
            $dados['itens'][$index]['base_calculo_ipi'] = $item['base_calculo_ipi'] ?? $item['valor_total'];
            $dados['itens'][$index]['aliquota_ipi'] = $item['aliquota_ipi'] ?? 0.00;
            $dados['itens'][$index]['valor_ipi'] = $item['valor_ipi'] ?? 0;

            // CST PIS
            $dados['itens'][$index]['cst_pis'] = $item['cst_pis'] ?? '01'; // Operação tributável (base de cálculo = valor da operação alíquota básica)
            $dados['itens'][$index]['base_calculo_pis'] = $item['base_calculo_pis'] ?? $item['valor_total'];
            // Só define aliquota_pis se não estiver presente no JSON
            if (!isset($item['aliquota_pis']) || empty($item['aliquota_pis'])) {
                $dados['itens'][$index]['aliquota_pis'] = 1.65;
            } else {
                $dados['itens'][$index]['aliquota_pis'] = $item['aliquota_pis'];
            }
            $dados['itens'][$index]['valor_pis'] = $item['valor_pis'] ?? 0;

            // CST COFINS
            $dados['itens'][$index]['cst_cofins'] = $item['cst_cofins'] ?? '01'; // Operação tributável (base de cálculo = valor da operação alíquota básica)
            $dados['itens'][$index]['base_calculo_cofins'] = $item['base_calculo_cofins'] ?? $item['valor_total'];
            // Só define aliquota_cofins se não estiver presente no JSON
            if (!isset($item['aliquota_cofins']) || empty($item['aliquota_cofins'])) {
                $dados['itens'][$index]['aliquota_cofins'] = 7.60;
            } else {
                $dados['itens'][$index]['aliquota_cofins'] = $item['aliquota_cofins'];
            }
            $dados['itens'][$index]['valor_cofins'] = $item['valor_cofins'] ?? 0;

            // Dados adicionais de tributação
            $dados['itens'][$index]['origem'] = $item['origem'] ?? '0'; // Nacional
            $dados['itens'][$index]['tributacao'] = $item['tributacao'] ?? 'T'; // Tributada
            $dados['itens'][$index]['modalidade_bc'] = $item['modalidade_bc'] ?? '0'; // Margem de valor agregado
            $dados['itens'][$index]['situacao_tributaria'] = $item['situacao_tributaria'] ?? '00'; // Tributada integralmente
        }

        return $dados;
    }

    /**
     * Gera um número sequencial para a NFe
     */
    private function gerarNumeroNFe()
    {
        // Usar timestamp + microtime para garantir unicidade
        return intval(substr(str_replace('.', '', microtime(true)), -8));
    }

    /**
     * Calcula o valor total dos itens
     */
    private function calcularValorTotal($itens)
    {
        $total = 0;
        foreach ($itens as $item) {
            $total += floatval($item['valor_total']);
        }
        return round($total, 2);
    }

    /**
     * Gera a chave de acesso da NFe (formato simplificado)
     */
    private function gerarChaveAcesso($dados)
    {
        // Formato correto da chave NFe: UF + AAMM + CNPJ + mod + serie + nNF + tpEmis + cNF + DV
        $uf = $this->getCodigoUF($dados['empresa']['estado']);
        $anoMes = date('ym', strtotime($dados['data_emissao']));
        $cnpj = preg_replace('/\D/', '', $dados['empresa']['cnpj']);
        $modelo = '55'; // NFe
        $serie = str_pad($dados['serie'], 3, '0', STR_PAD_LEFT);
        $numero = str_pad($dados['numero'], 9, '0', STR_PAD_LEFT);
        $tipoEmissao = '1'; // Normal
        $codigoNFe = str_pad(substr($dados['numero'], -8), 8, '0', STR_PAD_LEFT);

        // Montar chave sem DV (43 dígitos)
        $chave = $uf . $anoMes . $cnpj . $modelo . $serie . $numero . $tipoEmissao . $codigoNFe;

        // Adicionar dígito verificador
        $chave .= $this->calcularDVChave($chave);

        return $chave;
    }

    /**
     * Obtém o código da UF
     */
    private function getCodigoUF($estado)
    {
        $ufs = [
            'AC' => '12',
            'AL' => '27',
            'AP' => '16',
            'AM' => '13',
            'BA' => '29',
            'CE' => '23',
            'DF' => '53',
            'ES' => '32',
            'GO' => '52',
            'MA' => '21',
            'MT' => '51',
            'MS' => '50',
            'MG' => '31',
            'PA' => '15',
            'PB' => '25',
            'PR' => '41',
            'PE' => '26',
            'PI' => '22',
            'RJ' => '33',
            'RN' => '24',
            'RS' => '43',
            'RO' => '11',
            'RR' => '14',
            'SC' => '42',
            'SP' => '35',
            'SE' => '28',
            'TO' => '17'
        ];

        return $ufs[$estado] ?? '35'; // Default SP
    }

    /**
     * Calcula o dígito verificador da chave (algoritmo Módulo 11)
     */
    private function calcularDVChave($chave)
    {
        // Verificar se a chave tem exatamente 43 dígitos
        if (strlen($chave) !== 43) {
            throw new \Exception("Chave deve ter exatamente 43 dígitos para calcular DV. Atual: " . strlen($chave));
        }

        // Pesos para cálculo do DV (padrão SEFAZ)
        $pesos = [2, 3, 4, 5, 6, 7, 2, 3, 4, 5, 6, 7, 2, 3, 4, 5, 6, 7, 2, 3, 4, 5, 6, 7, 2, 3, 4, 5, 6, 7, 2, 3, 4, 5, 6, 7, 2, 3, 4, 5, 6, 7, 2];

        $soma = 0;
        for ($i = 0; $i < 43; $i++) {
            $soma += intval($chave[$i]) * $pesos[$i];
        }
        $resto = $soma % 11;
        $dv = ($resto < 2) ? 0 : (11 - $resto);

        return (string) $dv;
    }
}
