<?php
/**
 * SCRIPT DE DIAGNÓSTICO - ERRO 500
 * Acesse: https://seu-dominio.com.br/pontti_nfe/diagnostico_erro_500.php
 */

// Ativar exibição de erros
error_reporting(E_ALL);
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);

header('Content-Type: text/html; charset=utf-8');
?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Diagnóstico - Erro 500</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: #1a1a1a;
            color: #fff;
            padding: 20px;
            line-height: 1.6;
        }
        .container { max-width: 1200px; margin: 0 auto; }
        h1 {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            padding: 20px;
            border-radius: 10px;
            margin-bottom: 20px;
            text-align: center;
        }
        .section {
            background: #2d2d2d;
            padding: 20px;
            margin: 15px 0;
            border-radius: 8px;
            border-left: 4px solid #667eea;
        }
        .section h2 {
            color: #667eea;
            margin-bottom: 15px;
            font-size: 1.3em;
        }
        .success { color: #00ff88; }
        .error { color: #ff4757; }
        .warning { color: #ffa502; }
        .info { color: #3742fa; }
        pre {
            background: #1a1a1a;
            padding: 15px;
            border-radius: 5px;
            overflow-x: auto;
            margin: 10px 0;
            border: 1px solid #444;
        }
        table {
            width: 100%;
            border-collapse: collapse;
            margin: 10px 0;
        }
        th, td {
            padding: 12px;
            text-align: left;
            border-bottom: 1px solid #444;
        }
        th {
            background: #1a1a1a;
            color: #667eea;
            font-weight: 600;
        }
        .status-ok::before { content: "✅ "; }
        .status-error::before { content: "❌ "; }
        .status-warning::before { content: "⚠️ "; }
        .btn {
            display: inline-block;
            padding: 12px 24px;
            background: #667eea;
            color: white;
            text-decoration: none;
            border-radius: 5px;
            margin: 10px 5px;
            font-weight: 600;
            transition: all 0.3s;
        }
        .btn:hover {
            background: #764ba2;
            transform: translateY(-2px);
        }
        .code {
            background: #1a1a1a;
            padding: 2px 6px;
            border-radius: 3px;
            font-family: 'Courier New', monospace;
            font-size: 0.9em;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>🔍 Diagnóstico Completo - Erro 500</h1>

        <?php
        $baseDir = __DIR__;
        $erros = [];
        $avisos = [];
        $sucessos = [];

        // ============================================
        // 1. INFORMAÇÕES DO SERVIDOR
        // ============================================
        ?>
        <div class="section">
            <h2>1️⃣ Informações do Servidor</h2>
            <table>
                <tr>
                    <th>Item</th>
                    <th>Valor</th>
                </tr>
                <tr>
                    <td>Versão PHP</td>
                    <td class="success"><?= PHP_VERSION ?></td>
                </tr>
                <tr>
                    <td>Sistema Operacional</td>
                    <td><?= PHP_OS ?></td>
                </tr>
                <tr>
                    <td>Servidor Web</td>
                    <td><?= $_SERVER['SERVER_SOFTWARE'] ?? 'Desconhecido' ?></td>
                </tr>
                <tr>
                    <td>Diretório Base</td>
                    <td class="code"><?= $baseDir ?></td>
                </tr>
                <tr>
                    <td>Document Root</td>
                    <td class="code"><?= $_SERVER['DOCUMENT_ROOT'] ?? 'N/A' ?></td>
                </tr>
            </table>
        </div>

        <?php
        // ============================================
        // 2. EXTENSÕES PHP OBRIGATÓRIAS
        // ============================================
        ?>
        <div class="section">
            <h2>2️⃣ Extensões PHP Necessárias</h2>
            <table>
                <tr>
                    <th>Extensão</th>
                    <th>Status</th>
                    <th>Necessária para</th>
                </tr>
                <?php
                $extensoesNecessarias = [
                    'curl' => 'Comunicação com SEFAZ',
                    'openssl' => 'Certificado Digital',
                    'soap' => 'Web Services SEFAZ',
                    'xml' => 'Manipulação de XML',
                    'dom' => 'Processamento XML',
                    'zip' => 'Compactação de arquivos',
                    'json' => 'API REST',
                    'mbstring' => 'Strings multibyte',
                    'fileinfo' => 'Informações de arquivo',
                    'gd' => 'Geração de QR Code'
                ];

                foreach ($extensoesNecessarias as $ext => $desc) {
                    $instalada = extension_loaded($ext);
                    $class = $instalada ? 'status-ok' : 'status-error';
                    if (!$instalada) {
                        $erros[] = "Extensão PHP '$ext' não instalada";
                    } else {
                        $sucessos[] = "Extensão '$ext' OK";
                    }
                    echo "<tr>";
                    echo "<td class='code'>$ext</td>";
                    echo "<td class='$class'>" . ($instalada ? 'Instalada' : 'NÃO INSTALADA') . "</td>";
                    echo "<td>$desc</td>";
                    echo "</tr>";
                }
                ?>
            </table>
        </div>

        <?php
        // ============================================
        // 3. ARQUIVOS ESSENCIAIS
        // ============================================
        ?>
        <div class="section">
            <h2>3️⃣ Arquivos Essenciais do Sistema</h2>
            <table>
                <tr>
                    <th>Arquivo</th>
                    <th>Status</th>
                    <th>Permissões</th>
                </tr>
                <?php
                $arquivosEssenciais = [
                    'index.php',
                    'api_router.php',
                    'composer.json',
                    'vendor/autoload.php',
                    'src/Core/Router.php',
                    'src/Core/Database.php',
                    'src/Controllers/NFeController.php',
                    'src/Services/NFeService.php',
                    'config_producao.php',
                    'config_certificado.php'
                ];

                foreach ($arquivosEssenciais as $arquivo) {
                    $caminho = $baseDir . '/' . $arquivo;
                    $existe = file_exists($caminho);
                    $class = $existe ? 'status-ok' : 'status-error';
                    $perms = $existe ? substr(sprintf('%o', fileperms($caminho)), -4) : 'N/A';

                    if (!$existe && strpos($arquivo, 'vendor') === false) {
                        $erros[] = "Arquivo '$arquivo' não encontrado";
                    } elseif ($existe) {
                        $sucessos[] = "Arquivo '$arquivo' encontrado";
                    }

                    echo "<tr>";
                    echo "<td class='code'>$arquivo</td>";
                    echo "<td class='$class'>" . ($existe ? 'Existe' : 'NÃO EXISTE') . "</td>";
                    echo "<td class='code'>$perms</td>";
                    echo "</tr>";
                }
                ?>
            </table>
        </div>

        <?php
        // ============================================
        // 4. TESTE DO AUTOLOAD
        // ============================================
        ?>
        <div class="section">
            <h2>4️⃣ Teste do Autoload do Composer</h2>
            <?php
            $autoloadPath = $baseDir . '/vendor/autoload.php';
            if (file_exists($autoloadPath)) {
                echo "<p class='status-ok'>Arquivo autoload.php encontrado</p>";

                try {
                    require_once $autoloadPath;
                    echo "<p class='status-ok'>Autoload carregado com SUCESSO!</p>";
                    $sucessos[] = "Autoload funcionando";

                    // Testar se as classes podem ser encontradas
                    $classes = [
                        'App\\Core\\Router',
                        'App\\Core\\Database',
                        'App\\Controllers\\NFeController',
                        'App\\Services\\NFeService'
                    ];

                    echo "<h3>Classes do Sistema:</h3>";
                    echo "<table>";
                    echo "<tr><th>Classe</th><th>Status</th></tr>";
                    foreach ($classes as $classe) {
                        $existe = class_exists($classe);
                        $class = $existe ? 'status-ok' : 'status-error';
                        echo "<tr>";
                        echo "<td class='code'>$classe</td>";
                        echo "<td class='$class'>" . ($existe ? 'Encontrada' : 'NÃO ENCONTRADA') . "</td>";
                        echo "</tr>";

                        if (!$existe) {
                            $erros[] = "Classe '$classe' não encontrada";
                        }
                    }
                    echo "</table>";

                } catch (Exception $e) {
                    echo "<p class='status-error'>ERRO ao carregar autoload:</p>";
                    echo "<pre class='error'>" . htmlspecialchars($e->getMessage()) . "</pre>";
                    $erros[] = "Erro no autoload: " . $e->getMessage();
                }
            } else {
                echo "<p class='status-error'>Arquivo vendor/autoload.php NÃO ENCONTRADO!</p>";
                echo "<p class='warning'>Execute: <span class='code'>composer install --ignore-platform-reqs</span></p>";
                $erros[] = "vendor/autoload.php não encontrado - Execute composer install";
            }
            ?>
        </div>

        <?php
        // ============================================
        // 5. PERMISSÕES DE DIRETÓRIOS
        // ============================================
        ?>
        <div class="section">
            <h2>5️⃣ Permissões de Diretórios</h2>
            <table>
                <tr>
                    <th>Diretório</th>
                    <th>Permissão</th>
                    <th>Gravável</th>
                </tr>
                <?php
                $diretorios = [
                    'arquivos',
                    'logs',
                    'storage',
                    'certificados'
                ];

                foreach ($diretorios as $dir) {
                    $caminho = $baseDir . '/' . $dir;
                    $existe = is_dir($caminho);
                    $gravavel = $existe && is_writable($caminho);
                    $perms = $existe ? substr(sprintf('%o', fileperms($caminho)), -4) : 'N/A';
                    $class = $gravavel ? 'status-ok' : 'status-error';

                    if (!$gravavel && $existe) {
                        $avisos[] = "Diretório '$dir' não é gravável";
                    }

                    echo "<tr>";
                    echo "<td class='code'>$dir/</td>";
                    echo "<td class='code'>$perms</td>";
                    echo "<td class='$class'>" . ($gravavel ? 'SIM' : 'NÃO') . "</td>";
                    echo "</tr>";
                }
                ?>
            </table>
            <p class="info"><strong>Recomendado:</strong> Permissão 755 para diretórios e 644 para arquivos</p>
        </div>

        <?php
        // ============================================
        // 6. VERIFICAR LOGS DE ERRO
        // ============================================
        ?>
        <div class="section">
            <h2>6️⃣ Logs de Erro do Sistema</h2>
            <?php
            $logsDir = $baseDir . '/logs';
            if (is_dir($logsDir)) {
                $arquivosLog = glob($logsDir . '/*.log');
                if (!empty($arquivosLog)) {
                    foreach ($arquivosLog as $logFile) {
                        $nomeLog = basename($logFile);
                        $tamanho = filesize($logFile);
                        echo "<h3>📄 $nomeLog (" . number_format($tamanho / 1024, 2) . " KB)</h3>";

                        if ($tamanho > 0 && $tamanho < 50000) {
                            $conteudo = file_get_contents($logFile);
                            $linhas = explode("\n", $conteudo);
                            $ultimasLinhas = array_slice($linhas, -20);
                            echo "<pre>" . htmlspecialchars(implode("\n", $ultimasLinhas)) . "</pre>";
                        } elseif ($tamanho > 50000) {
                            echo "<p class='warning'>Log muito grande, mostrando últimas 50 linhas:</p>";
                            $linhas = file($logFile);
                            $ultimasLinhas = array_slice($linhas, -50);
                            echo "<pre>" . htmlspecialchars(implode("", $ultimasLinhas)) . "</pre>";
                        } else {
                            echo "<p class='info'>Log vazio</p>";
                        }
                    }
                } else {
                    echo "<p class='info'>Nenhum arquivo de log encontrado</p>";
                }
            } else {
                echo "<p class='warning'>Diretório de logs não existe</p>";
            }

            // Verificar log de erro do PHP
            $phpErrorLog = ini_get('error_log');
            if ($phpErrorLog && file_exists($phpErrorLog)) {
                echo "<h3>📄 PHP Error Log</h3>";
                echo "<p class='code'>$phpErrorLog</p>";
            }
            ?>
        </div>

        <?php
        // ============================================
        // 7. TESTE DE CONEXÃO COM BANCO (se aplicável)
        // ============================================
        ?>
        <div class="section">
            <h2>7️⃣ Configuração de Banco de Dados</h2>
            <?php
            $configDb = $baseDir . '/src/Core/Database.php';
            if (file_exists($configDb)) {
                echo "<p class='status-ok'>Arquivo Database.php encontrado</p>";
                echo "<p class='info'>Verificar manualmente se as credenciais estão corretas</p>";
            } else {
                echo "<p class='status-warning'>Database.php não encontrado (pode não ser necessário)</p>";
            }
            ?>
        </div>

        <?php
        // ============================================
        // 8. RESUMO FINAL
        // ============================================
        ?>
        <div class="section">
            <h2>📊 Resumo do Diagnóstico</h2>

            <?php if (empty($erros)): ?>
                <div style="background: #27ae60; padding: 20px; border-radius: 8px; margin: 10px 0;">
                    <h3 style="color: white;">✅ NENHUM ERRO CRÍTICO ENCONTRADO!</h3>
                    <p>O sistema parece estar configurado corretamente.</p>
                </div>
            <?php else: ?>
                <div style="background: #c0392b; padding: 20px; border-radius: 8px; margin: 10px 0;">
                    <h3 style="color: white;">❌ ERROS ENCONTRADOS (<?= count($erros) ?>)</h3>
                    <ul style="margin-top: 10px;">
                        <?php foreach ($erros as $erro): ?>
                            <li style="margin: 5px 0;"><?= htmlspecialchars($erro) ?></li>
                        <?php endforeach; ?>
                    </ul>
                </div>
            <?php endif; ?>

            <?php if (!empty($avisos)): ?>
                <div style="background: #f39c12; padding: 20px; border-radius: 8px; margin: 10px 0;">
                    <h3 style="color: white;">⚠️ AVISOS (<?= count($avisos) ?>)</h3>
                    <ul style="margin-top: 10px;">
                        <?php foreach ($avisos as $aviso): ?>
                            <li style="margin: 5px 0;"><?= htmlspecialchars($aviso) ?></li>
                        <?php endforeach; ?>
                    </ul>
                </div>
            <?php endif; ?>

            <div style="background: #16a085; padding: 20px; border-radius: 8px; margin: 10px 0;">
                <h3 style="color: white;">✅ TUDO OK (<?= count($sucessos) ?>)</h3>
                <p>Componentes funcionando corretamente</p>
            </div>
        </div>

        <?php
        // ============================================
        // 9. PRÓXIMOS PASSOS
        // ============================================
        ?>
        <div class="section">
            <h2>🔧 Próximos Passos</h2>

            <?php if (!empty($erros)): ?>
                <h3>Resolver Erros Críticos:</h3>
                <ol style="margin-left: 20px;">
                    <?php if (in_array(true, array_map(function($e) { return strpos($e, 'vendor/autoload') !== false; }, $erros))): ?>
                        <li style="margin: 10px 0;">
                            <strong>Instalar dependências do Composer:</strong>
                            <pre>cd <?= $baseDir ?>
composer install --ignore-platform-reqs --no-dev</pre>
                        </li>
                    <?php endif; ?>

                    <?php if (in_array(true, array_map(function($e) { return strpos($e, 'Extensão PHP') !== false; }, $erros))): ?>
                        <li style="margin: 10px 0;">
                            <strong>Instalar extensões PHP faltantes</strong>
                            <p>Entre em contato com seu provedor de hospedagem ou administrador do servidor</p>
                        </li>
                    <?php endif; ?>
                </ol>
            <?php else: ?>
                <p class="success">✅ Sistema aparentemente OK! Se ainda houver erro 500:</p>
                <ol style="margin-left: 20px;">
                    <li style="margin: 10px 0;">Verifique os logs do servidor web (Apache/Nginx)</li>
                    <li style="margin: 10px 0;">Verifique o arquivo .htaccess (se existir)</li>
                    <li style="margin: 10px 0;">Teste acessar: <a href="/pontti_nfe/status.php" class="btn">status.php</a></li>
                </ol>
            <?php endif; ?>
        </div>

        <div style="text-align: center; margin: 30px 0;">
            <a href="/pontti_nfe/" class="btn">← Voltar ao Sistema</a>
            <a href="?refresh=1" class="btn">🔄 Executar Novamente</a>
        </div>

        <div style="text-align: center; margin: 20px 0; color: #888; font-size: 0.9em;">
            <p>Diagnóstico gerado em <?= date('d/m/Y H:i:s') ?></p>
            <p>Após corrigir os erros, você pode deletar este arquivo</p>
        </div>
    </div>
</body>
</html>

