<?php
/**
 * Endpoint para inutilização de numeração de NF-e
 */

// Headers CORS
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With, Accept, Origin');
header('Content-Type: application/json');

// Tratar requisições OPTIONS (preflight)
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// Verificar método HTTP
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode([
        'success' => false,
        'error' => 'Método não permitido. Use POST.'
    ], JSON_UNESCAPED_UNICODE);
    exit;
}

try {
    // Carregar autoload e configurações
    require_once __DIR__ . '/../../vendor/autoload.php';
    require_once __DIR__ . '/../../config_producao.php';

    // Definir constantes SOAP se necessário
    if (!defined('SOAP_1_2')) {
        define('SOAP_1_2', 2);
        define('SOAP_1_1', 1);
    }

    // Ler dados do body
    $input = file_get_contents('php://input');
    $dados = json_decode($input, true);

    if (!$dados) {
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'error' => 'Dados inválidos ou JSON mal formatado'
        ], JSON_UNESCAPED_UNICODE);
        exit;
    }

    // Validar campos obrigatórios
    $camposObrigatorios = ['empresa', 'serie', 'numero_inicial', 'numero_final', 'justificativa'];
    foreach ($camposObrigatorios as $campo) {
        if (!isset($dados[$campo]) || empty($dados[$campo])) {
            http_response_code(400);
            echo json_encode([
                'success' => false,
                'error' => "Campo obrigatório não informado: {$campo}"
            ], JSON_UNESCAPED_UNICODE);
            exit;
        }
    }

    // Validar empresa
    $camposEmpresa = ['nome', 'cnpj', 'inscricao_estadual', 'estado'];
    foreach ($camposEmpresa as $campo) {
        if (!isset($dados['empresa'][$campo]) || empty($dados['empresa'][$campo])) {
            http_response_code(400);
            echo json_encode([
                'success' => false,
                'error' => "Campo obrigatório da empresa não informado: {$campo}"
            ], JSON_UNESCAPED_UNICODE);
            exit;
        }
    }

    // Validar justificativa (mínimo 15 caracteres)
    if (strlen($dados['justificativa']) < 15) {
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'error' => 'Justificativa deve ter pelo menos 15 caracteres'
        ], JSON_UNESCAPED_UNICODE);
        exit;
    }

    // Validar que numero_inicial <= numero_final
    if (intval($dados['numero_inicial']) > intval($dados['numero_final'])) {
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'error' => 'Número inicial não pode ser maior que número final'
        ], JSON_UNESCAPED_UNICODE);
        exit;
    }

    // Instanciar serviço de NF-e
    $nfeService = new \App\Services\NFeService();

    // Inutilizar numeração
    $resultado = $nfeService->inutilizarNFe($dados);

    if ($resultado['success']) {
        http_response_code(200);
        echo json_encode([
            'success' => true,
            'protocolo_inutilizacao' => $resultado['protocolo_inutilizacao'],
            'numero_inicial' => $dados['numero_inicial'],
            'numero_final' => $dados['numero_final'],
            'serie' => $dados['serie'],
            'cStat' => $resultado['cStat'] ?? null,
            'message' => 'Numeração inutilizada com sucesso'
        ], JSON_UNESCAPED_UNICODE);
    } else {
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'cStat' => $resultado['cStat'] ?? null,
            'error' => $resultado['error']
        ], JSON_UNESCAPED_UNICODE);
    }
} catch (\Exception $e) {
    error_log("Erro ao inutilizar numeração: " . $e->getMessage());
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => 'Erro interno: ' . $e->getMessage()
    ], JSON_UNESCAPED_UNICODE);
}

