<?php
/**
 * Endpoint para gerar PDF de Carta de Correção a partir do XML salvo
 */

header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');
header('Content-Type: application/json; charset=utf-8');

// Aumentar timeout e memória
set_time_limit(120);
ini_set('memory_limit', '512M');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

require_once __DIR__ . '/../../vendor/autoload.php';

try {
    // Receber JSON
    $json = file_get_contents('php://input');
    $dados = json_decode($json, true);

    if (!$dados) {
        throw new Exception("Dados inválidos ou JSON mal formatado");
    }

    // Validar campos obrigatórios
    if (empty($dados['chave_acesso'])) {
        throw new Exception("Campo obrigatório não informado: chave_acesso");
    }

    if (empty($dados['numero_sequencial'])) {
        throw new Exception("Campo obrigatório não informado: numero_sequencial");
    }

    if (empty($dados['empresa']['cnpj'])) {
        throw new Exception("Campo obrigatório não informado: empresa.cnpj");
    }

    $cnpj = preg_replace('/\D/', '', $dados['empresa']['cnpj']);
    $chaveAcesso = $dados['chave_acesso'];
    $numeroSeq = $dados['numero_sequencial'];
    $ano = date('Y');
    $mes = date('m');

    // Caminho do XML
    $xmlPath = __DIR__ . "/../../arquivos/{$cnpj}/cce/xml/{$ano}_{$mes}/CCe_{$chaveAcesso}_v{$numeroSeq}.xml";

    if (!file_exists($xmlPath)) {
        throw new Exception("XML não encontrado: {$xmlPath}");
    }

    // Carregar XML
    $xmlEvento = file_get_contents($xmlPath);

    // Preparar dados do emitente
    $dadosEmitente = (object)[
        'razao' => $dados['empresa']['nome'] ?? 'Empresa',
        'logradouro' => $dados['empresa']['endereco'] ?? 'Rua',
        'numero' => $dados['empresa']['numero'] ?? 'S/N',
        'complemento' => $dados['empresa']['complemento'] ?? '',
        'bairro' => $dados['empresa']['bairro'] ?? 'Centro',
        'CEP' => preg_replace('/\D/', '', $dados['empresa']['cep'] ?? '00000000'),
        'municipio' => $dados['empresa']['cidade'] ?? 'Cidade',
        'UF' => $dados['empresa']['estado'] ?? 'PE',
        'telefone' => $dados['empresa']['telefone'] ?? '(00) 0000-0000',
        'email' => $dados['empresa']['email'] ?? 'contato@empresa.com',
        'cnpj' => $cnpj,
        'IE' => preg_replace('/\D/', '', $dados['empresa']['inscricao_estadual'] ?? '000000000')
    ];

    // Gerar PDF
    $daevento = new \NFePHP\DA\NFe\Daevento($xmlEvento, $dadosEmitente);
    $pdf = $daevento->render();

    // Salvar PDF
    $pdfDir = __DIR__ . "/../../arquivos/{$cnpj}/cce/pdf/{$ano}_{$mes}";
    if (!is_dir($pdfDir)) {
        mkdir($pdfDir, 0755, true);
    }

    $pdfPath = "{$pdfDir}/CCe_{$chaveAcesso}_v{$numeroSeq}.pdf";
    file_put_contents($pdfPath, $pdf);

    $response = [
        'success' => true,
        'chave_acesso' => $chaveAcesso,
        'numero_sequencial' => $numeroSeq,
        'xml_path' => "/pontti_nfe/arquivos/{$cnpj}/cce/xml/{$ano}_{$mes}/CCe_{$chaveAcesso}_v{$numeroSeq}.xml",
        'pdf_path' => "/pontti_nfe/arquivos/{$cnpj}/cce/pdf/{$ano}_{$mes}/CCe_{$chaveAcesso}_v{$numeroSeq}.pdf",
        'pdf_size' => strlen($pdf),
        'message' => 'PDF gerado com sucesso'
    ];

    http_response_code(200);
    echo json_encode($response, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);

} catch (\Exception $e) {
    $response = [
        'success' => false,
        'error' => $e->getMessage(),
        'file' => $e->getFile(),
        'line' => $e->getLine()
    ];

    http_response_code(400);
    echo json_encode($response, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
}

