<?php
/**
 * Endpoint para Carta de Correção Eletrônica (CC-e)
 */

// Headers CORS
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With, Accept, Origin');
header('Content-Type: application/json');

// Tratar requisições OPTIONS (preflight)
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// Verificar método HTTP
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode([
        'success' => false,
        'error' => 'Método não permitido. Use POST.'
    ], JSON_UNESCAPED_UNICODE);
    exit;
}

try {
    // Carregar autoload e configurações
    require_once __DIR__ . '/../../vendor/autoload.php';
    require_once __DIR__ . '/../../config_producao.php';

    // Definir constantes SOAP se necessário
    if (!defined('SOAP_1_2')) {
        define('SOAP_1_2', 2);
        define('SOAP_1_1', 1);
    }

    // Ler dados do body
    $input = file_get_contents('php://input');
    $dados = json_decode($input, true);

    if (!$dados) {
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'error' => 'Dados inválidos ou JSON mal formatado'
        ], JSON_UNESCAPED_UNICODE);
        exit;
    }

    // Validar campos obrigatórios
    $camposObrigatorios = ['chave_acesso', 'correcao', 'empresa'];
    foreach ($camposObrigatorios as $campo) {
        if (!isset($dados[$campo]) || empty($dados[$campo])) {
            http_response_code(400);
            echo json_encode([
                'success' => false,
                'error' => "Campo obrigatório não informado: {$campo}"
            ], JSON_UNESCAPED_UNICODE);
            exit;
        }
    }

    // Validar empresa
    $camposEmpresa = ['nome', 'cnpj', 'inscricao_estadual', 'estado'];
    foreach ($camposEmpresa as $campo) {
        if (!isset($dados['empresa'][$campo]) || empty($dados['empresa'][$campo])) {
            http_response_code(400);
            echo json_encode([
                'success' => false,
                'error' => "Campo obrigatório da empresa não informado: {$campo}"
            ], JSON_UNESCAPED_UNICODE);
            exit;
        }
    }

    // Validar correção (mínimo 15 caracteres)
    if (strlen($dados['correcao']) < 15) {
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'error' => 'Correção deve ter pelo menos 15 caracteres'
        ], JSON_UNESCAPED_UNICODE);
        exit;
    }

    // Validar correção (máximo 1000 caracteres)
    if (strlen($dados['correcao']) > 1000) {
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'error' => 'Correção não pode ter mais de 1000 caracteres'
        ], JSON_UNESCAPED_UNICODE);
        exit;
    }

    // Instanciar serviço de NF-e
    $nfeService = new \App\Services\NFeService();

    // Enviar carta de correção
    $resultado = $nfeService->cartaCorrecao($dados);

    if ($resultado['success']) {
        http_response_code(200);
        echo json_encode([
            'success' => true,
            'chave_acesso' => $resultado['chave_acesso'],
            'protocolo_cce' => $resultado['protocolo_cce'],
            'numero_sequencial' => $resultado['numero_sequencial'],
            'data_registro' => $resultado['data_registro'],
            'cStat' => $resultado['cStat'] ?? null,
            'xml_path' => $resultado['xml_path'] ?? null,
            'pdf_path' => $resultado['pdf_path'] ?? null,
            'message' => 'Carta de Correção registrada com sucesso'
        ], JSON_UNESCAPED_UNICODE);
    } else {
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'cStat' => $resultado['cStat'] ?? null,
            'error' => $resultado['error']
        ], JSON_UNESCAPED_UNICODE);
    }
} catch (\Exception $e) {
    error_log("Erro ao enviar carta de correção: " . $e->getMessage());
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => 'Erro interno: ' . $e->getMessage()
    ], JSON_UNESCAPED_UNICODE);
}

