<?php

/**
 * API de Emissão de NFC-e (Nota Fiscal de Consumidor Eletrônica)
 *
 * Endpoint: POST /api/nfce/emitir.php
 * Content-Type: application/json
 */

// ATIVAR EXIBIÇÃO DE ERROS PARA DEBUG
error_reporting(E_ALL);
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

// Lidar com preflight OPTIONS
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// Apenas POST é permitido
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode([
        'success' => false,
        'error' => 'Método não permitido. Use POST.'
    ]);
    exit;
}

// Autoload
try {
    require_once __DIR__ . '/../../vendor/autoload.php';
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => 'Erro ao carregar autoload',
        'message' => $e->getMessage(),
        'file' => $e->getFile(),
        'line' => $e->getLine()
    ]);
    exit;
}

use App\Services\NFCeService;

try {
    // Ler dados JSON
    $json = file_get_contents('php://input');
    $dados = json_decode($json, true);

    if (!$dados) {
        throw new Exception('JSON inválido');
    }

    // Validações básicas
    if (!isset($dados['empresa'])) {
        throw new Exception('Dados da empresa são obrigatórios');
    }

    if (!isset($dados['itens']) || empty($dados['itens'])) {
        throw new Exception('É necessário informar pelo menos 1 item');
    }

    // Validar empresa
    $camposEmpresa = ['nome', 'cnpj', 'inscricao_estadual', 'estado', 'cidade', 'endereco', 'numero', 'bairro', 'cep'];
    foreach ($camposEmpresa as $campo) {
        if (!isset($dados['empresa'][$campo]) || empty($dados['empresa'][$campo])) {
            throw new Exception("Campo empresa.$campo é obrigatório");
        }
    }

    // Validar CSC (obrigatório para NFC-e) - pode estar em "nfce" ou "empresa"
    $csc = null;
    $csc_id = null;

    // Prioridade: bloco "nfce", depois bloco "empresa"
    if (isset($dados['nfce']['csc']) && !empty($dados['nfce']['csc'])) {
        $csc = $dados['nfce']['csc'];
        $csc_id = $dados['nfce']['csc_id'] ?? null;
    } elseif (isset($dados['empresa']['csc']) && !empty($dados['empresa']['csc'])) {
        $csc = $dados['empresa']['csc'];
        $csc_id = $dados['empresa']['csc_id'] ?? null;
    }

    if (!$csc) {
        throw new Exception('CSC (Código de Segurança do Contribuinte) é obrigatório para NFC-e. Informe em "nfce.csc" ou "empresa.csc"');
    }

    if (!$csc_id) {
        throw new Exception('CSC ID é obrigatório para NFC-e. Informe em "nfce.csc_id" ou "empresa.csc_id"');
    }

    // Normalizar CSC para o bloco empresa (para compatibilidade com o serviço)
    if (isset($dados['nfce']['csc'])) {
        $dados['empresa']['csc'] = $dados['nfce']['csc'];
        $dados['empresa']['csc_id'] = $dados['nfce']['csc_id'];
    }

    // Validar itens
    foreach ($dados['itens'] as $index => $item) {
        $camposItem = ['codigo', 'descricao', 'ncm', 'cfop', 'unidade_comercial', 'quantidade_comercial', 'valor_unitario', 'valor_total'];
        foreach ($camposItem as $campo) {
            if (!isset($item[$campo])) {
                throw new Exception("Campo itens[$index].$campo é obrigatório");
            }
        }
    }

    // Criar serviço e emitir NFC-e
    $service = new NFCeService();
    $resultado = $service->emitir($dados);

    // Retornar resultado
    if ($resultado['success']) {
        http_response_code(200);
    } else {
        http_response_code(400);
    }

    echo json_encode($resultado, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);

} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage(),
        'type' => get_class($e),
        'file' => $e->getFile(),
        'line' => $e->getLine(),
        'trace' => $e->getTraceAsString()
    ], JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
} catch (Error $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => 'Erro fatal: ' . $e->getMessage(),
        'type' => get_class($e),
        'file' => $e->getFile(),
        'line' => $e->getLine(),
        'trace' => $e->getTraceAsString()
    ], JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
}

