<?php

/**
 * API de Consulta de NFC-e pela Chave de Acesso
 *
 * Endpoint: POST /api/nfce/consultar.php
 * Content-Type: application/json
 */

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

// Lidar com preflight OPTIONS
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// Apenas POST é permitido
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode([
        'success' => false,
        'error' => 'Método não permitido. Use POST.'
    ]);
    exit;
}

// Autoload
require_once __DIR__ . '/../../vendor/autoload.php';

use NFePHP\NFe\Tools;
use NFePHP\NFe\Common\Standardize;
use NFePHP\Common\Certificate;

try {
    // Ler dados JSON
    $json = file_get_contents('php://input');
    $dados = json_decode($json, true);

    if (!$dados) {
        throw new Exception('JSON inválido');
    }

    // Validações básicas
    if (!isset($dados['chave_acesso']) || empty($dados['chave_acesso'])) {
        throw new Exception('Chave de acesso é obrigatória');
    }

    if (!isset($dados['empresa']) || !isset($dados['empresa']['cnpj'])) {
        throw new Exception('CNPJ da empresa é obrigatório');
    }

    // Validar chave de acesso (44 dígitos)
    $chaveAcesso = preg_replace('/[^0-9]/', '', $dados['chave_acesso']);
    if (strlen($chaveAcesso) != 44) {
        throw new Exception('Chave de acesso deve ter 44 dígitos');
    }

    // Carregar certificado
    $cnpj = preg_replace('/[^0-9]/', '', $dados['empresa']['cnpj']);
    $certificadoPath = __DIR__ . "/../../certificados/empresas/{$cnpj}/{$cnpj}.pfx";

    if (!file_exists($certificadoPath)) {
        throw new Exception("Certificado não encontrado para o CNPJ {$cnpj}");
    }

    $configCertificado = require __DIR__ . '/../../config_certificado.php';
    $certificado = \App\Core\CertificateHelper::loadPfx($certificadoPath, $configCertificado['senha_certificado']);

    // Configuração
    $config = [
        "atualizacao" => date('Y-m-d H:i:s'),
        "tpAmb" => 1,
        "razaosocial" => $dados['empresa']['nome'] ?? '',
        "cnpj" => $cnpj,
        "siglaUF" => $dados['empresa']['estado'] ?? 'PE',
        "schemes" => "PL_009_V4",
        "versao" => '4.00'
    ];

    $configJson = json_encode($config);
    $tools = new Tools($configJson, $certificado);
    $tools->model(65); // Modelo 65 = NFC-e

    // Consultar na SEFAZ
    $response = $tools->sefazConsultaChave($chaveAcesso);

    $st = new Standardize($response);
    $std = $st->toStd();

    // Verificar resultado
    if (isset($std->protNFe->infProt->cStat)) {
        $cStat = $std->protNFe->infProt->cStat;
        $xMotivo = $std->protNFe->infProt->xMotivo;

        if (in_array($cStat, [100, 150])) {
            // NFC-e autorizada
            http_response_code(200);
            echo json_encode([
                'success' => true,
                'chave_acesso' => $chaveAcesso,
                'cStat' => $cStat,
                'xMotivo' => $xMotivo,
                'protocolo' => $std->protNFe->infProt->nProt ?? null,
                'data_autorizacao' => $std->protNFe->infProt->dhRecbto ?? null,
                'situacao' => 'Autorizada'
            ], JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
        } elseif ($cStat == 101) {
            // NFC-e cancelada
            http_response_code(200);
            echo json_encode([
                'success' => true,
                'chave_acesso' => $chaveAcesso,
                'cStat' => $cStat,
                'xMotivo' => $xMotivo,
                'situacao' => 'Cancelada'
            ], JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
        } else {
            // Outro status
            http_response_code(200);
            echo json_encode([
                'success' => false,
                'chave_acesso' => $chaveAcesso,
                'cStat' => $cStat,
                'xMotivo' => $xMotivo,
                'situacao' => 'Outro'
            ], JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
        }
    } else {
        throw new Exception('Resposta inválida da SEFAZ');
    }

} catch (Exception $e) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ], JSON_UNESCAPED_UNICODE);
}

