<?php

/**
 * API de Cancelamento de NFC-e
 *
 * Endpoint: POST /api/nfce/cancelar.php
 * Content-Type: application/json
 */

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

// Lidar com preflight OPTIONS
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// Apenas POST é permitido
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode([
        'success' => false,
        'error' => 'Método não permitido. Use POST.'
    ]);
    exit;
}

// Autoload
require_once __DIR__ . '/../../vendor/autoload.php';

use App\Services\NFCeService;

try {
    // Ler dados JSON
    $json = file_get_contents('php://input');
    $dados = json_decode($json, true);

    if (!$dados) {
        throw new Exception('JSON inválido');
    }

    // Validações básicas
    $camposObrigatorios = ['chave_acesso', 'justificativa', 'protocolo_autorizacao', 'empresa'];
    foreach ($camposObrigatorios as $campo) {
        if (!isset($dados[$campo]) || empty($dados[$campo])) {
            throw new Exception("Campo $campo é obrigatório");
        }
    }

    // Validar justificativa (mínimo 15 caracteres)
    if (strlen($dados['justificativa']) < 15) {
        throw new Exception('Justificativa deve ter no mínimo 15 caracteres');
    }

    // Validar empresa
    $camposEmpresa = ['nome', 'cnpj', 'inscricao_estadual', 'estado'];
    foreach ($camposEmpresa as $campo) {
        if (!isset($dados['empresa'][$campo]) || empty($dados['empresa'][$campo])) {
            throw new Exception("Campo empresa.$campo é obrigatório");
        }
    }

    // Criar serviço e cancelar NFC-e
    $service = new NFCeService();
    $resultado = $service->cancelar($dados);

    // Retornar resultado
    if ($resultado['success']) {
        http_response_code(200);
    } else {
        http_response_code(400);
    }

    echo json_encode($resultado, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);

} catch (Exception $e) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ], JSON_UNESCAPED_UNICODE);
}

