<?php

/**
 * API de Emissão de MDF-e (Manifesto Eletrônico de Documentos Fiscais)
 *
 * Endpoint: POST /api/mdfe/emitir.php
 * Content-Type: application/json
 */

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

// Lidar com preflight OPTIONS
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// Apenas POST é permitido
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode([
        'success' => false,
        'error' => 'Método não permitido. Use POST.'
    ]);
    exit;
}

// Autoload
require_once __DIR__ . '/../../vendor/autoload.php';

use App\Services\MDFeService;

try {
    // Ler dados JSON
    $json = file_get_contents('php://input');
    $dados = json_decode($json, true);

    if (!$dados) {
        throw new Exception('JSON inválido');
    }

    // Validações básicas
    if (!isset($dados['emitente'])) {
        throw new Exception('Dados do emitente são obrigatórios');
    }

    if (!isset($dados['veiculo'])) {
        throw new Exception('Dados do veículo são obrigatórios');
    }

    if (!isset($dados['condutores']) || empty($dados['condutores'])) {
        throw new Exception('É necessário informar pelo menos 1 condutor');
    }

    if (!isset($dados['municipios_carregamento']) || empty($dados['municipios_carregamento'])) {
        throw new Exception('É necessário informar pelo menos 1 município de carregamento');
    }

    if (!isset($dados['municipios_descarregamento']) || empty($dados['municipios_descarregamento'])) {
        throw new Exception('É necessário informar pelo menos 1 município de descarregamento');
    }

    if (!isset($dados['uf_inicio']) || !isset($dados['uf_fim'])) {
        throw new Exception('UF de início e fim são obrigatórias');
    }

    // Validar emitente
    $camposEmitente = ['nome', 'cnpj', 'inscricao_estadual', 'estado', 'cidade', 'endereco', 'numero', 'bairro', 'cep'];
    foreach ($camposEmitente as $campo) {
        if (!isset($dados['emitente'][$campo]) || empty($dados['emitente'][$campo])) {
            throw new Exception("Campo emitente.$campo é obrigatório");
        }
    }

    // Validar veículo
    $camposVeiculo = ['placa', 'tara', 'capacidade_kg', 'uf'];
    foreach ($camposVeiculo as $campo) {
        if (!isset($dados['veiculo'][$campo]) || empty($dados['veiculo'][$campo])) {
            throw new Exception("Campo veiculo.$campo é obrigatório");
        }
    }

    // Validar condutores
    foreach ($dados['condutores'] as $index => $condutor) {
        if (!isset($condutor['nome']) || !isset($condutor['cpf'])) {
            throw new Exception("Condutor $index: nome e CPF são obrigatórios");
        }
    }

    // Criar serviço e emitir MDF-e
    $service = new MDFeService();
    $resultado = $service->emitir($dados);

    // Retornar resultado
    if ($resultado['success']) {
        http_response_code(200);
    } else {
        http_response_code(400);
    }

    echo json_encode($resultado, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);

} catch (Exception $e) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ], JSON_UNESCAPED_UNICODE);
}

