<?php

declare(strict_types=1);

namespace App\Helpers;

/**
 * Helper para gerenciar URLs
 * Funciona tanto com VirtualHost quanto sem
 */
class UrlHelper
{
    /**
     * Retorna o base path da aplicação
     * IMPORTANTE: Detecta dinamicamente baseado no SCRIPT_NAME
     * NUNCA força um diretório específico via APP_URL
     * Cada sistema (/printjetdigital, /Systhema, etc) detecta seu próprio diretório
     */
    public static function basePath(): string
    {
        static $cachedBasePath = null;

        if ($cachedBasePath !== null) {
            return $cachedBasePath;
        }

        // PRIORIDADE: Detecta pelo SCRIPT_NAME (sempre correto)
        // Isso garante que cada sistema detecte seu próprio diretório
        $scriptName = $_SERVER['SCRIPT_NAME'] ?? '';
        $basePath = str_replace('\\', '/', dirname($scriptName));
        $basePath = rtrim($basePath, '/');

        // Se está na raiz (/) tenta detectar via SCRIPT_FILENAME/DOCUMENT_ROOT
        if ($basePath === '' || $basePath === '/') {
            $scriptFilename = $_SERVER['SCRIPT_FILENAME'] ?? '';
            $documentRoot = $_SERVER['DOCUMENT_ROOT'] ?? '';

            if ($scriptFilename && $documentRoot) {
                $normalizedScript = str_replace('\\', '/', $scriptFilename);
                $normalizedRoot = rtrim(str_replace('\\', '/', $documentRoot), '/');

                if (strpos($normalizedScript, $normalizedRoot) === 0) {
                    $relativePath = trim(dirname(substr($normalizedScript, strlen($normalizedRoot))), '/');
                    if ($relativePath !== '') {
                        $basePath = '/' . $relativePath;
                    }
                }
            }

            // Fallback adicional: usar REQUEST_URI se disponível
            if (($basePath === '' || $basePath === '/') && isset($_SERVER['REQUEST_URI'])) {
                $requestUri = parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);
                if ($requestUri && $requestUri !== '/') {
                    $uriParts = explode('/', trim($requestUri, '/'));
                    // Se o primeiro segmento não é um arquivo conhecido, pode ser o base path
                    if (!empty($uriParts[0]) && !preg_match('/\.(php|html|htm)$/i', $uriParts[0])) {
                        $basePath = '/' . $uriParts[0];
                    }
                }
            }

            // Se ainda estiver vazio, não há base path
            if ($basePath === '' || $basePath === '/') {
                return $cachedBasePath = '';
            }
        }

        // Retorna o base path detectado dinamicamente
        // NUNCA usa APP_URL para forçar um diretório - isso quebraria sistemas em outros diretórios
        return $cachedBasePath = $basePath;
    }

    /**
     * Retorna URL completa com base path
     */
    public static function url(string $path = '/'): string
    {
        $basePath = self::basePath();
        $normalizedPath = '/' . ltrim($path, '/');

        if ($basePath === '') {
            return $normalizedPath;
        }

        return $basePath . $normalizedPath;
    }

    /**
     * Retorna URL do asset
     * Assets sempre precisam do base path quando o sistema está em subdiretório
     */
    public static function asset(string $path): string
    {
        // Normalizar o caminho (garantir que comece com /)
        $path = '/' . ltrim($path, '/');

        // Sempre usar url() para garantir que o base path seja incluído quando necessário
        return self::url($path);
    }
}

