<?php

namespace App\Helpers;

use Exception;
use PDO;

/**
 * Helper para gerenciar skins do sistema
 */
class SkinHelper
{
    private static ?PDO $db = null;
    private static ?string $cachedSkin = null;
    private static ?array $cachedConfig = null;

    /**
     * Define a conexão com o banco de dados
     */
    public static function setDatabase(PDO $db): void
    {
        self::$db = $db;
    }

    /**
     * Verifica se o banco de dados foi configurado
     */
    public static function hasDatabase(): bool
    {
        return self::$db !== null;
    }

    /**
     * Obtém a skin ativa do sistema
     * Ordem de prioridade:
     * 1. Banco de dados (configuracoes_sistema)
     * 2. Variável de ambiente (APP_SKIN)
     * 3. Default (aurion)
     *
     * NOTA: Cache desabilitado para permitir mudanças dinâmicas no banco
     */
    public static function getActiveSkin(): string
    {
        $config = require ROOT_PATH . '/config/skins.php';

        // 1. Tentar buscar do banco de dados
        if (self::$db && ($config['detection']['source'] === 'database' || self::$db)) {
            try {
                $skin = self::getSkinFromDatabase($config);
                if ($skin && self::skinExists($skin, $config)) {
                    // Log para debug (remover em produção se necessário)
                    error_log("SkinHelper: Skin ativa detectada do banco: " . $skin);
                    return $skin;
                } else {
                    error_log("SkinHelper: Skin do banco não encontrada ou inválida: " . ($skin ?? 'null'));
                }
            } catch (Exception $e) {
                error_log("SkinHelper: Erro ao buscar skin do banco: " . $e->getMessage());
                error_log("SkinHelper: Stack trace: " . $e->getTraceAsString());
            }
        } else {
            error_log("SkinHelper: Banco de dados não disponível ou source não é 'database'");
        }

        // 2. Fallback para variável de ambiente
        if (!empty($config['detection']['fallback_env'])) {
            $envSkin = trim($config['detection']['fallback_env']);
            if (self::skinExists($envSkin, $config)) {
                error_log("SkinHelper: Usando skin do ambiente: " . $envSkin);
                return $envSkin;
            }
        }

        // 3. Default
        error_log("SkinHelper: Usando skin padrão: " . $config['default']);
        return $config['default'];
    }

    /**
     * Busca a skin do banco de dados
     */
    private static function getSkinFromDatabase(array $config): ?string
    {
        if (!self::$db) {
            return null;
        }

        $table = $config['detection']['table'] ?? 'configuracoes_sistema';
        $key = $config['detection']['key'] ?? 'skin_ativa';

        try {
            // Verificar se a tabela existe
            $stmt = self::$db->query("SHOW TABLES LIKE '{$table}'");
            if ($stmt->rowCount() === 0) {
                return null;
            }

            // Buscar a configuração
            $stmt = self::$db->prepare("SELECT valor FROM {$table} WHERE chave = :chave LIMIT 1");
            $stmt->execute(['chave' => $key]);
            $result = $stmt->fetch(PDO::FETCH_ASSOC);

            if ($result && !empty($result['valor'])) {
                return trim($result['valor']);
            }
        } catch (Exception $e) {
            error_log("SkinHelper: Erro ao buscar do banco: " . $e->getMessage());
        }

        return null;
    }

    /**
     * Verifica se uma skin existe na configuração
     */
    private static function skinExists(string $skin, array $config): bool
    {
        return isset($config['skins'][$skin]);
    }

    /**
     * Obtém a configuração completa da skin ativa
     */
    public static function getSkinConfig(): array
    {
        $config = require ROOT_PATH . '/config/skins.php';
        $active = self::getActiveSkin();

        $skinConfig = $config['skins'][$active] ?? $config['skins'][$config['default']];

        // Verificar se os arquivos da skin existem
        if (isset($skinConfig['css_variables'])) {
            $cssPath = ROOT_PATH . $skinConfig['css_variables'];
            if (!file_exists($cssPath)) {
                error_log("SkinHelper: Arquivo CSS não encontrado: " . $cssPath);
            }
        }

        return $skinConfig;
    }

    /**
     * Obtém o caminho do logo da skin ativa
     */
    public static function getLogo(): string
    {
        try {
            $skin = self::getSkinConfig();
            $logoPath = $skin['logo'] ?? '/storage/uploads/logo.png';

            // Verificar se o arquivo existe
            $fullPath = ROOT_PATH . $logoPath;
            if (file_exists($fullPath)) {
                return asset($logoPath);
            }

            // Fallback 1: Logo genérico na raiz de storage
            $fallbackPath = '/storage/uploads/logo.png';
            if (file_exists(ROOT_PATH . $fallbackPath)) {
                return asset($fallbackPath);
            }

            // Fallback 2: Logo tupan
            $fallbackTupan = '/storage/uploads/logos/tupan.png';
            if (file_exists(ROOT_PATH . $fallbackTupan)) {
                return asset($fallbackTupan);
            }

            // Fallback 3: Retornar o caminho mesmo que não exista (o onerror no HTML vai tratar)
            return asset($logoPath);
        } catch (Exception $e) {
            error_log("SkinHelper::getLogo() - Erro: " . $e->getMessage());
            // Retornar fallback seguro
            return asset('/storage/uploads/logo.png');
        }
    }

    /**
     * Obtém o caminho do favicon da skin ativa
     */
    public static function getFavicon(): string
    {
        $skin = self::getSkinConfig();
        $faviconPath = $skin['favicon'] ?? null;

        // Verificar se o arquivo de favicon existe
        if ($faviconPath) {
            $fullPath = ROOT_PATH . $faviconPath;
            if (file_exists($fullPath)) {
                return asset($faviconPath);
            }
        }

        // Se não existe, usar o logo como fallback
        $logoPath = $skin['logo'] ?? '/storage/uploads/logo.png';
        $fullLogoPath = ROOT_PATH . $logoPath;
        if (file_exists($fullLogoPath)) {
            return asset($logoPath);
        }

        // Fallback padrão (retornar o logo mesmo que não exista, para evitar 404)
        return asset($logoPath);
    }

    /**
     * Obtém o nome de exibição da skin ativa
     */
    public static function getDisplayName(): string
    {
        $skin = self::getSkinConfig();
        return $skin['display_name'];
    }

    /**
     * Obtém o caminho do arquivo CSS de variáveis da skin ativa
     */
    public static function getCssVariables(): string
    {
        $skin = self::getSkinConfig();
        return asset($skin['css_variables']);
    }

    /**
     * Obtém uma cor específica da skin ativa
     */
    public static function getColor(string $colorName): ?string
    {
        $skin = self::getSkinConfig();
        return $skin['colors'][$colorName] ?? null;
    }

    /**
     * Obtém uma variável de dark mode da skin ativa
     */
    public static function getDarkModeVar(string $varName): ?string
    {
        $skin = self::getSkinConfig();
        return $skin['dark_mode'][$varName] ?? null;
    }

    /**
     * Limpa o cache (útil após atualizar a skin)
     */
    public static function clearCache(): void
    {
        self::$cachedSkin = null;
        self::$cachedConfig = null;
    }

    /**
     * Lista todas as skins disponíveis
     */
    public static function getAvailableSkins(): array
    {
        $config = require ROOT_PATH . '/config/skins.php';
        return array_keys($config['skins']);
    }
}

