<?php

namespace App\Helpers;

use App\Services\InadimplenciaService;

class InadimplenciaHelper
{
    private static $inadimplenciaService;

    /**
     * Verifica se um cliente está inadimplente universalmente
     */
    public static function verificarInadimplenciaUniversal($clienteId, $companyId = null)
    {
        if (!self::$inadimplenciaService) {
            self::$inadimplenciaService = new InadimplenciaService();
        }

        // Se não foi passado companyId, usar da sessão
        if (!$companyId) {
            $companyId = $_SESSION['company_id'] ?? null;
        }

        if (!$companyId) {
            return ['inadimplente' => false];
        }

        // Verificar se bloqueio universal está ativo
        if (!self::$inadimplenciaService->isBloqueioUniversalAtivo($companyId)) {
            return ['inadimplente' => false];
        }

        $diasBloqueio = self::$inadimplenciaService->getDiasBloqueio($companyId);

        return self::$inadimplenciaService->isClienteInadimplenteUniversal($clienteId, $diasBloqueio);
    }

    /**
     * Obtém mensagem de bloqueio formatada
     */
    public static function getMensagemBloqueio($resultado)
    {
        if (!$resultado['inadimplente']) {
            return null;
        }

        $mensagens = [
            'default' => 'Cliente inadimplente. Operação bloqueada.',
            'universal' => 'Cliente inadimplente em outras empresas. Bloqueio universal ativo.',
            'empresa_especifica' => 'Cliente inadimplente na empresa {empresa}.'
        ];

        if (isset($resultado['empresa'])) {
            return str_replace('{empresa}', $resultado['empresa'], $mensagens['empresa_especifica']);
        }

        return $mensagens['universal'];
    }

    /**
     * Gera alerta JavaScript para inadimplência
     */
    public static function getAlertaJavaScript($resultado)
    {
        if (!$resultado['inadimplente']) {
            return '';
        }

        $mensagem = self::getMensagemBloqueio($resultado);

        return "
        <script>
        document.addEventListener('DOMContentLoaded', function() {
            Swal.fire({
                title: 'Cliente Inadimplente',
                text: '{$mensagem}',
                icon: 'warning',
                confirmButtonText: 'Entendi',
                confirmButtonColor: '#dc3545'
            });
        });
        </script>";
    }

    /**
     * Verifica e exibe alerta de inadimplência para um cliente
     */
    public static function verificarEExibirAlerta($clienteId, $companyId = null)
    {
        $resultado = self::verificarInadimplenciaUniversal($clienteId, $companyId);

        if ($resultado['inadimplente']) {
            echo self::getAlertaJavaScript($resultado);
            return true;
        }

        return false;
    }

    /**
     * Obtém status de inadimplência para exibição em listas
     */
    public static function getStatusInadimplencia($clienteId, $companyId = null)
    {
        $resultado = self::verificarInadimplenciaUniversal($clienteId, $companyId);

        if ($resultado['inadimplente']) {
            return [
                'status' => 'inadimplente',
                'badge' => 'danger',
                'texto' => 'Inadimplente',
                'icone' => 'bi-exclamation-triangle',
                'motivo' => $resultado['motivo'] ?? 'Inadimplência detectada'
            ];
        }

        return [
            'status' => 'regular',
            'badge' => 'success',
            'texto' => 'Regular',
            'icone' => 'bi-check-circle',
            'motivo' => null
        ];
    }

    /**
     * Verifica se deve bloquear uma operação
     */
    public static function deveBloquearOperacao($clienteId, $companyId = null)
    {
        $resultado = self::verificarInadimplenciaUniversal($clienteId, $companyId);
        return $resultado['inadimplente'];
    }

    /**
     * Obtém dados para relatório de inadimplência universal
     */
    public static function getRelatorioInadimplenciaUniversal()
    {
        if (!self::$inadimplenciaService) {
            self::$inadimplenciaService = new InadimplenciaService();
        }

        // Implementar lógica de relatório
        // Retornar lista de clientes inadimplentes em todas as empresas
        return [
            'total_inadimplentes' => 0,
            'empresas_afetadas' => [],
            'clientes' => []
        ];
    }
}
