<?php

/**
 * Funções globais para formatação e utilitários
 */

if (!function_exists('url')) {
    /**
     * Gera URL com base path correto
     */
    function url(string $path = '/'): string
    {
        return \App\Helpers\UrlHelper::url($path);
    }
}

if (!function_exists('asset')) {
    /**
     * Gera URL para assets
     */
    function asset(string $path): string
    {
        return \App\Helpers\UrlHelper::asset($path);
    }
}

if (!function_exists('formatCurrency')) {
    /**
     * Formata um valor como moeda brasileira
     */
    function formatCurrency($value): string
    {
        if (!is_numeric($value)) {
            return '0,00';
        }

        return number_format((float) $value, 2, ',', '.');
    }
}

if (!function_exists('formatDate')) {
    /**
     * Formata uma data para o padrão brasileiro
     */
    function formatDate($date, $format = 'd/m/Y'): string
    {
        if (empty($date)) {
            return '';
        }

        try {
            $dateObj = new DateTime($date);
            return $dateObj->format($format);
        } catch (Exception $e) {
            return '';
        }
    }
}

if (!function_exists('formatDateTime')) {
    /**
     * Formata uma data e hora para o padrão brasileiro
     */
    function formatDateTime($datetime, $format = 'd/m/Y H:i'): string
    {
        if (empty($datetime)) {
            return '';
        }

        try {
            $dateObj = new DateTime($datetime);
            return $dateObj->format($format);
        } catch (Exception $e) {
            return '';
        }
    }
}

if (!function_exists('formatCPF')) {
    /**
     * Formata CPF
     */
    function formatCPF($cpf): string
    {
        $cpf = preg_replace('/[^0-9]/', '', $cpf);
        if (strlen($cpf) === 11) {
            return substr($cpf, 0, 3) . '.' . substr($cpf, 3, 3) . '.' . substr($cpf, 6, 3) . '-' . substr($cpf, 9, 2);
        }
        return $cpf;
    }
}

if (!function_exists('formatCNPJ')) {
    /**
     * Formata CNPJ
     */
    function formatCNPJ($cnpj): string
    {
        $cnpj = preg_replace('/[^0-9]/', '', $cnpj);
        if (strlen($cnpj) === 14) {
            return substr($cnpj, 0, 2) . '.' . substr($cnpj, 2, 3) . '.' . substr($cnpj, 5, 3) . '/' . substr($cnpj, 8, 4) . '-' . substr($cnpj, 12, 2);
        }
        return $cnpj;
    }
}

if (!function_exists('formatDocument')) {
    /**
     * Formata documento automaticamente (CPF ou CNPJ)
     */
    function formatDocument($document): string
    {
        if (empty($document)) {
            return '';
        }

        $doc = preg_replace('/[^0-9]/', '', $document);
        $length = strlen($doc);

        if ($length === 11) {
            return formatCPF($doc);
        } elseif ($length === 14) {
            return formatCNPJ($doc);
        }

        // Se já está formatado ou não corresponde a CPF/CNPJ, retorna como está
        return $document;
    }
}

if (!function_exists('formatPhone')) {
    /**
     * Formata telefone
     */
    function formatPhone($phone): string
    {
        $phone = preg_replace('/[^0-9]/', '', $phone);
        if (strlen($phone) === 11) {
            return '(' . substr($phone, 0, 2) . ') ' . substr($phone, 2, 5) . '-' . substr($phone, 7, 4);
        } elseif (strlen($phone) === 10) {
            return '(' . substr($phone, 0, 2) . ') ' . substr($phone, 2, 4) . '-' . substr($phone, 6, 4);
        }
        return $phone;
    }
}

if (!function_exists('formatCEP')) {
    /**
     * Formata CEP
     */
    function formatCEP($cep): string
    {
        $cep = preg_replace('/[^0-9]/', '', $cep);
        if (strlen($cep) === 8) {
            return substr($cep, 0, 5) . '-' . substr($cep, 5, 3);
        }
        return $cep;
    }
}

if (!function_exists('sanitizeInput')) {
    /**
     * Sanitiza entrada do usuário
     */
    function sanitizeInput($input): string
    {
        return htmlspecialchars(trim($input), ENT_QUOTES, 'UTF-8');
    }
}

if (!function_exists('getStatusBadge')) {
    /**
     * Retorna badge HTML para status
     */
    function getStatusBadge($status): string
    {
        $badges = [
            'ativo' => '<span class="badge bg-success">Ativo</span>',
            'inativo' => '<span class="badge bg-secondary">Inativo</span>',
            'pendente' => '<span class="badge bg-warning">Pendente</span>',
            'concluido' => '<span class="badge bg-success">Concluído</span>',
            'concluida' => '<span class="badge bg-success">Concluída</span>',
            'cancelado' => '<span class="badge bg-danger">Cancelado</span>',
            'cancelada' => '<span class="badge bg-danger">Cancelada</span>',
            'pago' => '<span class="badge bg-success">Pago</span>',
            'vencido' => '<span class="badge bg-danger">Vencido</span>',
            'recebido' => '<span class="badge bg-success">Recebido</span>',
            'aguardando' => '<span class="badge bg-warning">Aguardando</span>',
        ];

        return $badges[$status] ?? '<span class="badge bg-secondary">' . ucfirst($status) . '</span>';
    }
}

if (!function_exists('getTypeBadge')) {
    /**
     * Retorna badge HTML para tipo
     */
    function getTypeBadge($type): string
    {
        $badges = [
            'entrada' => '<span class="badge bg-success">Entrada</span>',
            'saida' => '<span class="badge bg-danger">Saída</span>',
            'cliente' => '<span class="badge bg-primary">Cliente</span>',
            'fornecedor' => '<span class="badge bg-info">Fornecedor</span>',
            'funcionario' => '<span class="badge bg-warning">Funcionário</span>',
            'produto' => '<span class="badge bg-success">Produto</span>',
            'servico' => '<span class="badge bg-info">Serviço</span>',
        ];

        return $badges[$type] ?? '<span class="badge bg-secondary">' . ucfirst($type) . '</span>';
    }
}

if (!function_exists('moduloFrenteLojaAtivo')) {
    /**
     * Verifica se o módulo Frente de Loja (id = 18) está ativo
     */
    function moduloFrenteLojaAtivo(): bool
    {
        static $cache = null;

        if ($cache !== null) {
            return $cache;
        }

        $cache = false;

        try {
            $dbConfig = require \ROOT_PATH . '/config/database.php';
            $config = $dbConfig['master'];

            $dsn = sprintf(
                '%s:host=%s;port=%d;dbname=%s;charset=%s',
                $config['driver'],
                $config['host'],
                $config['port'],
                $config['database'],
                $config['charset']
            );

            $db = new \PDO($dsn, $config['username'], $config['password'], $config['options']);

            // Verificar se a tabela modulos existe
            $stmt = $db->query("SHOW TABLES LIKE 'modulos'");
            if (!$stmt->fetchColumn()) {
                return $cache = false;
            }

            // Verificar se o módulo 18 está ativo
            $stmt = $db->prepare("SELECT ativo FROM modulos WHERE id = 18 LIMIT 1");
            $stmt->execute();
            $result = $stmt->fetch(\PDO::FETCH_ASSOC);

            if ($result) {
                $ativo = strtoupper((string) ($result['ativo'] ?? 'NAO'));
                $cache = in_array($ativo, ['SIM', '1', 'TRUE', 'S', 'YES'], true);
            }
        } catch (\Exception $e) {
            error_log("Erro ao verificar módulo Frente de Loja: " . $e->getMessage());
            $cache = false;
        }

        return $cache;
    }
}