<?php

declare(strict_types=1);

namespace App\Helpers;

/**
 * Helper para consulta de CNPJ
 * Utiliza API ReceitaWS
 */
class CnpjHelper
{
    /**
     * Busca dados da empresa por CNPJ
     *
     * @param string $cnpj CNPJ sem formatação
     * @return array|null Dados da empresa ou null se não encontrado
     */
    public static function buscar(string $cnpj): ?array
    {
        // Remove formatação
        $cnpj = preg_replace('/[^0-9]/', '', $cnpj);

        if (strlen($cnpj) !== 14) {
            return null;
        }

        try {
            // API ReceitaWS (gratuita)
            $url = "https://www.receitaws.com.br/v1/cnpj/{$cnpj}";

            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $url);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_TIMEOUT, 15);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                'Accept: application/json',
                'User-Agent: Systhema ERP'
            ]);

            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);

            if ($httpCode !== 200) {
                return null;
            }

            $data = json_decode($response, true);

            if (isset($data['status']) && $data['status'] === 'ERROR') {
                return null;
            }

            return [
                'cnpj' => $data['cnpj'] ?? '',
                'razao_social' => $data['nome'] ?? '',
                'nome_fantasia' => $data['fantasia'] ?? '',
                'abertura' => $data['abertura'] ?? '',
                'situacao' => $data['situacao'] ?? '',
                'tipo' => $data['tipo'] ?? '',
                'porte' => $data['porte'] ?? '',
                'natureza_juridica' => $data['natureza_juridica'] ?? '',
                'email' => $data['email'] ?? '',
                'telefone' => $data['telefone'] ?? '',
                'capital_social' => $data['capital_social'] ?? '',
                'logradouro' => $data['logradouro'] ?? '',
                'numero' => $data['numero'] ?? '',
                'complemento' => $data['complemento'] ?? '',
                'bairro' => $data['bairro'] ?? '',
                'municipio' => $data['municipio'] ?? '',
                'uf' => $data['uf'] ?? '',
                'cep' => $data['cep'] ?? '',
                'atividade_principal' => $data['atividade_principal'][0]['text'] ?? '',
            ];

        } catch (\Exception $e) {
            error_log("Erro ao buscar CNPJ: " . $e->getMessage());
            return null;
        }
    }

    /**
     * Valida CNPJ
     */
    public static function validar(string $cnpj): bool
    {
        $cnpj = preg_replace('/[^0-9]/', '', $cnpj);

        if (strlen($cnpj) != 14) {
            return false;
        }

        if (preg_match('/(\d)\1{13}/', $cnpj)) {
            return false;
        }

        $soma = 0;
        $multiplicador = 5;

        for ($i = 0; $i < 12; $i++) {
            $soma += $cnpj[$i] * $multiplicador;
            $multiplicador = ($multiplicador == 2) ? 9 : $multiplicador - 1;
        }

        $resto = $soma % 11;
        $digito1 = ($resto < 2) ? 0 : 11 - $resto;

        if ($cnpj[12] != $digito1) {
            return false;
        }

        $soma = 0;
        $multiplicador = 6;

        for ($i = 0; $i < 13; $i++) {
            $soma += $cnpj[$i] * $multiplicador;
            $multiplicador = ($multiplicador == 2) ? 9 : $multiplicador - 1;
        }

        $resto = $soma % 11;
        $digito2 = ($resto < 2) ? 0 : 11 - $resto;

        return $cnpj[13] == $digito2;
    }
}

