<?php

declare(strict_types=1);

namespace App\Helpers;

/**
 * Helper para consulta de CEP
 * Utiliza API ViaCEP
 */
class CepHelper
{
    /**
     * Busca endereço por CEP
     *
     * @param string $cep CEP sem formatação
     * @return array|null Dados do endereço ou null se não encontrado
     */
    public static function buscar(string $cep): ?array
    {
        // Remove formatação
        $cep = preg_replace('/[^0-9]/', '', $cep);

        if (strlen($cep) !== 8) {
            return null;
        }

        try {
            // API ViaCEP
            $url = "https://viacep.com.br/ws/{$cep}/json/";

            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $url);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_TIMEOUT, 10);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);

            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);

            if ($httpCode !== 200) {
                return null;
            }

            $data = json_decode($response, true);

            if (isset($data['erro']) && $data['erro'] === true) {
                return null;
            }

            return [
                'cep' => $data['cep'] ?? '',
                'logradouro' => $data['logradouro'] ?? '',
                'complemento' => $data['complemento'] ?? '',
                'bairro' => $data['bairro'] ?? '',
                'localidade' => $data['localidade'] ?? '',
                'uf' => $data['uf'] ?? '',
                'ibge' => $data['ibge'] ?? '',
                'gia' => $data['gia'] ?? '',
                'ddd' => $data['ddd'] ?? '',
                'siafi' => $data['siafi'] ?? ''
            ];

        } catch (\Exception $e) {
            error_log("Erro ao buscar CEP: " . $e->getMessage());
            return null;
        }
    }
}

