<?php

declare(strict_types=1);

namespace App\Controllers;

use Exception;

/**
 * Controller de Configuração Inicial do Tenant
 * Permite criar o primeiro usuário admin de um tenant novo
 */
class TenantSetupController extends BaseController
{
    /**
     * Exibe formulário de setup inicial
     */
    public function showSetup(): void
    {
        // Verifica se já tem usuários cadastrados
        if ($this->db) {
            try {
                $stmt = $this->db->query("SELECT COUNT(*) as count FROM users");
                $result = $stmt->fetch();

                if ($result && $result['count'] > 0) {
                    // Já tem usuários, redireciona para login
                    $this->redirect('login');
                    return;
                }
            } catch (Exception $e) {
                // Tabela não existe ainda, precisa criar schema
            }
        }

        // Pega informações do tenant
        $tenantName = 'Novo Tenant';
        $tenantSubdomain = 'desconhecido';

        if ($this->tenantManager && $this->tenantManager->getCurrentTenant()) {
            $tenant = $this->tenantManager->getCurrentTenant();
            $tenantName = $tenant['name'] ?? $tenantName;
            $tenantSubdomain = $tenant['subdomain'] ?? $tenantSubdomain;
        }

        $this->view('tenant/setup', [
            'pageTitle' => 'Configuração Inicial',
            'tenantName' => $tenantName,
            'tenantSubdomain' => $tenantSubdomain
        ]);
    }

    /**
     * Cria o primeiro usuário admin do tenant
     */
    public function createFirstUser(): void
    {
        // Valida dados
        $name = $this->request->post('name');
        $email = $this->request->post('email');
        $password = $this->request->post('password');
        $passwordConfirmation = $this->request->post('password_confirmation');

        if (empty($name) || empty($email) || empty($password)) {
            $this->error('Todos os campos são obrigatórios');
            return;
        }

        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $this->error('Email inválido');
            return;
        }

        if (strlen($password) < 6) {
            $this->error('A senha deve ter no mínimo 6 caracteres');
            return;
        }

        if ($password !== $passwordConfirmation) {
            $this->error('As senhas não conferem');
            return;
        }

        try {
            // Verifica se já tem usuários
            if ($this->db) {
                try {
                    $stmt = $this->db->query("SELECT COUNT(*) as count FROM users");
                    $result = $stmt->fetch();

                    if ($result && $result['count'] > 0) {
                        $this->error('Tenant já possui usuários cadastrados');
                        return;
                    }
                } catch (Exception $e) {
                    // Tabela não existe, precisa criar
                    $this->createTenantSchema();
                }
            }

            // Pega ID do tenant atual
            $tenant = $this->tenantManager->getCurrentTenant();
            if (!$tenant) {
                $this->error('Tenant não identificado');
                return;
            }

            // Cria o usuário admin
            $passwordHash = password_hash($password, PASSWORD_BCRYPT);

            $stmt = $this->db->prepare("
                INSERT INTO users (tenant_id, name, email, password, role, status, created_at, updated_at)
                VALUES (:tenant_id, :name, :email, :password, 'admin', 'active', NOW(), NOW())
            ");

            $stmt->execute([
                'tenant_id' => $tenant['id'],
                'name' => $name,
                'email' => $email,
                'password' => $passwordHash
            ]);

            $this->success('Usuário admin criado com sucesso! Faça login para acessar o sistema.', [
                'redirect' => 'login'
            ]);

        } catch (Exception $e) {
            error_log("Erro ao criar primeiro usuário: " . $e->getMessage());
            $this->error('Erro ao criar usuário: ' . $e->getMessage());
        }
    }

    /**
     * Cria o schema básico do tenant (tabela users)
     */
    private function createTenantSchema(): void
    {
        if (!$this->db) {
            return;
        }

        // Cria tabela users
        $this->db->exec("
            CREATE TABLE IF NOT EXISTS users (
                id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
                tenant_id INT UNSIGNED NOT NULL,
                name VARCHAR(255) NOT NULL,
                email VARCHAR(255) NOT NULL,
                password VARCHAR(255) NOT NULL,
                role ENUM('admin', 'user', 'manager') DEFAULT 'user',
                status ENUM('active', 'inactive', 'suspended') DEFAULT 'active',
                company_id INT UNSIGNED NULL,
                last_login DATETIME NULL,
                created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
                updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                UNIQUE KEY unique_email (email),
                INDEX idx_tenant (tenant_id),
                INDEX idx_email (email),
                INDEX idx_status (status)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
        ");

        // Aqui você pode adicionar outras tabelas essenciais
        // companies, products, etc.
    }
}

