<?php

namespace App\Controllers;

use Exception;

class SuporteController extends BaseController
{
    /**
     * Página inicial do suporte
     */
    public function index(): void
    {
        // Verificar se o usuário é admin ou tem permissão de suporte
        $user = $this->session->get('user');
        if (!$user || ($user['role'] !== 'admin' && $user['role'] !== 'suporte')) {
            $this->response->forbidden('Acesso negado. Área restrita ao suporte.');
            return;
        }

        $this->view('suporte/index', [
            'pageTitle' => 'Suporte',
            'activeMenu' => 'suporte'
        ]);
    }

    /**
     * Exibe o contrato de licença
     */
    public function contrato(): void
    {
        // Verificar se o usuário é admin ou tem permissão de suporte
        $user = $this->session->get('user');
        if (!$user || ($user['role'] !== 'admin' && $user['role'] !== 'suporte')) {
            $this->response->forbidden('Acesso negado. Área restrita ao suporte.');
            return;
        }

        // Buscar dados da empresa atual
        $companyId = $this->getCompanyId();
        $empresa = null;

        if ($companyId) {
            try {
                $stmt = $this->db->prepare("SELECT * FROM empresas WHERE id = :id LIMIT 1");
                $stmt->execute(['id' => $companyId]);
                $empresa = $stmt->fetch();
            } catch (Exception $e) {
                error_log("Erro ao buscar empresa: " . $e->getMessage());
            }
        }

        $this->view('suporte/contrato', [
            'empresa' => $empresa,
            'pageTitle' => 'Contrato de Licença',
            'activeMenu' => 'suporte'
        ]);
    }

    /**
     * Exibe página para limpar tabelas
     */
    public function limparTabelas(): void
    {
        // Verificar se o usuário é admin ou tem permissão de suporte
        $user = $this->session->get('user');
        if (!$user || ($user['role'] !== 'admin' && $user['role'] !== 'suporte')) {
            $this->response->forbidden('Acesso negado. Área restrita ao suporte.');
            return;
        }

        $companyId = $this->getCompanyId();

        // Buscar todas as tabelas do banco de dados
        $tabelas = $this->listarTabelas($companyId);

        $this->view('suporte/limpar-tabelas', [
            'tabelas' => $tabelas,
            'pageTitle' => 'Limpar Tabelas',
            'activeMenu' => 'suporte'
        ]);
    }

    /**
     * Executa a limpeza de uma tabela específica (chamada individual)
     */
    public function executarLimpeza(): void
    {
        // Verificar se o usuário é admin ou tem permissão de suporte
        $user = $this->session->get('user');
        if (!$user || ($user['role'] !== 'admin' && $user['role'] !== 'suporte')) {
            $this->error('Acesso negado. Área restrita ao suporte.');
            return;
        }

        try {
            $companyId = $this->getCompanyId();
            $tabela = $this->request->post('tabela');

            if (empty($tabela)) {
                $this->error('Tabela não informada');
                return;
            }

            $tabelasAuxiliares = $this->getTabelasAuxiliares();
            $auxiliares = $tabelasAuxiliares[$tabela] ?? [];

            $this->db->beginTransaction();

            $limpas = [];
            $erros = [];

            // Limpar tabela principal
            try {
                $stmt = $this->db->prepare("SHOW COLUMNS FROM `{$tabela}` LIKE 'company_id'");
                $stmt->execute();
                $temCompanyId = $stmt->rowCount() > 0;

                if ($temCompanyId) {
                    $stmtCount = $this->db->prepare("SELECT COUNT(*) FROM `{$tabela}` WHERE company_id = :company_id");
                    $stmtCount->execute(['company_id' => $companyId]);
                    $totalAntes = (int) $stmtCount->fetchColumn();

                    $stmt = $this->db->prepare("DELETE FROM `{$tabela}` WHERE company_id = :company_id");
                    $stmt->execute(['company_id' => $companyId]);

                    $limpas[] = [
                        'tabela' => $tabela,
                        'linhas' => $totalAntes
                    ];
                } else {
                    $stmtCount = $this->db->query("SELECT COUNT(*) FROM `{$tabela}`");
                    $totalAntes = (int) $stmtCount->fetchColumn();

                    $this->db->exec("TRUNCATE TABLE `{$tabela}`");

                    $limpas[] = [
                        'tabela' => $tabela,
                        'linhas' => $totalAntes
                    ];
                }
            } catch (Exception $e) {
                $erros[] = [
                    'tabela' => $tabela,
                    'erro' => $e->getMessage()
                ];
            }

            // Limpar tabelas auxiliares
            foreach ($auxiliares as $auxiliar) {
                try {
                    // Verificar se a tabela existe
                    $stmtCheck = $this->db->query("SHOW TABLES LIKE '{$auxiliar}'");
                    if ($stmtCheck->rowCount() === 0) {
                        continue;
                    }

                    $stmt = $this->db->prepare("SHOW COLUMNS FROM `{$auxiliar}` LIKE 'company_id'");
                    $stmt->execute();
                    $auxTemCompanyId = $stmt->rowCount() > 0;

                    if ($auxTemCompanyId) {
                        $stmtCount = $this->db->prepare("SELECT COUNT(*) FROM `{$auxiliar}` WHERE company_id = :company_id");
                        $stmtCount->execute(['company_id' => $companyId]);
                        $totalAntes = (int) $stmtCount->fetchColumn();

                        $stmt = $this->db->prepare("DELETE FROM `{$auxiliar}` WHERE company_id = :company_id");
                        $stmt->execute(['company_id' => $companyId]);

                        $limpas[] = [
                            'tabela' => $auxiliar,
                            'linhas' => $totalAntes
                        ];
                    } else {
                        $stmtCount = $this->db->query("SELECT COUNT(*) FROM `{$auxiliar}`");
                        $totalAntes = (int) $stmtCount->fetchColumn();

                        $this->db->exec("TRUNCATE TABLE `{$auxiliar}`");

                        $limpas[] = [
                            'tabela' => $auxiliar,
                            'linhas' => $totalAntes
                        ];
                    }
                } catch (Exception $e) {
                    $erros[] = [
                        'tabela' => $auxiliar,
                        'erro' => $e->getMessage()
                    ];
                }
            }

            $this->db->commit();

            $this->success('Limpeza executada com sucesso', [
                'limpas' => $limpas,
                'erros' => $erros
            ]);

        } catch (Exception $e) {
            if ($this->db && $this->db->inTransaction()) {
                $this->db->rollBack();
            }
            error_log("Erro ao executar limpeza: " . $e->getMessage());
            $this->error('Erro ao executar limpeza: ' . $e->getMessage());
        }
    }

    /**
     * Exibe página para importar banco de dados
     */
    public function importarBanco(): void
    {
        // Verificar se o usuário é admin ou tem permissão de suporte
        $user = $this->session->get('user');
        if (!$user || ($user['role'] !== 'admin' && $user['role'] !== 'suporte')) {
            $this->response->forbidden('Acesso negado. Área restrita ao suporte.');
            return;
        }

        $this->view('suporte/importar-banco', [
            'pageTitle' => 'Importar Banco de Dados',
            'activeMenu' => 'suporte'
        ]);
    }

    /**
     * Processa o upload e importação do arquivo SQL
     */
    public function processarImportacao(): void
    {
        // Verificar se o usuário é admin ou tem permissão de suporte
        $user = $this->session->get('user');
        if (!$user || ($user['role'] !== 'admin' && $user['role'] !== 'suporte')) {
            $this->error('Acesso negado. Área restrita ao suporte.');
            return;
        }

        try {
            if (!isset($_FILES['arquivo_sql']) || $_FILES['arquivo_sql']['error'] !== UPLOAD_ERR_OK) {
                $this->error('Erro ao fazer upload do arquivo');
                return;
            }

            $arquivo = $_FILES['arquivo_sql'];
            $nomeArquivo = $arquivo['name'];
            $caminhoTemporario = $arquivo['tmp_name'];

            // Validar extensão
            $extensao = strtolower(pathinfo($nomeArquivo, PATHINFO_EXTENSION));
            if ($extensao !== 'sql') {
                $this->error('Apenas arquivos .sql são permitidos');
                return;
            }

            // Ler conteúdo do arquivo
            $conteudo = file_get_contents($caminhoTemporario);
            if ($conteudo === false) {
                $this->error('Erro ao ler arquivo');
                return;
            }

            // Dividir em comandos SQL individuais
            $comandos = $this->dividirComandosSQL($conteudo);

            $this->db->beginTransaction();

            $executados = 0;
            $erros = [];

            foreach ($comandos as $comando) {
                $comando = trim($comando);
                if (empty($comando)) {
                    continue;
                }

                try {
                    $this->db->exec($comando);
                    $executados++;
                } catch (Exception $e) {
                    $erros[] = [
                        'comando' => substr($comando, 0, 100),
                        'erro' => $e->getMessage()
                    ];
                }
            }

            $this->db->commit();

            $this->success('Importação concluída', [
                'executados' => $executados,
                'erros' => $erros
            ]);

        } catch (Exception $e) {
            if ($this->db->inTransaction()) {
                $this->db->rollBack();
            }
            error_log("Erro ao processar importação: " . $e->getMessage());
            $this->error('Erro ao processar importação: ' . $e->getMessage());
        }
    }

    /**
     * Retorna mapeamento de tabelas principais e suas auxiliares
     */
    private function getTabelasAuxiliares(): array
    {
        return [
            'vendas' => ['vendas_itens', 'vendas_pagamentos'],
            'compras' => ['compras_itens'],
            'produtos' => ['estoque_movimentos', 'produtos_kit'],
            'pessoas' => ['pessoa_veiculos'],
            'contas_receber' => [],
            'contas_pagar' => [],
            'metodos_pagamento' => ['metodos_pagamento_prazos'],
            'impostos' => ['impostos_configuracoes'],
            'estoque' => ['estoque_movimentos'],
            'pedidos_quebras_envaze' => [],
            'ordens_servico' => ['ordem_servico_itens'],
            'modulo_status' => [],
            'grupos' => ['grupos_acesso'],
            'categorias' => [],
            'plano_contas' => [],
            'centro_custos' => [],
            'gestor_estoque' => [],
            'fluxo_caixa' => [],
            'nfe_distribuicao' => [],
            'users' => [],
            'companies' => []
        ];
    }

    /**
     * Lista todas as tabelas principais do banco de dados
     */
    private function listarTabelas(int $companyId): array
    {
        $tabelasAuxiliares = $this->getTabelasAuxiliares();
        $tabelasPrincipais = array_keys($tabelasAuxiliares);

        $stmt = $this->db->query("SHOW TABLES");
        $resultado = $stmt->fetchAll(\PDO::FETCH_NUM);

        $todasTabelas = [];
        foreach ($resultado as $row) {
            $todasTabelas[] = $row[0];
        }

        $tabelas = [];
        foreach ($tabelasPrincipais as $nomeTabela) {
            // Verificar se a tabela existe no banco
            if (!in_array($nomeTabela, $todasTabelas)) {
                continue;
            }

            // Verificar se tem company_id
            $stmtCol = $this->db->prepare("SHOW COLUMNS FROM `{$nomeTabela}` LIKE 'company_id'");
            $stmtCol->execute();
            $temCompanyId = $stmtCol->rowCount() > 0;

            // Contar registros
            if ($temCompanyId) {
                $stmtCount = $this->db->prepare("SELECT COUNT(*) FROM `{$nomeTabela}` WHERE company_id = :company_id");
                $stmtCount->execute(['company_id' => $companyId]);
                $total = $stmtCount->fetchColumn();
            } else {
                $stmtCount = $this->db->query("SELECT COUNT(*) FROM `{$nomeTabela}`");
                $total = $stmtCount->fetchColumn();
            }

            // Contar registros nas tabelas auxiliares
            $auxiliares = $tabelasAuxiliares[$nomeTabela] ?? [];
            $totalAuxiliares = 0;
            foreach ($auxiliares as $auxiliar) {
                if (in_array($auxiliar, $todasTabelas)) {
                    try {
                        // Verificar se a tabela auxiliar tem company_id
                        $stmtAux = $this->db->prepare("SHOW COLUMNS FROM `{$auxiliar}` LIKE 'company_id'");
                        $stmtAux->execute();
                        $auxTemCompanyId = $stmtAux->rowCount() > 0;

                        if ($auxTemCompanyId) {
                            $stmtCountAux = $this->db->prepare("SELECT COUNT(*) FROM `{$auxiliar}` WHERE company_id = :company_id");
                            $stmtCountAux->execute(['company_id' => $companyId]);
                            $totalAuxiliares += (int) $stmtCountAux->fetchColumn();
                        } else {
                            $stmtCountAux = $this->db->query("SELECT COUNT(*) FROM `{$auxiliar}`");
                            $totalAuxiliares += (int) $stmtCountAux->fetchColumn();
                        }
                    } catch (Exception $e) {
                        // Ignora erros ao contar auxiliares
                    }
                }
            }

            $tabelas[] = [
                'nome' => $nomeTabela,
                'total' => (int) $total,
                'total_auxiliares' => $totalAuxiliares,
                'tem_company_id' => $temCompanyId,
                'auxiliares' => $auxiliares
            ];
        }

        return $tabelas;
    }

    /**
     * Divide uma string SQL em comandos individuais
     */
    private function dividirComandosSQL(string $sql): array
    {
        // Remove comentários
        $sql = preg_replace('/--.*$/m', '', $sql);
        $sql = preg_replace('/\/\*.*?\*\//s', '', $sql);

        // Divide por ponto e vírgula, mas respeita strings
        $comandos = [];
        $comandoAtual = '';
        $dentroString = false;
        $delimitador = '';

        for ($i = 0; $i < strlen($sql); $i++) {
            $char = $sql[$i];
            $charProximo = $i < strlen($sql) - 1 ? $sql[$i + 1] : '';

            if (!$dentroString && ($char === '"' || $char === "'" || $char === '`')) {
                $dentroString = true;
                $delimitador = $char;
            } elseif ($dentroString && $char === $delimitador && $charProximo !== $delimitador) {
                $dentroString = false;
                $delimitador = '';
            } elseif ($dentroString && $char === $delimitador && $charProximo === $delimitador) {
                $comandoAtual .= $char . $charProximo;
                $i++; // Pula o próximo caractere
                continue;
            }

            $comandoAtual .= $char;

            if (!$dentroString && $char === ';') {
                $comandoAtual = trim($comandoAtual);
                if (!empty($comandoAtual)) {
                    $comandos[] = $comandoAtual;
                }
                $comandoAtual = '';
            }
        }

        // Adiciona último comando se não terminar com ;
        $comandoAtual = trim($comandoAtual);
        if (!empty($comandoAtual)) {
            $comandos[] = $comandoAtual;
        }

        return $comandos;
    }

    /**
     * Exibe página de backup
     */
    public function backup(): void
    {
        // Verificar se o usuário é admin ou tem permissão de suporte
        $user = $this->session->get('user');
        if (!$user || ($user['role'] !== 'admin' && $user['role'] !== 'suporte')) {
            $this->response->forbidden('Acesso negado. Área restrita ao suporte.');
            return;
        }

        // Listar backups existentes
        $backupsDir = \ROOT_PATH . '/storage/backups';
        $backups = [];

        // Criar diretório se não existir
        if (!is_dir($backupsDir)) {
            @mkdir($backupsDir, 0755, true);
        }

        if (is_dir($backupsDir)) {
            $files = scandir($backupsDir);
            foreach ($files as $file) {
                if ($file !== '.' && $file !== '..' && pathinfo($file, PATHINFO_EXTENSION) === 'sql') {
                    $filePath = $backupsDir . '/' . $file;
                    if (file_exists($filePath)) {
                        $backups[] = [
                            'nome' => $file,
                            'tamanho' => filesize($filePath),
                            'data' => date('d/m/Y H:i:s', filemtime($filePath))
                        ];
                    }
                }
            }
            // Ordenar por data (mais recente primeiro)
            usort($backups, function($a, $b) {
                return strcmp($b['nome'], $a['nome']);
            });
        }

        $this->view('suporte/backup', [
            'backups' => $backups,
            'pageTitle' => 'Backup do Banco de Dados',
            'activeMenu' => 'backup'
        ]);
    }

    /**
     * Cria um backup do banco de dados
     */
    public function criarBackup(): void
    {
        // Verificar se o usuário é admin ou tem permissão de suporte
        $user = $this->session->get('user');
        if (!$user || ($user['role'] !== 'admin' && $user['role'] !== 'suporte')) {
            $this->error('Acesso negado. Área restrita ao suporte.');
            return;
        }

        try {
            // Obter configurações do banco
            $dbConfig = require \ROOT_PATH . '/config/database.php';
            $config = $dbConfig['master'];

            // Criar diretório de backups se não existir
            $backupsDir = \ROOT_PATH . '/storage/backups';
            if (!is_dir($backupsDir)) {
                mkdir($backupsDir, 0755, true);
            }

            // Gerar nome do arquivo: bkp_[dia_mes_ano_h_m_s].sql
            $dataHora = date('d_m_Y_H_i_s');
            $nomeArquivo = "bkp_{$dataHora}.sql";
            $caminhoArquivo = $backupsDir . '/' . $nomeArquivo;

            // Verificar se mysqldump está disponível e se funções de shell estão habilitadas
            $mysqldump = $this->encontrarMysqldump();
            $funcoesShellDisponiveis = function_exists('exec') || function_exists('shell_exec');

            if (!$mysqldump || !$funcoesShellDisponiveis) {
                // Se mysqldump não estiver disponível ou funções de shell desabilitadas, usar método alternativo via PDO
                $this->criarBackupPDO($config, $caminhoArquivo);
            } else {
                // Usar mysqldump (método mais eficiente)
                $comando = sprintf(
                    '%s --host=%s --port=%d --user=%s --password=%s %s > %s 2>&1',
                    escapeshellarg($mysqldump),
                    escapeshellarg($config['host']),
                    escapeshellarg($config['port']),
                    escapeshellarg($config['username']),
                    escapeshellarg($config['password']),
                    escapeshellarg($config['database']),
                    escapeshellarg($caminhoArquivo)
                );

                $output = [];
                $returnCode = 0;
                $sucesso = false;

                // Tentar usar exec primeiro
                if (function_exists('exec')) {
                    try {
                        $execFunc = '\\exec';
                        $execFunc($comando, $output, $returnCode);
                        if ($returnCode === 0 && file_exists($caminhoArquivo) && filesize($caminhoArquivo) > 0) {
                            $sucesso = true;
                        }
                    } catch (\Exception $e) {
                        // Ignorar e tentar próximo método
                    }
                }

                // Se exec não funcionou, tentar shell_exec
                if (!$sucesso && function_exists('shell_exec')) {
                    try {
                        $shellExecFunc = '\\shell_exec';
                        $result = $shellExecFunc($comando . ' 2>&1');
                        if ($result !== null && file_exists($caminhoArquivo) && filesize($caminhoArquivo) > 0) {
                            $sucesso = true;
                        }
                    } catch (\Exception $e) {
                        // Ignorar e usar método PDO
                    }
                }

                // Se nenhum método de shell funcionou, usar método PDO
                if (!$sucesso) {
                    $this->criarBackupPDO($config, $caminhoArquivo);
                }
            }

            $tamanhoArquivo = filesize($caminhoArquivo);

            $this->success('Backup criado com sucesso', [
                'arquivo' => $nomeArquivo,
                'tamanho' => $tamanhoArquivo,
                'data' => date('d/m/Y H:i:s')
            ]);

        } catch (Exception $e) {
            error_log("Erro ao criar backup: " . $e->getMessage());
            $this->error('Erro ao criar backup: ' . $e->getMessage());
        }
    }

    /**
     * Cria backup usando PDO (método alternativo)
     */
    private function criarBackupPDO(array $config, string $caminhoArquivo): void
    {
        $arquivo = fopen($caminhoArquivo, 'w');

        if (!$arquivo) {
            throw new Exception('Não foi possível criar o arquivo de backup.');
        }

        // Cabeçalho do backup
        fwrite($arquivo, "-- Backup do Banco de Dados\n");
        fwrite($arquivo, "-- Data: " . date('d/m/Y H:i:s') . "\n");
        fwrite($arquivo, "-- Banco: {$config['database']}\n\n");
        fwrite($arquivo, "SET FOREIGN_KEY_CHECKS=0;\n");
        fwrite($arquivo, "SET SQL_MODE='NO_AUTO_VALUE_ON_ZERO';\n");
        fwrite($arquivo, "SET AUTOCOMMIT=0;\n");
        fwrite($arquivo, "START TRANSACTION;\n\n");

        // Obter todas as tabelas
        $stmt = $this->db->query("SHOW TABLES");
        $tabelas = $stmt->fetchAll(\PDO::FETCH_NUM);

        foreach ($tabelas as $row) {
            $tabela = $row[0];

            // Estrutura da tabela
            $stmt = $this->db->query("SHOW CREATE TABLE `{$tabela}`");
            $createTable = $stmt->fetch(\PDO::FETCH_ASSOC);

            if ($createTable) {
                // A chave pode ser 'Create Table' ou 'Create Table' dependendo da versão do MySQL
                $createTableSql = $createTable['Create Table'] ?? $createTable['CREATE TABLE'] ?? '';

                if (!empty($createTableSql)) {
                    fwrite($arquivo, "-- Estrutura da tabela `{$tabela}`\n");
                    fwrite($arquivo, "DROP TABLE IF EXISTS `{$tabela}`;\n");
                    fwrite($arquivo, $createTableSql . ";\n\n");
                }
            }

            // Dados da tabela
            $stmt = $this->db->query("SELECT * FROM `{$tabela}`");
            $registros = $stmt->fetchAll(\PDO::FETCH_ASSOC);

            if (count($registros) > 0) {
                fwrite($arquivo, "-- Dados da tabela `{$tabela}`\n");
                fwrite($arquivo, "LOCK TABLES `{$tabela}` WRITE;\n");

                foreach ($registros as $registro) {
                    $campos = array_keys($registro);
                    $valores = array_map(function($valor) {
                        if ($valor === null) {
                            return 'NULL';
                        }
                        return $this->db->quote($valor);
                    }, array_values($registro));

                    $sql = "INSERT INTO `{$tabela}` (`" . implode('`, `', $campos) . "`) VALUES (" . implode(', ', $valores) . ");\n";
                    fwrite($arquivo, $sql);
                }

                fwrite($arquivo, "UNLOCK TABLES;\n\n");
            }
        }

        fwrite($arquivo, "COMMIT;\n");
        fwrite($arquivo, "SET FOREIGN_KEY_CHECKS=1;\n");

        fclose($arquivo);
    }

    /**
     * Encontra o caminho do mysqldump
     */
    private function encontrarMysqldump(): ?string
    {
        $possiveisCaminhos = [
            'mysqldump',
            '/usr/bin/mysqldump',
            '/usr/local/bin/mysqldump',
            '/opt/lampp/bin/mysqldump',
            'C:\\xampp\\mysql\\bin\\mysqldump.exe',
            'C:\\wamp\\bin\\mysql\\mysql5.7.36\\bin\\mysqldump.exe',
        ];

        foreach ($possiveisCaminhos as $caminho) {
            if (is_executable($caminho) || (strtoupper(substr(PHP_OS, 0, 3)) === 'WIN' && file_exists($caminho))) {
                return $caminho;
            }
        }

        // Tentar encontrar via which/where (apenas se funções de shell estiverem disponíveis)
        if (!function_exists('exec') && !function_exists('shell_exec')) {
            return null;
        }

        $comando = strtoupper(substr(PHP_OS, 0, 3)) === 'WIN' ? 'where mysqldump' : 'which mysqldump';
        $output = [];
        $returnCode = 0;

        // Usar função global exec
        if (function_exists('exec')) {
            try {
                $execFunc = '\\exec';
                $execFunc($comando, $output, $returnCode);
            } catch (\Exception $e) {
                // Ignorar erro
                $returnCode = 1;
            }
        } elseif (function_exists('shell_exec')) {
            try {
                $shellExecFunc = '\\shell_exec';
                $result = $shellExecFunc($comando . ' 2>&1');
                if ($result !== null) {
                    $output = [trim($result)];
                    $returnCode = 0;
                } else {
                    $returnCode = 1;
                }
            } catch (\Exception $e) {
                // Ignorar erro
                $returnCode = 1;
            }
        }

        if ($returnCode === 0 && !empty($output[0]) && file_exists($output[0])) {
            return $output[0];
        }

        return null;
    }

    /**
     * Faz download de um backup
     */
    public function downloadBackup(): void
    {
        // Verificar se o usuário é admin ou tem permissão de suporte
        $user = $this->session->get('user');
        if (!$user || ($user['role'] !== 'admin' && $user['role'] !== 'suporte')) {
            $this->response->forbidden('Acesso negado. Área restrita ao suporte.');
            return;
        }

        $nomeArquivo = $this->request->get('arquivo');

        if (empty($nomeArquivo)) {
            $this->response->forbidden('Arquivo não especificado.');
            return;
        }

        // Validar nome do arquivo (prevenir path traversal)
        $nomeArquivo = basename($nomeArquivo);

        if (!preg_match('/^bkp_\d{2}_\d{2}_\d{4}_\d{2}_\d{2}_\d{2}\.sql$/', $nomeArquivo)) {
            $this->response->forbidden('Nome de arquivo inválido.');
            return;
        }

        $caminhoArquivo = \ROOT_PATH . '/storage/backups/' . $nomeArquivo;

        if (!file_exists($caminhoArquivo)) {
            $this->response->notFound('Arquivo não encontrado.');
            return;
        }

        // Enviar arquivo para download
        header('Content-Type: application/octet-stream');
        header('Content-Disposition: attachment; filename="' . $nomeArquivo . '"');
        header('Content-Length: ' . filesize($caminhoArquivo));
        header('Cache-Control: must-revalidate');
        header('Pragma: public');

        readfile($caminhoArquivo);
        exit;
    }
}
