<?php

declare(strict_types=1);

namespace App\Controllers;

use Exception;
use App\Helpers\UrlHelper;

/**
 * Controller de Perfis de Acesso
 */
class PerfisAcessoController extends BaseController
{
    /**
     * Lista todos os perfis de acesso
     */
    public function index(): void
    {
        try {
            // Verificar permissão de visualização (se não tiver permissão configurada, permite acesso)
            if (method_exists($this, 'canView') && !$this->canView('perfis-acesso')) {
                $this->response->forbidden('Você não tem permissão para visualizar perfis de acesso.');
                return;
            }

            $companyId = $this->getCompanyId();

            if (!$this->db) {
                error_log("[PerfisAcessoController] Erro: Conexão com o banco de dados não estabelecida.");
                $this->session->setFlash('error_message', 'Erro de conexão com o banco de dados.');
                $this->redirect('/dashboard');
                return;
            }

            if (!$companyId) {
                error_log("[PerfisAcessoController] Erro: companyId não encontrado na sessão.");
                $this->session->setFlash('error_message', 'Sessão expirada ou inválida. Faça login novamente.');
                $this->redirect('/login');
                return;
            }

            // Verificar se a tabela existe
            $tabelaExiste = false;
            try {
                $stmt = $this->db->query("SHOW TABLES LIKE 'perfis_acesso'");
                $tabelaExiste = $stmt->rowCount() > 0;
            } catch (\Exception $e) {
                // Se houver erro ao verificar a tabela, assumir que não existe
                error_log("[PerfisAcessoController] Erro ao verificar tabela: " . $e->getMessage());
            }

            if (!$tabelaExiste) {
                // Tabela não existe, mostrar mensagem
                $this->view('perfis-acesso/index', [
                    'perfis' => [],
                    'pageTitle' => 'Perfis de Acesso',
                    'activeMenu' => 'perfis-acesso',
                    'page' => 1,
                    'perPage' => 50,
                    'total' => 0,
                    'totalPages' => 1,
                    'tabela_nao_existe' => true
                ]);
                return;
            }

            // Paginação
            $perPage = 50;
            $page = max(1, (int) ($this->request->get('page') ?? 1));
            $offset = ($page - 1) * $perPage;

            // Contar total de perfis
            $total = 0;
            $totalPages = 1;
            try {
                $stmtCount = $this->db->prepare("
                    SELECT COUNT(*) as total
                    FROM perfis_acesso
                    WHERE company_id = :company_id
                ");
                $stmtCount->execute(['company_id' => $companyId]);
                $result = $stmtCount->fetch();
                if ($result) {
                    $total = (int) $result['total'];
                    $totalPages = max(1, (int) ceil($total / $perPage));
                }
            } catch (\Exception $e) {
                error_log("[PerfisAcessoController] Erro ao contar perfis: " . $e->getMessage());
                // Se houver erro, assumir que não há perfis
                $total = 0;
                $totalPages = 1;
            }

            // Buscar perfis (paginado)
            // Verificar se a coluna perfil_acesso_id existe na tabela usuarios antes de usar na subquery
            $temColunaPerfil = false;
            try {
                $stmtCheck = $this->db->query("SHOW COLUMNS FROM usuarios LIKE 'perfil_acesso_id'");
                $temColunaPerfil = $stmtCheck->rowCount() > 0;
            } catch (\Exception $e) {
                // Se houver erro ao verificar, assumir que não existe
                error_log("[PerfisAcessoController] Erro ao verificar coluna perfil_acesso_id: " . $e->getMessage());
            }

            try {
                if ($temColunaPerfil) {
                    $stmt = $this->db->prepare("
                        SELECT p.*,
                            (SELECT COUNT(*) FROM usuarios u WHERE u.perfil_acesso_id = p.id AND u.company_id = p.company_id) as total_usuarios
                        FROM perfis_acesso p
                        WHERE p.company_id = :company_id
                        ORDER BY p.nome ASC
                        LIMIT :limit OFFSET :offset
                    ");
                } else {
                    // Se a coluna não existe, buscar sem contar usuários
                    $stmt = $this->db->prepare("
                        SELECT p.*, 0 as total_usuarios
                        FROM perfis_acesso p
                        WHERE p.company_id = :company_id
                        ORDER BY p.nome ASC
                        LIMIT :limit OFFSET :offset
                    ");
                }

                $stmt->bindValue(':company_id', $companyId, \PDO::PARAM_INT);
                $stmt->bindValue(':limit', $perPage, \PDO::PARAM_INT);
                $stmt->bindValue(':offset', $offset, \PDO::PARAM_INT);
                $stmt->execute();
                $perfis = $stmt->fetchAll();
            } catch (\Exception $e) {
                error_log("[PerfisAcessoController] Erro ao buscar perfis: " . $e->getMessage());
                // Se houver erro, retornar array vazio
                $perfis = [];
            }

            $this->view('perfis-acesso/index', [
                'perfis' => $perfis,
                'pageTitle' => 'Perfis de Acesso',
                'activeMenu' => 'perfis-acesso',
                'page' => $page,
                'perPage' => $perPage,
                'total' => $total,
                'totalPages' => $totalPages
            ]);

        } catch (\Exception $e) {
            error_log("[PerfisAcessoController::index] Erro: " . $e->getMessage());
            error_log("[PerfisAcessoController::index] Stack trace: " . $e->getTraceAsString());

            // Se for requisição AJAX, retornar JSON
            if ($this->request->isAjax()) {
                $this->error('Erro ao carregar perfis de acesso: ' . $e->getMessage());
                return;
            }

            // Caso contrário, mostrar view com mensagem de erro
            try {
                $this->session->setFlash('error_message', 'Erro ao carregar perfis de acesso: ' . $e->getMessage());
                $this->view('perfis-acesso/index', [
                    'perfis' => [],
                    'pageTitle' => 'Perfis de Acesso',
                    'activeMenu' => 'perfis-acesso',
                    'page' => 1,
                    'perPage' => 50,
                    'total' => 0,
                    'totalPages' => 1,
                    'erro' => $e->getMessage()
                ]);
            } catch (\Exception $viewError) {
                // Se houver erro ao renderizar a view, redirecionar para dashboard
                error_log("[PerfisAcessoController::index] Erro ao renderizar view: " . $viewError->getMessage());
                $this->session->setFlash('error_message', 'Erro ao carregar perfis de acesso. Verifique se a tabela foi criada.');
                $this->redirect('/dashboard');
            }
        }
    }

    /**
     * Exibe formulário de criação
     */
    public function create(): void
    {
        try {
            // Verificar permissão de criação (se não tiver permissão configurada, permite acesso)
            if (method_exists($this, 'canCreate') && !$this->canCreate('perfis-acesso')) {
                $this->response->forbidden('Você não tem permissão para criar perfis de acesso.');
                return;
            }

            $this->view('perfis-acesso/create', [
                'pageTitle' => 'Novo Perfil de Acesso',
                'activeMenu' => 'perfis-acesso'
            ]);
        } catch (\Exception $e) {
            error_log("[PerfisAcessoController::create] Erro: " . $e->getMessage());
            error_log("[PerfisAcessoController::create] Stack trace: " . $e->getTraceAsString());

            // Se for requisição AJAX, retornar JSON
            if ($this->request->isAjax()) {
                $this->error('Erro ao exibir formulário: ' . $e->getMessage());
                return;
            }

            // Caso contrário, redirecionar para a listagem com mensagem de erro
            $this->session->setFlash('error_message', 'Erro ao exibir formulário de criação: ' . $e->getMessage());
            $this->redirect('/perfis-acesso');
        }
    }

    /**
     * Salva novo perfil de acesso
     */
    public function store(): void
    {
        // Verificar permissão de criação
        if (!$this->canCreate('perfis-acesso')) {
            $this->response->forbidden('Você não tem permissão para criar perfis de acesso.');
            return;
        }

        try {
            $companyId = $this->getCompanyId();

            $data = [
                'company_id' => $companyId,
                'nome' => $this->request->post('nome'),
                'descricao' => $this->request->post('descricao') ?: null,
                'ativo' => $this->request->post('ativo') ? 1 : 0,
            ];

            // Validação
            if (empty($data['nome'])) {
                $this->error('O campo Nome é obrigatório');
                return;
            }

            // Verificar se já existe um perfil com o mesmo nome
            $stmt = $this->db->prepare("
                SELECT id FROM perfis_acesso
                WHERE company_id = :company_id AND nome = :nome
            ");
            $stmt->execute([
                'company_id' => $companyId,
                'nome' => $data['nome']
            ]);
            if ($stmt->fetch()) {
                $this->error('Já existe um perfil de acesso com este nome');
                return;
            }

            $this->db->beginTransaction();

            $stmt = $this->db->prepare("
                INSERT INTO perfis_acesso (company_id, nome, descricao, ativo, created_at, updated_at)
                VALUES (:company_id, :nome, :descricao, :ativo, NOW(), NOW())
            ");

            $stmt->execute($data);
            $perfilId = (int) $this->db->lastInsertId();

            $this->logActivity('create', 'perfis_acesso', $perfilId, $data);
            $this->db->commit();

            $this->success('Perfil de acesso cadastrado com sucesso', [
                'id' => $perfilId,
                'redirect' => UrlHelper::url('/perfis-acesso')
            ]);

        } catch (Exception $e) {
            if ($this->db->inTransaction()) {
                $this->db->rollBack();
            }
            error_log("Erro ao criar perfil de acesso: " . $e->getMessage());
            $this->error('Erro ao cadastrar perfil de acesso: ' . $e->getMessage());
        }
    }

    /**
     * Exibe formulário de edição
     */
    public function edit(): void
    {
        // Verificar permissão de edição
        if (!$this->canEdit('perfis-acesso')) {
            $this->response->forbidden('Você não tem permissão para editar perfis de acesso.');
            return;
        }

        try {
            $id = (int) $this->request->get('id');
            $perfil = $this->getPerfil($id);

            if (!$perfil) {
                $this->redirect('/perfis-acesso');
                return;
            }

            // Buscar permissões do perfil
            $permissoes = $this->getPermissoes((int) $perfil['id']);

            // Lista de módulos disponíveis
            $modulos = $this->getModulosDisponiveis();

            $this->view('perfis-acesso/edit', [
                'perfil' => $perfil,
                'permissoes' => $permissoes,
                'modulos' => $modulos,
                'pageTitle' => 'Editar ' . $perfil['nome'],
                'activeMenu' => 'perfis-acesso'
            ]);

        } catch (Exception $e) {
            error_log("Erro ao exibir formulário: " . $e->getMessage());
            $this->redirect('/perfis-acesso');
        }
    }

    /**
     * Atualiza um perfil de acesso
     */
    public function update(): void
    {
        // Verificar permissão de edição
        if (!$this->canEdit('perfis-acesso')) {
            $this->response->forbidden('Você não tem permissão para editar perfis de acesso.');
            return;
        }

        try {
            $id = (int) $this->request->post('id');
            $perfil = $this->getPerfil($id);

            if (!$perfil) {
                $this->error('Perfil de acesso não encontrado');
                return;
            }

            $data = [
                'nome' => $this->request->post('nome'),
                'descricao' => $this->request->post('descricao') ?: null,
                'ativo' => $this->request->post('ativo') ? 1 : 0,
                'id' => $id,
                'company_id' => $this->getCompanyId()
            ];

            if (empty($data['nome'])) {
                $this->error('O campo Nome é obrigatório');
                return;
            }

            // Verificar se já existe outro perfil com o mesmo nome
            $stmt = $this->db->prepare("
                SELECT id FROM perfis_acesso
                WHERE company_id = :company_id AND nome = :nome AND id != :id
            ");
            $stmt->execute([
                'company_id' => $this->getCompanyId(),
                'nome' => $data['nome'],
                'id' => $id
            ]);
            if ($stmt->fetch()) {
                $this->error('Já existe um perfil de acesso com este nome');
                return;
            }

            $this->db->beginTransaction();

            $stmt = $this->db->prepare("
                UPDATE perfis_acesso SET
                    nome = :nome,
                    descricao = :descricao,
                    ativo = :ativo,
                    updated_at = NOW()
                WHERE id = :id AND company_id = :company_id
            ");

            $stmt->execute($data);

            $this->logActivity('update', 'perfis_acesso', $id, $data);
            $this->db->commit();

            $this->success('Perfil de acesso atualizado com sucesso', [
                'redirect' => UrlHelper::url('/perfis-acesso')
            ]);

        } catch (Exception $e) {
            if ($this->db->inTransaction()) {
                $this->db->rollBack();
            }
            error_log("Erro ao atualizar perfil de acesso: " . $e->getMessage());
            $this->error('Erro ao atualizar perfil de acesso');
        }
    }

    /**
     * Exclui um perfil de acesso
     */
    public function delete(): void
    {
        // Verificar permissão de exclusão
        if (!$this->canDelete('perfis-acesso')) {
            $this->response->forbidden('Você não tem permissão para excluir perfis de acesso.');
            return;
        }

        try {
            $id = (int) $this->request->post('id');
            $perfil = $this->getPerfil($id);

            if (!$perfil) {
                $this->error('Perfil de acesso não encontrado');
                return;
            }

            // Verificar se há usuários usando este perfil
            $stmt = $this->db->prepare("
                SELECT COUNT(*) as total FROM usuarios
                WHERE perfil_acesso_id = :id AND company_id = :company_id
            ");
            $stmt->execute([
                'id' => $id,
                'company_id' => $this->getCompanyId()
            ]);
            $result = $stmt->fetch();

            if ($result && $result['total'] > 0) {
                $this->error('Não é possível excluir este perfil. Existem ' . $result['total'] . ' usuário(s) utilizando este perfil.');
                return;
            }

            $this->db->beginTransaction();

            $stmt = $this->db->prepare("DELETE FROM perfis_acesso WHERE id = :id AND company_id = :company_id");
            $stmt->execute(['id' => $id, 'company_id' => $this->getCompanyId()]);

            $this->logActivity('delete', 'perfis_acesso', $id, $perfil);
            $this->db->commit();

            $this->success('Perfil de acesso excluído com sucesso');

        } catch (Exception $e) {
            if ($this->db->inTransaction()) {
                $this->db->rollBack();
            }
            error_log("Erro ao excluir perfil de acesso: " . $e->getMessage());
            $this->error('Erro ao excluir perfil de acesso');
        }
    }

    /**
     * Método auxiliar para buscar um perfil
     */
    private function getPerfil(int $id): ?array
    {
        if (!$this->db) {
            return null;
        }

        try {
            // Verificar se a tabela existe
            $stmtCheck = $this->db->query("SHOW TABLES LIKE 'perfis_acesso'");
            if ($stmtCheck->rowCount() == 0) {
                return null;
            }

            $stmt = $this->db->prepare("
                SELECT * FROM perfis_acesso WHERE id = :id AND company_id = :company_id
            ");
            $stmt->execute(['id' => $id, 'company_id' => $this->getCompanyId()]);
            return $stmt->fetch() ?: null;
        } catch (\Exception $e) {
            error_log("[PerfisAcessoController::getPerfil] Erro: " . $e->getMessage());
            return null;
        }
    }

    /**
     * Busca permissões do perfil
     */
    private function getPermissoes(int $perfilId): array
    {
        try {
            // Verificar se a tabela existe
            $stmt = $this->db->query("SHOW TABLES LIKE 'perfis_acesso_permissoes'");
            if (!$stmt->fetch()) {
                return [];
            }

            $stmt = $this->db->prepare("
                SELECT id, module, can_view, can_create, can_edit, can_delete
                FROM perfis_acesso_permissoes
                WHERE perfil_acesso_id = :perfil_id AND company_id = :company_id
            ");
            $stmt->execute([
                'perfil_id' => $perfilId,
                'company_id' => $this->getCompanyId()
            ]);
            $permissoes = $stmt->fetchAll() ?: [];

            // Transformar em array associativo por módulo
            $result = [];
            foreach ($permissoes as $perm) {
                $result[$perm['module']] = [
                    'id' => (int) $perm['id'],
                    'can_view' => (bool) $perm['can_view'],
                    'can_create' => (bool) $perm['can_create'],
                    'can_edit' => (bool) $perm['can_edit'],
                    'can_delete' => (bool) $perm['can_delete']
                ];
            }

            return $result;
        } catch (Exception $e) {
            error_log("Erro ao buscar permissões do perfil: " . $e->getMessage());
            return [];
        }
    }

    /**
     * Lista de módulos disponíveis no sistema
     * Busca da tabela modulos onde ativo = 'Sim'
     */
    private function getModulosDisponiveis(): array
    {
        try {
            if (!$this->db) {
                return [];
            }

            // Verificar se a tabela modulos existe
            $stmtCheck = $this->db->query("SHOW TABLES LIKE 'modulos'");
            if ($stmtCheck->rowCount() === 0) {
                error_log("[PerfisAcessoController::getModulosDisponiveis] Tabela modulos não existe");
                return [];
            }

            // Buscar módulos ativos (usando nome_interface como chave e nome como valor)
            $stmt = $this->db->query("
                SELECT nome_interface, nome
                FROM modulos
                WHERE ativo = 'Sim'
                ORDER BY nome ASC
            ");

            $modulos = $stmt->fetchAll() ?: [];

            // Transformar em array associativo: nome_interface => nome
            $result = [];
            foreach ($modulos as $modulo) {
                $nomeInterface = $modulo['nome_interface'] ?? '';
                $nome = $modulo['nome'] ?? $nomeInterface;

                if (!empty($nomeInterface)) {
                    $result[$nomeInterface] = $nome;
                }
            }

            return $result;
        } catch (Exception $e) {
            error_log("Erro ao buscar módulos disponíveis: " . $e->getMessage());
            return [];
        }
    }

    /**
     * Salva permissões do perfil
     */
    public function salvarPermissoes(): void
    {
        // Verificar permissão de edição
        if (!$this->canEdit('perfis-acesso')) {
            $this->error('Você não tem permissão para editar perfis de acesso.');
            return;
        }

        try {
            $perfilId = (int) $this->request->post('perfil_id');
            $permissoes = $this->request->post('permissoes') ?? [];

            // Verificar se o perfil existe
            $perfil = $this->getPerfil($perfilId);
            if (!$perfil) {
                $this->error('Perfil de acesso não encontrado');
                return;
            }

            // Verificar se a tabela existe
            $stmt = $this->db->query("SHOW TABLES LIKE 'perfis_acesso_permissoes'");
            if (!$stmt->fetch()) {
                $this->error('Tabela de permissões não existe. Execute o script de criação da tabela primeiro.');
                return;
            }

            $this->db->beginTransaction();

            // Deletar permissões existentes
            $stmt = $this->db->prepare("
                DELETE FROM perfis_acesso_permissoes
                WHERE perfil_acesso_id = :perfil_id AND company_id = :company_id
            ");
            $stmt->execute([
                'perfil_id' => $perfilId,
                'company_id' => $this->getCompanyId()
            ]);

            // Inserir novas permissões
            if (!empty($permissoes)) {
                $stmt = $this->db->prepare("
                    INSERT INTO perfis_acesso_permissoes
                    (company_id, perfil_acesso_id, module, can_view, can_create, can_edit, can_delete, created_at, updated_at)
                    VALUES (:company_id, :perfil_id, :module, :can_view, :can_create, :can_edit, :can_delete, NOW(), NOW())
                ");

                foreach ($permissoes as $module => $perms) {
                    if (empty($module)) {
                        continue;
                    }
                    $stmt->execute([
                        'company_id' => $this->getCompanyId(),
                        'perfil_id' => $perfilId,
                        'module' => $module,
                        'can_view' => isset($perms['can_view']) && $perms['can_view'] ? 1 : 0,
                        'can_create' => isset($perms['can_create']) && $perms['can_create'] ? 1 : 0,
                        'can_edit' => isset($perms['can_edit']) && $perms['can_edit'] ? 1 : 0,
                        'can_delete' => isset($perms['can_delete']) && $perms['can_delete'] ? 1 : 0,
                    ]);
                }
            }

            $this->logActivity('update', 'perfis_acesso', $perfilId, [], ['permissoes' => 'atualizadas']);
            $this->db->commit();

            $this->success('Permissões salvas com sucesso', [
                'redirect' => UrlHelper::url('/perfis-acesso')
            ]);

        } catch (Exception $e) {
            if ($this->db->inTransaction()) {
                $this->db->rollBack();
            }
            error_log("Erro ao salvar permissões: " . $e->getMessage());
            $this->error('Erro ao salvar permissões: ' . $e->getMessage());
        }
    }

    /**
     * Busca movimentos de uma permissão
     */
    public function buscarMovimentos(): void
    {
        try {
            $permissaoId = (int) $this->request->post('permissao_id');

            if (!$permissaoId) {
                $this->error('ID da permissão não informado');
                return;
            }

            // Buscar a permissão para obter o módulo
            $stmt = $this->db->prepare("
                SELECT module
                FROM perfis_acesso_permissoes
                WHERE id = :permissao_id AND company_id = :company_id
            ");
            $stmt->execute([
                'permissao_id' => $permissaoId,
                'company_id' => $this->getCompanyId()
            ]);
            $permissao = $stmt->fetch();

            if (!$permissao) {
                $this->error('Permissão não encontrada');
                return;
            }

            $modulo = $permissao['module'];

            // Verificar se a tabela existe
            $stmt = $this->db->query("SHOW TABLES LIKE 'perfis_acesso_permissoes_movimentos'");
            $tabelaExiste = $stmt->fetch() ? true : false;

            $movimentosSalvos = [];
            if ($tabelaExiste) {
                // Buscar movimentos da permissão
                $stmt = $this->db->prepare("
                    SELECT movimento, permitido
                    FROM perfis_acesso_permissoes_movimentos
                    WHERE permissao_id = :permissao_id AND company_id = :company_id
                ");
                $stmt->execute([
                    'permissao_id' => $permissaoId,
                    'company_id' => $this->getCompanyId()
                ]);
                $movimentosSalvos = $stmt->fetchAll() ?: [];
            }

            $movimentosDisponiveis = $this->getMovimentosDisponiveis($modulo);

            $this->success('Movimentos carregados com sucesso', [
                'movimentos' => $movimentosSalvos,
                'movimentos_disponiveis' => $movimentosDisponiveis,
                'modulo' => $modulo
            ]);

        } catch (Exception $e) {
            error_log("Erro ao buscar movimentos: " . $e->getMessage());
            $this->error('Erro ao buscar movimentos: ' . $e->getMessage());
        }
    }

    /**
     * Salva movimentos de uma permissão
     */
    public function salvarMovimentos(): void
    {
        try {
            $permissaoId = (int) $this->request->post('permissao_id');
            $modulo = $this->request->post('modulo');
            $movimentos = $this->request->post('movimentos') ?? [];

            if (!$permissaoId || !$modulo) {
                $this->error('Dados incompletos');
                return;
            }

            // Verificar se a permissão existe e obter perfil_id
            $stmt = $this->db->prepare("
                SELECT perfil_acesso_id
                FROM perfis_acesso_permissoes
                WHERE id = :permissao_id AND company_id = :company_id
            ");
            $stmt->execute([
                'permissao_id' => $permissaoId,
                'company_id' => $this->getCompanyId()
            ]);
            $permissao = $stmt->fetch();

            if (!$permissao) {
                $this->error('Permissão não encontrada');
                return;
            }

            $perfilId = (int) $permissao['perfil_acesso_id'];

            // Verificar se a tabela existe
            $stmt = $this->db->query("SHOW TABLES LIKE 'perfis_acesso_permissoes_movimentos'");
            if (!$stmt->fetch()) {
                $this->error('Tabela de movimentos não existe. Execute o script de criação da tabela primeiro.');
                return;
            }

            $this->db->beginTransaction();

            // Deletar movimentos existentes
            $stmt = $this->db->prepare("
                DELETE FROM perfis_acesso_permissoes_movimentos
                WHERE permissao_id = :permissao_id AND company_id = :company_id
            ");
            $stmt->execute([
                'permissao_id' => $permissaoId,
                'company_id' => $this->getCompanyId()
            ]);

            // Inserir novos movimentos
            if (!empty($movimentos)) {
                $stmt = $this->db->prepare("
                    INSERT INTO perfis_acesso_permissoes_movimentos
                    (company_id, perfil_acesso_id, permissao_id, module, movimento, permitido, created_at, updated_at)
                    VALUES (:company_id, :perfil_id, :permissao_id, :module, :movimento, 1, NOW(), NOW())
                ");

                foreach ($movimentos as $movimento) {
                    if (empty($movimento)) {
                        continue;
                    }
                    $stmt->execute([
                        'company_id' => $this->getCompanyId(),
                        'perfil_id' => $perfilId,
                        'permissao_id' => $permissaoId,
                        'module' => $modulo,
                        'movimento' => $movimento
                    ]);
                }
            }

            $this->logActivity('update', 'perfis_acesso', $perfilId, [], ['movimentos' => 'atualizados']);
            $this->db->commit();

            $this->success('Movimentos salvos com sucesso');

        } catch (Exception $e) {
            if ($this->db->inTransaction()) {
                $this->db->rollBack();
            }
            error_log("Erro ao salvar movimentos: " . $e->getMessage());
            $this->error('Erro ao salvar movimentos: ' . $e->getMessage());
        }
    }

    /**
     * Lista de movimentos/permissões específicas disponíveis por módulo
     */
    private function getMovimentosDisponiveis(string $modulo): array
    {
        // Normalizar o nome do módulo (pode vir como 'pessoas', 'modulo_pessoas', etc)
        $moduloNormalizado = $modulo;
        if (strpos($modulo, 'modulo_') === 0) {
            $moduloNormalizado = str_replace('modulo_', '', $modulo);
        }

        $movimentos = [
            'pessoas' => [
                'clientes' => [
                    'limite_credito' => 'Limite de Crédito',
                    'vendedor' => 'Vendedor',
                    'vincular_empresas' => 'Vincular Empresa(s)'
                ],
                'fornecedor' => [],
                'transportadora' => []
            ]
        ];

        return $movimentos[$moduloNormalizado] ?? [];
    }
}

