<?php

declare(strict_types=1);

namespace App\Controllers;

use Exception;

class OrdemServicoPmocController extends BaseController
{
    protected function moduleOrigin(): string
    {
        return 'os_pmoc';
    }

    public function index(): void
    {
        // Verificar permissão de visualização
        if (!$this->canView('ordem-servico-pmoc')) {
            $this->response->forbidden('Você não tem permissão para visualizar ordens de serviço PMOC.');
            return;
        }

        try {
            $companyId = $this->getCompanyId();
            $search = $this->request->get('search', '');
            $status = $this->request->get('status', '');

            $query = "
                SELECT v.*,
                    COALESCE(p.name, v.customer_name, 'Consumidor Final') as customer_name,
                    COALESCE(p.document, v.customer_document) as customer_document,
                    mp.name as payment_method_name,
                    u.name as vendor_name,
                    pmoc.numero_pmoc,
                    pmoc.data_emissao as pmoc_data_emissao,
                    pmoc.data_validade as pmoc_data_validade,
                    pmoc.responsavel_tecnico,
                    pmoc.crea_cau
                FROM vendas v
                LEFT JOIN pessoas p ON v.customer_id = p.id
                LEFT JOIN metodos_pagamento mp ON v.payment_method_id = mp.id
                LEFT JOIN users u ON v.vendor_id = u.id
                INNER JOIN ordem_servico_pmoc osp ON v.id = osp.venda_id
                LEFT JOIN pmoc pmoc ON osp.pmoc_id = pmoc.id
                WHERE v.company_id = :company_id
            ";

            $params = ['company_id' => $companyId];

            if (!empty($search)) {
                $query .= " AND (v.sale_number LIKE :search OR p.name LIKE :search OR pmoc.numero_pmoc LIKE :search)";
                $params['search'] = "%{$search}%";
            }

            if (!empty($status)) {
                $query .= " AND v.status = :status";
                $params['status'] = $status;
            }

            $query .= " ORDER BY v.id DESC";

            $stmt = $this->db->prepare($query);
            $stmt->execute($params);
            $ordens = $stmt->fetchAll();

            // Busca status do módulo
            $stmt = $this->db->prepare("
                SELECT codigo, nome, emite_nfe
                FROM modulo_status
                WHERE company_id = :company_id
                  AND modulo = 'vendas'
                  AND ativo = 1
            ");
            $stmt->execute(['company_id' => $companyId]);
            $statusOrdens = $stmt->fetchAll();

            $emiteMap = [];
            foreach ($statusOrdens as $statusLinha) {
                $codigo = $statusLinha['codigo'] ?? '';
                if ($codigo !== '') {
                    $emiteMap[$codigo] = !empty($statusLinha['emite_nfe']) && (int) $statusLinha['emite_nfe'] === 1;
                }
            }

            $emitirPorStatus = [];
            foreach ($ordens as $ordem) {
                $codigoStatus = $ordem['status'] ?? '';
                $emitirPorStatus[$ordem['id']] = $codigoStatus !== '' && !empty($emiteMap[$codigoStatus]);
            }

            $this->view('ordem-servico-pmoc/index', [
                'ordens' => $ordens,
                'search' => $search,
                'status' => $status,
                'statusOrdens' => $statusOrdens,
                'podeEmitirNfePorVenda' => $emitirPorStatus,
                'pageTitle' => 'Ordens de Serviço PMOC',
                'activeMenu' => 'ordem-servico-pmoc'
            ]);
        } catch (Exception $e) {
            error_log("Erro ao carregar ordens de serviço PMOC: " . $e->getMessage());
            $this->error('Erro ao carregar ordens de serviço PMOC: ' . $e->getMessage());
        }
    }

    public function create(): void
    {
        try {
            $companyId = $this->getCompanyId();

            // Busca clientes
            $stmt = $this->db->prepare("
                SELECT id, name, trade_name, document, phone, mobile, email FROM pessoas
                WHERE company_id = :company_id AND type IN ('cliente', 'ambos') AND is_active = 1
                ORDER BY name ASC
            ");
            $stmt->execute(['company_id' => $companyId]);
            $clientes = $stmt->fetchAll();

            // Busca PMOCs existentes
            $stmt = $this->db->prepare("
                SELECT id, numero_pmoc, data_emissao, data_validade, cliente_id
                FROM pmoc
                WHERE company_id = :company_id
                ORDER BY data_emissao DESC
            ");
            $stmt->execute(['company_id' => $companyId]);
            $pmocs = $stmt->fetchAll();

            // Busca produtos/serviços
            $stmt = $this->db->prepare("
                SELECT id, name, sku, price, unit
                FROM produtos
                WHERE company_id = :company_id AND is_active = 1
                ORDER BY name ASC
            ");
            $stmt->execute(['company_id' => $companyId]);
            $produtos = $stmt->fetchAll();

            // Busca serviços disponíveis para PMOC
            $stmt = $this->db->prepare("
                SELECT id, name, description
                FROM produtos
                WHERE company_id = :company_id
                  AND type = 'servico'
                  AND is_active = 1
                ORDER BY name ASC
            ");
            $stmt->execute(['company_id' => $companyId]);
            $servicosPmoc = $stmt->fetchAll();

            // Busca métodos de pagamento
            $stmt = $this->db->prepare("
                SELECT id, name, type FROM metodos_pagamento
                WHERE company_id = :company_id AND is_active = 1
                ORDER BY name ASC
            ");
            $stmt->execute(['company_id' => $companyId]);
            $metodosPagamento = $stmt->fetchAll();

            // Busca vendedores
            $stmt = $this->db->prepare("
                SELECT id, name FROM users
                WHERE company_id = :company_id AND is_active = 1
                ORDER BY name ASC
            ");
            $stmt->execute(['company_id' => $companyId]);
            $vendedores = $stmt->fetchAll();

            // Busca status
            $stmt = $this->db->prepare("
                SELECT codigo, nome, descricao, cor, icone, ordem, is_default
                FROM modulo_status
                WHERE company_id = :company_id
                  AND modulo = 'vendas'
                  AND ativo = 1
                ORDER BY ordem ASC
            ");
            $stmt->execute(['company_id' => $companyId]);
            $statusVendas = $stmt->fetchAll();

            // Gerar próximo número
            $proximoNumero = $this->gerarProximoNumeroVenda($companyId);

            // Status padrão
            $statusPadrao = !empty($statusVendas) ? $statusVendas[0]['codigo'] : 'orcamento';
            foreach ($statusVendas as $status) {
                if (!empty($status['is_default'])) {
                    $statusPadrao = $status['codigo'];
                    break;
                }
            }

            $this->view('ordem-servico-pmoc/create', [
                'clientes' => $clientes,
                'pmocs' => $pmocs,
                'produtos' => $produtos,
                'servicosPmoc' => $servicosPmoc,
                'metodosPagamento' => $metodosPagamento,
                'vendedores' => $vendedores,
                'statusVendas' => $statusVendas,
                'statusPadrao' => $statusPadrao,
                'proximoNumero' => $proximoNumero,
                'pageTitle' => 'Nova Ordem de Serviço PMOC',
                'activeMenu' => 'ordem-servico-pmoc'
            ]);
        } catch (Exception $e) {
            error_log("Erro ao carregar formulário de criação: " . $e->getMessage());
            $this->error('Erro ao carregar formulário: ' . $e->getMessage());
        }
    }

    public function store(): void
    {
        // Verificar permissão de criação
        if (!$this->canCreate('ordem-servico-pmoc')) {
            $this->response->forbidden('Você não tem permissão para criar ordens de serviço PMOC.');
            return;
        }

        try {
            $companyId = $this->getCompanyId();
            $userId = $this->getUserId();

            $data = $this->request->post();

            // Validações básicas
            if (empty($data['customer_id'])) {
                $this->response->json(['success' => false, 'message' => 'Cliente é obrigatório']);
                return;
            }

            if (empty($data['sale_number'])) {
                $this->response->json(['success' => false, 'message' => 'Número da ordem é obrigatório']);
                return;
            }

            // Iniciar transação
            $this->db->beginTransaction();

            try {
                // Criar ou buscar PMOC
                $pmocId = null;
                if (!empty($data['pmoc_id'])) {
                    $pmocId = (int) $data['pmoc_id'];
                } elseif (!empty($data['criar_pmoc']) && $data['criar_pmoc'] === '1') {
                    // Criar novo PMOC
                    $pmocId = $this->criarPMOC($companyId, $userId, $data);
                }

                // Criar venda (ordem de serviço)
                $stmt = $this->db->prepare("
                    INSERT INTO vendas (
                        company_id, sale_number, sale_date, customer_id, customer_name, customer_document,
                        vendor_id, payment_method_id, status, total, subtotal, discount,
                        notes, created_at
                    ) VALUES (
                        :company_id, :sale_number, :sale_date, :customer_id, :customer_name, :customer_document,
                        :vendor_id, :payment_method_id, :status, :total, :subtotal, :discount,
                        :notes, NOW()
                    )
                ");

                $stmt->execute([
                    'company_id' => $companyId,
                    'sale_number' => $data['sale_number'],
                    'sale_date' => $data['sale_date'] ?? date('Y-m-d'),
                    'customer_id' => (int) $data['customer_id'],
                    'customer_name' => $data['customer_name'] ?? '',
                    'customer_document' => $data['customer_document'] ?? '',
                    'vendor_id' => !empty($data['vendor_id']) ? (int) $data['vendor_id'] : $userId,
                    'payment_method_id' => !empty($data['payment_method_id']) ? (int) $data['payment_method_id'] : null,
                    'status' => $data['status'] ?? 'orcamento',
                    'total' => (float) ($data['total'] ?? 0),
                    'subtotal' => (float) ($data['subtotal'] ?? 0),
                    'discount' => (float) ($data['discount'] ?? 0),
                    'notes' => $data['observations'] ?? ''
                ]);

                $vendaId = (int) $this->db->lastInsertId();

                // Vincular PMOC à ordem de serviço
                if ($pmocId) {
                    $stmt = $this->db->prepare("
                        INSERT INTO ordem_servico_pmoc (venda_id, pmoc_id, company_id, created_at)
                        VALUES (:venda_id, :pmoc_id, :company_id, NOW())
                    ");
                    $stmt->execute([
                        'venda_id' => $vendaId,
                        'pmoc_id' => $pmocId,
                        'company_id' => $companyId
                    ]);
                }

                // Salvar itens
                if (!empty($data['itens']) && is_array($data['itens'])) {
                    $this->salvarItens($vendaId, $data['itens']);
                }

                $this->db->commit();

                $this->response->json([
                    'success' => true,
                    'message' => 'Ordem de Serviço PMOC criada com sucesso!',
                    'id' => $vendaId
                ]);
            } catch (Exception $e) {
                $this->db->rollBack();
                throw $e;
            }
        } catch (Exception $e) {
            error_log("Erro ao criar ordem de serviço PMOC: " . $e->getMessage());
            $this->response->json([
                'success' => false,
                'message' => 'Erro ao criar ordem de serviço PMOC: ' . $e->getMessage()
            ]);
        }
    }

    public function edit(): void
    {
        // Verificar permissão de edição
        if (!$this->canEdit('ordem-servico-pmoc')) {
            $this->response->forbidden('Você não tem permissão para editar ordens de serviço PMOC.');
            return;
        }

        try {
            $companyId = $this->getCompanyId();
            $id = (int) $this->request->get('id');

            if (empty($id)) {
                $this->error('ID da ordem de serviço não informado');
                return;
            }

            // Buscar ordem de serviço
            $stmt = $this->db->prepare("
                SELECT v.*, osp.pmoc_id
                FROM vendas v
                INNER JOIN ordem_servico_pmoc osp ON v.id = osp.venda_id
                WHERE v.id = :id AND v.company_id = :company_id
            ");
            $stmt->execute(['id' => $id, 'company_id' => $companyId]);
            $ordem = $stmt->fetch();

            if (!$ordem) {
                $this->error('Ordem de serviço não encontrada');
                return;
            }

            // Buscar dados para o formulário (similar ao create)
            $stmt = $this->db->prepare("
                SELECT id, name, trade_name, document, phone, mobile, email FROM pessoas
                WHERE company_id = :company_id AND type IN ('cliente', 'ambos') AND is_active = 1
                ORDER BY name ASC
            ");
            $stmt->execute(['company_id' => $companyId]);
            $clientes = $stmt->fetchAll();

            $stmt = $this->db->prepare("
                SELECT id, numero_pmoc, data_emissao, data_validade, cliente_id
                FROM pmoc
                WHERE company_id = :company_id
                ORDER BY data_emissao DESC
            ");
            $stmt->execute(['company_id' => $companyId]);
            $pmocs = $stmt->fetchAll();

            $stmt = $this->db->prepare("
                SELECT id, name, sku, price, unit
                FROM produtos
                WHERE company_id = :company_id AND is_active = 1
                ORDER BY name ASC
            ");
            $stmt->execute(['company_id' => $companyId]);
            $produtos = $stmt->fetchAll();

            // Busca serviços disponíveis para PMOC
            $stmt = $this->db->prepare("
                SELECT id, name, description
                FROM produtos
                WHERE company_id = :company_id
                  AND type = 'servico'
                  AND is_active = 1
                ORDER BY name ASC
            ");
            $stmt->execute(['company_id' => $companyId]);
            $servicosPmoc = $stmt->fetchAll();

            $stmt = $this->db->prepare("
                SELECT id, name, type FROM metodos_pagamento
                WHERE company_id = :company_id AND is_active = 1
                ORDER BY name ASC
            ");
            $stmt->execute(['company_id' => $companyId]);
            $metodosPagamento = $stmt->fetchAll();

            $stmt = $this->db->prepare("
                SELECT id, name FROM users
                WHERE company_id = :company_id AND is_active = 1
                ORDER BY name ASC
            ");
            $stmt->execute(['company_id' => $companyId]);
            $vendedores = $stmt->fetchAll();

            $stmt = $this->db->prepare("
                SELECT codigo, nome, descricao, cor, icone, ordem, is_default
                FROM modulo_status
                WHERE company_id = :company_id
                  AND modulo = 'vendas'
                  AND ativo = 1
                ORDER BY ordem ASC
            ");
            $stmt->execute(['company_id' => $companyId]);
            $statusVendas = $stmt->fetchAll();

            // Buscar itens
            $stmt = $this->db->prepare("
                SELECT * FROM vendas_itens
                WHERE venda_id = :venda_id
                ORDER BY id ASC
            ");
            $stmt->execute(['venda_id' => $id]);
            $itens = $stmt->fetchAll();

            $this->view('ordem-servico-pmoc/edit', [
                'ordem' => $ordem,
                'clientes' => $clientes,
                'pmocs' => $pmocs,
                'produtos' => $produtos,
                'servicosPmoc' => $servicosPmoc,
                'metodosPagamento' => $metodosPagamento,
                'vendedores' => $vendedores,
                'statusVendas' => $statusVendas,
                'itens' => $itens,
                'pageTitle' => 'Editar Ordem de Serviço PMOC',
                'activeMenu' => 'ordem-servico-pmoc'
            ]);
        } catch (Exception $e) {
            error_log("Erro ao carregar ordem de serviço PMOC: " . $e->getMessage());
            $this->error('Erro ao carregar ordem de serviço: ' . $e->getMessage());
        }
    }

    public function update(): void
    {
        // Verificar permissão de edição
        if (!$this->canEdit('ordem-servico-pmoc')) {
            $this->response->forbidden('Você não tem permissão para editar ordens de serviço PMOC.');
            return;
        }

        try {
            $companyId = $this->getCompanyId();
            $id = (int) $this->request->post('id');
            $data = $this->request->post();

            if (empty($id)) {
                $this->response->json(['success' => false, 'message' => 'ID não informado']);
                return;
            }

            $this->db->beginTransaction();

            try {
                // Atualizar venda
                $stmt = $this->db->prepare("
                    UPDATE vendas SET
                        sale_date = :sale_date,
                        customer_id = :customer_id,
                        customer_name = :customer_name,
                        customer_document = :customer_document,
                        vendor_id = :vendor_id,
                        payment_method_id = :payment_method_id,
                        status = :status,
                        total = :total,
                        subtotal = :subtotal,
                        discount = :discount,
                        notes = :notes,
                        updated_at = NOW()
                    WHERE id = :id AND company_id = :company_id
                      AND EXISTS (SELECT 1 FROM ordem_servico_pmoc WHERE venda_id = vendas.id)
                ");

                $stmt->execute([
                    'id' => $id,
                    'company_id' => $companyId,
                    'sale_date' => $data['sale_date'] ?? date('Y-m-d'),
                    'customer_id' => (int) $data['customer_id'],
                    'customer_name' => $data['customer_name'] ?? '',
                    'customer_document' => $data['customer_document'] ?? '',
                    'vendor_id' => !empty($data['vendor_id']) ? (int) $data['vendor_id'] : null,
                    'payment_method_id' => !empty($data['payment_method_id']) ? (int) $data['payment_method_id'] : null,
                    'status' => $data['status'] ?? 'orcamento',
                    'total' => (float) ($data['total'] ?? 0),
                    'subtotal' => (float) ($data['subtotal'] ?? 0),
                    'discount' => (float) ($data['discount'] ?? 0),
                    'notes' => $data['observations'] ?? ''
                ]);

                // Atualizar vínculo PMOC
                if (!empty($data['pmoc_id'])) {
                    $stmt = $this->db->prepare("
                        INSERT INTO ordem_servico_pmoc (venda_id, pmoc_id, company_id, created_at)
                        VALUES (:venda_id, :pmoc_id, :company_id, NOW())
                        ON DUPLICATE KEY UPDATE pmoc_id = :pmoc_id, updated_at = NOW()
                    ");
                    $stmt->execute([
                        'venda_id' => $id,
                        'pmoc_id' => (int) $data['pmoc_id'],
                        'company_id' => $companyId
                    ]);
                }

                // Remover itens antigos e adicionar novos
                $stmt = $this->db->prepare("DELETE FROM vendas_itens WHERE venda_id = :venda_id");
                $stmt->execute(['venda_id' => $id]);

                if (!empty($data['itens']) && is_array($data['itens'])) {
                    $this->salvarItens($id, $data['itens']);
                }

                $this->db->commit();

                $this->response->json([
                    'success' => true,
                    'message' => 'Ordem de Serviço PMOC atualizada com sucesso!'
                ]);
            } catch (Exception $e) {
                $this->db->rollBack();
                throw $e;
            }
        } catch (Exception $e) {
            error_log("Erro ao atualizar ordem de serviço PMOC: " . $e->getMessage());
            $this->response->json([
                'success' => false,
                'message' => 'Erro ao atualizar: ' . $e->getMessage()
            ]);
        }
    }

    public function show(): void
    {
        try {
            $companyId = $this->getCompanyId();
            $id = (int) $this->request->get('id');

            $stmt = $this->db->prepare("
                SELECT v.*,
                    p.name as customer_name_full,
                    p.document as customer_document_full,
                    mp.name as payment_method_name,
                    u.name as vendor_name,
                    pmoc.*,
                    osp.pmoc_id
                FROM vendas v
                LEFT JOIN pessoas p ON v.customer_id = p.id
                LEFT JOIN metodos_pagamento mp ON v.payment_method_id = mp.id
                LEFT JOIN users u ON v.vendor_id = u.id
                INNER JOIN ordem_servico_pmoc osp ON v.id = osp.venda_id
                LEFT JOIN pmoc pmoc ON osp.pmoc_id = pmoc.id
                WHERE v.id = :id AND v.company_id = :company_id
            ");
            $stmt->execute(['id' => $id, 'company_id' => $companyId]);
            $ordem = $stmt->fetch();

            if (!$ordem) {
                $this->error('Ordem de serviço não encontrada');
                return;
            }

            // Buscar itens
            $stmt = $this->db->prepare("
                SELECT vi.*, p.name as product_name
                FROM vendas_itens vi
                LEFT JOIN produtos p ON vi.product_id = p.id
                WHERE vi.venda_id = :venda_id
                ORDER BY vi.id ASC
            ");
            $stmt->execute(['venda_id' => $id]);
            $itens = $stmt->fetchAll();

            // Buscar equipamentos do PMOC
            $equipamentos = [];
            if (!empty($ordem['pmoc_id'])) {
                $stmt = $this->db->prepare("
                    SELECT * FROM pmoc_equipamentos
                    WHERE pmoc_id = :pmoc_id
                    ORDER BY id ASC
                ");
                $stmt->execute(['pmoc_id' => $ordem['pmoc_id']]);
                $equipamentos = $stmt->fetchAll();
            }

            $this->view('ordem-servico-pmoc/show', [
                'ordem' => $ordem,
                'itens' => $itens,
                'equipamentos' => $equipamentos,
                'pageTitle' => 'Visualizar Ordem de Serviço PMOC',
                'activeMenu' => 'ordem-servico-pmoc'
            ]);
        } catch (Exception $e) {
            error_log("Erro ao visualizar ordem de serviço PMOC: " . $e->getMessage());
            $this->error('Erro ao visualizar: ' . $e->getMessage());
        }
    }

    public function delete(): void
    {
        // Verificar permissão de exclusão
        if (!$this->canDelete('ordem-servico-pmoc')) {
            $this->response->forbidden('Você não tem permissão para excluir ordens de serviço PMOC.');
            return;
        }

        try {
            $companyId = $this->getCompanyId();
            $id = (int) $this->request->post('id');

            $this->db->beginTransaction();

            try {
                // Remover vínculo PMOC
                $stmt = $this->db->prepare("
                    DELETE FROM ordem_servico_pmoc
                    WHERE venda_id = :venda_id
                ");
                $stmt->execute(['venda_id' => $id]);

                // Remover itens
                $stmt = $this->db->prepare("DELETE FROM vendas_itens WHERE venda_id = :venda_id");
                $stmt->execute(['venda_id' => $id]);

                // Remover venda
                $stmt = $this->db->prepare("
                    DELETE FROM vendas
                    WHERE id = :id AND company_id = :company_id
                      AND EXISTS (SELECT 1 FROM ordem_servico_pmoc WHERE venda_id = vendas.id)
                ");
                $stmt->execute(['id' => $id, 'company_id' => $companyId]);

                $this->db->commit();

                $this->response->json([
                    'success' => true,
                    'message' => 'Ordem de Serviço PMOC excluída com sucesso!'
                ]);
            } catch (Exception $e) {
                $this->db->rollBack();
                throw $e;
            }
        } catch (Exception $e) {
            error_log("Erro ao excluir ordem de serviço PMOC: " . $e->getMessage());
            $this->response->json([
                'success' => false,
                'message' => 'Erro ao excluir: ' . $e->getMessage()
            ]);
        }
    }

    public function print(): void
    {
        try {
            $companyId = $this->getCompanyId();
            $id = (int) $this->request->get('id');

            $stmt = $this->db->prepare("
                SELECT v.*,
                    p.name as customer_name_full,
                    p.document as customer_document_full,
                    mp.name as payment_method_name,
                    u.name as vendor_name,
                    pmoc.*,
                    osp.pmoc_id
                FROM vendas v
                LEFT JOIN pessoas p ON v.customer_id = p.id
                LEFT JOIN metodos_pagamento mp ON v.payment_method_id = mp.id
                LEFT JOIN users u ON v.vendor_id = u.id
                INNER JOIN ordem_servico_pmoc osp ON v.id = osp.venda_id
                LEFT JOIN pmoc pmoc ON osp.pmoc_id = pmoc.id
                WHERE v.id = :id AND v.company_id = :company_id
            ");
            $stmt->execute(['id' => $id, 'company_id' => $companyId]);
            $ordem = $stmt->fetch();

            if (!$ordem) {
                $this->error('Ordem de serviço não encontrada');
                return;
            }

            // Buscar itens
            $stmt = $this->db->prepare("
                SELECT vi.*, p.name as product_name
                FROM vendas_itens vi
                LEFT JOIN produtos p ON vi.product_id = p.id
                WHERE vi.venda_id = :venda_id
                ORDER BY vi.id ASC
            ");
            $stmt->execute(['venda_id' => $id]);
            $itens = $stmt->fetchAll();

            // Buscar equipamentos
            $equipamentos = [];
            if (!empty($ordem['pmoc_id'])) {
                $stmt = $this->db->prepare("
                    SELECT * FROM pmoc_equipamentos
                    WHERE pmoc_id = :pmoc_id
                    ORDER BY id ASC
                ");
                $stmt->execute(['pmoc_id' => $ordem['pmoc_id']]);
                $equipamentos = $stmt->fetchAll();
            }

            $this->view('ordem-servico-pmoc/print', [
                'ordem' => $ordem,
                'itens' => $itens,
                'equipamentos' => $equipamentos,
                'pageTitle' => 'Imprimir Ordem de Serviço PMOC'
            ]);
        } catch (Exception $e) {
            error_log("Erro ao imprimir ordem de serviço PMOC: " . $e->getMessage());
            $this->error('Erro ao imprimir: ' . $e->getMessage());
        }
    }

    // Métodos auxiliares privados

    private function gerarProximoNumeroVenda(int $companyId): string
    {
        $ano = date('Y');
        $stmt = $this->db->prepare("
            SELECT COUNT(*) as total
            FROM vendas v
            INNER JOIN ordem_servico_pmoc osp ON v.id = osp.venda_id
            WHERE v.company_id = :company_id
              AND YEAR(v.sale_date) = :ano
        ");
        $stmt->execute([
            'company_id' => $companyId,
            'ano' => $ano
        ]);
        $result = $stmt->fetch();
        $numero = (int)($result['total'] ?? 0) + 1;
        return "OS-PMOC/{$ano}/" . str_pad((string)$numero, 4, '0', STR_PAD_LEFT);
    }

    private function criarPMOC(int $companyId, int $userId, array $data): int
    {
        $ano = date('Y');
        $stmt = $this->db->prepare("
            SELECT COUNT(*) as total
            FROM pmoc
            WHERE company_id = :company_id
              AND YEAR(data_emissao) = :ano
        ");
        $stmt->execute(['company_id' => $companyId, 'ano' => $ano]);
        $result = $stmt->fetch();
        $numero = ($result['total'] ?? 0) + 1;
        $numeroPMOC = "PMOC/{$ano}/" . str_pad($numero, 4, '0', STR_PAD_LEFT);
        $hashPMOC = md5($companyId . $numeroPMOC . time());

        $stmt = $this->db->prepare("
            INSERT INTO pmoc (
                company_id, hash_pmoc, numero_pmoc, data_emissao, data_validade,
                cliente_id, responsavel_tecnico, crea_cau, operador_id, situacao_id,
                observacoes, valor_total, origem, created_at
            ) VALUES (
                :company_id, :hash_pmoc, :numero_pmoc, :data_emissao, :data_validade,
                :cliente_id, :responsavel_tecnico, :crea_cau, :operador_id, :situacao_id,
                :observacoes, :valor_total, 'pmoc', NOW()
            )
        ");

        $dataValidade = !empty($data['pmoc_data_validade'])
            ? $data['pmoc_data_validade']
            : date('Y-m-d', strtotime('+1 year'));

        $stmt->execute([
            'company_id' => $companyId,
            'hash_pmoc' => $hashPMOC,
            'numero_pmoc' => $numeroPMOC,
            'data_emissao' => $data['pmoc_data_emissao'] ?? date('Y-m-d'),
            'data_validade' => $dataValidade,
            'cliente_id' => (int) $data['customer_id'],
            'responsavel_tecnico' => $data['pmoc_responsavel_tecnico'] ?? '',
            'crea_cau' => $data['pmoc_crea_cau'] ?? '',
            'operador_id' => $userId,
            'situacao_id' => 1, // Em elaboração
            'observacoes' => $data['pmoc_observacoes'] ?? '',
            'valor_total' => (float) ($data['total'] ?? 0)
        ]);

        return (int) $this->db->lastInsertId();
    }

    private function salvarItens(int $vendaId, array $itens): void
    {
        $companyId = $this->getCompanyId();

        $stmt = $this->db->prepare("
            INSERT INTO vendas_itens (
                company_id, venda_id, product_id, product_name, quantity, unit_price, total_price,
                discount, created_at
            ) VALUES (
                :company_id, :venda_id, :product_id, :product_name, :quantity, :unit_price, :total_price,
                :discount, NOW()
            )
        ");

        foreach ($itens as $item) {
            $stmt->execute([
                'company_id' => $companyId,
                'venda_id' => $vendaId,
                'product_id' => !empty($item['product_id']) ? (int) $item['product_id'] : null,
                'product_name' => $item['product_name'] ?? '',
                'quantity' => (float) ($item['quantity'] ?? 1),
                'unit_price' => (float) ($item['unit_price'] ?? 0),
                'total_price' => (float) ($item['total_price'] ?? 0),
                'discount' => (float) ($item['discount'] ?? 0)
            ]);
        }
    }
}
