<?php

declare(strict_types=1);

namespace App\Controllers;

use Exception;
use PDO;

class ModuloStatusController extends BaseController
{
    public function index(): void
    {
        // Verificar permissão de visualização
        if (!$this->canView('modulo-status')) {
            $this->response->forbidden('Você não tem permissão para visualizar status de módulos.');
            return;
        }

        try {
            $companyId = $this->getCompanyId();
            if (!$companyId) {
                $this->response->redirect('/login');
                return;
            }

            $search = trim((string) $this->request->get('search', ''));
            $modulo = trim((string) $this->request->get('modulo', ''));
            $ativo = $this->request->get('ativo', '');
            $padrao = $this->request->get('padrao', '');
            $final = $this->request->get('final', '');

            $query = "
                SELECT ms.*
                FROM modulo_status ms
                WHERE ms.company_id = :company_id
            ";
            $params = ['company_id' => $companyId];

            if ($search !== '') {
                $query .= "
                    AND (
                        ms.nome LIKE :search
                        OR ms.codigo LIKE :search
                        OR ms.descricao LIKE :search
                    )
                ";
                $params['search'] = '%' . $search . '%';
            }

            if ($modulo !== '') {
                $query .= " AND ms.modulo = :modulo";
                $params['modulo'] = $modulo;
            }

            if ($ativo !== '') {
                $query .= " AND ms.ativo = :ativo";
                $params['ativo'] = (int) ($ativo === '1' ? 1 : 0);
            }

            if ($padrao !== '') {
                $query .= " AND ms.is_default = :is_default";
                $params['is_default'] = (int) ($padrao === '1' ? 1 : 0);
            }

            if ($final !== '') {
                $query .= " AND ms.is_final = :is_final";
                $params['is_final'] = (int) ($final === '1' ? 1 : 0);
            }

            $query .= "
                ORDER BY ms.modulo ASC,
                         ms.ordem ASC,
                         ms.nome ASC
            ";

            $stmt = $this->db->prepare($query);
            $stmt->execute($params);
            $status = $stmt->fetchAll(PDO::FETCH_ASSOC);

            $modulos = $this->buscarModulos($companyId);

            $this->view('modulo-status/index', [
                'status' => $status,
                'modulos' => $modulos,
                'filters' => [
                    'search' => $search,
                    'modulo' => $modulo,
                    'ativo' => $ativo,
                    'padrao' => $padrao,
                    'final' => $final,
                ],
                'pageTitle' => 'Gestão de Status',
                'activeMenu' => 'modulo-status',
            ]);
        } catch (Exception $e) {
            error_log('Erro ao listar status do módulo: ' . $e->getMessage());
            $this->error('Erro ao carregar os status do módulo');
        }
    }

    public function edit(): void
    {
        try {
            $id = (int) $this->request->get('id');
            $status = $this->buscarStatus($id);

            if (!$status) {
                $this->response->notFound('Status não encontrado');
                return;
            }

            $this->view('modulo-status/edit', [
                'status' => $status,
                'pageTitle' => 'Editar Status',
                'activeMenu' => 'modulo-status',
            ]);
        } catch (Exception $e) {
            error_log('Erro ao carregar formulário de status: ' . $e->getMessage());
            $this->error('Erro ao carregar formulário de edição');
        }
    }

    public function update(): void
    {
        // Verificar permissão de edição
        if (!$this->canEdit('modulo-status')) {
            $this->response->forbidden('Você não tem permissão para editar status de módulos.');
            return;
        }

        $this->db->beginTransaction();

        try {
            $id = (int) $this->request->post('id');
            $statusAtual = $this->buscarStatus($id);

            if (!$statusAtual) {
                $this->error('Status não encontrado');
                $this->db->rollBack();
                return;
            }

            $data = [
                'nome' => trim((string) $this->request->post('nome')),
                'descricao' => trim((string) $this->request->post('descricao')),
                'gera_financeiro' => $this->checkboxValue('gera_financeiro'),
                'movimenta_estoque' => $this->checkboxValue('movimenta_estoque'),
                'reserva_estoque' => $this->checkboxValue('reserva_estoque'),
                'previsao_estoque' => $this->checkboxValue('previsao_estoque'),
                'emite_nfe' => $this->checkboxValue('emite_nfe'),
                'cor' => trim((string) $this->request->post('cor')),
                'icone' => trim((string) $this->request->post('icone')),
                'ordem' => (int) ($this->request->post('ordem') !== '' ? $this->request->post('ordem') : 0),
                'is_default' => $this->checkboxValue('is_default'),
                'is_final' => $this->checkboxValue('is_final'),
                'ativo' => $this->checkboxValue('ativo'),
                'id' => $id,
                'company_id' => $this->getCompanyId(),
            ];

            $errors = $this->validate([
                'nome' => 'required|min:2',
                'cor' => 'required',
                'icone' => 'required',
            ]);

            if (!empty($errors)) {
                $this->db->rollBack();
                $this->error('Dados inválidos', $errors);
                return;
            }

            $stmt = $this->db->prepare("
                UPDATE modulo_status SET
                    nome = :nome,
                    descricao = :descricao,
                    gera_financeiro = :gera_financeiro,
                    movimenta_estoque = :movimenta_estoque,
                    reserva_estoque = :reserva_estoque,
                    previsao_estoque = :previsao_estoque,
                    emite_nfe = :emite_nfe,
                    cor = :cor,
                    icone = :icone,
                    ordem = :ordem,
                    is_default = :is_default,
                    is_final = :is_final,
                    ativo = :ativo,
                    updated_at = NOW()
                WHERE id = :id
                  AND company_id = :company_id
            ");

            $stmt->execute($data);

            $this->logActivity('update', 'modulo_status', $id, $data);

            $this->db->commit();

            $this->success('Status atualizado com sucesso', [
                'redirect' => '/gestao/status',
            ]);
        } catch (Exception $e) {
            $this->db->rollBack();
            error_log('Erro ao atualizar status: ' . $e->getMessage());
            $this->error('Erro ao atualizar status');
        }
    }

    private function buscarStatus(int $id): ?array
    {
        if (!$id) {
            return null;
        }

        try {
            $stmt = $this->db->prepare("
                SELECT *
                FROM modulo_status
                WHERE id = :id
                  AND company_id = :company_id
                LIMIT 1
            ");
            $stmt->execute([
                'id' => $id,
                'company_id' => $this->getCompanyId(),
            ]);

            $result = $stmt->fetch(PDO::FETCH_ASSOC);

            return $result ?: null;
        } catch (Exception $e) {
            error_log('Erro ao buscar status: ' . $e->getMessage());
            return null;
        }
    }

    private function buscarModulos(int $companyId): array
    {
        try {
            $stmt = $this->db->prepare("
                SELECT DISTINCT modulo
                FROM modulo_status
                WHERE company_id = :company_id
                ORDER BY modulo ASC
            ");
            $stmt->execute(['company_id' => $companyId]);

            return array_column($stmt->fetchAll(PDO::FETCH_ASSOC), 'modulo');
        } catch (Exception $e) {
            error_log('Erro ao buscar módulos de status: ' . $e->getMessage());
            return [];
        }
    }

    private function checkboxValue(string $field): int
    {
        return $this->request->post($field, '0') === '1' ? 1 : 0;
    }
}


