<?php

declare(strict_types=1);

namespace App\Controllers;

use Exception;

class ModelosPmocController extends BaseController
{
    /**
     * Lista todos os modelos PMOC
     */
    public function index(): void
    {
        // Verificar permissão de visualização
        if (!$this->canView('modelos-pmoc')) {
            $this->response->forbidden('Você não tem permissão para visualizar modelos PMOC.');
            return;
        }

        try {
            $companyId = $this->getCompanyId();

            $stmt = $this->db->prepare("
                SELECT *
                FROM modelos_pmoc
                WHERE company_id = :company_id
                ORDER BY nome ASC
            ");
            $stmt->execute(['company_id' => $companyId]);
            $modelos = $stmt->fetchAll();

            $this->view('modelos-pmoc/index', [
                'modelos' => $modelos,
                'pageTitle' => 'Modelos PMOC',
                'activeMenu' => 'ordem-servico-pmoc'
            ]);
        } catch (Exception $e) {
            error_log("Erro ao carregar modelos PMOC: " . $e->getMessage());
            $this->error('Erro ao carregar modelos PMOC');
        }
    }

    /**
     * Exibe formulário de criação
     */
    public function create(): void
    {
        try {
            $companyId = $this->getCompanyId();

            // Buscar serviços cadastrados (produtos do tipo servico)
            $stmt = $this->db->prepare("
                SELECT id, name, description
                FROM produtos
                WHERE company_id = :company_id
                  AND type = 'servico'
                  AND is_active = 1
                ORDER BY name ASC
            ");
            $stmt->execute(['company_id' => $companyId]);
            $servicosDisponiveis = $stmt->fetchAll();

            $this->view('modelos-pmoc/create', [
                'servicos' => [],
                'servicosDisponiveis' => $servicosDisponiveis,
                'pageTitle' => 'Novo Modelo PMOC',
                'activeMenu' => 'ordem-servico-pmoc'
            ]);
        } catch (Exception $e) {
            error_log("Erro ao carregar formulário de criação: " . $e->getMessage());
            $this->error('Erro ao carregar formulário');
        }
    }

    /**
     * Salva novo modelo PMOC
     */
    public function store(): void
    {
        // Verificar permissão de criação
        if (!$this->canCreate('modelos-pmoc')) {
            $this->response->forbidden('Você não tem permissão para criar modelos PMOC.');
            return;
        }

        try {
            $companyId = $this->getCompanyId();
            $userId = $this->getUserId();

            $data = [
                'company_id' => $companyId,
                'nome' => $this->request->post('nome'),
                'tipo_equipamento' => $this->request->post('tipo_equipamento') ?: null,
                'descricao' => $this->request->post('descricao') ?: null,
                'observacoes' => $this->request->post('observacoes') ?: null,
                'is_active' => $this->request->post('is_active') ? 1 : 0,
            ];

            // Validação simples
            $errors = [];
            if (empty($data['nome']) || strlen($data['nome']) < 2) {
                $errors['nome'] = 'O nome do modelo é obrigatório e deve ter no mínimo 2 caracteres';
            }

            if (!empty($errors)) {
                if ($this->request->isAjax()) {
                    $this->error('Dados inválidos', $errors);
                    return;
                }

                // Buscar serviços disponíveis novamente para exibir no formulário
                $stmt = $this->db->prepare("
                    SELECT id, name, description
                    FROM produtos
                    WHERE company_id = :company_id
                      AND type = 'servico'
                      AND is_active = 1
                    ORDER BY name ASC
                ");
                $stmt->execute(['company_id' => $companyId]);
                $servicosDisponiveis = $stmt->fetchAll();

                $this->view('modelos-pmoc/create', [
                    'modelo' => $data,
                    'servicos' => $this->request->post('servicos') ?? [],
                    'servicosDisponiveis' => $servicosDisponiveis,
                    'errors' => $errors,
                    'pageTitle' => 'Novo Modelo PMOC',
                    'activeMenu' => 'ordem-servico-pmoc'
                ]);
                return;
            }

            $this->db->beginTransaction();

            $stmt = $this->db->prepare("
                INSERT INTO modelos_pmoc (
                    company_id, nome, tipo_equipamento, descricao, observacoes, is_active,
                    created_at, updated_at
                ) VALUES (
                    :company_id, :nome, :tipo_equipamento, :descricao, :observacoes, :is_active,
                    NOW(), NOW()
                )
            ");

            $stmt->execute($data);
            $modeloId = $this->db->lastInsertId();

            // Salvar serviços
            $servicos = $this->request->post('servicos') ?? [];
            if (!empty($servicos)) {
                $stmtServico = $this->db->prepare("
                    INSERT INTO modelos_pmoc_servicos (
                        modelo_pmoc_id, product_id, nome, descricao, frequencia, ordem,
                        created_at, updated_at
                    ) VALUES (
                        :modelo_pmoc_id, :product_id, :nome, :descricao, :frequencia, :ordem,
                        NOW(), NOW()
                    )
                ");

                foreach ($servicos as $index => $servico) {
                    if (!empty($servico['nome'])) {
                        $stmtServico->execute([
                            'modelo_pmoc_id' => $modeloId,
                            'product_id' => !empty($servico['product_id']) ? (int)$servico['product_id'] : null,
                            'nome' => $servico['nome'],
                            'descricao' => $servico['descricao'] ?? null,
                            'frequencia' => $servico['frequencia'] ?? null,
                            'ordem' => $index
                        ]);
                    }
                }
            }

            $this->logActivity('create', 'modelos_pmoc', (int)$modeloId, $data);

            $this->db->commit();

            if ($this->request->isAjax()) {
                $this->success('Modelo PMOC criado com sucesso', [
                    'id' => $modeloId,
                    'redirect' => '/modelos-pmoc'
                ]);
                return;
            }

            $this->redirect('/modelos-pmoc?success=1');
        } catch (Exception $e) {
            if ($this->db->inTransaction()) {
                $this->db->rollBack();
            }
            error_log("Erro ao criar modelo PMOC: " . $e->getMessage());

            if ($this->request->isAjax()) {
                $this->error('Erro ao cadastrar modelo PMOC: ' . $e->getMessage());
                return;
            }

            $this->view('modelos-pmoc/create', [
                'modelo' => $data ?? [],
                'errors' => ['Erro ao cadastrar modelo PMOC'],
                'pageTitle' => 'Novo Modelo PMOC',
                'activeMenu' => 'ordem-servico-pmoc'
            ]);
        }
    }

    /**
     * Exibe formulário de edição
     */
    public function edit(): void
    {
        // Verificar permissão de edição
        if (!$this->canEdit('modelos-pmoc')) {
            $this->response->forbidden('Você não tem permissão para editar modelos PMOC.');
            return;
        }

        try {
            $id = (int) $this->request->get('id');
            $companyId = $this->getCompanyId();
            $modelo = $this->getModelo($id, $companyId);

            if (!$modelo) {
                $this->response->notFound('Modelo PMOC não encontrado');
                return;
            }

            // Buscar serviços do modelo
            $servicos = $this->getServicosModelo($id);

            // Buscar serviços cadastrados (produtos do tipo servico)
            $stmt = $this->db->prepare("
                SELECT id, name, description
                FROM produtos
                WHERE company_id = :company_id
                  AND type = 'servico'
                  AND is_active = 1
                ORDER BY name ASC
            ");
            $stmt->execute(['company_id' => $companyId]);
            $servicosDisponiveis = $stmt->fetchAll();

            $this->view('modelos-pmoc/edit', [
                'modelo' => $modelo,
                'servicos' => $servicos,
                'servicosDisponiveis' => $servicosDisponiveis,
                'isEdit' => true,
                'pageTitle' => 'Editar Modelo PMOC',
                'activeMenu' => 'ordem-servico-pmoc'
            ]);
        } catch (Exception $e) {
            error_log("Erro ao exibir formulário de edição: " . $e->getMessage());
            $this->error('Erro ao carregar formulário');
        }
    }

    /**
     * Atualiza um modelo PMOC
     */
    public function update(): void
    {
        // Verificar permissão de edição
        if (!$this->canEdit('modelos-pmoc')) {
            $this->response->forbidden('Você não tem permissão para editar modelos PMOC.');
            return;
        }

        try {
            $id = (int) $this->request->post('id');
            $companyId = $this->getCompanyId();
            $modelo = $this->getModelo($id, $companyId);

            if (!$modelo) {
                if ($this->request->isAjax()) {
                    $this->error('Modelo PMOC não encontrado');
                    return;
                }
                $this->redirect('/modelos-pmoc?error=not_found');
                return;
            }

            $data = [
                'nome' => $this->request->post('nome'),
                'tipo_equipamento' => $this->request->post('tipo_equipamento') ?: null,
                'descricao' => $this->request->post('descricao') ?: null,
                'observacoes' => $this->request->post('observacoes') ?: null,
                'is_active' => $this->request->post('is_active') ? 1 : 0,
                'id' => $id
            ];

            // Validação simples
            $errors = [];
            if (empty($data['nome']) || strlen($data['nome']) < 2) {
                $errors['nome'] = 'O nome do modelo é obrigatório e deve ter no mínimo 2 caracteres';
            }

            if (!empty($errors)) {
                if ($this->request->isAjax()) {
                    $this->error('Dados inválidos', $errors);
                    return;
                }

                // Buscar serviços disponíveis novamente para exibir no formulário
                $stmt = $this->db->prepare("
                    SELECT id, name, description
                    FROM produtos
                    WHERE company_id = :company_id
                      AND type = 'servico'
                      AND is_active = 1
                    ORDER BY name ASC
                ");
                $stmt->execute(['company_id' => $companyId]);
                $servicosDisponiveis = $stmt->fetchAll();

                $this->view('modelos-pmoc/edit', [
                    'modelo' => array_merge($modelo, $data),
                    'servicos' => $this->request->post('servicos') ?? [],
                    'servicosDisponiveis' => $servicosDisponiveis,
                    'isEdit' => true,
                    'errors' => $errors,
                    'pageTitle' => 'Editar Modelo PMOC',
                    'activeMenu' => 'ordem-servico-pmoc'
                ]);
                return;
            }

            $this->db->beginTransaction();

            $stmt = $this->db->prepare("
                UPDATE modelos_pmoc SET
                    nome = :nome,
                    tipo_equipamento = :tipo_equipamento,
                    descricao = :descricao,
                    observacoes = :observacoes,
                    is_active = :is_active,
                    updated_at = NOW()
                WHERE id = :id AND company_id = :company_id
            ");

            $stmt->execute(array_merge($data, ['company_id' => $companyId]));

            // Atualizar serviços - remover todos e inserir novamente
            $stmtDelete = $this->db->prepare("DELETE FROM modelos_pmoc_servicos WHERE modelo_pmoc_id = :modelo_pmoc_id");
            $stmtDelete->execute(['modelo_pmoc_id' => $id]);

            $servicos = $this->request->post('servicos') ?? [];
            if (!empty($servicos)) {
                $stmtServico = $this->db->prepare("
                    INSERT INTO modelos_pmoc_servicos (
                        modelo_pmoc_id, product_id, nome, descricao, frequencia, ordem,
                        created_at, updated_at
                    ) VALUES (
                        :modelo_pmoc_id, :product_id, :nome, :descricao, :frequencia, :ordem,
                        NOW(), NOW()
                    )
                ");

                foreach ($servicos as $index => $servico) {
                    if (!empty($servico['nome'])) {
                        $stmtServico->execute([
                            'modelo_pmoc_id' => $id,
                            'product_id' => !empty($servico['product_id']) ? (int)$servico['product_id'] : null,
                            'nome' => $servico['nome'],
                            'descricao' => $servico['descricao'] ?? null,
                            'frequencia' => $servico['frequencia'] ?? null,
                            'ordem' => $index
                        ]);
                    }
                }
            }

            $this->logActivity('update', 'modelos_pmoc', (int)$id, $data);

            $this->db->commit();

            if ($this->request->isAjax()) {
                $this->success('Modelo PMOC atualizado com sucesso', [
                    'redirect' => '/modelos-pmoc'
                ]);
                return;
            }

            $this->redirect('/modelos-pmoc?success=1');
        } catch (Exception $e) {
            if ($this->db->inTransaction()) {
                $this->db->rollBack();
            }
            error_log("Erro ao atualizar modelo PMOC: " . $e->getMessage());

            if ($this->request->isAjax()) {
                $this->error('Erro ao atualizar modelo PMOC: ' . $e->getMessage());
                return;
            }

            $this->view('modelos-pmoc/edit', [
                'modelo' => $modelo ?? [],
                'errors' => ['Erro ao atualizar modelo PMOC'],
                'pageTitle' => 'Editar Modelo PMOC',
                'activeMenu' => 'ordem-servico-pmoc'
            ]);
        }
    }

    /**
     * Exclui um modelo PMOC
     */
    public function delete(): void
    {
        // Verificar permissão de exclusão
        if (!$this->canDelete('modelos-pmoc')) {
            $this->response->forbidden('Você não tem permissão para excluir modelos PMOC.');
            return;
        }

        try {
            $id = (int) $this->request->post('id');
            $companyId = $this->getCompanyId();
            $modelo = $this->getModelo($id, $companyId);

            if (!$modelo) {
                $this->error('Modelo PMOC não encontrado');
                return;
            }

            $this->db->beginTransaction();

            $stmt = $this->db->prepare("
                DELETE FROM modelos_pmoc
                WHERE id = :id AND company_id = :company_id
            ");

            $stmt->execute(['id' => $id, 'company_id' => $companyId]);

            $this->logActivity('delete', 'modelos_pmoc', $id, $modelo);

            $this->db->commit();

            $this->success('Modelo PMOC excluído com sucesso');
        } catch (Exception $e) {
            if ($this->db->inTransaction()) {
                $this->db->rollBack();
            }
            error_log("Erro ao excluir modelo PMOC: " . $e->getMessage());
            $this->error('Erro ao excluir modelo PMOC');
        }
    }

    /**
     * Busca um modelo PMOC por ID
     */
    private function getModelo(int $id, int $companyId): ?array
    {
        $stmt = $this->db->prepare("
            SELECT *
            FROM modelos_pmoc
            WHERE id = :id AND company_id = :company_id
            LIMIT 1
        ");
        $stmt->execute(['id' => $id, 'company_id' => $companyId]);
        $result = $stmt->fetch();
        return $result ?: null;
    }

    /**
     * Busca serviços de um modelo PMOC
     */
    private function getServicosModelo(int $modeloId): array
    {
        $stmt = $this->db->prepare("
            SELECT *
            FROM modelos_pmoc_servicos
            WHERE modelo_pmoc_id = :modelo_pmoc_id
            ORDER BY ordem ASC, id ASC
        ");
        $stmt->execute(['modelo_pmoc_id' => $modeloId]);
        return $stmt->fetchAll() ?: [];
    }
}
