<?php

namespace App\Controllers;

use Exception;

class ImpostosController extends BaseController
{
    /**
     * Lista todos os grupos tributários
     */
    public function index(): void
    {
        // Verificar permissão de visualização
        if (!$this->canView('impostos')) {
            $this->response->forbidden('Você não tem permissão para visualizar impostos.');
            return;
        }

        try {
            $companyId = $this->getCompanyId();

            // Busca grupos tributários
            $stmt = $this->db->prepare("
                SELECT i.*
                FROM impostos i
                WHERE i.company_id = :company_id
                ORDER BY i.nome ASC, i.uf_origem ASC
            ");
            $stmt->execute(['company_id' => $companyId]);
            $impostos = $stmt->fetchAll();

            $this->view('impostos/index', [
                'impostos' => $impostos,
                'pageTitle' => 'Gestão de Impostos',
                'activeMenu' => 'impostos'
            ]);

        } catch (Exception $e) {
            error_log("Erro ao carregar impostos: " . $e->getMessage());
            $this->error('Erro ao carregar impostos');
        }
    }

    /**
     * Exibe formulário de criação
     */
    public function create(): void
    {
        $empresas = $this->getEmpresas();
        $estados = $this->getEstados();
        $cfops = $this->getCfops();

        $this->view('impostos/create', [
            'empresas' => $empresas,
            'estados' => $estados,
            'cfops' => $cfops,
            'pageTitle' => 'Novo Grupo Tributário',
            'activeMenu' => 'impostos'
        ]);
    }

    /**
     * Salva novo grupo tributário
     */
    public function store(): void
    {
        // Verificar permissão de criação
        if (!$this->canCreate('impostos')) {
            $this->response->forbidden('Você não tem permissão para criar impostos.');
            return;
        }

        try {
            $companyId = $this->getCompanyId();
            // Usar o hash do POST se existir, senão gerar um novo
            $hashImposto = $this->request->post('hash_imposto') ?: hash('sha256', uniqid(mt_rand(), true));

            $data = [
                'company_id' => $companyId,
                'hash_imposto' => $hashImposto,
                'nome' => $this->request->post('nome'),
                'empresa' => $this->request->post('empresa') ?: null,
                'ativo' => $this->request->post('ativo') ?: 'Sim',

                // Dados Fiscais Gerais
                'ncm' => $this->request->post('ncm'),
                'cest' => $this->request->post('cest'),
                'ibpt' => $this->request->post('ibpt') ? str_replace(',', '.', $this->request->post('ibpt')) : 0,
                'atividade_produto_sped' => $this->request->post('atividade_produto_sped'),
                'informacoes_adicionais_sped' => $this->request->post('informacoes_adicionais_sped'),
                'genero_produto_edoc' => $this->request->post('genero_produto_edoc'),

                // ICMS Dentro do Estado
                'uf_origem' => $this->request->post('uf_origem'),
                'cst_csosn' => $this->request->post('cst_csosn'),
                'icms' => $this->request->post('icms') ?: 0,
                'modalidade_bc_icms' => $this->request->post('modalidade_bc_icms'),
                'origem_mercadoria' => $this->request->post('origem_mercadoria'),
                'motivo_desicms' => $this->request->post('motivo_desicms'),
                'reducao_bc' => $this->request->post('reducao_bc') ?: 0,
                'fcp' => $this->request->post('fcp') ?: 0,
                'cfop' => $this->request->post('cfop'),
                'valor_pauta' => $this->request->post('valor_pauta') ?: 0,

                // ICMS ST Fora do Estado
                'uf_destino' => $this->request->post('uf_destino'),
                'cst_csosn_st' => $this->request->post('cst_csosn_st'),
                'icms_st' => $this->request->post('icms_st') ?: 0,
                'mva' => $this->request->post('mva') ?: 0,
                'modalidade_bc_icms_st' => $this->request->post('modalidade_bc_icms_st'),
                'reducao_bc_st' => $this->request->post('reducao_bc_st') ?: 0,
                'fcp_st' => $this->request->post('fcp_st') ?: 0,
                'cfop_st' => $this->request->post('cfop_st'),

                // IPI
                'cst_ipi' => $this->request->post('cst_ipi'),
                'ipi' => $this->request->post('ipi') ?: 0,
                'classe_enquadramento_ipi' => $this->request->post('classe_enquadramento_ipi'),
                'enquadramento_ipi' => $this->request->post('enquadramento_ipi'),

                // PIS/COFINS
                'cst_pis' => $this->request->post('cst_pis'),
                'pis' => $this->request->post('pis') ?: 0,
                'cst_cofins' => $this->request->post('cst_cofins'),
                'cofins' => $this->request->post('cofins') ?: 0,
                'natureza_receita_pis_cofins' => $this->request->post('natureza_receita_pis_cofins'),
            ];

            // Validação
            if (empty($data['nome'])) {
                $this->error('O campo Nome do Grupo é obrigatório');
                return;
            }

            $this->db->beginTransaction();

            $stmt = $this->db->prepare("
                INSERT INTO impostos (
                    company_id, hash_imposto, nome, empresa, ativo,
                    ncm, cest, ibpt, atividade_produto_sped, informacoes_adicionais_sped, genero_produto_edoc,
                    uf_origem, cst_csosn, icms, modalidade_bc_icms, origem_mercadoria, motivo_desicms, reducao_bc, fcp, cfop, valor_pauta,
                    uf_destino, cst_csosn_st, icms_st, mva, modalidade_bc_icms_st, reducao_bc_st, fcp_st, cfop_st,
                    cst_ipi, ipi, classe_enquadramento_ipi, enquadramento_ipi,
                    cst_pis, pis, cst_cofins, cofins, natureza_receita_pis_cofins,
                    created_at, updated_at
                ) VALUES (
                    :company_id, :hash_imposto, :nome, :empresa, :ativo,
                    :ncm, :cest, :ibpt, :atividade_produto_sped, :informacoes_adicionais_sped, :genero_produto_edoc,
                    :uf_origem, :cst_csosn, :icms, :modalidade_bc_icms, :origem_mercadoria, :motivo_desicms, :reducao_bc, :fcp, :cfop, :valor_pauta,
                    :uf_destino, :cst_csosn_st, :icms_st, :mva, :modalidade_bc_icms_st, :reducao_bc_st, :fcp_st, :cfop_st,
                    :cst_ipi, :ipi, :classe_enquadramento_ipi, :enquadramento_ipi,
                    :cst_pis, :pis, :cst_cofins, :cofins, :natureza_receita_pis_cofins,
                    NOW(), NOW()
                )
            ");

            $stmt->execute($data);
            $impostoId = $this->db->lastInsertId();

            $this->logActivity('create', 'impostos', $impostoId, $data);
            $this->db->commit();

            $this->success('Grupo tributário cadastrado com sucesso', [
                'id' => $impostoId,
                'redirect' => '/impostos'
            ]);

        } catch (Exception $e) {
            $this->db->rollBack();
            error_log("Erro ao criar imposto: " . $e->getMessage());
            $this->error('Erro ao cadastrar grupo tributário: ' . $e->getMessage());
        }
    }

    /**
     * Exibe formulário de edição
     */
    public function edit(): void
    {
        // Verificar permissão de edição
        if (!$this->canEdit('impostos')) {
            $this->response->forbidden('Você não tem permissão para editar impostos.');
            return;
        }

        try {
            $id = (int) $this->request->get('id');
            $imposto = $this->getImposto($id);

            if (!$imposto) {
                $this->redirect('/impostos');
                return;
            }

            $empresas = $this->getEmpresas();
            $estados = $this->getEstados();
            $cfops = $this->getCfops();

            // BUSCAR CONFIGURAÇÕES EXISTENTES
            $configuracoes = [];
            if (!empty($imposto['hash_imposto'])) {
                $stmt = $this->db->prepare("
                    SELECT
                        c.*,
                        e_origem.nome_estado as nome_estado_origem,
                        e_destino.nome_estado as nome_estado_destino
                    FROM impostos_configuracoes c
                    LEFT JOIN estados e_origem ON c.uf_origem = e_origem.codigo_uf
                    LEFT JOIN estados e_destino ON c.uf_destino = e_destino.codigo_uf
                    WHERE c.hash_imposto = :hash AND c.company_id = :company_id
                    ORDER BY c.uf_origem
                ");
                $stmt->execute([
                    'hash' => $imposto['hash_imposto'],
                    'company_id' => $this->getCompanyId()
                ]);
                $configuracoes = $stmt->fetchAll();
            }

            $this->view('impostos/edit', [
                'imposto' => $imposto,
                'configuracoes' => $configuracoes,
                'empresas' => $empresas,
                'estados' => $estados,
                'cfops' => $cfops,
                'pageTitle' => 'Editar ' . $imposto['nome'],
                'activeMenu' => 'impostos'
            ]);

        } catch (Exception $e) {
            error_log("Erro ao exibir formulário: " . $e->getMessage());
            $this->redirect('/impostos');
        }
    }

    /**
     * Atualiza um grupo tributário
     */
    public function update(): void
    {
        // Verificar permissão de edição
        if (!$this->canEdit('impostos')) {
            $this->response->forbidden('Você não tem permissão para editar impostos.');
            return;
        }

        try {
            $id = (int) $this->request->post('id');
            $imposto = $this->getImposto($id);

            if (!$imposto) {
                $this->error('Grupo tributário não encontrado');
                return;
            }

            $data = [
                'nome' => $this->request->post('nome'),
                'empresa' => $this->request->post('empresa') ?: null,
                'ativo' => $this->request->post('ativo') ?: 'Sim',

                'ncm' => $this->request->post('ncm'),
                'cest' => $this->request->post('cest'),
                'ibpt' => $this->request->post('ibpt') ?: 0,
                'atividade_produto_sped' => $this->request->post('atividade_produto_sped'),
                'informacoes_adicionais_sped' => $this->request->post('informacoes_adicionais_sped'),
                'genero_produto_edoc' => $this->request->post('genero_produto_edoc'),

                'uf_origem' => $this->request->post('uf_origem'),
                'cst_csosn' => $this->request->post('cst_csosn'),
                'icms' => $this->request->post('icms') ?: 0,
                'modalidade_bc_icms' => $this->request->post('modalidade_bc_icms'),
                'origem_mercadoria' => $this->request->post('origem_mercadoria'),
                'motivo_desicms' => $this->request->post('motivo_desicms'),
                'reducao_bc' => $this->request->post('reducao_bc') ?: 0,
                'fcp' => $this->request->post('fcp') ?: 0,
                'cfop' => $this->request->post('cfop'),
                'valor_pauta' => $this->request->post('valor_pauta') ?: 0,

                'uf_destino' => $this->request->post('uf_destino'),
                'cst_csosn_st' => $this->request->post('cst_csosn_st'),
                'icms_st' => $this->request->post('icms_st') ?: 0,
                'mva' => $this->request->post('mva') ?: 0,
                'modalidade_bc_icms_st' => $this->request->post('modalidade_bc_icms_st'),
                'reducao_bc_st' => $this->request->post('reducao_bc_st') ?: 0,
                'fcp_st' => $this->request->post('fcp_st') ?: 0,
                'cfop_st' => $this->request->post('cfop_st'),

                'cst_ipi' => $this->request->post('cst_ipi'),
                'ipi' => $this->request->post('ipi') ?: 0,
                'classe_enquadramento_ipi' => $this->request->post('classe_enquadramento_ipi'),
                'enquadramento_ipi' => $this->request->post('enquadramento_ipi'),

                'cst_pis' => $this->request->post('cst_pis'),
                'pis' => $this->request->post('pis') ?: 0,
                'cst_cofins' => $this->request->post('cst_cofins'),
                'cofins' => $this->request->post('cofins') ?: 0,
                'natureza_receita_pis_cofins' => $this->request->post('natureza_receita_pis_cofins'),

                'id' => $id,
                'company_id' => $this->getCompanyId()
            ];

            if (empty($data['nome'])) {
                $this->error('O campo Nome do Grupo é obrigatório');
                return;
            }

            $this->db->beginTransaction();

            $stmt = $this->db->prepare("
                UPDATE impostos SET
                    nome = :nome, empresa = :empresa, ativo = :ativo,
                    ncm = :ncm, cest = :cest, ibpt = :ibpt,
                    atividade_produto_sped = :atividade_produto_sped,
                    informacoes_adicionais_sped = :informacoes_adicionais_sped,
                    genero_produto_edoc = :genero_produto_edoc,
                    uf_origem = :uf_origem, cst_csosn = :cst_csosn, icms = :icms,
                    modalidade_bc_icms = :modalidade_bc_icms, origem_mercadoria = :origem_mercadoria,
                    motivo_desicms = :motivo_desicms, reducao_bc = :reducao_bc, fcp = :fcp,
                    cfop = :cfop, valor_pauta = :valor_pauta,
                    uf_destino = :uf_destino, cst_csosn_st = :cst_csosn_st, icms_st = :icms_st,
                    mva = :mva, modalidade_bc_icms_st = :modalidade_bc_icms_st,
                    reducao_bc_st = :reducao_bc_st, fcp_st = :fcp_st, cfop_st = :cfop_st,
                    cst_ipi = :cst_ipi, ipi = :ipi,
                    classe_enquadramento_ipi = :classe_enquadramento_ipi,
                    enquadramento_ipi = :enquadramento_ipi,
                    cst_pis = :cst_pis, pis = :pis, cst_cofins = :cst_cofins, cofins = :cofins,
                    natureza_receita_pis_cofins = :natureza_receita_pis_cofins,
                    updated_at = NOW()
                WHERE id = :id AND company_id = :company_id
            ");

            $stmt->execute($data);

            $this->logActivity('update', 'impostos', $id, $data);
            $this->db->commit();

            $this->success('Grupo tributário atualizado com sucesso', [
                'redirect' => '/impostos'
            ]);

        } catch (Exception $e) {
            $this->db->rollBack();
            error_log("Erro ao atualizar imposto: " . $e->getMessage());
            $this->error('Erro ao atualizar grupo tributário');
        }
    }

    /**
     * Exclui um grupo tributário
     */
    public function delete(): void
    {
        // Verificar permissão de exclusão
        if (!$this->canDelete('impostos')) {
            $this->response->forbidden('Você não tem permissão para excluir impostos.');
            return;
        }

        try {
            $id = (int) $this->request->post('id');
            $imposto = $this->getImposto($id);

            if (!$imposto) {
                $this->error('Grupo tributário não encontrado');
                return;
            }

            $this->db->beginTransaction();

            $stmt = $this->db->prepare("DELETE FROM impostos WHERE id = :id AND company_id = :company_id");
            $stmt->execute(['id' => $id, 'company_id' => $this->getCompanyId()]);

            $this->logActivity('delete', 'impostos', $id, $imposto);
            $this->db->commit();

            $this->success('Grupo tributário excluído com sucesso');

        } catch (Exception $e) {
            $this->db->rollBack();
            error_log("Erro ao excluir imposto: " . $e->getMessage());
            $this->error('Erro ao excluir grupo tributário');
        }
    }

    /**
     * Clona uma configuração de imposto para outro(s) estado(s)
     */
    public function clonar(): void
    {
        try {
            $hashImposto = $this->request->post('hash_imposto');
            $ufOrigem = $this->request->post('uf_origem');
            $estadoDestino = $this->request->post('estado');
            $todosEstados = $this->request->post('todos_estados');
            $companyId = $this->getCompanyId();

            // Busca a configuração original da tabela auxiliar
            $stmt = $this->db->prepare("
                SELECT * FROM impostos_configuracoes
                WHERE hash_imposto = :hash AND uf_origem = :uf AND company_id = :company_id
                LIMIT 1
            ");
            $stmt->execute([
                'hash' => $hashImposto,
                'uf' => $ufOrigem,
                'company_id' => $companyId
            ]);
            $configOriginal = $stmt->fetch();

            if (!$configOriginal) {
                $this->error('Configuração original não encontrada');
                return;
            }

            $clonados = 0;
            $estados = $todosEstados ? $this->getEstados() : [['codigo_uf' => $estadoDestino]];

            $this->db->beginTransaction();

            foreach ($estados as $estado) {
                $uf = $estado['codigo_uf'] ?? $estado;

                // Não clonar para a mesma UF de destino que já existe
                if ($uf == $configOriginal['uf_destino']) continue;

                // Verifica se já existe configuração com esta UF de destino
                $stmt = $this->db->prepare("
                    SELECT id FROM impostos_configuracoes
                    WHERE hash_imposto = :hash AND uf_origem = :uf_origem AND uf_destino = :uf_destino AND company_id = :company_id
                ");
                $stmt->execute([
                    'hash' => $hashImposto,
                    'uf_origem' => $ufOrigem, // Mantém a origem original
                    'uf_destino' => $uf, // Verifica o novo destino
                    'company_id' => $companyId
                ]);

                if ($stmt->fetch()) continue; // Já existe, pula

                // Clona a configuração para a nova UF
                $novaConfig = $configOriginal;
                unset($novaConfig['id']); // Remove o ID para criar novo registro
                unset($novaConfig['created_at']); // Remove timestamp
                // PRESERVA a UF de origem (não muda)
                // ALTERA apenas a UF de destino
                $novaConfig['uf_destino'] = $uf; // Altera a UF de destino para o novo estado

                // Monta query de inserção
                $campos = array_keys($novaConfig);
                $placeholders = ':' . implode(', :', $campos);
                $sql = "INSERT INTO impostos_configuracoes (" . implode(', ', $campos) . ") VALUES ($placeholders)";

                $stmt = $this->db->prepare($sql);
                $stmt->execute($novaConfig);

                $clonados++;
            }

            $this->db->commit();

            $this->success("Configuração clonada para $clonados estado(s)");

        } catch (Exception $e) {
            if ($this->db->inTransaction()) {
                $this->db->rollBack();
            }
            error_log("Erro ao clonar configuração: " . $e->getMessage());
            $this->error('Erro ao clonar configuração: ' . $e->getMessage());
        }
    }

    // Métodos auxiliares
    private function getImposto(int $id): ?array
    {
        $stmt = $this->db->prepare("
            SELECT * FROM impostos WHERE id = :id AND company_id = :company_id
        ");
        $stmt->execute(['id' => $id, 'company_id' => $this->getCompanyId()]);
        return $stmt->fetch() ?: null;
    }

    private function getEmpresas(): array
    {
        $stmt = $this->db->prepare("SELECT id, razao_social, cnpj FROM empresas WHERE company_id = :company_id ORDER BY razao_social");
        $stmt->execute(['company_id' => $this->getCompanyId()]);
        return $stmt->fetchAll();
    }

    private function getEstados(): array
    {
        $stmt = $this->db->query("SELECT codigo_uf, nome_estado FROM estados ORDER BY codigo_uf");
        return $stmt->fetchAll();
    }

    private function getCfops(): array
    {
        $stmt = $this->db->query("SELECT id, descricao FROM cfop WHERE ativo = 'Sim' ORDER BY id");
        return $stmt->fetchAll();
    }

    /**
     * Adiciona uma configuração de UF na tabela auxiliar
     */
    public function adicionarConfiguracao(): void
    {
        try {
            $companyId = $this->getCompanyId();
            $hashImposto = $this->request->post('hash_imposto');

            $data = [
                'company_id' => $companyId,
                'hash_imposto' => $hashImposto,
                'uf_origem' => $this->request->post('uf_origem'),
                'cst_csosn' => $this->request->post('cst_csosn'),
                'icms' => $this->request->post('icms') ?: 0,
                'modalidade_bc_icms' => $this->request->post('modalidade_bc_icms'),
                'origem_mercadoria' => $this->request->post('origem_mercadoria'),
                'motivo_desicms' => $this->request->post('motivo_desicms'),
                'reducao_bc' => $this->request->post('reducao_bc') ?: 0,
                'fcp' => $this->request->post('fcp') ?: 0,
                'cfop' => $this->request->post('cfop'),
                'valor_pauta' => $this->request->post('valor_pauta') ?: 0,
                'uf_destino' => $this->request->post('uf_destino'),
                'cst_csosn_st' => $this->request->post('cst_csosn_st'),
                'icms_st' => $this->request->post('icms_st') ?: 0,
                'mva' => $this->request->post('mva') ?: 0,
                'modalidade_bc_icms_st' => $this->request->post('modalidade_bc_icms_st'),
                'reducao_bc_st' => $this->request->post('reducao_bc_st') ?: 0,
                'fcp_st' => $this->request->post('fcp_st') ?: 0,
                'cfop_st' => $this->request->post('cfop_st'),
                'cst_ipi' => $this->request->post('cst_ipi'),
                'ipi' => $this->request->post('ipi') ?: 0,
                'classe_enquadramento_ipi' => $this->request->post('classe_enquadramento_ipi'),
                'enquadramento_ipi' => $this->request->post('enquadramento_ipi'),
                'cst_pis' => $this->request->post('cst_pis'),
                'pis' => $this->request->post('pis') ?: 0,
                'cst_cofins' => $this->request->post('cst_cofins'),
                'cofins' => $this->request->post('cofins') ?: 0,
                'natureza_receita_pis_cofins' => $this->request->post('natureza_receita_pis_cofins'),

                // IBS/CBS
                'cclass' => $this->request->post('cclass'),
                'aliquota_ibs' => $this->request->post('aliquota_ibs') ? str_replace(',', '.', $this->request->post('aliquota_ibs')) : 0,
                'reducao_aliquota_ibs' => $this->request->post('reducao_aliquota_ibs') ? str_replace(',', '.', $this->request->post('reducao_aliquota_ibs')) : 0,
                'aliquota_cbs' => $this->request->post('aliquota_cbs') ? str_replace(',', '.', $this->request->post('aliquota_cbs')) : 0,
                'reducao_aliquota_cbs' => $this->request->post('reducao_aliquota_cbs') ? str_replace(',', '.', $this->request->post('reducao_aliquota_cbs')) : 0,
                'aliquota_ibs_municipal' => $this->request->post('aliquota_ibs_municipal') ? str_replace(',', '.', $this->request->post('aliquota_ibs_municipal')) : 0,
                'reducao_aliquota_ibs_municipal' => $this->request->post('reducao_aliquota_ibs_municipal') ? str_replace(',', '.', $this->request->post('reducao_aliquota_ibs_municipal')) : 0,
            ];

            if (!$data['hash_imposto'] || !$data['uf_origem']) {
                $this->error('Hash do imposto e UF de origem são obrigatórios');
                return;
            }

            $stmt = $this->db->prepare("
                INSERT INTO impostos_configuracoes (
                    company_id, hash_imposto, uf_origem, cst_csosn, icms, modalidade_bc_icms, origem_mercadoria,
                    motivo_desicms, reducao_bc, fcp, cfop, valor_pauta, uf_destino,
                    cst_csosn_st, icms_st, mva, modalidade_bc_icms_st, reducao_bc_st,
                    fcp_st, cfop_st, cst_ipi, ipi, classe_enquadramento_ipi,
                    enquadramento_ipi, cst_pis, pis, cst_cofins, cofins,
                    natureza_receita_pis_cofins,
                    cclass, aliquota_ibs, reducao_aliquota_ibs, aliquota_cbs, reducao_aliquota_cbs, aliquota_ibs_municipal, reducao_aliquota_ibs_municipal
                ) VALUES (
                    :company_id, :hash_imposto, :uf_origem, :cst_csosn, :icms, :modalidade_bc_icms, :origem_mercadoria,
                    :motivo_desicms, :reducao_bc, :fcp, :cfop, :valor_pauta, :uf_destino,
                    :cst_csosn_st, :icms_st, :mva, :modalidade_bc_icms_st, :reducao_bc_st,
                    :fcp_st, :cfop_st, :cst_ipi, :ipi, :classe_enquadramento_ipi,
                    :enquadramento_ipi, :cst_pis, :pis, :cst_cofins, :cofins,
                    :natureza_receita_pis_cofins,
                    :cclass, :aliquota_ibs, :reducao_aliquota_ibs, :aliquota_cbs, :reducao_aliquota_cbs, :aliquota_ibs_municipal, :reducao_aliquota_ibs_municipal
                )
            ");

            $stmt->execute($data);

            $this->success('Configuração adicionada com sucesso');

        } catch (Exception $e) {
            error_log("Erro ao adicionar configuração: " . $e->getMessage());
            $this->error('Erro ao adicionar configuração: ' . $e->getMessage());
        }
    }

    /**
     * Lista configurações de um grupo (por hash)
     */
    public function listarConfiguracoes(): void
    {
        try {
            $hash = $this->request->get('hash');
            $companyId = $this->getCompanyId();

            if (!$hash) {
                $this->error('Hash não informado');
                return;
            }

            $stmt = $this->db->prepare("
                SELECT
                    c.*,
                    e_origem.nome_estado as nome_estado_origem,
                    e_destino.nome_estado as nome_estado_destino
                FROM impostos_configuracoes c
                LEFT JOIN estados e_origem ON c.uf_origem = e_origem.codigo_uf
                LEFT JOIN estados e_destino ON c.uf_destino = e_destino.codigo_uf
                WHERE c.hash_imposto = :hash AND c.company_id = :company_id
                ORDER BY c.uf_origem
            ");
            $stmt->execute(['hash' => $hash, 'company_id' => $companyId]);
            $configuracoes = $stmt->fetchAll();

            $this->success('Configurações carregadas', $configuracoes);

        } catch (Exception $e) {
            error_log("Erro ao listar configurações: " . $e->getMessage());
            $this->error('Erro ao listar configurações');
        }
    }

    /**
     * Busca uma configuração específica para edição
     */
    public function buscarConfiguracao(): void
    {
        try {
            $id = (int) $this->request->get('id');
            $companyId = $this->getCompanyId();

            $stmt = $this->db->prepare("
                SELECT * FROM impostos_configuracoes
                WHERE id = :id AND company_id = :company_id
            ");
            $stmt->execute(['id' => $id, 'company_id' => $companyId]);
            $config = $stmt->fetch();

            if ($config) {
                $this->success('Configuração encontrada', $config);
            } else {
                $this->error('Configuração não encontrada');
            }

        } catch (Exception $e) {
            error_log("Erro ao buscar configuração: " . $e->getMessage());
            $this->error('Erro ao buscar configuração');
        }
    }

    /**
     * Atualiza uma configuração específica
     */
    public function atualizarConfiguracao(): void
    {
        try {
            $id = (int) $this->request->post('id');
            $companyId = $this->getCompanyId();

            $data = [
                'uf_origem' => $this->request->post('uf_origem'),
                'cst_csosn' => $this->request->post('cst_csosn'),
                'icms' => $this->request->post('icms') ?: 0,
                'modalidade_bc_icms' => $this->request->post('modalidade_bc_icms'),
                'origem_mercadoria' => $this->request->post('origem_mercadoria'),
                'motivo_desicms' => $this->request->post('motivo_desicms'),
                'reducao_bc' => $this->request->post('reducao_bc') ?: 0,
                'fcp' => $this->request->post('fcp') ?: 0,
                'cfop' => $this->request->post('cfop'),
                'valor_pauta' => $this->request->post('valor_pauta') ?: 0,
                'uf_destino' => $this->request->post('uf_destino'),
                'cst_csosn_st' => $this->request->post('cst_csosn_st'),
                'icms_st' => $this->request->post('icms_st') ?: 0,
                'mva' => $this->request->post('mva') ?: 0,
                'modalidade_bc_icms_st' => $this->request->post('modalidade_bc_icms_st'),
                'reducao_bc_st' => $this->request->post('reducao_bc_st') ?: 0,
                'fcp_st' => $this->request->post('fcp_st') ?: 0,
                'cfop_st' => $this->request->post('cfop_st'),
                'cst_ipi' => $this->request->post('cst_ipi'),
                'ipi' => $this->request->post('ipi') ?: 0,
                'classe_enquadramento_ipi' => $this->request->post('classe_enquadramento_ipi'),
                'enquadramento_ipi' => $this->request->post('enquadramento_ipi'),
                'cst_pis' => $this->request->post('cst_pis'),
                'pis' => $this->request->post('pis') ?: 0,
                'cst_cofins' => $this->request->post('cst_cofins'),
                'cofins' => $this->request->post('cofins') ?: 0,
                'natureza_receita_pis_cofins' => $this->request->post('natureza_receita_pis_cofins'),

                // IBS/CBS
                'cclass' => $this->request->post('cclass'),
                'aliquota_ibs' => $this->request->post('aliquota_ibs') ? str_replace(',', '.', $this->request->post('aliquota_ibs')) : 0,
                'reducao_aliquota_ibs' => $this->request->post('reducao_aliquota_ibs') ? str_replace(',', '.', $this->request->post('reducao_aliquota_ibs')) : 0,
                'aliquota_cbs' => $this->request->post('aliquota_cbs') ? str_replace(',', '.', $this->request->post('aliquota_cbs')) : 0,
                'reducao_aliquota_cbs' => $this->request->post('reducao_aliquota_cbs') ? str_replace(',', '.', $this->request->post('reducao_aliquota_cbs')) : 0,
                'aliquota_ibs_municipal' => $this->request->post('aliquota_ibs_municipal') ? str_replace(',', '.', $this->request->post('aliquota_ibs_municipal')) : 0,
                'reducao_aliquota_ibs_municipal' => $this->request->post('reducao_aliquota_ibs_municipal') ? str_replace(',', '.', $this->request->post('reducao_aliquota_ibs_municipal')) : 0,

                'id' => $id,
                'company_id' => $companyId
            ];

            $stmt = $this->db->prepare("
                UPDATE impostos_configuracoes SET
                    uf_origem = :uf_origem,
                    cst_csosn = :cst_csosn,
                    icms = :icms,
                    modalidade_bc_icms = :modalidade_bc_icms,
                    origem_mercadoria = :origem_mercadoria,
                    motivo_desicms = :motivo_desicms,
                    reducao_bc = :reducao_bc,
                    fcp = :fcp,
                    cfop = :cfop,
                    valor_pauta = :valor_pauta,
                    uf_destino = :uf_destino,
                    cst_csosn_st = :cst_csosn_st,
                    icms_st = :icms_st,
                    mva = :mva,
                    modalidade_bc_icms_st = :modalidade_bc_icms_st,
                    reducao_bc_st = :reducao_bc_st,
                    fcp_st = :fcp_st,
                    cfop_st = :cfop_st,
                    cst_ipi = :cst_ipi,
                    ipi = :ipi,
                    classe_enquadramento_ipi = :classe_enquadramento_ipi,
                    enquadramento_ipi = :enquadramento_ipi,
                    cst_pis = :cst_pis,
                    pis = :pis,
                    cst_cofins = :cst_cofins,
                    cofins = :cofins,
                    natureza_receita_pis_cofins = :natureza_receita_pis_cofins,
                    cclass = :cclass,
                    aliquota_ibs = :aliquota_ibs,
                    reducao_aliquota_ibs = :reducao_aliquota_ibs,
                    aliquota_cbs = :aliquota_cbs,
                    reducao_aliquota_cbs = :reducao_aliquota_cbs,
                    aliquota_ibs_municipal = :aliquota_ibs_municipal,
                    reducao_aliquota_ibs_municipal = :reducao_aliquota_ibs_municipal
                WHERE id = :id AND company_id = :company_id
            ");

            $stmt->execute($data);

            $this->success('Configuração atualizada com sucesso');

        } catch (Exception $e) {
            error_log("Erro ao atualizar configuração: " . $e->getMessage());
            $this->error('Erro ao atualizar configuração');
        }
    }

    /**
     * Exclui uma configuração específica
     */
    public function excluirConfiguracao(): void
    {
        try {
            $id = (int) $this->request->post('id');
            $companyId = $this->getCompanyId();

            $stmt = $this->db->prepare("DELETE FROM impostos_configuracoes WHERE id = :id AND company_id = :company_id");
            $stmt->execute(['id' => $id, 'company_id' => $companyId]);

            $this->success('Configuração excluída com sucesso');

        } catch (Exception $e) {
            error_log("Erro ao excluir configuração: " . $e->getMessage());
            $this->error('Erro ao excluir configuração');
        }
    }
}
