<?php

declare(strict_types=1);

namespace App\Controllers;

use Exception;
use App\Helpers\UrlHelper;

/**
 * Controller de Grupos de Pessoas
 */
class GruposPessoasController extends BaseController
{
    /**
     * Lista grupos de pessoas
     */
    public function index(): void
    {
        // Verificar permissão de visualização
        if (!$this->canView('grupos-pessoas')) {
            $this->response->forbidden('Você não tem permissão para visualizar grupos de pessoas.');
            return;
        }

        try {
            $companyId = $this->getCompanyId();

            // Paginação
            $perPage = 50;
            $page = max(1, (int) ($this->request->get('page') ?? 1));
            $offset = ($page - 1) * $perPage;

            // Contar total de grupos
            $stmtCount = $this->db->prepare("
                SELECT COUNT(*) as total
                FROM grupos_pessoas g
                WHERE g.company_id = :company_id
            ");
            $stmtCount->execute(['company_id' => $companyId]);
            $total = (int) $stmtCount->fetch()['total'];
            $totalPages = max(1, (int) ceil($total / $perPage));

            // Buscar grupos com contagem de subgrupos (paginado)
            $stmt = $this->db->prepare("
                SELECT g.*,
                    (SELECT COUNT(*) FROM subgrupos_pessoas sp WHERE sp.grupo_id = g.id AND sp.ativo = 1) as total_subgrupos
                FROM grupos_pessoas g
                WHERE g.company_id = :company_id
                ORDER BY g.ordem ASC, g.nome ASC
                LIMIT :limit OFFSET :offset
            ");
            $stmt->bindValue(':company_id', $companyId, \PDO::PARAM_INT);
            $stmt->bindValue(':limit', $perPage, \PDO::PARAM_INT);
            $stmt->bindValue(':offset', $offset, \PDO::PARAM_INT);
            $stmt->execute();
            $grupos = $stmt->fetchAll();

            // Buscar subgrupos para cada grupo
            $gruposComSubgrupos = [];
            foreach ($grupos as $grupo) {
                $stmt = $this->db->prepare("
                    SELECT * FROM subgrupos_pessoas
                    WHERE grupo_id = :grupo_id AND company_id = :company_id
                    ORDER BY ordem ASC, nome ASC
                ");
                $stmt->execute(['grupo_id' => $grupo['id'], 'company_id' => $companyId]);
                $grupo['subgrupos'] = $stmt->fetchAll();
                $gruposComSubgrupos[] = $grupo;
            }

            $this->view('grupos_pessoas/index', [
                'grupos' => $gruposComSubgrupos,
                'pageTitle' => 'Grupos',
                'activeMenu' => 'grupos-pessoas',
                'page' => $page,
                'perPage' => $perPage,
                'total' => $total,
                'totalPages' => $totalPages
            ]);

        } catch (Exception $e) {
            error_log("Erro ao listar grupos: " . $e->getMessage());
            $this->error('Erro ao carregar grupos');
        }
    }

    /**
     * Exibe formulário de criação
     */
    public function create(): void
    {
        $this->view('grupos_pessoas/create', [
            'pageTitle' => 'Novo Grupo de Pessoas',
            'activeMenu' => 'grupos-pessoas'
        ]);
    }

    /**
     * Salva novo grupo
     */
    public function store(): void
    {
        // Verificar permissão de criação
        if (!$this->canCreate('grupos-pessoas')) {
            $this->response->forbidden('Você não tem permissão para criar grupos de pessoas.');
            return;
        }

        try {
            $companyId = $this->getCompanyId();

            $data = [
                'company_id' => $companyId,
                'tipo' => $this->request->post('tipo') ?: 'pessoas',
                'subtipo' => $this->request->post('subtipo') ?: null,
                'nome' => $this->request->post('nome'),
                'descricao' => $this->request->post('descricao'),
                'cor' => $this->request->post('cor') ?: '#007bff',
                'icone' => $this->request->post('icone') ?: 'bi-people',
                'ativo' => $this->request->post('ativo') ? 1 : 0,
                'ordem' => (int) $this->request->post('ordem') ?: 0
            ];

            $stmt = $this->db->prepare("
                INSERT INTO grupos_pessoas (company_id, tipo, subtipo, nome, descricao, cor, icone, ativo, ordem, created_at)
                VALUES (:company_id, :tipo, :subtipo, :nome, :descricao, :cor, :icone, :ativo, :ordem, NOW())
            ");

            $stmt->execute($data);

            $this->success('Grupo criado com sucesso', [
                'redirect' => UrlHelper::url('/grupos-pessoas')
            ]);

        } catch (Exception $e) {
            error_log("Erro ao criar grupo: " . $e->getMessage());
            $this->error('Erro ao criar grupo');
        }
    }

    /**
     * Exibe formulário de edição
     */
    public function edit(): void
    {
        // Verificar permissão de edição
        if (!$this->canEdit('grupos-pessoas')) {
            $this->response->forbidden('Você não tem permissão para editar grupos de pessoas.');
            return;
        }

        try {
            $id = (int) $this->request->get('id');
            $companyId = $this->getCompanyId();

            $stmt = $this->db->prepare("
                SELECT * FROM grupos_pessoas
                WHERE id = :id AND company_id = :company_id
            ");
            $stmt->execute(['id' => $id, 'company_id' => $companyId]);
            $grupo = $stmt->fetch();

            if (!$grupo) {
                $this->response->notFound('Grupo não encontrado');
                return;
            }

            // Buscar subgrupos
            $stmt = $this->db->prepare("
                SELECT * FROM subgrupos_pessoas
                WHERE grupo_id = :grupo_id AND company_id = :company_id
                ORDER BY ordem ASC, nome ASC
            ");
            $stmt->execute(['grupo_id' => $id, 'company_id' => $companyId]);
            $subgrupos = $stmt->fetchAll();

            $this->view('grupos_pessoas/edit', [
                'grupo' => $grupo,
                'subgrupos' => $subgrupos,
                'pageTitle' => 'Editar Grupo',
                'activeMenu' => 'grupos-pessoas'
            ]);

        } catch (Exception $e) {
            error_log("Erro ao editar grupo: " . $e->getMessage());
            $this->error('Erro ao carregar grupo');
        }
    }

    /**
     * Atualiza grupo
     */
    public function update(): void
    {
        // Verificar permissão de edição
        if (!$this->canEdit('grupos-pessoas')) {
            $this->response->forbidden('Você não tem permissão para editar grupos de pessoas.');
            return;
        }

        try {
            $id = (int) $this->request->post('id');
            $companyId = $this->getCompanyId();

            $data = [
                'tipo' => $this->request->post('tipo') ?: 'pessoas',
                'subtipo' => $this->request->post('subtipo') ?: null,
                'nome' => $this->request->post('nome'),
                'descricao' => $this->request->post('descricao'),
                'cor' => $this->request->post('cor') ?: '#007bff',
                'icone' => $this->request->post('icone') ?: 'bi-people',
                'ativo' => $this->request->post('ativo') ? 1 : 0,
                'ordem' => (int) $this->request->post('ordem') ?: 0,
                'id' => $id,
                'company_id' => $companyId
            ];

            $stmt = $this->db->prepare("
                UPDATE grupos_pessoas SET
                    tipo = :tipo,
                    subtipo = :subtipo,
                    nome = :nome,
                    descricao = :descricao,
                    cor = :cor,
                    icone = :icone,
                    ativo = :ativo,
                    ordem = :ordem,
                    updated_at = NOW()
                WHERE id = :id AND company_id = :company_id
            ");

            $stmt->execute($data);

            $this->success('Grupo atualizado com sucesso', [
                'redirect' => UrlHelper::url('/grupos-pessoas')
            ]);

        } catch (Exception $e) {
            error_log("Erro ao atualizar grupo: " . $e->getMessage());
            $this->error('Erro ao atualizar grupo');
        }
    }

    /**
     * Exclui grupo
     */
    public function delete(): void
    {
        // Verificar permissão de exclusão
        if (!$this->canDelete('grupos-pessoas')) {
            $this->response->forbidden('Você não tem permissão para excluir grupos de pessoas.');
            return;
        }

        try {
            $id = (int) $this->request->post('id');
            $companyId = $this->getCompanyId();

            // Verifica se há pessoas usando este grupo
            $stmt = $this->db->prepare("
                SELECT COUNT(*) as total FROM pessoas
                WHERE grupo_id = :id AND company_id = :company_id
            ");
            $stmt->execute(['id' => $id, 'company_id' => $companyId]);
            $result = $stmt->fetch();

            if ($result['total'] > 0) {
                $this->error('Não é possível excluir. Existem ' . $result['total'] . ' pessoa(s) usando este grupo.');
                return;
            }

            // Exclui subgrupos primeiro
            $stmt = $this->db->prepare("
                DELETE FROM subgrupos_pessoas
                WHERE grupo_id = :id AND company_id = :company_id
            ");
            $stmt->execute(['id' => $id, 'company_id' => $companyId]);

            // Exclui grupo
            $stmt = $this->db->prepare("
                DELETE FROM grupos_pessoas
                WHERE id = :id AND company_id = :company_id
            ");
            $stmt->execute(['id' => $id, 'company_id' => $companyId]);

            $this->success('Grupo excluído com sucesso');

        } catch (Exception $e) {
            error_log("Erro ao excluir grupo: " . $e->getMessage());
            $this->error('Erro ao excluir grupo');
        }
    }

    /**
     * Adiciona um subgrupo
     */
    public function adicionarSubgrupo(): void
    {
        try {
            $companyId = $this->getCompanyId();
            $grupoId = (int) $this->request->post('grupo_id');

            $data = [
                'company_id' => $companyId,
                'grupo_id' => $grupoId,
                'nome' => $this->request->post('nome'),
                'descricao' => $this->request->post('descricao') ?: null,
                'cor' => $this->request->post('cor') ?: '#6c757d',
                'icone' => $this->request->post('icone') ?: 'bi-tag',
                'ativo' => $this->request->post('ativo') ? 1 : 0,
                'ordem' => (int) $this->request->post('ordem') ?: 0
            ];

            $stmt = $this->db->prepare("
                INSERT INTO subgrupos_pessoas (company_id, grupo_id, nome, descricao, cor, icone, ativo, ordem, created_at)
                VALUES (:company_id, :grupo_id, :nome, :descricao, :cor, :icone, :ativo, :ordem, NOW())
            ");

            $stmt->execute($data);

            $this->success('Subgrupo adicionado com sucesso');

        } catch (Exception $e) {
            error_log("Erro ao adicionar subgrupo: " . $e->getMessage());
            $this->error('Erro ao adicionar subgrupo');
        }
    }

    /**
     * Exclui um subgrupo
     */
    public function deletarSubgrupo(): void
    {
        try {
            $id = (int) $this->request->post('id');
            $companyId = $this->getCompanyId();

            // Verifica se há pessoas usando este subgrupo
            $stmt = $this->db->prepare("
                SELECT COUNT(*) as total FROM pessoas
                WHERE subgrupo_id = :id AND company_id = :company_id
            ");
            $stmt->execute(['id' => $id, 'company_id' => $companyId]);
            $result = $stmt->fetch();

            if ($result['total'] > 0) {
                $this->error('Não é possível excluir. Existem ' . $result['total'] . ' pessoa(s) usando este subgrupo.');
                return;
            }

            // Exclui subgrupo
            $stmt = $this->db->prepare("
                DELETE FROM subgrupos_pessoas
                WHERE id = :id AND company_id = :company_id
            ");
            $stmt->execute(['id' => $id, 'company_id' => $companyId]);

            $this->success('Subgrupo excluído com sucesso');

        } catch (Exception $e) {
            error_log("Erro ao excluir subgrupo: " . $e->getMessage());
            $this->error('Erro ao excluir subgrupo');
        }
    }
}