<?php

namespace App\Controllers;

use Exception;

class GruposController extends BaseController
{
    /**
     * Lista todos os grupos
     */
    public function index(): void
    {
        // Verificar permissão de visualização
        if (!$this->canView('grupos')) {
            $this->response->forbidden('Você não tem permissão para visualizar grupos.');
            return;
        }

        try {
            $companyId = $this->getCompanyId();

            $stmt = $this->db->prepare("
                SELECT * FROM grupos
                WHERE company_id = :company_id
                ORDER BY name ASC
            ");
            $stmt->execute(['company_id' => $companyId]);
            $grupos = $stmt->fetchAll();

            $this->view('grupos/index', [
                'grupos' => $grupos,
                'pageTitle' => 'Grupos de Pessoas'
            ]);

        } catch (Exception $e) {
            error_log("Erro ao carregar grupos: " . $e->getMessage());
            $this->error('Erro ao carregar grupos');
        }
    }

    /**
     * Exibe formulário de criação
     */
    public function create(): void
    {
        // Verificar permissão de criação
        if (!$this->canCreate('grupos')) {
            $this->response->forbidden('Você não tem permissão para criar grupos.');
            return;
        }

        $this->view('grupos/form', [
            'pageTitle' => 'Novo Grupo'
        ]);
    }

    /**
     * Salva novo grupo
     */
    public function store(): void
    {
        // Verificar permissão de criação
        if (!$this->canCreate('grupos')) {
            $this->response->forbidden('Você não tem permissão para criar grupos.');
            return;
        }

        try {
            $companyId = $this->getCompanyId();

            $data = [
                'company_id' => $companyId,
                'type' => $this->request->post('type'),
                'name' => $this->request->post('name'),
                'description' => $this->request->post('description'),
                'is_active' => $this->request->post('is_active', '0') === '1' ? 1 : 0,
            ];

            $errors = $this->validate([
                'type' => 'required',
                'name' => 'required|min:2',
            ]);

            if (!empty($errors)) {
                $this->error('Dados inválidos', $errors);
                return;
            }

            $this->db->beginTransaction();

            $stmt = $this->db->prepare("
                INSERT INTO grupos (
                    company_id, type, name, description, is_active, created_at, updated_at
                ) VALUES (
                    :company_id, :type, :name, :description, :is_active, NOW(), NOW()
                )
            ");

            $stmt->execute($data);
            $grupoId = $this->db->lastInsertId();

            $this->logActivity('create', 'grupos', $grupoId, $data);

            $this->db->commit();

            $this->success('Grupo criado com sucesso', [
                'id' => $grupoId,
                'redirect' => '/grupos'
            ]);

        } catch (Exception $e) {
            $this->db->rollBack();
            error_log("Erro ao criar grupo: " . $e->getMessage());
            $this->error('Erro ao cadastrar grupo');
        }
    }

    /**
     * Exibe formulário de edição
     */
    public function edit(): void
    {
        // Verificar permissão de edição
        if (!$this->canEdit('grupos')) {
            $this->response->forbidden('Você não tem permissão para editar grupos.');
            return;
        }

        try {
            $id = (int) $this->request->get('id');
            $grupo = $this->getGrupo($id);

            if (!$grupo) {
                $this->response->notFound('Grupo não encontrado');
                return;
            }

            $this->view('grupos/edit', [
                'grupo' => $grupo,
                'pageTitle' => 'Editar ' . $grupo['name']
            ]);

        } catch (Exception $e) {
            error_log("Erro ao exibir formulário de edição: " . $e->getMessage());
            $this->error('Erro ao carregar formulário');
        }
    }

    /**
     * Atualiza um grupo
     */
    public function update(): void
    {
        // Verificar permissão de edição
        if (!$this->canEdit('grupos')) {
            $this->response->forbidden('Você não tem permissão para editar grupos.');
            return;
        }

        try {
            $id = (int) $this->request->post('id');
            $grupo = $this->getGrupo($id);

            if (!$grupo) {
                $this->error('Grupo não encontrado');
                return;
            }

            $data = [
                'type' => $this->request->post('type'),
                'name' => $this->request->post('name'),
                'description' => $this->request->post('description'),
                'is_active' => $this->request->post('is_active', '0') === '1' ? 1 : 0,
                'id' => $id
            ];

            $errors = $this->validate([
                'type' => 'required',
                'name' => 'required|min:2',
            ]);

            if (!empty($errors)) {
                $this->error('Dados inválidos', $errors);
                return;
            }

            $this->db->beginTransaction();

            $stmt = $this->db->prepare("
                UPDATE grupos SET
                    type = :type,
                    name = :name,
                    description = :description,
                    is_active = :is_active,
                    updated_at = NOW()
                WHERE id = :id AND company_id = :company_id
            ");

            $data['company_id'] = $this->getCompanyId();
            $stmt->execute($data);

            $this->logActivity('update', 'grupos', $id, $data);

            $this->db->commit();

            $this->success('Grupo atualizado com sucesso', [
                'redirect' => '/grupos'
            ]);

        } catch (Exception $e) {
            $this->db->rollBack();
            error_log("Erro ao atualizar grupo: " . $e->getMessage());
            $this->error('Erro ao atualizar grupo');
        }
    }

    /**
     * Exclui um grupo
     */
    public function delete(): void
    {
        // Verificar permissão de exclusão
        if (!$this->canDelete('grupos')) {
            $this->response->forbidden('Você não tem permissão para excluir grupos.');
            return;
        }

        try {
            $id = (int) $this->request->post('id');
            $grupo = $this->getGrupo($id);

            if (!$grupo) {
                $this->error('Grupo não encontrado');
                return;
            }

            $this->db->beginTransaction();

            $stmt = $this->db->prepare("DELETE FROM grupos WHERE id = :id AND company_id = :company_id");
            $stmt->execute(['id' => $id, 'company_id' => $this->getCompanyId()]);

            $this->logActivity('delete', 'grupos', $id, $grupo);

            $this->db->commit();

            $this->success('Grupo excluído com sucesso');

        } catch (Exception $e) {
            $this->db->rollBack();
            error_log("Erro ao excluir grupo: " . $e->getMessage());
            $this->error('Erro ao excluir grupo');
        }
    }

    /**
     * Busca um grupo por ID
     */
    private function getGrupo(int $id): ?array
    {
        $stmt = $this->db->prepare("
            SELECT * FROM grupos
            WHERE id = :id AND company_id = :company_id
        ");
        $stmt->execute([
            'id' => $id,
            'company_id' => $this->getCompanyId()
        ]);

        return $stmt->fetch() ?: null;
    }
}
