<?php

declare(strict_types=1);

namespace App\Controllers;

use Exception;

class AssinaturaController extends BaseController
{
    public function index(): void
    {
        try {
            $companyId = $this->getCompanyId();

            // Busca informações da empresa/assinatura
            $stmt = $this->db->prepare("SELECT * FROM companies WHERE id = :company_id");
            $stmt->execute(['company_id' => $companyId]);
            $empresa = $stmt->fetch();

            // Simula dados de assinatura (em um sistema real, isso viria de uma tabela de assinaturas)
            $assinatura = [
                'plan' => 'Básico',
                'status' => 'ativa',
                'expires_at' => date('Y-m-d', strtotime('+30 days')),
                'users_limit' => 5,
                'storage_limit' => '10GB',
                'features' => [
                    'Gestão de Pessoas',
                    'Controle de Estoque',
                    'Relatórios Básicos',
                    'Suporte por Email'
                ]
            ];

            $this->view('assinatura/index', [
                'empresa' => $empresa,
                'assinatura' => $assinatura,
                'pageTitle' => 'Assinatura',
                'activeMenu' => 'assinatura'
            ]);

        } catch (Exception $e) {
            error_log("Erro ao carregar assinatura: " . $e->getMessage());
            $this->error('Erro ao carregar informações da assinatura');
        }
    }

    public function upgrade(): void
    {
        try {
            $companyId = $this->getCompanyId();

            // Busca informações da empresa
            $stmt = $this->db->prepare("SELECT * FROM companies WHERE id = :company_id");
            $stmt->execute(['company_id' => $companyId]);
            $empresa = $stmt->fetch();

            // Planos disponíveis
            $planos = [
                'basico' => [
                    'name' => 'Básico',
                    'price' => 29.90,
                    'users_limit' => 5,
                    'storage_limit' => '10GB',
                    'features' => [
                        'Gestão de Pessoas',
                        'Controle de Estoque',
                        'Relatórios Básicos',
                        'Suporte por Email'
                    ]
                ],
                'profissional' => [
                    'name' => 'Profissional',
                    'price' => 59.90,
                    'users_limit' => 15,
                    'storage_limit' => '50GB',
                    'features' => [
                        'Todas as funcionalidades do Básico',
                        'Relatórios Avançados',
                        'Integração com APIs',
                        'Suporte Prioritário',
                        'Backup Automático'
                    ]
                ],
                'empresarial' => [
                    'name' => 'Empresarial',
                    'price' => 99.90,
                    'users_limit' => 50,
                    'storage_limit' => '200GB',
                    'features' => [
                        'Todas as funcionalidades do Profissional',
                        'Multi-empresas',
                        'API Completa',
                        'Suporte 24/7',
                        'Treinamento Personalizado',
                        'SLA Garantido'
                    ]
                ]
            ];

            $this->view('assinatura/upgrade', [
                'empresa' => $empresa,
                'planos' => $planos,
                'pageTitle' => 'Upgrade de Plano',
                'activeMenu' => 'assinatura'
            ]);

        } catch (Exception $e) {
            error_log("Erro ao carregar upgrade: " . $e->getMessage());
            $this->error('Erro ao carregar planos de upgrade');
        }
    }

    public function processUpgrade(): void
    {
        try {
            $companyId = $this->getCompanyId();
            $plano = $this->request->post('plano');
            $paymentMethod = $this->request->post('payment_method');

            // Validação básica
            $planosValidos = ['basico', 'profissional', 'empresarial'];
            if (!in_array($plano, $planosValidos)) {
                $this->error('Plano inválido');
                return;
            }

            // Em um sistema real, aqui você integraria com um gateway de pagamento
            // Por enquanto, apenas simula o upgrade
            $this->db->beginTransaction();

            // Aqui você salvaria as informações do upgrade em uma tabela de assinaturas
            // Por simplicidade, vamos apenas logar a ação
            $this->logActivity('upgrade', 'assinatura', $companyId, [
                'plano' => $plano,
                'payment_method' => $paymentMethod
            ]);

            $this->db->commit();

            $this->success('Upgrade processado com sucesso! Em breve você receberá um email de confirmação.', [
                'redirect' => '/assinatura'
            ]);

        } catch (Exception $e) {
            $this->db->rollBack();
            error_log("Erro ao processar upgrade: " . $e->getMessage());
            $this->error('Erro ao processar upgrade');
        }
    }

    public function invoices(): void
    {
        try {
            $companyId = $this->getCompanyId();

            // Simula faturas (em um sistema real, isso viria de uma tabela de faturas)
            $faturas = [
                [
                    'id' => 1,
                    'number' => 'FAT-2024-001',
                    'date' => '2024-01-01',
                    'amount' => 29.90,
                    'status' => 'pago',
                    'plan' => 'Básico'
                ],
                [
                    'id' => 2,
                    'number' => 'FAT-2024-002',
                    'date' => '2024-02-01',
                    'amount' => 29.90,
                    'status' => 'pago',
                    'plan' => 'Básico'
                ],
                [
                    'id' => 3,
                    'number' => 'FAT-2024-003',
                    'date' => '2024-03-01',
                    'amount' => 29.90,
                    'status' => 'pendente',
                    'plan' => 'Básico'
                ]
            ];

            $this->view('assinatura/invoices', [
                'faturas' => $faturas,
                'pageTitle' => 'Faturas',
                'activeMenu' => 'assinatura'
            ]);

        } catch (Exception $e) {
            error_log("Erro ao carregar faturas: " . $e->getMessage());
            $this->error('Erro ao carregar faturas');
        }
    }
}