<?php

namespace App\Controllers;

use App\Helpers\CnpjHelper;

class ApiController extends BaseController
{
    /**
     * Busca dados de CNPJ na Receita Federal
     * Rota: GET /api-cnpj.php?cnpj=00000000000000
     */
    public function buscarCnpj(): void
    {
        try {
            $cnpj = $this->request->get('cnpj') ?? '';

            // Remove formatação do CNPJ
            $cnpj = preg_replace('/[^0-9]/', '', $cnpj);

            error_log("[ApiController::buscarCnpj] CNPJ recebido: " . $cnpj);

            // Valida CNPJ
            if (empty($cnpj) || strlen($cnpj) !== 14) {
                error_log("[ApiController::buscarCnpj] CNPJ inválido (vazio ou tamanho incorreto)");
                $this->response->json([
                    'success' => false,
                    'message' => 'CNPJ inválido. Deve ter 14 dígitos.'
                ], 400);
                return;
            }

            if (!CnpjHelper::validar($cnpj)) {
                error_log("[ApiController::buscarCnpj] CNPJ falhou na validação");
                $this->response->json([
                    'success' => false,
                    'message' => 'CNPJ inválido'
                ], 400);
                return;
            }

            error_log("[ApiController::buscarCnpj] Buscando dados do CNPJ na Receita Federal...");

            // Busca dados do CNPJ
            $resultado = CnpjHelper::buscar($cnpj);

            if ($resultado) {
                error_log("[ApiController::buscarCnpj] CNPJ encontrado com sucesso");
                $this->response->json([
                    'success' => true,
                    'data' => $resultado
                ]);
            } else {
                error_log("[ApiController::buscarCnpj] CNPJ não encontrado ou serviço indisponível");
                $this->response->json([
                    'success' => false,
                    'message' => 'CNPJ não encontrado ou serviço indisponível'
                ], 404);
            }
        } catch (\Exception $e) {
            error_log("[ApiController::buscarCnpj] Erro: " . $e->getMessage());
            error_log("[ApiController::buscarCnpj] Stack trace: " . $e->getTraceAsString());
            $this->response->json([
                'success' => false,
                'message' => 'Erro ao consultar CNPJ: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Busca endereço por CEP via ViaCEP
     * Rota: GET /api-cep.php?cep=00000000
     */
    public function buscarCep(): void
    {
        $cep = preg_replace('/\D/', '', $this->request->get('cep') ?? '');

        if (strlen($cep) !== 8) {
            $this->response->json([
                'success' => false,
                'message' => 'CEP deve ter 8 dígitos'
            ]);
            return;
        }

        // Buscar via ViaCEP
        $url = "https://viacep.com.br/ws/{$cep}/json/";

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 10);
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        if ($httpCode === 200 && $response) {
            $data = json_decode($response, true);

            if (isset($data['erro'])) {
                $this->response->json([
                    'success' => false,
                    'message' => 'CEP não encontrado'
                ]);
                return;
            }

            $this->response->json([
                'success' => true,
                'data' => $data
            ]);
        } else {
            $this->response->json([
                'success' => false,
                'message' => 'Erro ao consultar CEP'
            ]);
        }
    }
}

