<?php

declare(strict_types=1);

namespace App\Controllers;

use Exception;

class AgendaController extends BaseController
{
    public function index(): void
    {
        // Verificar permissão de visualização
        if (!$this->canView('agenda')) {
            $this->response->forbidden('Você não tem permissão para visualizar agenda.');
            return;
        }

        try {
            $companyId = $this->getCompanyId();
            $search = $this->request->get('search', '');
            $date = $this->request->get('date', '');

            $query = "
                SELECT ae.*, p.name as contact_name
                FROM agenda_eventos ae
                LEFT JOIN pessoas p ON ae.contact_id = p.id
                WHERE ae.company_id = :company_id
            ";

            $params = ['company_id' => $companyId];

            if (!empty($search)) {
                $query .= " AND (ae.title LIKE :search OR ae.description LIKE :search OR p.name LIKE :search)";
                $params['search'] = "%{$search}%";
            }

            if (!empty($date)) {
                $query .= " AND DATE(ae.start_date) = :date";
                $params['date'] = $date;
            }

            $query .= " ORDER BY ae.start_date ASC";

            $stmt = $this->db->prepare($query);
            $stmt->execute($params);
            $eventos = $stmt->fetchAll();

            $this->view('agenda/index', [
                'eventos' => $eventos,
                'search' => $search,
                'date' => $date,
                'pageTitle' => 'Agenda',
                'activeMenu' => 'agenda'
            ]);

        } catch (Exception $e) {
            error_log("Erro ao carregar agenda: " . $e->getMessage());
            $this->error('Erro ao carregar agenda');
        }
    }

    public function create(): void
    {
        try {
            $companyId = $this->getCompanyId();

            // Busca contatos
            $stmt = $this->db->prepare("
                SELECT id, name FROM pessoas
                WHERE company_id = :company_id AND is_active = 1
                ORDER BY name ASC
            ");
            $stmt->execute(['company_id' => $companyId]);
            $contatos = $stmt->fetchAll();

            $this->view('agenda/create', [
                'contatos' => $contatos,
                'pageTitle' => 'Novo Evento',
                'activeMenu' => 'agenda'
            ]);

        } catch (Exception $e) {
            error_log("Erro ao carregar formulário de agenda: " . $e->getMessage());
            $this->error('Erro ao carregar formulário');
        }
    }

    public function store(): void
    {
        // Verificar permissão de criação
        if (!$this->canCreate('agenda')) {
            $this->response->forbidden('Você não tem permissão para criar eventos na agenda.');
            return;
        }

        try {
            $companyId = $this->getCompanyId();

            $data = [
                'company_id' => $companyId,
                'title' => $this->request->post('title'),
                'description' => $this->request->post('description'),
                'start_date' => $this->request->post('start_date'),
                'end_date' => $this->request->post('end_date') ?: null,
                'all_day' => $this->request->post('all_day') ? 1 : 0,
                'location' => $this->request->post('location'),
                'contact_id' => $this->request->post('contact_id') ?: null,
            ];

            $this->db->beginTransaction();

            $stmt = $this->db->prepare("
                INSERT INTO agenda_eventos (company_id, title, description, start_date, end_date, all_day, location, contact_id, created_at, updated_at)
                VALUES (:company_id, :title, :description, :start_date, :end_date, :all_day, :location, :contact_id, NOW(), NOW())
            ");
            $stmt->execute($data);
            $eventoId = $this->db->lastInsertId();

            $this->logActivity('create', 'agenda_eventos', $eventoId, $data);
            $this->db->commit();

            $this->success('Evento criado com sucesso', [
                'id' => $eventoId,
                'redirect' => '/agenda'
            ]);

        } catch (Exception $e) {
            $this->db->rollBack();
            error_log("Erro ao criar evento: " . $e->getMessage());
            $this->error('Erro ao criar evento');
        }
    }

    public function edit(): void
    {
        // Verificar permissão de edição
        if (!$this->canEdit('agenda')) {
            $this->response->forbidden('Você não tem permissão para editar eventos na agenda.');
            return;
        }

        try {
            $id = (int) $this->request->get('id');
            $evento = $this->getEvento($id);

            if (!$evento) {
                $this->response->notFound('Evento não encontrado');
                return;
            }

            $companyId = $this->getCompanyId();

            // Busca contatos
            $stmt = $this->db->prepare("
                SELECT id, name FROM pessoas
                WHERE company_id = :company_id AND is_active = 1
                ORDER BY name ASC
            ");
            $stmt->execute(['company_id' => $companyId]);
            $contatos = $stmt->fetchAll();

            $this->view('agenda/edit', [
                'evento' => $evento,
                'contatos' => $contatos,
                'pageTitle' => 'Editar Evento',
                'activeMenu' => 'agenda'
            ]);

        } catch (Exception $e) {
            error_log("Erro ao editar evento: " . $e->getMessage());
            $this->error('Erro ao carregar formulário');
        }
    }

    public function update(): void
    {
        // Verificar permissão de edição
        if (!$this->canEdit('agenda')) {
            $this->response->forbidden('Você não tem permissão para editar eventos na agenda.');
            return;
        }

        try {
            $id = (int) $this->request->post('id');
            $evento = $this->getEvento($id);

            if (!$evento) {
                $this->error('Evento não encontrado');
                return;
            }

            $data = [
                'title' => $this->request->post('title'),
                'description' => $this->request->post('description'),
                'start_date' => $this->request->post('start_date'),
                'end_date' => $this->request->post('end_date') ?: null,
                'all_day' => $this->request->post('all_day') ? 1 : 0,
                'location' => $this->request->post('location'),
                'contact_id' => $this->request->post('contact_id') ?: null,
                'id' => $id,
                'company_id' => $this->getCompanyId()
            ];

            $this->db->beginTransaction();

            $stmt = $this->db->prepare("
                UPDATE agenda_eventos SET
                    title = :title,
                    description = :description,
                    start_date = :start_date,
                    end_date = :end_date,
                    all_day = :all_day,
                    location = :location,
                    contact_id = :contact_id,
                    updated_at = NOW()
                WHERE id = :id AND company_id = :company_id
            ");
            $stmt->execute($data);

            $this->logActivity('update', 'agenda_eventos', $id, $data);
            $this->db->commit();

            $this->success('Evento atualizado com sucesso', [
                'redirect' => '/agenda'
            ]);

        } catch (Exception $e) {
            $this->db->rollBack();
            error_log("Erro ao atualizar evento: " . $e->getMessage());
            $this->error('Erro ao atualizar evento');
        }
    }

    public function delete(): void
    {
        // Verificar permissão de exclusão
        if (!$this->canDelete('agenda')) {
            $this->response->forbidden('Você não tem permissão para excluir eventos da agenda.');
            return;
        }

        try {
            $id = (int) $this->request->post('id');
            $evento = $this->getEvento($id);

            if (!$evento) {
                $this->error('Evento não encontrado');
                return;
            }

            $this->db->beginTransaction();

            $stmt = $this->db->prepare("DELETE FROM agenda_eventos WHERE id = :id AND company_id = :company_id");
            $stmt->execute(['id' => $id, 'company_id' => $this->getCompanyId()]);

            $this->logActivity('delete', 'agenda_eventos', $id, $evento);
            $this->db->commit();

            $this->success('Evento excluído com sucesso');

        } catch (Exception $e) {
            $this->db->rollBack();
            error_log("Erro ao excluir evento: " . $e->getMessage());
            $this->error('Erro ao excluir evento');
        }
    }

    private function getEvento(int $id): ?array
    {
        $stmt = $this->db->prepare("SELECT * FROM agenda_eventos WHERE id = :id AND company_id = :company_id");
        $stmt->execute(['id' => $id, 'company_id' => $this->getCompanyId()]);
        return $stmt->fetch() ?: null;
    }
}