<?php

/**
 * Rotas Web
 */

use App\Core\Router;

// Router é passado pela Application
if (!isset($router)) {
    return;
}

// Rotas públicas
$router->get('/', 'HomeController@index');
$router->get('/login', 'AuthController@showLogin');
$router->post('/login', 'AuthController@login');
// Registro desabilitado - modo simples
// $router->get('/register', 'AuthController@showRegister');
// $router->post('/register', 'AuthController@register');
$router->get('/logout', 'AuthController@logout');

// APIs públicas (helpers)
$router->get('/api-cnpj.php', 'ApiController@buscarCnpj');
$router->get('/api-cep.php', 'ApiController@buscarCep');

// Rotas do PDV (Ponto de Venda) - Login público
$router->get('/pdv/login', 'PDVController@login');
$router->post('/pdv/autenticar', 'PDVController@autenticar');
$router->get('/pdv/abertura-caixa', 'PDVController@aberturaCaixa');
$router->post('/pdv/abrir-caixa', 'PDVController@abrirCaixa');
$router->get('/pdv/caixa', 'PDVController@caixa');
$router->get('/pdv/listar-grupos', 'PDVController@listarGrupos');
$router->get('/pdv/buscar-produtos', 'PDVController@buscarProdutos');
$router->post('/pdv/gerar-pix-shipay', 'PDVController@gerarPixShipay');
$router->post('/pdv/consultar-status-pix', 'PDVController@consultarStatusPix');
$router->post('/pdv/cancelar-pix-shipay', 'PDVController@cancelarPixShipay');
$router->get('/pdv/listar-pix-cancelaveis', 'PDVController@listarPixCancelaveis');
$router->get('/pdv/listar-pix-estornaveis', 'PDVController@listarPixEstornaveis');
$router->get('/pdv/listar-cobrancas-shipay', 'PDVController@listarCobrancasShipay');
$router->post('/pdv/estornar-pix-shipay', 'PDVController@estornarPixShipay');
$router->post('/pdv/processar-venda', 'PDVController@processarVenda');
$router->get('/pdv/imprimir-cupom', 'PDVController@imprimirCupom');
$router->get('/pdv/imprimir-abertura-caixa', 'PDVController@imprimirAberturaCaixa');
$router->get('/pdv/imprimir-fechamento-caixa', 'PDVController@imprimirFechamentoCaixa');
$router->post('/pdv/registrar-sangria', 'PDVController@registrarSangria');
$router->post('/pdv/registrar-suprimento', 'PDVController@registrarSuprimento');
$router->post('/pdv/fechar-caixa', 'PDVController@fecharCaixa');
$router->post('/pdv/registrar-pagamento', 'PDVController@registrarPagamento');
$router->get('/pdv/listar-vendas-pendentes', 'PDVController@listarVendasPendentes');
$router->get('/pdv/carregar-venda', 'PDVController@carregarVenda');
$router->post('/pdv/emitir-nfce', 'PDVController@emitirNFCe');
$router->get('/pdv/logout', 'PDVController@logout');

// Setup inicial de tenant (quando não tem usuários)
$router->get('/setup', 'TenantSetupController@showSetup');
$router->post('/setup', 'TenantSetupController@createFirstUser');

// Painel de Revendas
$router->get('/revenda/login', 'RevendaController@showLogin');
$router->post('/revenda/login', 'RevendaController@login');
$router->get('/revenda/logout', 'RevendaController@logout');
$router->get('/revenda', 'RevendaController@index');
$router->get('/revenda/dashboard', 'RevendaController@index');
$router->get('/revenda/comissoes', 'RevendaController@comissoes');
$router->get('/revenda/clientes', 'RevendaController@clientes');
$router->get('/revenda/vendas', 'RevendaController@vendas');

// Rota de verificação de disponibilidade
$router->post('/check-subdomain', 'AuthController@checkSubdomain');

// Módulo: Suporte (Reservado ao Suporte)
$router->get('/suporte', 'SuporteController@index', ['AuthMiddleware']);
$router->get('/suporte/contrato', 'SuporteController@contrato', ['AuthMiddleware']);
$router->get('/suporte/limpar-tabelas', 'SuporteController@limparTabelas', ['AuthMiddleware']);
$router->post('/suporte/limpar-tabelas', 'SuporteController@executarLimpeza', ['AuthMiddleware']);
$router->get('/suporte/importar-banco', 'SuporteController@importarBanco', ['AuthMiddleware']);
$router->post('/suporte/importar-banco', 'SuporteController@processarImportacao', ['AuthMiddleware']);

// Módulo: Agenda Técnica (Reservado ao Suporte)
$router->get('/agenda-tecnica', 'AgendaTecnicaController@index', ['AuthMiddleware']);
$router->get('/agenda-tecnica/criar', 'AgendaTecnicaController@criar', ['AuthMiddleware']);
$router->post('/agenda-tecnica/criar', 'AgendaTecnicaController@store', ['AuthMiddleware']);
$router->get('/agenda-tecnica/editar', 'AgendaTecnicaController@editar', ['AuthMiddleware']);
$router->post('/agenda-tecnica/editar', 'AgendaTecnicaController@update', ['AuthMiddleware']);
$router->post('/agenda-tecnica/deletar', 'AgendaTecnicaController@deletar', ['AuthMiddleware']);
$router->get('/agenda-tecnica/buscar-clientes', 'AgendaTecnicaController@buscarClientes', ['AuthMiddleware']);

// Módulo: Gestão - Backup
$router->get('/gestao/backup', 'SuporteController@backup', ['AuthMiddleware']);
$router->post('/gestao/backup/criar', 'SuporteController@criarBackup', ['AuthMiddleware']);
$router->get('/gestao/backup/download', 'SuporteController@downloadBackup', ['AuthMiddleware']);

// Módulo: Gestão - Importação
$router->get('/importacao', 'ImportacaoController@index', ['AuthMiddleware']);
$router->get('/importacao/template', 'ImportacaoController@gerarTemplate', ['AuthMiddleware']);
$router->post('/importacao/processar', 'ImportacaoController@processarImportacao', ['AuthMiddleware']);

// Módulo: Pedidos
$router->get('/pedidos', 'PedidosController@index', ['AuthMiddleware']);
$router->get('/pedidos/criar', 'PedidosController@create', ['AuthMiddleware']);
$router->post('/pedidos/criar', 'PedidosController@store', ['AuthMiddleware']);
$router->get('/pedidos/editar', 'PedidosController@edit', ['AuthMiddleware']);
$router->post('/pedidos/editar', 'PedidosController@update', ['AuthMiddleware']);
$router->post('/pedidos/atualizar-status', 'PedidosController@atualizarStatus', ['AuthMiddleware']);
$router->get('/pedidos/resumo', 'PedidosController@resumo', ['AuthMiddleware']);
$router->post('/pedidos/processar-pagamento', 'PedidosController@processarPagamento', ['AuthMiddleware']);
$router->post('/pedidos/processar-envaze', 'PedidosController@processarEnvaze', ['AuthMiddleware']);
$router->post('/pedidos/atualizar-linha-temporal', 'PedidosController@atualizarLinhaTemporalPedido', ['AuthMiddleware']);
$router->get('/pedidos/buscar-veiculos-motoristas', 'PedidosController@buscarVeiculosMotoristas', ['AuthMiddleware']);

// Rotas protegidas (requerem autenticação)
$router->get('/dashboard', 'DashboardController@index', ['AuthMiddleware']);

// Módulo: Empresas
$router->get('/empresas', 'EmpresasController@index', ['AuthMiddleware']);
$router->get('/empresas/criar', 'EmpresasController@create', ['AuthMiddleware']);
$router->post('/empresas/criar', 'EmpresasController@store', ['AuthMiddleware']);
$router->get('/empresas/editar', 'EmpresasController@edit', ['AuthMiddleware']);
$router->post('/empresas/editar', 'EmpresasController@update', ['AuthMiddleware']);
$router->post('/empresas/deletar', 'EmpresasController@delete', ['AuthMiddleware']);
$router->get('/empresas/testar-certificado', 'EmpresasController@testarCertificado', ['AuthMiddleware']);
$router->get('/empresas/download-certificado', 'EmpresasController@downloadCertificado', ['AuthMiddleware']);
$router->get('/empresas/testar-shipay', 'EmpresasController@testarShipay', ['AuthMiddleware']);

// Rotas para Tabelas de Preço
$router->get('/empresas/tabela-preco/listar', 'EmpresasController@listarTabelasPreco', ['AuthMiddleware']);
$router->get('/empresas/tabela-preco/buscar', 'EmpresasController@buscarTabelaPreco', ['AuthMiddleware']);
$router->post('/empresas/tabela-preco/criar', 'EmpresasController@criarTabelaPreco', ['AuthMiddleware']);
$router->post('/empresas/tabela-preco/atualizar', 'EmpresasController@atualizarTabelaPreco', ['AuthMiddleware']);
$router->post('/empresas/tabela-preco/excluir', 'EmpresasController@excluirTabelaPreco', ['AuthMiddleware']);

// Rotas para Fórmulas de Custo
$router->get('/empresas/formulas-custo/buscar', 'EmpresasController@buscarFormulasCusto', ['AuthMiddleware']);
$router->post('/empresas/formulas-custo/salvar', 'EmpresasController@salvarFormulaCusto', ['AuthMiddleware']);
$router->post('/empresas/formulas-custo/remover', 'EmpresasController@removerFormulaCusto', ['AuthMiddleware']);

// Endpoint para retornar dados da empresa atual em JSON (usado pelo PDV para NFC-e)
$router->get('/empresas/atual/json', 'EmpresasController@atualJson', ['AuthMiddleware']);

// Módulo: Pessoas
$router->get('/pessoas', 'PessoasController@index', ['AuthMiddleware']);
$router->get('/pessoas/criar', 'PessoasController@create', ['AuthMiddleware']);
$router->post('/pessoas/criar', 'PessoasController@store', ['AuthMiddleware']);
$router->get('/pessoas/editar', 'PessoasController@edit', ['AuthMiddleware']);
$router->post('/pessoas/editar', 'PessoasController@update', ['AuthMiddleware']);
$router->post('/pessoas/deletar', 'PessoasController@delete', ['AuthMiddleware']);
$router->get('/pessoas/extrato', 'PessoasController@extrato', ['AuthMiddleware']);
$router->get('/pessoas/ficha', 'PessoasController@show', ['AuthMiddleware']);
$router->get('/pessoas/dados-rapidos', 'PessoasController@dadosRapidos', ['AuthMiddleware']);

// Módulo: Categorias
$router->get('/categorias', 'CategoriasController@index', ['AuthMiddleware']);
$router->get('/categorias/criar', 'CategoriasController@create', ['AuthMiddleware']);
$router->post('/categorias/criar', 'CategoriasController@store', ['AuthMiddleware']);
$router->get('/categorias/editar', 'CategoriasController@edit', ['AuthMiddleware']);
$router->post('/categorias/editar', 'CategoriasController@update', ['AuthMiddleware']);
$router->post('/categorias/deletar', 'CategoriasController@delete', ['AuthMiddleware']);

// Módulo: Tabelas de Preço
$router->get('/tabelas-preco', 'TabelasPrecoController@index', ['AuthMiddleware']);
$router->get('/tabelas-preco/criar', 'TabelasPrecoController@create', ['AuthMiddleware']);
$router->post('/tabelas-preco/criar', 'TabelasPrecoController@store', ['AuthMiddleware']);
$router->get('/tabelas-preco/editar', 'TabelasPrecoController@edit', ['AuthMiddleware']);
$router->post('/tabelas-preco/editar', 'TabelasPrecoController@update', ['AuthMiddleware']);
$router->post('/tabelas-preco/excluir', 'TabelasPrecoController@delete', ['AuthMiddleware']);
$router->post('/tabelas-preco/clonar', 'TabelasPrecoController@clonar', ['AuthMiddleware']);

// Módulo: Áreas Comerciais
$router->get('/areas', 'AreaController@index', ['AuthMiddleware']);
$router->get('/areas/criar', 'AreaController@create', ['AuthMiddleware']);
$router->post('/areas/criar', 'AreaController@store', ['AuthMiddleware']);
$router->get('/areas/editar', 'AreaController@edit', ['AuthMiddleware']);
$router->post('/areas/editar', 'AreaController@update', ['AuthMiddleware']);
$router->post('/areas/deletar', 'AreaController@delete', ['AuthMiddleware']);

// Módulo: Produtos
$router->get('/produtos', 'ProdutosController@index', ['AuthMiddleware']);
$router->get('/produtos/criar', 'ProdutosController@create', ['AuthMiddleware']);
$router->post('/produtos/criar', 'ProdutosController@store', ['AuthMiddleware']);
$router->get('/produtos/editar', 'ProdutosController@edit', ['AuthMiddleware']);
$router->post('/produtos/editar', 'ProdutosController@update', ['AuthMiddleware']);
$router->post('/produtos/deletar', 'ProdutosController@delete', ['AuthMiddleware']);
$router->get('/produtos/buscar', 'ProdutosController@buscarProdutos', ['AuthMiddleware']);
$router->get('/produtos/buscar-posicoes-estoque', 'ProdutosController@buscarPosicoesEstoque', ['AuthMiddleware']);

// Módulo: Vendas
$router->get('/vendas', 'VendasController@index', ['AuthMiddleware']);
$router->get('/vendas/conferencia', 'VendasController@conferencia', ['AuthMiddleware']);
$router->get('/vendas/criar', 'VendasController@create', ['AuthMiddleware']);
$router->post('/vendas/criar', 'VendasController@store', ['AuthMiddleware']);
$router->get('/vendas/editar', 'VendasController@edit', ['AuthMiddleware']);
$router->post('/vendas/editar', 'VendasController@update', ['AuthMiddleware']);
$router->post('/vendas/deletar', 'VendasController@delete', ['AuthMiddleware']);
$router->post('/vendas/duplicar', 'VendasController@duplicar', ['AuthMiddleware']);
$router->get('/vendas/visualizar', 'VendasController@show', ['AuthMiddleware']);
$router->get('/vendas/imprimir', 'VendasController@print', ['AuthMiddleware']);
$router->get('/vendas/romaneio-separacao', 'VendasController@romaneioSeparacao', ['AuthMiddleware']);
$router->post('/vendas/conferir-item-romaneio', 'VendasController@conferirItemRomaneio', ['AuthMiddleware']);
$router->post('/vendas/buscar-item-conferencia', 'VendasController@buscarItemConferencia', ['AuthMiddleware']);
$router->post('/vendas/enviar-email', 'VendasController@enviarEmail', ['AuthMiddleware']);
$router->get('/vendas/buscar-itens', 'VendasController@buscarItens', ['AuthMiddleware']);
$router->get('/vendas/buscar-comissao-usuario', 'VendasController@buscarComissaoUsuario', ['AuthMiddleware']);
$router->get('/vendas/buscar-clientes-por-empresa', 'VendasController@buscarClientesPorEmpresa', ['AuthMiddleware']);
$router->get('/vendas/buscar-produtos-por-empresa', 'VendasController@buscarProdutosPorEmpresa', ['AuthMiddleware']);
$router->get('/vendas/buscar-licencas-cliente', 'VendasController@buscarLicencasCliente', ['AuthMiddleware']);
$router->post('/vendas/emitir-nfe', 'VendasController@emitirNfe', ['AuthMiddleware']);
$router->get('/vendas/preview-danfe', 'VendasController@previewDanfe', ['AuthMiddleware']);
$router->get('/vendas/visualizar-pdf-preview', 'VendasController@visualizarPdfPreview', ['AuthMiddleware']);
$router->get('/vendas/reimprimir-nfe', 'VendasController@reimprimirNfe', ['AuthMiddleware']);
$router->get('/vendas/reimprimir-nfse', 'VendasController@reimprimirNfse', ['AuthMiddleware']);
$router->get('/vendas/visualizar-pdf-nfse', 'VendasController@visualizarPdfNfse', ['AuthMiddleware']);
$router->get('/vendas/visualizar-xml-nfse', 'VendasController@visualizarXmlNfse', ['AuthMiddleware']);
$router->get('/vendas/visualizar-pdf', 'VendasController@visualizarPdf', ['AuthMiddleware']);
$router->get('/vendas/visualizar-pdf-cancelamento', 'VendasController@visualizarPdfCancelamento', ['AuthMiddleware']);
$router->get('/vendas/visualizar-xml', 'VendasController@visualizarXml', ['AuthMiddleware']);
$router->get('/vendas/download-xml', 'VendasController@downloadXml', ['AuthMiddleware']);
$router->post('/vendas/cancelar-nfe', 'VendasController@cancelarNfe', ['AuthMiddleware']);
$router->post('/vendas/carta-correcao-nfe', 'VendasController@cartaCorrecaoNfe', ['AuthMiddleware']);
$router->post('/vendas/emitir-nfse', 'VendasController@emitirNfse', ['AuthMiddleware']);
$router->get('/vendas/buscar-preco-produto', 'VendasController@buscarPrecoProduto', ['AuthMiddleware']);
$router->get('/vendas/simular-precos-produto', 'VendasController@simularPrecosProduto', ['AuthMiddleware']);
$router->get('/vendas/buscar-conferencia', 'VendasController@buscarVendaConferencia', ['AuthMiddleware']);
$router->post('/vendas/salvar-conferencia', 'VendasController@salvarConferencia', ['AuthMiddleware']);

// Módulo: Contas a Receber (Entradas)
// NOTA: Rotas de /entradas movidas para o final do arquivo (EntradasController)
// As rotas abaixo são mantidas apenas para /contas-receber (compatibilidade)

// Rotas de compatibilidade (contas-receber -> entradas)
$router->get('/contas-receber', 'ContasReceberController@index', ['AuthMiddleware']);
$router->get('/contas-receber/criar', 'ContasReceberController@create', ['AuthMiddleware']);
$router->post('/contas-receber/criar', 'ContasReceberController@store', ['AuthMiddleware']);
$router->get('/contas-receber/editar', 'ContasReceberController@edit', ['AuthMiddleware']);
$router->post('/contas-receber/editar', 'ContasReceberController@update', ['AuthMiddleware']);
$router->post('/contas-receber/receber', 'ContasReceberController@receive', ['AuthMiddleware']);
$router->get('/contas-receber/visualizar', 'ContasReceberController@show', ['AuthMiddleware']);

// Módulo: Métodos de Pagamento
$router->get('/metodos-pagamento', 'MetodosPagamentoController@index', ['AuthMiddleware']);
$router->get('/metodos-pagamento/criar', 'MetodosPagamentoController@create', ['AuthMiddleware']);
$router->post('/metodos-pagamento/criar', 'MetodosPagamentoController@store', ['AuthMiddleware']);
$router->get('/metodos-pagamento/editar', 'MetodosPagamentoController@edit', ['AuthMiddleware']);
$router->post('/metodos-pagamento/editar', 'MetodosPagamentoController@update', ['AuthMiddleware']);
$router->post('/metodos-pagamento/deletar', 'MetodosPagamentoController@delete', ['AuthMiddleware']);
$router->get('/metodos-pagamento/buscar-prazos', 'MetodosPagamentoController@buscarPrazos', ['AuthMiddleware']);
$router->post('/metodos-pagamento/adicionar-prazo', 'MetodosPagamentoController@adicionarPrazo', ['AuthMiddleware']);
$router->post('/metodos-pagamento/excluir-prazo', 'MetodosPagamentoController@excluirPrazo', ['AuthMiddleware']);

// Módulo: Relatórios
$router->get('/relatorios', 'RelatoriosController@index', ['AuthMiddleware']);
$router->get('/relatorios/vendas-por-periodo', 'RelatoriosController@vendasPeriodo', ['AuthMiddleware']);
$router->get('/relatorios/vendas-por-produto', 'RelatoriosController@vendasProduto', ['AuthMiddleware']);
$router->get('/relatorios/vendas-por-cliente', 'RelatoriosController@vendasCliente', ['AuthMiddleware']);
$router->get('/relatorios/vendas-por-vendedor', 'RelatoriosController@vendasVendedor', ['AuthMiddleware']);
$router->get('/relatorios/fluxo-caixa', 'RelatoriosController@fluxoCaixa', ['AuthMiddleware']);
$router->get('/relatorios/entradas', 'RelatoriosController@contasReceber', ['AuthMiddleware']);
$router->get('/relatorios/contas-receber', 'RelatoriosController@contasReceber', ['AuthMiddleware']); // Compatibilidade
$router->get('/relatorios/contas-pagar', 'RelatoriosController@contasPagar', ['AuthMiddleware']);
$router->get('/relatorios/dre', 'RelatoriosController@dre', ['AuthMiddleware']);
$router->get('/relatorios/balanco', 'RelatoriosController@balanco', ['AuthMiddleware']);
$router->get('/relatorios/estoque-atual', 'RelatoriosController@estoqueAtual', ['AuthMiddleware']);
$router->get('/relatorios/movimentacao-estoque', 'RelatoriosController@movimentacaoEstoque', ['AuthMiddleware']);
$router->get('/relatorios/produtos-zerados', 'RelatoriosController@produtosZerados', ['AuthMiddleware']);
$router->get('/relatorios/produtos-proximos-vencer', 'RelatoriosController@produtosProximosVencer', ['AuthMiddleware']);
$router->get('/relatorios/inventario', 'RelatoriosController@inventario', ['AuthMiddleware']);
$router->get('/relatorios/valorizacao-estoque', 'RelatoriosController@valorizacaoEstoque', ['AuthMiddleware']);
$router->get('/relatorios/apuracao-impostos', 'RelatoriosController@apuracaoImpostos', ['AuthMiddleware']);

// Relatórios de Comissões
$router->get('/relatorios/comissoes-vendedor', 'RelatoriosController@comissoesVendedor', ['AuthMiddleware']);
$router->get('/relatorios/comissoes-profissional', 'RelatoriosController@comissoesProfissional', ['AuthMiddleware']);
$router->get('/relatorios/comissoes-periodo', 'RelatoriosController@comissoesPeriodo', ['AuthMiddleware']);
$router->get('/relatorios/comissoes-produto', 'RelatoriosController@comissoesProduto', ['AuthMiddleware']);
$router->get('/relatorios/comissoes-resumo', 'RelatoriosController@comissoesResumo', ['AuthMiddleware']);

// Módulo: Grupos
$router->get('/grupos', 'GruposController@index', ['AuthMiddleware']);
$router->get('/grupos/criar', 'GruposController@create', ['AuthMiddleware']);
$router->post('/grupos/criar', 'GruposController@store', ['AuthMiddleware']);
$router->get('/grupos/editar', 'GruposController@edit', ['AuthMiddleware']);
$router->post('/grupos/editar', 'GruposController@update', ['AuthMiddleware']);
$router->post('/grupos/deletar', 'GruposController@delete', ['AuthMiddleware']);

// Módulo: Grupos
$router->get('/grupos-pessoas', 'GruposPessoasController@index', ['AuthMiddleware']);
$router->get('/grupos-pessoas/criar', 'GruposPessoasController@create', ['AuthMiddleware']);
$router->post('/grupos-pessoas/criar', 'GruposPessoasController@store', ['AuthMiddleware']);
$router->get('/grupos-pessoas/editar', 'GruposPessoasController@edit', ['AuthMiddleware']);
$router->post('/grupos-pessoas/editar', 'GruposPessoasController@update', ['AuthMiddleware']);
$router->post('/grupos-pessoas/deletar', 'GruposPessoasController@delete', ['AuthMiddleware']);
$router->post('/grupos-pessoas/adicionar-subgrupo', 'GruposPessoasController@adicionarSubgrupo', ['AuthMiddleware']);
$router->post('/grupos-pessoas/deletar-subgrupo', 'GruposPessoasController@deletarSubgrupo', ['AuthMiddleware']);

// Módulo: Contas Bancárias
$router->get('/contas', 'ContasController@index', ['AuthMiddleware']);
$router->get('/contas/criar', 'ContasController@create', ['AuthMiddleware']);
$router->post('/contas/criar', 'ContasController@store', ['AuthMiddleware']);
$router->get('/contas/editar', 'ContasController@edit', ['AuthMiddleware']);
$router->post('/contas/editar', 'ContasController@update', ['AuthMiddleware']);
$router->post('/contas/deletar', 'ContasController@delete', ['AuthMiddleware']);

// Módulo: Itens/Produtos (alias)
$router->get('/itens', 'ProdutosController@index', ['AuthMiddleware']);
$router->get('/itens/buscar', 'ProdutosController@buscar', ['AuthMiddleware']);
$router->get('/itens/criar', 'ProdutosController@create', ['AuthMiddleware']);
$router->post('/itens/criar', 'ProdutosController@store', ['AuthMiddleware']);
$router->get('/itens/editar', 'ProdutosController@edit', ['AuthMiddleware']);
$router->post('/itens/editar', 'ProdutosController@update', ['AuthMiddleware']);
$router->get('/itens/buscar-posicoes-estoque', 'ProdutosController@buscarPosicoesEstoque', ['AuthMiddleware']);
$router->get('/itens/extrato-produto', 'ProdutosController@extratoProduto', ['AuthMiddleware']);
$router->post('/itens/deletar', 'ProdutosController@delete', ['AuthMiddleware']);

// Módulo: Plano de Contas
$router->get('/plano-contas', 'PlanoContasController@index', ['AuthMiddleware']);
$router->get('/plano-contas/criar', 'PlanoContasController@create', ['AuthMiddleware']);
$router->post('/plano-contas/criar', 'PlanoContasController@store', ['AuthMiddleware']);
$router->get('/plano-contas/editar', 'PlanoContasController@edit', ['AuthMiddleware']);
$router->post('/plano-contas/editar', 'PlanoContasController@update', ['AuthMiddleware']);
$router->post('/plano-contas/deletar', 'PlanoContasController@delete', ['AuthMiddleware']);

// Módulo: Centro de Custos
$router->get('/centro-custos', 'CentroCustosController@index', ['AuthMiddleware']);
$router->get('/centro-custos/criar', 'CentroCustosController@create', ['AuthMiddleware']);
$router->post('/centro-custos/criar', 'CentroCustosController@store', ['AuthMiddleware']);
$router->get('/centro-custos/editar', 'CentroCustosController@edit', ['AuthMiddleware']);
$router->post('/centro-custos/editar', 'CentroCustosController@update', ['AuthMiddleware']);
$router->post('/centro-custos/deletar', 'CentroCustosController@delete', ['AuthMiddleware']);

// Módulo: Gestor de Estoque (Locais)
$router->get('/gestor-estoque', 'GestorEstoqueController@index', ['AuthMiddleware']);
$router->get('/gestor-estoque/criar', 'GestorEstoqueController@create', ['AuthMiddleware']);
$router->post('/gestor-estoque/criar', 'GestorEstoqueController@store', ['AuthMiddleware']);
$router->get('/gestor-estoque/editar', 'GestorEstoqueController@edit', ['AuthMiddleware']);
$router->post('/gestor-estoque/editar', 'GestorEstoqueController@update', ['AuthMiddleware']);
$router->post('/gestor-estoque/deletar', 'GestorEstoqueController@delete', ['AuthMiddleware']);

// Módulo: Transferência de Estoque
$router->get('/estoque/transferencia', 'TransferenciaEstoqueController@index', ['AuthMiddleware']);
$router->get('/estoque/transferencia/criar', 'TransferenciaEstoqueController@create', ['AuthMiddleware']);
$router->post('/estoque/transferencia/criar', 'TransferenciaEstoqueController@store', ['AuthMiddleware']);
$router->get('/estoque/transferencia/editar', 'TransferenciaEstoqueController@edit', ['AuthMiddleware']);
$router->post('/estoque/transferencia/editar', 'TransferenciaEstoqueController@update', ['AuthMiddleware']);
$router->post('/estoque/transferencia/deletar', 'TransferenciaEstoqueController@delete', ['AuthMiddleware']);
$router->get('/estoque/transferencia/visualizar', 'TransferenciaEstoqueController@visualizar', ['AuthMiddleware']);
$router->get('/estoque/transferencia/buscar-produtos', 'TransferenciaEstoqueController@buscarProdutos', ['AuthMiddleware']);
$router->get('/estoque/transferencia/buscar-lotes', 'TransferenciaEstoqueController@buscarLotesProduto', ['AuthMiddleware']);

// Módulo: Impostos
$router->get('/impostos', 'ImpostosController@index', ['AuthMiddleware']);
$router->get('/impostos/criar', 'ImpostosController@create', ['AuthMiddleware']);
$router->post('/impostos/criar', 'ImpostosController@store', ['AuthMiddleware']);
$router->get('/impostos/editar', 'ImpostosController@edit', ['AuthMiddleware']);
$router->post('/impostos/editar', 'ImpostosController@update', ['AuthMiddleware']);
$router->post('/impostos/deletar', 'ImpostosController@delete', ['AuthMiddleware']);
$router->post('/impostos/clonar', 'ImpostosController@clonar', ['AuthMiddleware']);
$router->post('/impostos/adicionar-configuracao', 'ImpostosController@adicionarConfiguracao', ['AuthMiddleware']);
$router->get('/impostos/listar-configuracoes', 'ImpostosController@listarConfiguracoes', ['AuthMiddleware']);

// Módulo: Perfis de Acesso
$router->get('/perfis-acesso', 'PerfisAcessoController@index', ['AuthMiddleware']);
$router->get('/perfis-acesso/criar', 'PerfisAcessoController@create', ['AuthMiddleware']);
$router->post('/perfis-acesso/criar', 'PerfisAcessoController@store', ['AuthMiddleware']);
$router->get('/perfis-acesso/editar', 'PerfisAcessoController@edit', ['AuthMiddleware']);
$router->post('/perfis-acesso/editar', 'PerfisAcessoController@update', ['AuthMiddleware']);
$router->post('/perfis-acesso/deletar', 'PerfisAcessoController@delete', ['AuthMiddleware']);
$router->post('/perfis-acesso/salvar-permissoes', 'PerfisAcessoController@salvarPermissoes', ['AuthMiddleware']);
$router->post('/perfis-acesso/buscar-movimentos', 'PerfisAcessoController@buscarMovimentos', ['AuthMiddleware']);
$router->post('/perfis-acesso/salvar-movimentos', 'PerfisAcessoController@salvarMovimentos', ['AuthMiddleware']);
$router->get('/impostos/buscar-configuracao', 'ImpostosController@buscarConfiguracao', ['AuthMiddleware']);
$router->post('/impostos/atualizar-configuracao', 'ImpostosController@atualizarConfiguracao', ['AuthMiddleware']);
$router->post('/impostos/excluir-configuracao', 'ImpostosController@excluirConfiguracao', ['AuthMiddleware']);

// Módulo: Compras
$router->get('/compras', 'ComprasController@index', ['AuthMiddleware']);
$router->get('/compras/criar', 'ComprasController@create', ['AuthMiddleware']);
$router->post('/compras/criar', 'ComprasController@store', ['AuthMiddleware']);
$router->get('/compras/visualizar', 'ComprasController@show', ['AuthMiddleware']);
$router->get('/compras/imprimir', 'ComprasController@print', ['AuthMiddleware']);
$router->get('/compras/editar', 'ComprasController@edit', ['AuthMiddleware']);
$router->post('/compras/editar', 'ComprasController@update', ['AuthMiddleware']);
$router->post('/compras/deletar', 'ComprasController@delete', ['AuthMiddleware']);
$router->post('/compras/enviar-email', 'ComprasController@enviarEmail', ['AuthMiddleware']);
$router->get('/compras/monitor-nfe', 'ComprasController@monitorNFe', ['AuthMiddleware']);
$router->post('/compras/manifestar-nfe', 'ComprasController@manifestarNFe', ['AuthMiddleware']);
$router->post('/compras/importar-nfe-manual', 'ComprasController@importarNFeManual', ['AuthMiddleware']);
$router->post('/compras/cadastrar-fornecedor-e-criar-compra', 'ComprasController@cadastrarFornecedorECriarCompra', ['AuthMiddleware']);
$router->get('/compras/relacionar-produtos-xml', 'ComprasController@relacionarProdutosXml', ['AuthMiddleware']);
$router->post('/compras/atualizar-sku-produto', 'ComprasController@atualizarSkuProduto', ['AuthMiddleware']);
$router->post('/compras/salvar-relacionamento-produtos', 'ComprasController@salvarRelacionamentoProdutos', ['AuthMiddleware']);
$router->post('/compras/processar-xml', 'ComprasController@processarXml', ['AuthMiddleware']);
$router->get('/compras/nfe/pdf', 'ComprasController@downloadNFePdf', ['AuthMiddleware']);
$router->get('/compras/nfe/xml', 'ComprasController@downloadNFeXml', ['AuthMiddleware']);

// Módulo: Gestão de Estoque
// Rotas específicas primeiro (antes da rota genérica /estoque)
$router->get('/estoque/expedicao/buscar-vendas', 'EstoqueController@buscarVendasExpedicao', ['AuthMiddleware']);
$router->get('/estoque/inventario/buscar-produtos', 'EstoqueController@buscarProdutosInventario', ['AuthMiddleware']);
$router->get('/estoque/inventario/exportar', 'EstoqueController@exportarInventarioTemplate', ['AuthMiddleware']);
$router->get('/estoque/inventario/empresas', 'EstoqueController@listarEmpresasParaInventario', ['AuthMiddleware']);
$router->get('/estoque/inventario/detalhes', 'EstoqueController@detalhesInventario', ['AuthMiddleware']);
$router->get('/estoque/inventario/criar', 'EstoqueController@createInventario', ['AuthMiddleware']);
$router->get('/estoque/inventario', 'EstoqueController@indexInventarios', ['AuthMiddleware']);
$router->get('/estoque/movimentacoes', 'EstoqueController@movimentacoes', ['AuthMiddleware']);
$router->get('/estoque/relatorios', 'EstoqueController@relatorios', ['AuthMiddleware']);
$router->get('/estoque/lotes', 'EstoqueController@lotes', ['AuthMiddleware']);
$router->post('/estoque/inventario/criar', 'EstoqueController@storeInventario', ['AuthMiddleware']);
$router->post('/estoque/inventario/importar', 'EstoqueController@importarInventarioPlanilha', ['AuthMiddleware']);
// Rota genérica por último
$router->get('/estoque', 'EstoqueController@index', ['AuthMiddleware']);

// Módulo: Contratos
$router->get('/contratos', 'ContratosController@index', ['AuthMiddleware']);
$router->get('/contratos/criar', 'ContratosController@create', ['AuthMiddleware']);
$router->post('/contratos/criar', 'ContratosController@store', ['AuthMiddleware']);
$router->get('/contratos/editar', 'ContratosController@edit', ['AuthMiddleware']);
$router->post('/contratos/editar', 'ContratosController@update', ['AuthMiddleware']);
$router->get('/contratos/visualizar', 'ContratosController@show', ['AuthMiddleware']);
$router->get('/contratos/imprimir', 'ContratosController@print', ['AuthMiddleware']);
$router->post('/contratos/deletar', 'ContratosController@delete', ['AuthMiddleware']);

// Módulo: Tickets (Chamados)
$router->get('/tickets', 'TicketsController@index', ['AuthMiddleware']);
$router->get('/tickets/abrir', 'TicketsController@abrir', ['AuthMiddleware']);
$router->post('/tickets/abrir', 'TicketsController@storeAbrir', ['AuthMiddleware']);
$router->get('/tickets/acompanhar', 'TicketsController@acompanhar', ['AuthMiddleware']);
$router->get('/tickets/acompanhar/visualizar', 'TicketsController@acompanharShow', ['AuthMiddleware']);
$router->get('/tickets/criar', 'TicketsController@create', ['AuthMiddleware']);
$router->post('/tickets/criar', 'TicketsController@store', ['AuthMiddleware']);
$router->get('/tickets/editar', 'TicketsController@edit', ['AuthMiddleware']);
$router->post('/tickets/editar', 'TicketsController@update', ['AuthMiddleware']);
$router->get('/tickets/visualizar', 'TicketsController@show', ['AuthMiddleware']);
$router->post('/tickets/deletar', 'TicketsController@delete', ['AuthMiddleware']);
$router->post('/tickets/adicionar-comentario', 'TicketsController@adicionarComentario', ['AuthMiddleware']);
$router->post('/tickets/atualizar-status', 'TicketsController@atualizarStatus', ['AuthMiddleware']);

// Rotas Ordem de Serviço PMOC (ANTES de ordem-servico para evitar conflito)
$router->get('/ordem-servico-pmoc', 'OrdemServicoPmocController@index', ['AuthMiddleware']);
$router->get('/ordem-servico-pmoc/criar', 'OrdemServicoPmocController@create', ['AuthMiddleware']);
$router->post('/ordem-servico-pmoc/criar', 'OrdemServicoPmocController@store', ['AuthMiddleware']);
$router->get('/ordem-servico-pmoc/editar', 'OrdemServicoPmocController@edit', ['AuthMiddleware']);
$router->post('/ordem-servico-pmoc/editar', 'OrdemServicoPmocController@update', ['AuthMiddleware']);
$router->get('/ordem-servico-pmoc/visualizar', 'OrdemServicoPmocController@show', ['AuthMiddleware']);
$router->get('/ordem-servico-pmoc/imprimir', 'OrdemServicoPmocController@print', ['AuthMiddleware']);
$router->post('/ordem-servico-pmoc/deletar', 'OrdemServicoPmocController@delete', ['AuthMiddleware']);

// Módulo: Modelos PMOC
$router->get('/modelos-pmoc', 'ModelosPmocController@index', ['AuthMiddleware']);
$router->get('/modelos-pmoc/criar', 'ModelosPmocController@create', ['AuthMiddleware']);
$router->post('/modelos-pmoc/criar', 'ModelosPmocController@store', ['AuthMiddleware']);
$router->get('/modelos-pmoc/editar', 'ModelosPmocController@edit', ['AuthMiddleware']);
$router->post('/modelos-pmoc/editar', 'ModelosPmocController@update', ['AuthMiddleware']);
$router->post('/modelos-pmoc/deletar', 'ModelosPmocController@delete', ['AuthMiddleware']);

// Módulo: Ordem de Serviço
// Módulo: Ordem de Serviço (completo - baseado em Vendas)
$router->get('/ordem-servico', 'OrdemServicoController@index', ['AuthMiddleware']);
$router->get('/ordem-servico/criar', 'OrdemServicoController@create', ['AuthMiddleware']);
$router->post('/ordem-servico/criar', 'OrdemServicoController@store', ['AuthMiddleware']);
$router->get('/ordem-servico/editar', 'OrdemServicoController@edit', ['AuthMiddleware']);
$router->post('/ordem-servico/editar', 'OrdemServicoController@update', ['AuthMiddleware']);
$router->post('/ordem-servico/deletar', 'OrdemServicoController@delete', ['AuthMiddleware']);
$router->get('/ordem-servico/visualizar', 'OrdemServicoController@show', ['AuthMiddleware']);
$router->get('/ordem-servico/imprimir', 'OrdemServicoController@print', ['AuthMiddleware']);
$router->get('/ordem-servico/imprimir-producao', 'OrdemServicoController@printProducao', ['AuthMiddleware']);
$router->post('/ordem-servico/enviar-email', 'OrdemServicoController@enviarEmail', ['AuthMiddleware']);
$router->get('/ordem-servico/buscar-itens', 'OrdemServicoController@buscarItens', ['AuthMiddleware']);
$router->post('/ordem-servico/emitir-nfe', 'OrdemServicoController@emitirNfe', ['AuthMiddleware']);
$router->get('/ordem-servico/preview-danfe', 'OrdemServicoController@previewDanfe', ['AuthMiddleware']);
$router->get('/ordem-servico/visualizar-pdf-preview', 'OrdemServicoController@visualizarPdfPreview', ['AuthMiddleware']);
$router->get('/ordem-servico/reimprimir-nfe', 'OrdemServicoController@reimprimirNfe', ['AuthMiddleware']);
$router->get('/ordem-servico/visualizar-pdf', 'OrdemServicoController@visualizarPdf', ['AuthMiddleware']);
$router->get('/ordem-servico/visualizar-pdf-cancelamento', 'OrdemServicoController@visualizarPdfCancelamento', ['AuthMiddleware']);
$router->get('/ordem-servico/visualizar-xml', 'OrdemServicoController@visualizarXml', ['AuthMiddleware']);
$router->get('/ordem-servico/download-xml', 'OrdemServicoController@downloadXml', ['AuthMiddleware']);
$router->post('/ordem-servico/cancelar-nfe', 'OrdemServicoController@cancelarNfe', ['AuthMiddleware']);
$router->post('/ordem-servico/carta-correcao-nfe', 'OrdemServicoController@cartaCorrecaoNfe', ['AuthMiddleware']);
$router->post('/ordem-servico/status', 'OrdemServicoController@updateStatus', ['AuthMiddleware']);

// Módulo: DAV (Pré-vendas)
$router->get('/pre-vendas', 'PreVendasController@index', ['AuthMiddleware']);
$router->get('/pre-vendas/criar', 'PreVendasController@create', ['AuthMiddleware']);
$router->post('/pre-vendas/criar', 'PreVendasController@store', ['AuthMiddleware']);
$router->get('/pre-vendas/editar', 'PreVendasController@edit', ['AuthMiddleware']);
$router->post('/pre-vendas/editar', 'PreVendasController@update', ['AuthMiddleware']);
$router->post('/pre-vendas/deletar', 'PreVendasController@delete', ['AuthMiddleware']);
$router->get('/pre-vendas/visualizar', 'PreVendasController@show', ['AuthMiddleware']);
$router->get('/pre-vendas/imprimir', 'PreVendasController@print', ['AuthMiddleware']);
$router->post('/pre-vendas/converter', 'PreVendasController@converterParaVenda', ['AuthMiddleware']);
$router->post('/pre-vendas/enviar-email', 'PreVendasController@enviarEmail', ['AuthMiddleware']);
$router->get('/pre-vendas/buscar-itens', 'PreVendasController@buscarItens', ['AuthMiddleware']);
$router->post('/pre-vendas/status', 'PreVendasController@updateStatus', ['AuthMiddleware']);

// Módulo: Orçamentos de Água Mineral
$router->get('/orcamentos-agua', 'OrcamentosAguaController@index', ['AuthMiddleware']);
$router->get('/orcamentos-agua/criar', 'OrcamentosAguaController@create', ['AuthMiddleware']);
$router->post('/orcamentos-agua/criar', 'OrcamentosAguaController@store', ['AuthMiddleware']);
$router->get('/orcamentos-agua/editar', 'OrcamentosAguaController@edit', ['AuthMiddleware']);
$router->post('/orcamentos-agua/editar', 'OrcamentosAguaController@update', ['AuthMiddleware']);
$router->post('/orcamentos-agua/status', 'OrcamentosAguaController@updateStatus', ['AuthMiddleware']);
$router->get('/orcamentos-agua/itens', 'OrcamentosAguaController@getItens', ['AuthMiddleware']);
$router->post('/orcamentos-agua/itens/atualizar', 'OrcamentosAguaController@atualizarQuebraTroca', ['AuthMiddleware']);

// Módulo: Expedição
$router->get('/expedicao', 'ExpedicaoController@index', ['AuthMiddleware']);
$router->get('/expedicao/visualizar', 'ExpedicaoController@show', ['AuthMiddleware']);
$router->post('/expedicao/conferir', 'ExpedicaoController@conferir', ['AuthMiddleware']);
$router->post('/expedicao/emitir-nfe', 'ExpedicaoController@emitirNfe', ['AuthMiddleware']);
$router->post('/expedicao/despachar', 'ExpedicaoController@despachar', ['AuthMiddleware']);

// Módulo: Agenda
$router->get('/agenda', 'AgendaController@index', ['AuthMiddleware']);
$router->get('/agenda/eventos', 'AgendaController@eventos', ['AuthMiddleware']);
$router->post('/agenda/eventos/criar', 'AgendaController@criarEvento', ['AuthMiddleware']);
$router->post('/agenda/eventos/editar', 'AgendaController@editarEvento', ['AuthMiddleware']);
$router->post('/agenda/eventos/deletar', 'AgendaController@deletarEvento', ['AuthMiddleware']);

// Módulo: Entradas (Contas a Receber)
$router->get('/entradas', 'EntradasController@index', ['AuthMiddleware']);
$router->get('/entradas/criar', 'EntradasController@create', ['AuthMiddleware']);
$router->post('/entradas/criar', 'EntradasController@store', ['AuthMiddleware']);
$router->get('/entradas/editar', 'EntradasController@edit', ['AuthMiddleware']);
$router->post('/entradas/editar', 'EntradasController@update', ['AuthMiddleware']);
$router->post('/entradas/atualizar', 'EntradasController@updateAndRedirect', ['AuthMiddleware']);
$router->get('/entradas/visualizar', 'EntradasController@show', ['AuthMiddleware']);
$router->post('/entradas/receber', 'EntradasController@receive', ['AuthMiddleware']);
$router->get('/entradas/recibo', 'EntradasController@recibo', ['AuthMiddleware']);
$router->get('/entradas/duplicata', 'EntradasController@duplicata', ['AuthMiddleware']);
$router->post('/entradas/gerar-boleto-inter', 'EntradasController@gerarBoletoInter', ['AuthMiddleware']);
$router->post('/entradas/gerar-boletos-lote', 'EntradasController@gerarBoletosEmLote', ['AuthMiddleware']);
$router->post('/entradas/enviar-boleto-email', 'EntradasController@enviarBoletoEmail', ['AuthMiddleware']);
// Rotas Shipay
$router->post('/entradas/gerar-boleto-shipay', 'EntradasController@gerarBoletoShipay', ['AuthMiddleware']);
$router->get('/entradas/consultar-boleto-shipay', 'EntradasController@consultarBoletoShipay', ['AuthMiddleware']);
$router->post('/entradas/enviar-boleto-shipay-email', 'EntradasController@enviarBoletoShipayEmail', ['AuthMiddleware']);
$router->post('/entradas/reemitir-boleto-shipay', 'EntradasController@reemitirBoletoShipay', ['AuthMiddleware']);
$router->get('/entradas/listar-cobrancas-shipay', 'EntradasController@listarCobrancasShipay', ['AuthMiddleware']);
$router->get('/entradas/listar-wallets-shipay', 'EntradasController@listarWalletsShipay', ['AuthMiddleware']);
$router->get('/entradas/recebimento-diario', 'EntradasController@recebimentoDiario', ['AuthMiddleware']);
$router->get('/entradas/sicacred/consultar-datas', 'EntradasController@sicacredConsultarDatas', ['AuthMiddleware']);
$router->post('/entradas/sicacred/enviar-dados', 'EntradasController@sicacredEnviarDados', ['AuthMiddleware']);
$router->post('/entradas/deletar', 'EntradasController@delete', ['AuthMiddleware']);
$router->post('/entradas/anexos/upload', 'EntradasController@uploadAnexo', ['AuthMiddleware']);
$router->get('/entradas/anexos/listar', 'EntradasController@listarAnexos', ['AuthMiddleware']);
$router->post('/entradas/anexos/deletar', 'EntradasController@deletarAnexo', ['AuthMiddleware']);
$router->get('/entradas/anexos/download', 'EntradasController@downloadAnexo', ['AuthMiddleware']);

// Webhook Banco Inter (rota pública - sem autenticação)
$router->post('/webhook/banco-inter', 'WebhookBancoInterController@receberNotificacao');

// Webhook Itaú (rota pública - sem autenticação)
$router->post('/webhook/itau', 'WebhookItauController@receberNotificacao');

// Webhook Shipay (rota pública - sem autenticação)
$router->post('/webhook/shipay', 'ShipayWebhookController@receive');

// Módulo: Saídas (Contas a Pagar)
$router->get('/saidas', 'SaidasController@index', ['AuthMiddleware']);
$router->get('/saidas/criar', 'SaidasController@create', ['AuthMiddleware']);
$router->post('/saidas/criar', 'SaidasController@store', ['AuthMiddleware']);
$router->get('/saidas/editar', 'SaidasController@edit', ['AuthMiddleware']);
$router->post('/saidas/editar', 'SaidasController@update', ['AuthMiddleware']);
$router->post('/saidas/deletar', 'SaidasController@delete', ['AuthMiddleware']);
$router->get('/saidas/visualizar', 'SaidasController@show', ['AuthMiddleware']);
$router->post('/saidas/pagar', 'SaidasController@pagar', ['AuthMiddleware']);
$router->post('/saidas/anexos/upload', 'SaidasController@uploadAnexo', ['AuthMiddleware']);
$router->get('/saidas/anexos/listar', 'SaidasController@listarAnexos', ['AuthMiddleware']);
$router->post('/saidas/anexos/deletar', 'SaidasController@deletarAnexo', ['AuthMiddleware']);
$router->get('/saidas/anexos/download', 'SaidasController@downloadAnexo', ['AuthMiddleware']);
$router->get('/saidas/bordero', 'SaidasController@bordero', ['AuthMiddleware']);
$router->get('/saidas/bordero/print', 'SaidasController@borderoPrint', ['AuthMiddleware']);
$router->get('/saidas/provisionamento', 'SaidasController@provisionamento', ['AuthMiddleware']);
$router->get('/saidas/folha-pagamento', 'SaidasController@folhaPagamento', ['AuthMiddleware']);

// Módulo: Fluxo de Caixa
$router->get('/fluxo-caixa', 'FluxoCaixaController@index', ['AuthMiddleware']);
$router->get('/fluxo-caixa/criar', 'FluxoCaixaController@create', ['AuthMiddleware']);
$router->post('/fluxo-caixa/criar', 'FluxoCaixaController@store', ['AuthMiddleware']);
$router->get('/fluxo-caixa/editar', 'FluxoCaixaController@edit', ['AuthMiddleware']);
$router->post('/fluxo-caixa/editar', 'FluxoCaixaController@update', ['AuthMiddleware']);
$router->get('/fluxo-caixa/planos-contas', 'FluxoCaixaController@getPlanosContasAjax', ['AuthMiddleware']);
$router->get('/fluxo-caixa/centros-custos', 'FluxoCaixaController@getCentrosCustosAjax', ['AuthMiddleware']);
$router->post('/fluxo-caixa/atualizar-plano-contas', 'FluxoCaixaController@updatePlanoContas', ['AuthMiddleware']);
$router->post('/fluxo-caixa/atualizar-centro-custo', 'FluxoCaixaController@updateCentroCusto', ['AuthMiddleware']);
$router->post('/fluxo-caixa/excluir', 'FluxoCaixaController@delete', ['AuthMiddleware']);
        $router->get('/fluxo-caixa/conciliacao', 'FluxoCaixaController@conciliacao', ['AuthMiddleware']);
        $router->post('/fluxo-caixa/conciliacao/processar', 'FluxoCaixaController@processarConciliacao', ['AuthMiddleware']);
        $router->post('/fluxo-caixa/conciliacao/criar-lancamento', 'FluxoCaixaController@criarLancamentoOfx', ['AuthMiddleware']);
$router->get('/fluxo-caixa/saldo-bancario', 'FluxoCaixaController@saldoBancario', ['AuthMiddleware']);
$router->get('/fluxo-caixa/movimentacao-dia', 'FluxoCaixaController@movimentacaoDia', ['AuthMiddleware']);
$router->get('/fluxo-caixa/projecao', 'FluxoCaixaController@projecao', ['AuthMiddleware']);
$router->get('/fluxo-caixa/relatorio', 'FluxoCaixaController@relatorio', ['AuthMiddleware']);

// Módulo: Usuários
$router->get('/usuarios', 'UsuariosController@index', ['AuthMiddleware']);
$router->get('/usuarios/criar', 'UsuariosController@create', ['AuthMiddleware']);
$router->post('/usuarios/criar', 'UsuariosController@store', ['AuthMiddleware']);
$router->get('/usuarios/editar', 'UsuariosController@edit', ['AuthMiddleware']);
$router->post('/usuarios/editar', 'UsuariosController@update', ['AuthMiddleware']);
$router->post('/usuarios/toggle-status', 'UsuariosController@toggleStatus', ['AuthMiddleware']);
$router->get('/usuarios/permissoes', 'UsuariosController@permissions', ['AuthMiddleware']);
$router->post('/usuarios/permissoes', 'UsuariosController@updatePermissions', ['AuthMiddleware']);

// Módulo: Parâmetros
$router->get('/parametros', 'ParametrosController@index', ['AuthMiddleware']);
$router->get('/parametros-edit', 'ParametrosController@edit', ['AuthMiddleware']);
$router->get('/parametros/editar', 'ParametrosController@edit', ['AuthMiddleware']);
$router->post('/parametros-salvar', 'ParametrosController@save', ['AuthMiddleware']);
$router->post('/parametros/boolean', 'ParametrosController@salvarBoolean', ['AuthMiddleware']);
$router->post('/parametros-upload', 'ParametrosController@upload', ['AuthMiddleware']);
$router->post('/parametros/modulos/toggle', 'ParametrosController@toggleModulo', ['AuthMiddleware']);
$router->get('/parametros/grupos-acesso', 'ParametrosController@gruposAcesso', ['AuthMiddleware']);
$router->post('/parametros/grupos-acesso', 'ParametrosController@gruposAcesso', ['AuthMiddleware']);

// Módulo: Gestão de Status
$router->get('/gestao/status', 'ModuloStatusController@index', ['AuthMiddleware']);
$router->get('/gestao/status/editar', 'ModuloStatusController@edit', ['AuthMiddleware']);
$router->post('/gestao/status/editar', 'ModuloStatusController@update', ['AuthMiddleware']);

// IPs Válidos
$router->post('/parametros-adicionar-ip', 'ParametrosController@adicionarIpValido', ['AuthMiddleware']);
$router->post('/parametros-remover-ip', 'ParametrosController@removerIpValido', ['AuthMiddleware']);
$router->post('/parametros-toggle-ip', 'ParametrosController@toggleIpValido', ['AuthMiddleware']);

// Módulo: Assinatura e Pagamentos
$router->group('/assinatura', function ($router) {
    $router->get('', 'AssinaturaController@index');
    $router->get('/upgrade', 'AssinaturaController@upgrade');
    $router->post('/upgrade', 'AssinaturaController@processUpgrade');
    $router->get('/faturas', 'AssinaturaController@invoices');
}, ['AuthMiddleware']);

// Módulo: Arquivos Fiscais
$router->get('/arquivos-fiscais', 'ArquivosFiscaisController@index', ['AuthMiddleware']);
$router->get('/arquivos-fiscais/download-xml', 'ArquivosFiscaisController@downloadXml', ['AuthMiddleware']);
$router->get('/arquivos-fiscais/download-pdf', 'ArquivosFiscaisController@downloadPdf', ['AuthMiddleware']);
$router->post('/arquivos-fiscais/exportar-contador', 'ArquivosFiscaisController@exportarContador', ['AuthMiddleware']);
$router->post('/arquivos-fiscais/enviar-email-contador', 'ArquivosFiscaisController@enviarEmailContador', ['AuthMiddleware']);
$router->post('/arquivos-fiscais/gerar-sped-fiscal', 'ArquivosFiscaisController@gerarSpedFiscal', ['AuthMiddleware']);
$router->post('/arquivos-fiscais/gerar-sped-contribuicoes', 'ArquivosFiscaisController@gerarSpedContribuicoes', ['AuthMiddleware']);
$router->post('/arquivos-fiscais/deletar', 'ArquivosFiscaisController@deletar', ['AuthMiddleware']);
