/**
 * Systhema ERP - Main JavaScript
 */

(function () {
    'use strict';

    // Loading overlay
    const LoadingOverlay = {
        show: function () {
            const overlay = document.createElement('div');
            overlay.className = 'loading-overlay';
            overlay.id = 'loadingOverlay';
            overlay.innerHTML = '<div class="loading-spinner"></div>';
            document.body.appendChild(overlay);
        },

        hide: function () {
            const overlay = document.getElementById('loadingOverlay');
            if (overlay) {
                overlay.remove();
            }
        }
    };

    // SweetAlert2 Toast notifications
    const Toast = {
        show: function (message, type = 'info') {
            const iconMap = {
                success: 'success',
                danger: 'error',
                error: 'error',
                warning: 'warning',
                info: 'info'
            };

            Swal.fire({
                icon: iconMap[type] || 'info',
                title: message,
                toast: true,
                position: 'top-end',
                showConfirmButton: false,
                timer: 3000,
                timerProgressBar: true
            });
        },

        success: function (message) {
            Swal.fire({
                icon: 'success',
                title: message,
                toast: true,
                position: 'top-end',
                showConfirmButton: false,
                timer: 3000,
                timerProgressBar: true
            });
        },

        error: function (message) {
            Swal.fire({
                icon: 'error',
                title: 'Erro',
                text: message,
                confirmButtonColor: '#0066CC'
            });
        },

        warning: function (message) {
            Swal.fire({
                icon: 'warning',
                title: 'Atenção',
                text: message,
                confirmButtonColor: '#0066CC'
            });
        },

        info: function (message) {
            Swal.fire({
                icon: 'info',
                title: 'Informação',
                text: message,
                confirmButtonColor: '#0066CC'
            });
        }
    };

    // Confirmação de exclusão com SweetAlert2
    window.confirmDelete = async function (message = 'Tem certeza que deseja excluir?') {
        const result = await Swal.fire({
            title: 'Confirmar Exclusão',
            text: message,
            icon: 'warning',
            showCancelButton: true,
            confirmButtonColor: '#dc3545',
            cancelButtonColor: '#6c757d',
            confirmButtonText: 'Sim, excluir',
            cancelButtonText: 'Cancelar',
            reverseButtons: true
        });

        return result.isConfirmed;
    };

    // Formata moeda
    window.formatMoney = function (value) {
        return new Intl.NumberFormat('pt-BR', {
            style: 'currency',
            currency: 'BRL'
        }).format(value);
    };

    // Formata data
    window.formatDate = function (date) {
        return new Intl.DateTimeFormat('pt-BR').format(new Date(date));
    };

    // Mascara para CPF/CNPJ
    window.maskDocument = function (value) {
        value = value.replace(/\D/g, '');

        if (value.length <= 11) {
            // CPF: 000.000.000-00
            value = value.replace(/(\d{3})(\d)/, '$1.$2');
            value = value.replace(/(\d{3})(\d)/, '$1.$2');
            value = value.replace(/(\d{3})(\d{1,2})$/, '$1-$2');
        } else {
            // CNPJ: 00.000.000/0000-00
            value = value.replace(/^(\d{2})(\d)/, '$1.$2');
            value = value.replace(/^(\d{2})\.(\d{3})(\d)/, '$1.$2.$3');
            value = value.replace(/\.(\d{3})(\d)/, '.$1/$2');
            value = value.replace(/(\d{4})(\d)/, '$1-$2');
        }

        return value;
    };

    // Mascara para telefone
    window.maskPhone = function (value) {
        value = value.replace(/\D/g, '');

        if (value.length <= 10) {
            // (00) 0000-0000
            value = value.replace(/^(\d{2})(\d)/, '($1) $2');
            value = value.replace(/(\d{4})(\d)/, '$1-$2');
        } else {
            // (00) 00000-0000
            value = value.replace(/^(\d{2})(\d)/, '($1) $2');
            value = value.replace(/(\d{5})(\d)/, '$1-$2');
        }

        return value;
    };

    // Mascara para CEP
    window.maskZipCode = function (value) {
        value = value.replace(/\D/g, '');
        value = value.replace(/^(\d{5})(\d)/, '$1-$2');
        return value;
    };

    // Mascara para data (dd/mm/aaaa)
    window.maskDate = function (value) {
        value = value.replace(/\D/g, '');
        value = value.replace(/^(\d{2})(\d)/, '$1/$2');
        value = value.replace(/^(\d{2})\/(\d{2})(\d)/, '$1/$2/$3');
        return value.substring(0, 10);
    };

    // Mascara para moeda (R$ 0.000,00)
    window.maskMoney = function (value) {
        value = value.replace(/\D/g, '');
        if (value.length === 0) return '';
        value = value.padStart(3, '0');
        value = value.replace(/(\d)(\d{2})$/, '$1,$2');
        value = value.replace(/(?=(\d{3})+(\D))\B/g, '.');
        return value;
    };

    // Mascara para quantidade (0,000)
    window.maskQuantity = function (value) {
        value = value.replace(/[^\d,]/g, '');
        const parts = value.split(',');
        if (parts.length > 1) {
            return parts[0] + ',' + parts[1].substring(0, 3);
        }
        return value;
    };

    // Mascara para CPF (000.000.000-00)
    window.maskCPF = function (value) {
        value = value.replace(/\D/g, '');
        value = value.substring(0, 11);
        value = value.replace(/(\d{3})(\d)/, '$1.$2');
        value = value.replace(/(\d{3})(\d)/, '$1.$2');
        value = value.replace(/(\d{3})(\d{1,2})$/, '$1-$2');
        return value;
    };

    // Mascara para CNPJ (00.000.000/0000-00)
    window.maskCNPJ = function (value) {
        value = value.replace(/\D/g, '');
        value = value.substring(0, 14);
        value = value.replace(/^(\d{2})(\d)/, '$1.$2');
        value = value.replace(/^(\d{2})\.(\d{3})(\d)/, '$1.$2.$3');
        value = value.replace(/\.(\d{3})(\d)/, '.$1/$2');
        value = value.replace(/(\d{4})(\d)/, '$1-$2');
        return value;
    };

    // Mascara para Placa de Carro (ABC-1234 ou ABC1D23 - Mercosul)
    window.maskPlate = function (value) {
        value = value.toUpperCase().replace(/[^A-Z0-9]/g, '');
        value = value.substring(0, 7);

        if (value.length > 3) {
            // Detecta se é placa antiga ou Mercosul
            const hasLetter4 = /[A-Z]/.test(value.charAt(4));

            if (hasLetter4) {
                // Mercosul: ABC1D23
                value = value.replace(/^([A-Z]{3})([0-9])([A-Z])([0-9]{2})/, '$1$2$3$4');
            } else {
                // Antiga: ABC-1234
                value = value.replace(/^([A-Z]{3})([0-9]{1,4})/, '$1-$2');
            }
        }

        return value;
    };

    // Mascara para RG (00.000.000-0)
    window.maskRG = function (value) {
        value = value.replace(/\D/g, '');
        value = value.substring(0, 9);
        value = value.replace(/(\d{2})(\d)/, '$1.$2');
        value = value.replace(/(\d{3})(\d)/, '$1.$2');
        value = value.replace(/(\d{3})(\d{1})$/, '$1-$2');
        return value;
    };

    // Mascara para Hora (00:00)
    window.maskTime = function (value) {
        value = value.replace(/\D/g, '');
        value = value.substring(0, 4);
        value = value.replace(/^(\d{2})(\d)/, '$1:$2');
        return value;
    };

    // Mascara para Data e Hora (dd/mm/aaaa 00:00)
    window.maskDateTime = function (value) {
        value = value.replace(/[^\d/: ]/g, '');

        // Separa data e hora
        const parts = value.split(' ');
        let datePart = parts[0] || '';
        let timePart = parts[1] || '';

        // Formata data
        datePart = datePart.replace(/\D/g, '');
        datePart = datePart.replace(/^(\d{2})(\d)/, '$1/$2');
        datePart = datePart.replace(/^(\d{2})\/(\d{2})(\d)/, '$1/$2/$3');
        datePart = datePart.substring(0, 10);

        // Formata hora
        timePart = timePart.replace(/\D/g, '');
        timePart = timePart.replace(/^(\d{2})(\d)/, '$1:$2');
        timePart = timePart.substring(0, 5);

        if (timePart) {
            return datePart + ' ' + timePart;
        }
        return datePart;
    };

    // ========================================
    // APIS EXTERNAS
    // ========================================

    // Busca CEP automaticamente
    window.buscarCep = function (cep, callback) {
        cep = cep.replace(/\D/g, '');

        if (cep.length !== 8) {
            return;
        }

        LoadingOverlay.show();

        $.ajax({
            url: '/api-cep.php?cep=' + cep,
            type: 'GET',
            dataType: 'json',
            success: function (response) {
                LoadingOverlay.hide();
                if (response.success) {
                    callback(response.data);
                    Toast.success('CEP encontrado!');
                } else {
                    Toast.warning('CEP não encontrado');
                }
            },
            error: function () {
                LoadingOverlay.hide();
                Toast.error('Erro ao buscar CEP');
            }
        });
    };

    // Busca CNPJ automaticamente
    window.buscarCnpj = function (cnpj, callback) {
        cnpj = cnpj.replace(/\D/g, '');

        if (cnpj.length !== 14) {
            Toast.warning('CNPJ deve ter 14 dígitos');
            return;
        }

        LoadingOverlay.show();

        $.ajax({
            url: '/api-cnpj.php?cnpj=' + cnpj,
            type: 'GET',
            dataType: 'json',
            success: function (response) {
                LoadingOverlay.hide();
                if (response.success) {
                    callback(response.data);
                    Toast.success('CNPJ encontrado!');
                } else {
                    Toast.warning(response.message || 'CNPJ não encontrado');
                }
            },
            error: function () {
                LoadingOverlay.hide();
                Toast.error('Erro ao buscar CNPJ');
            }
        });
    };

    // ========================================
    // CONVERSÕES
    // ========================================

    // Converte string de moeda para float
    window.moneyToFloat = function (value) {
        if (typeof value === 'number') {
            return value;
        }
        value = value.toString().replace(/[^\d,]/g, '');
        value = value.replace(',', '.');
        return parseFloat(value) || 0;
    };

    // Converte float para string de moeda
    window.floatToMoney = function (value) {
        return parseFloat(value).toFixed(2).replace('.', ',').replace(/\B(?=(\d{3})+(?!\d))/g, '.');
    };

    // Converte data BR (dd/mm/yyyy) para SQL (yyyy-mm-dd)
    window.dateToSql = function (value) {
        const parts = value.split('/');
        if (parts.length === 3) {
            return parts[2] + '-' + parts[1] + '-' + parts[0];
        }
        return value;
    };

    // Converte data SQL (yyyy-mm-dd) para BR (dd/mm/yyyy)
    window.sqlToDate = function (value) {
        const parts = value.split('-');
        if (parts.length === 3) {
            return parts[2] + '/' + parts[1] + '/' + parts[0];
        }
        return value;
    };

    // ========================================
    // VALIDADORES
    // ========================================

    // Valida CPF
    window.validateCPF = function (cpf) {
        cpf = cpf.replace(/\D/g, '');

        if (cpf.length !== 11) return false;
        if (/^(\d)\1{10}$/.test(cpf)) return false; // Sequência repetida

        let sum = 0;
        let remainder;

        // Valida primeiro dígito
        for (let i = 1; i <= 9; i++) {
            sum += parseInt(cpf.substring(i - 1, i)) * (11 - i);
        }
        remainder = (sum * 10) % 11;
        if (remainder === 10 || remainder === 11) remainder = 0;
        if (remainder !== parseInt(cpf.substring(9, 10))) return false;

        // Valida segundo dígito
        sum = 0;
        for (let i = 1; i <= 10; i++) {
            sum += parseInt(cpf.substring(i - 1, i)) * (12 - i);
        }
        remainder = (sum * 10) % 11;
        if (remainder === 10 || remainder === 11) remainder = 0;
        if (remainder !== parseInt(cpf.substring(10, 11))) return false;

        return true;
    };

    // Valida CNPJ
    window.validateCNPJ = function (cnpj) {
        cnpj = cnpj.replace(/\D/g, '');

        if (cnpj.length !== 14) return false;
        if (/^(\d)\1{13}$/.test(cnpj)) return false; // Sequência repetida

        let length = cnpj.length - 2;
        let numbers = cnpj.substring(0, length);
        const digits = cnpj.substring(length);
        let sum = 0;
        let pos = length - 7;

        // Valida primeiro dígito
        for (let i = length; i >= 1; i--) {
            sum += numbers.charAt(length - i) * pos--;
            if (pos < 2) pos = 9;
        }
        let result = sum % 11 < 2 ? 0 : 11 - (sum % 11);
        if (result !== parseInt(digits.charAt(0))) return false;

        // Valida segundo dígito
        length = length + 1;
        numbers = cnpj.substring(0, length);
        sum = 0;
        pos = length - 7;
        for (let i = length; i >= 1; i--) {
            sum += numbers.charAt(length - i) * pos--;
            if (pos < 2) pos = 9;
        }
        result = sum % 11 < 2 ? 0 : 11 - (sum % 11);
        if (result !== parseInt(digits.charAt(1))) return false;

        return true;
    };

    // Valida Email
    window.validateEmail = function (email) {
        const regex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        return regex.test(email);
    };

    // Valida Data (dd/mm/yyyy)
    window.validateDate = function (date) {
        const parts = date.split('/');
        if (parts.length !== 3) return false;

        const day = parseInt(parts[0], 10);
        const month = parseInt(parts[1], 10);
        const year = parseInt(parts[2], 10);

        if (isNaN(day) || isNaN(month) || isNaN(year)) return false;
        if (month < 1 || month > 12) return false;
        if (day < 1 || day > 31) return false;
        if (year < 1900 || year > 2100) return false;

        // Valida dias do mês
        const daysInMonth = new Date(year, month, 0).getDate();
        return day <= daysInMonth;
    };

    // Valida Hora (00:00)
    window.validateTime = function (time) {
        const parts = time.split(':');
        if (parts.length !== 2) return false;

        const hour = parseInt(parts[0], 10);
        const minute = parseInt(parts[1], 10);

        if (isNaN(hour) || isNaN(minute)) return false;
        if (hour < 0 || hour > 23) return false;
        if (minute < 0 || minute > 59) return false;

        return true;
    };

    // Valida Placa (ABC-1234 ou ABC1D23)
    window.validatePlate = function (plate) {
        plate = plate.toUpperCase().replace(/[^A-Z0-9]/g, '');

        if (plate.length !== 7) return false;

        // Placa antiga: ABC1234
        const oldPattern = /^[A-Z]{3}[0-9]{4}$/;
        // Placa Mercosul: ABC1D23
        const newPattern = /^[A-Z]{3}[0-9][A-Z][0-9]{2}$/;

        return oldPattern.test(plate) || newPattern.test(plate);
    };

    // Toggle sidebar (mobile)
    window.toggleSidebar = function () {
        const sidebar = document.querySelector('.sidebar');
        if (sidebar) {
            sidebar.classList.toggle('show');
        }
    };

    // Auto-dismiss alerts
    document.addEventListener('DOMContentLoaded', function () {
        const alerts = document.querySelectorAll('.alert:not(.alert-permanent)');
        alerts.forEach(function (alert) {
            setTimeout(function () {
                const bsAlert = new bootstrap.Alert(alert);
                bsAlert.close();
            }, 5000);
        });
    });

    // Exporta funções globais
    window.LoadingOverlay = LoadingOverlay;
    window.Toast = Toast;

    // AJAX Setup
    if (typeof jQuery !== 'undefined') {
        $.ajaxSetup({
            headers: {
                'X-Requested-With': 'XMLHttpRequest'
            },
            beforeSend: function (xhr, settings) {
                const useOverlay = settings && settings.useLoadingOverlay;
                if (!useOverlay) {
                    xhr._useLoadingOverlay = false;
                    LoadingOverlay.hide();
                    return;
                }
                xhr._useLoadingOverlay = true;
                LoadingOverlay.show();
            },
            complete: function (xhr) {
                if (xhr && xhr._useLoadingOverlay) {
                    LoadingOverlay.hide();
                }
            },
            error: function (xhr) {
                const response = xhr.responseJSON;
                if (response && response.message) {
                    Toast.error(response.message);
                } else {
                    Toast.error('Ocorreu um erro. Tente novamente.');
                }
            }
        });
    }

    // Função para verificar inadimplência universal
    window.verificarInadimplenciaUniversal = function (clienteId, callback) {
        if (!clienteId) {
            if (callback) callback(null, 'ID do cliente é obrigatório');
            return;
        }

        fetch('/api-verificar-inadimplencia.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                cliente_id: clienteId
            })
        })
            .then(response => response.json())
            .then(data => {
                if (callback) {
                    callback(data);
                }
            })
            .catch(error => {
                console.error('Erro ao verificar inadimplência:', error);
                if (callback) callback(null, 'Erro de conexão');
            });
    };

    // Função para exibir alerta de inadimplência
    window.exibirAlertaInadimplencia = function (resultado) {
        if (!resultado || !resultado.inadimplente) {
            return;
        }

        Swal.fire({
            title: 'Cliente Inadimplente',
            text: resultado.mensagem || 'Cliente inadimplente. Operação bloqueada.',
            icon: 'warning',
            confirmButtonText: 'Entendi',
            confirmButtonColor: '#dc3545',
            showCancelButton: false,
            allowOutsideClick: false
        });
    };

    // Função para verificar inadimplência ao selecionar cliente
    window.verificarInadimplenciaAoSelecionarCliente = function (clienteId, callback) {
        verificarInadimplenciaUniversal(clienteId, function (resultado) {
            if (resultado && resultado.inadimplente) {
                exibirAlertaInadimplencia(resultado);

                // Se callback foi fornecido, executar com resultado
                if (callback) {
                    callback(resultado);
                }
            } else if (callback) {
                callback(resultado);
            }
        });
    };

    console.log('%c Aurion ERP ', 'background: #3788d8; color: #fff; font-size: 20px; font-weight: bold; padding: 10px;');
    console.log('%c Sistema carregado com sucesso! ', 'color: #28a745; font-size: 14px;');

})();

