<?php
require_once __DIR__ . '/../vendor/autoload.php';

use App\Helpers\CnpjHelper;

// Limpar qualquer output buffer
while (ob_get_level() > 0) {
    ob_end_clean();
}

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

// Tratar requisições OPTIONS (CORS preflight)
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

try {
    $cnpj = $_GET['cnpj'] ?? '';

    // Remove formatação do CNPJ
    $cnpj = preg_replace('/[^0-9]/', '', $cnpj);

    error_log("[api-cnpj.php] CNPJ recebido: " . $cnpj);

    // Valida se o CNPJ tem 14 dígitos
    if (empty($cnpj) || strlen($cnpj) !== 14) {
        error_log("[api-cnpj.php] CNPJ inválido (vazio ou tamanho incorreto): " . strlen($cnpj));
        echo json_encode([
            'success' => false,
            'message' => 'CNPJ inválido. Deve ter 14 dígitos.'
        ], JSON_UNESCAPED_UNICODE);
        exit;
    }

    // Valida CNPJ
    if (!CnpjHelper::validar($cnpj)) {
        error_log("[api-cnpj.php] CNPJ falhou na validação");
        echo json_encode([
            'success' => false,
            'message' => 'CNPJ inválido'
        ], JSON_UNESCAPED_UNICODE);
        exit;
    }

    error_log("[api-cnpj.php] Buscando dados do CNPJ na Receita Federal...");

    // Busca dados do CNPJ
    $resultado = CnpjHelper::buscar($cnpj);

    if ($resultado) {
        error_log("[api-cnpj.php] CNPJ encontrado com sucesso");
        echo json_encode([
            'success' => true,
            'data' => $resultado
        ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    } else {
        error_log("[api-cnpj.php] CNPJ não encontrado ou serviço indisponível");
        echo json_encode([
            'success' => false,
            'message' => 'CNPJ não encontrado ou serviço indisponível'
        ], JSON_UNESCAPED_UNICODE);
    }
} catch (Exception $e) {
    error_log("[api-cnpj.php] Erro: " . $e->getMessage());
    error_log("[api-cnpj.php] Stack trace: " . $e->getTraceAsString());
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Erro ao consultar CNPJ: ' . $e->getMessage()
    ], JSON_UNESCAPED_UNICODE);
}

