<?php

/**
 * Aurion ERP SaaS
 * Sistema Multi-tenant
 *
 * @package Systhema
 * @version 1.0.0
 * @author Systhema Team
 */

declare(strict_types=1);

// INICIAR OUTPUT BUFFERING IMEDIATAMENTE para capturar qualquer output indesejado
ob_start();

// Define o início da execução
define('APP_START', microtime(true));

// Define o diretório raiz
define('ROOT_PATH', __DIR__);

// Carrega o autoloader do Composer
require_once ROOT_PATH . '/vendor/autoload.php';

// Carrega as variáveis de ambiente
$dotenv = Dotenv\Dotenv::createImmutable(ROOT_PATH);
$dotenv->load();

// Carrega as configurações
require_once ROOT_PATH . '/config/app.php';

// Carrega os helpers globais
require_once ROOT_PATH . '/src/Helpers/GlobalHelpers.php';

// VERIFICAÇÃO CRÍTICA: Se alguém acessar /pdv/* diretamente (sem /Systhema/), redireciona
$requestUri = parse_url($_SERVER['REQUEST_URI'] ?? '/', PHP_URL_PATH);
$pathParts = array_filter(explode('/', trim($requestUri, '/')));

// LOG: Detalhes da requisição
error_log("==========================================");
error_log("[INDEX.PHP] Iniciando processamento da requisição");
error_log("[INDEX.PHP] REQUEST_URI: " . ($_SERVER['REQUEST_URI'] ?? 'N/A'));
error_log("[INDEX.PHP] REQUEST_URI (parsed): " . $requestUri);
error_log("[INDEX.PHP] SCRIPT_NAME: " . ($_SERVER['SCRIPT_NAME'] ?? 'N/A'));
error_log("[INDEX.PHP] Path parts: " . json_encode(array_values($pathParts)));

// Detecta o caminho base do script atual
$scriptName = $_SERVER['SCRIPT_NAME'] ?? '';
$basePath = str_replace('\\', '/', dirname($scriptName));
$basePath = rtrim($basePath, '/');
$basePathSegment = $basePath !== '/' && $basePath !== '' ? trim(str_replace('\\', '/', basename($basePath)), '/') : 'Systhema';

error_log("[INDEX.PHP] Base path detectado: " . $basePath);
error_log("[INDEX.PHP] Base path segment: " . $basePathSegment);
error_log("[INDEX.PHP] Primeiro path part: " . (isset(array_values($pathParts)[0]) ? array_values($pathParts)[0] : 'vazio'));

// Se a requisição é para /pdv/* mas o script está em /Systhema/, redireciona
if (!empty($pathParts) && $pathParts[0] === 'pdv' && ($basePath === '/' || $basePath === '' || strpos($requestUri, $basePath) !== 0)) {
    error_log("[INDEX.PHP] ⚠️ REDIRECIONAMENTO NECESSÁRIO DETECTADO!");
    error_log("[INDEX.PHP] Requisição para /pdv/* sem caminho base correto");

    // Construir URL correta com o caminho base
    $correctUrl = $basePath !== '/' && $basePath !== '' ? $basePath . $requestUri : '/' . $basePathSegment . $requestUri;
    $queryString = $_SERVER['QUERY_STRING'] ?? '';
    if ($queryString) {
        $correctUrl .= '?' . $queryString;
    }
    // Usar URL completa para garantir redirecionamento correto
    $protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
    $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
    $fullUrl = $protocol . '://' . $host . $correctUrl;

    error_log("[INDEX.PHP] URL original: " . $requestUri);
    error_log("[INDEX.PHP] URL corrigida: " . $correctUrl);
    error_log("[INDEX.PHP] URL completa: " . $fullUrl);
    error_log("[INDEX.PHP] 🔄 REDIRECIONANDO...");

    header('Location: ' . $fullUrl, true, 301);
    exit;
}

error_log("[INDEX.PHP] Requisição OK, continuando processamento normal");
error_log("==========================================");

// Inicializa o sistema
try {
    // Cria a aplicação
    $app = new App\Core\Application();

    // Registra globalmente para as rotas
    $GLOBALS['app'] = $app;

    // Executa a aplicação
    $app->run();
} catch (Exception $e) {
    // Registrar erro no log
    error_log("Erro na aplicação: " . $e->getMessage());
    error_log("Arquivo: " . $e->getFile() . ":" . $e->getLine());
    error_log("Stack trace: " . $e->getTraceAsString());

    // Verificar se é requisição AJAX
    $isAjax = isset($_SERVER['HTTP_X_REQUESTED_WITH'])
        && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) === 'xmlhttprequest';

    // Se for requisição AJAX, retornar JSON
    if ($isAjax) {
        http_response_code(500);
        header('Content-Type: application/json');
        echo json_encode([
            'success' => false,
            'message' => $_ENV['APP_ENV'] === 'production'
                ? 'Ocorreu um erro. Por favor, tente novamente mais tarde.'
                : $e->getMessage(),
            'errors' => []
        ]);
        exit;
    }

    // Caso contrário, renderizar página de erro
    try {
        $request = new \App\Core\Request();
        $response = new \App\Core\Response();
        $errorController = new \App\Controllers\ErrorController($request, $response);

        $message = $_ENV['APP_ENV'] === 'production'
            ? null
            : $e->getMessage() . "\n" . $e->getFile() . ":" . $e->getLine();

        $errorController->serverError($message);
    } catch (\Exception $fallbackException) {
        // Fallback final se não conseguir renderizar a página de erro
        http_response_code(500);
        echo "<h1>Erro interno do servidor</h1>";
        if ($_ENV['APP_ENV'] !== 'production') {
            echo "<p><strong>Mensagem:</strong> " . htmlspecialchars($e->getMessage()) . "</p>";
            echo "<p><strong>Arquivo:</strong> " . htmlspecialchars($e->getFile()) . ":" . $e->getLine() . "</p>";
            echo "<pre>" . htmlspecialchars($e->getTraceAsString()) . "</pre>";
        }
    }
}