# 🚀 Implementação: Portal Nacional de NFS-e (Substituindo Recife)

## 📊 Situação Atual

### ❌ **Descontinuação do Webservice da Prefeitura:**
- A Prefeitura de Recife está **descontinuando** seu webservice próprio
- Migração para o **Portal Nacional de NFS-e**
- Todas as emissões futuras serão via Portal Nacional

### ✅ **Vantagens da Migração:**
- ✅ Padrão único nacional (uma integração serve todos os municípios)
- ✅ API REST moderna (mais fácil que SOAP)
- ✅ JSON ao invés de XML complexo
- ✅ Ambiente de homologação oficial
- ✅ Documentação padronizada

---

## 🎯 Nova Estrutura de Implementação

### **O que MUDOU:**

| Item | Webservice Recife (Antigo) | Portal Nacional (Novo) |
|------|----------------------------|------------------------|
| **Protocolo** | SOAP/XML | REST/JSON |
| **Autenticação** | Certificado no XML | OAuth2 + Certificado |
| **Formato** | XML ABRASF complexo | JSON simplificado |
| **Envio** | EnviarLoteRps (SOAP) | POST /nfse (REST) |
| **Abrangência** | Só Recife | Todos os municípios aderentes |

---

## 📋 O que precisa implementar agora

### **1. Cadastro no Portal Nacional** ⏰ 1-2 dias
- Criar conta no portal
- Solicitar credenciais OAuth2
- Configurar certificado digital
- Obter Client ID e Client Secret

### **2. Autenticação OAuth2** ⏰ 1-2 dias
```php
// Obter token de acesso
POST https://api.nfse.gov.br/oauth/token
{
    "grant_type": "client_credentials",
    "client_id": "...",
    "client_secret": "..."
}
```

### **3. Serviço de Emissão** ⏰ 3-4 dias
- Estrutura JSON para emissão
- Validações fiscais
- Envio via API REST
- Tratamento de erros

### **4. Processamento de Resposta** ⏰ 1-2 dias
- Receber JSON de resposta
- Extrair dados da NFS-e autorizada
- Atualizar banco de dados

---

## 🏗️ Estrutura de Classes Proposta

```
Systhema/
├── src/
│   ├── Services/
│   │   └── NFSe/
│   │       ├── NFSeNacionalService.php          # Serviço principal
│   │       ├── NFSeNacionalAuthService.php      # OAuth2
│   │       └── NFSeNacionalClient.php           # Cliente HTTP
│   └── Controllers/
│       └── VendasController.php                 # Usar novo serviço
```

---

## 💻 Implementação Passo a Passo

### **PASSO 1: Service de Autenticação OAuth2**

```php
<?php
namespace App\Services\NFSe;

class NFSeNacionalAuthService
{
    private string $baseUrl;
    private string $clientId;
    private string $clientSecret;
    private ?string $accessToken = null;
    private ?int $tokenExpiresAt = null;

    public function __construct(string $clientId, string $clientSecret, bool $homologacao = false)
    {
        $this->clientId = $clientId;
        $this->clientSecret = $clientSecret;
        $this->baseUrl = $homologacao
            ? 'https://api-hml.nfse.gov.br'
            : 'https://api.nfse.gov.br';
    }

    /**
     * Obtém token de acesso OAuth2
     */
    public function obterToken(): string
    {
        // Se já tem token válido, retorna
        if ($this->accessToken && $this->tokenExpiresAt > time()) {
            return $this->accessToken;
        }

        // Obter novo token
        $response = $this->solicitarToken();

        $this->accessToken = $response['access_token'];
        $this->tokenExpiresAt = time() + ($response['expires_in'] ?? 3600) - 300; // -5min segurança

        return $this->accessToken;
    }

    private function solicitarToken(): array
    {
        $ch = curl_init($this->baseUrl . '/oauth/token');

        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => json_encode([
                'grant_type' => 'client_credentials',
                'client_id' => $this->clientId,
                'client_secret' => $this->clientSecret,
            ]),
            CURLOPT_HTTPHEADER => [
                'Content-Type: application/json',
                'Accept: application/json',
            ],
        ]);

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        if ($httpCode !== 200) {
            throw new \Exception("Erro ao obter token OAuth2: HTTP $httpCode - $response");
        }

        return json_decode($response, true);
    }
}
```

---

### **PASSO 2: Cliente HTTP para API**

```php
<?php
namespace App\Services\NFSe;

class NFSeNacionalClient
{
    private string $baseUrl;
    private NFSeNacionalAuthService $auth;

    public function __construct(NFSeNacionalAuthService $auth, bool $homologacao = false)
    {
        $this->auth = $auth;
        $this->baseUrl = $homologacao
            ? 'https://api-hml.nfse.gov.br'
            : 'https://api.nfse.gov.br';
    }

    /**
     * Emite NFS-e
     */
    public function emitirNfse(array $dadosNfse): array
    {
        $token = $this->auth->obterToken();

        $ch = curl_init($this->baseUrl . '/nfse');

        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => json_encode($dadosNfse),
            CURLOPT_HTTPHEADER => [
                'Authorization: Bearer ' . $token,
                'Content-Type: application/json',
                'Accept: application/json',
            ],
        ]);

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        if ($httpCode !== 200 && $httpCode !== 201) {
            $error = json_decode($response, true);
            throw new \Exception(
                $error['message'] ?? "Erro ao emitir NFS-e: HTTP $httpCode"
            );
        }

        return json_decode($response, true);
    }

    /**
     * Consulta NFS-e por número
     */
    public function consultarNfse(string $numeroNfse, string $codigoVerificacao): array
    {
        $token = $this->auth->obterToken();

        $ch = curl_init($this->baseUrl . '/nfse/' . $numeroNfse . '?codigo_verificacao=' . $codigoVerificacao);

        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HTTPHEADER => [
                'Authorization: Bearer ' . $token,
                'Accept: application/json',
            ],
        ]);

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        if ($httpCode !== 200) {
            throw new \Exception("Erro ao consultar NFS-e: HTTP $httpCode");
        }

        return json_decode($response, true);
    }

    /**
     * Cancela NFS-e
     */
    public function cancelarNfse(string $numeroNfse, string $codigoVerificacao, string $motivo): array
    {
        $token = $this->auth->obterToken();

        $ch = curl_init($this->baseUrl . '/nfse/' . $numeroNfse . '/cancelar');

        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => json_encode([
                'codigo_verificacao' => $codigoVerificacao,
                'motivo' => $motivo,
            ]),
            CURLOPT_HTTPHEADER => [
                'Authorization: Bearer ' . $token,
                'Content-Type: application/json',
                'Accept: application/json',
            ],
        ]);

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        if ($httpCode !== 200) {
            throw new \Exception("Erro ao cancelar NFS-e: HTTP $httpCode");
        }

        return json_decode($response, true);
    }
}
```

---

### **PASSO 3: Service Principal**

```php
<?php
namespace App\Services\NFSe;

class NFSeNacionalService
{
    private NFSeNacionalClient $client;
    private bool $homologacao;

    public function __construct(string $clientId, string $clientSecret, bool $homologacao = false)
    {
        $auth = new NFSeNacionalAuthService($clientId, $clientSecret, $homologacao);
        $this->client = new NFSeNacionalClient($auth, $homologacao);
        $this->homologacao = $homologacao;
    }

    /**
     * Emite NFS-e a partir de dados da venda
     */
    public function emitirNfse(array $dadosVenda, array $empresa, array $configuracao): array
    {
        // 1. Montar payload JSON no formato do Portal Nacional
        $payload = $this->montarPayloadNfse($dadosVenda, $empresa, $configuracao);

        // 2. Validar dados antes de enviar
        $this->validarDados($payload);

        // 3. Enviar para API
        try {
            $resultado = $this->client->emitirNfse($payload);

            return [
                'success' => true,
                'numero_nfse' => $resultado['numero'],
                'codigo_verificacao' => $resultado['codigo_verificacao'],
                'data_emissao' => $resultado['data_emissao'],
                'xml_path' => $resultado['xml_url'] ?? null,
                'pdf_path' => $resultado['pdf_url'] ?? null,
                'qr_code' => $resultado['qr_code'] ?? null,
            ];
        } catch (\Exception $e) {
            return [
                'success' => false,
                'erro' => $e->getMessage(),
            ];
        }
    }

    /**
     * Monta payload JSON no formato do Portal Nacional
     */
    private function montarPayloadNfse(array $venda, array $empresa, array $config): array
    {
        return [
            'prestador' => [
                'cnpj' => preg_replace('/\D/', '', $empresa['cnpj'] ?? $empresa['document']),
                'inscricao_municipal' => $empresa['inscricao_municipal'],
                'razao_social' => $empresa['razao_social'] ?? $empresa['name'],
                'endereco' => [
                    'logradouro' => $empresa['address'] ?? '',
                    'numero' => $empresa['address_number'] ?? 'S/N',
                    'bairro' => $empresa['neighborhood'] ?? '',
                    'codigo_municipio' => '2611606', // Recife
                    'uf' => $empresa['state'] ?? 'PE',
                    'cep' => preg_replace('/\D/', '', $empresa['zipcode'] ?? ''),
                ],
            ],
            'tomador' => $this->montarDadosTomador($venda),
            'servico' => [
                'codigo_servico' => $config['codigo_servico'],
                'codigo_tributacao' => $config['codigo_tributacao'] ?? $config['codigo_servico'],
                'discriminacao' => $config['descricao_servico'],
                'valor_servicos' => (float) $venda['total'],
                'aliquota' => (float) $config['aliquota_iss'],
                'iss_retido' => ($config['iss_retido'] ?? '2') === '1',
                'natureza_operacao' => (int) ($config['natureza_operacao'] ?? 1),
                'regime_tributacao' => $config['regime_tributacao'] ?? '',
                'optante_simples' => ($config['optante_simples'] ?? '2') === '1',
            ],
            'data_emissao' => date('Y-m-d\TH:i:s'),
        ];
    }

    private function montarDadosTomador(array $venda): array
    {
        $cliente = $venda['cliente'] ?? [];

        return [
            'identificacao' => [
                'cpf' => !empty($cliente['cpf']) ? preg_replace('/\D/', '', $cliente['cpf']) : null,
                'cnpj' => !empty($cliente['cnpj']) ? preg_replace('/\D/', '', $cliente['cnpj']) : null,
            ],
            'razao_social' => $cliente['name'] ?? $cliente['razao_social'] ?? 'Consumidor não identificado',
            'endereco' => [
                'logradouro' => $cliente['address'] ?? '',
                'numero' => $cliente['address_number'] ?? 'S/N',
                'bairro' => $cliente['neighborhood'] ?? '',
                'codigo_municipio' => $cliente['city_code'] ?? '2611606',
                'uf' => $cliente['state'] ?? 'PE',
                'cep' => !empty($cliente['zipcode']) ? preg_replace('/\D/', '', $cliente['zipcode']) : null,
            ],
        ];
    }

    private function validarDados(array $payload): void
    {
        if (empty($payload['prestador']['cnpj']) || strlen($payload['prestador']['cnpj']) !== 14) {
            throw new \Exception('CNPJ do prestador inválido');
        }

        if (empty($payload['prestador']['inscricao_municipal'])) {
            throw new \Exception('Inscrição municipal obrigatória');
        }

        if (empty($payload['servico']['codigo_servico'])) {
            throw new \Exception('Código do serviço obrigatório');
        }

        if (empty($payload['servico']['valor_servicos']) || $payload['servico']['valor_servicos'] <= 0) {
            throw new \Exception('Valor dos serviços deve ser maior que zero');
        }
    }
}
```

---

### **PASSO 4: Atualizar Controller**

```php
// Em VendasController::emitirNfse()

// ANTES (webservice Recife):
// $nfseService = new NFSeRecifeService($empresa);
// $xml = $nfseService->gerarXmlRps($payload);

// DEPOIS (Portal Nacional):
$nfseService = new \App\Services\NFSe\NFSeNacionalService(
    $empresa['nfse_client_id'] ?? '',
    $empresa['nfse_client_secret'] ?? '',
    $homologacao = false // true para testes
);

$resultado = $nfseService->emitirNfse($venda, $empresa, [
    'codigo_servico' => $codigoServico,
    'codigo_tributacao' => $codigoTributacao,
    'descricao_servico' => $descricaoServico,
    'aliquota_iss' => $aliquotaIss,
    'natureza_operacao' => $naturezaOperacao,
    'regime_tributacao' => $regimeTributacao,
    'optante_simples' => $optanteSimples,
    'iss_retido' => $issRetido,
]);

if ($resultado['success']) {
    // Atualizar banco de dados
    $this->atualizarDadosVendaNFSe($vendaId, [
        'nfse_numero' => $resultado['numero_nfse'],
        'nfse_serie' => '1',
        'nfse_status' => 'autorizado',
        'nfse_codigo_verificacao' => $resultado['codigo_verificacao'],
        'nfse_data_emissao' => $resultado['data_emissao'],
        'nfse_xml_path' => $resultado['xml_path'],
        'nfse_pdf_path' => $resultado['pdf_path'],
    ]);

    $this->success('NFS-e emitida com sucesso!', [
        'numero_nfse' => $resultado['numero_nfse'],
        'codigo_verificacao' => $resultado['codigo_verificacao'],
    ]);
} else {
    $this->error('Erro ao emitir NFS-e: ' . $resultado['erro']);
}
```

---

## 📊 Campos Necessários no Banco

### **Tabela `empresas` - Adicionar:**
```sql
ALTER TABLE empresas
ADD COLUMN nfse_nacional_client_id VARCHAR(255) NULL,
ADD COLUMN nfse_nacional_client_secret VARCHAR(255) NULL,
ADD COLUMN nfse_nacional_homologacao TINYINT(1) DEFAULT 0;
```

### **Tabela `vendas` - Já existe:**
- `nfse_numero`
- `nfse_serie`
- `nfse_status`
- `nfse_codigo_verificacao`
- `nfse_data_emissao`
- `nfse_xml_path`
- `nfse_pdf_path`

---

## ⏱️ Tempo Estimado

| Tarefa | Tempo |
|--------|-------|
| Cadastro no Portal Nacional | 1-2 dias |
| Service de Autenticação OAuth2 | 1-2 dias |
| Service de Emissão | 3-4 dias |
| Atualizar Controller | 1 dia |
| Testes em homologação | 2-3 dias |
| **TOTAL** | **8-12 dias** |

---

## ✅ Próximos Passos

1. **Cadastrar no Portal Nacional** (você precisa fazer)
   - Acessar https://www.gov.br/nfse/
   - Criar conta
   - Solicitar credenciais OAuth2

2. **Adicionar campos no banco** (eu posso fazer)
   - Campos para Client ID e Secret

3. **Criar formulário de configuração** (eu posso fazer)
   - Na tela de empresas, aba NFS-e

4. **Implementar services** (eu posso fazer)
   - Autenticação OAuth2
   - Cliente HTTP
   - Service principal

5. **Integrar no Controller** (eu posso fazer)
   - Atualizar método emitirNfse()

Quer que eu comece a implementar? Posso começar criando a estrutura de classes e depois você fornece as credenciais do Portal Nacional para testarmos!
