# 🛠️ Funções Globais - Systhema ERP

**Desenvolvido por:** Romildo Conegundes
**Empresa:** Systhema
**Última atualização:** 13 de Outubro de 2025

---

## 📋 Índice

1. [Máscaras de Entrada](#-máscaras-de-entrada)
2. [Validadores](#-validadores)
3. [APIs Externas](#-apis-externas)
4. [Conversões](#-conversões)
5. [Utilitários](#-utilitários)
6. [Exemplos de Uso](#-exemplos-de-uso)

---

## 🎭 Máscaras de Entrada

### `maskDocument(value)`
Máscara inteligente para CPF ou CNPJ (detecta automaticamente)
```javascript
// CPF: 000.000.000-00
// CNPJ: 00.000.000/0000-00

$('#document').on('input', function() {
    this.value = maskDocument(this.value);
});
```

### `maskCPF(value)`
Máscara específica para CPF
```javascript
// Formato: 000.000.000-00
$('#cpf').on('input', function() {
    this.value = maskCPF(this.value);
});
```

### `maskCNPJ(value)`
Máscara específica para CNPJ
```javascript
// Formato: 00.000.000/0000-00
$('#cnpj').on('input', function() {
    this.value = maskCNPJ(this.value);
});
```

### `maskPhone(value)`
Máscara para telefone fixo ou celular
```javascript
// Fixo: (00) 0000-0000
// Celular: (00) 00000-0000

$('#phone').on('input', function() {
    this.value = maskPhone(this.value);
});
```

### `maskZipCode(value)`
Máscara para CEP
```javascript
// Formato: 00000-000
$('#cep').on('input', function() {
    this.value = maskZipCode(this.value);
});
```

### `maskDate(value)`
Máscara para data
```javascript
// Formato: dd/mm/aaaa
$('#data').on('input', function() {
    this.value = maskDate(this.value);
});
```

### `maskTime(value)`
Máscara para hora
```javascript
// Formato: 00:00
$('#hora').on('input', function() {
    this.value = maskTime(this.value);
});
```

### `maskDateTime(value)`
Máscara para data e hora
```javascript
// Formato: dd/mm/aaaa 00:00
$('#dataHora').on('input', function() {
    this.value = maskDateTime(this.value);
});
```

### `maskMoney(value)`
Máscara para valores monetários
```javascript
// Formato: 1.234,56
$('#valor').on('input', function() {
    this.value = maskMoney(this.value);
});
```

### `maskQuantity(value)`
Máscara para quantidade com decimais
```javascript
// Formato: 123,456
$('#quantidade').on('input', function() {
    this.value = maskQuantity(this.value);
});
```

### `maskPlate(value)`
Máscara para placa de veículo (antiga e Mercosul)
```javascript
// Antiga: ABC-1234
// Mercosul: ABC1D23

$('#placa').on('input', function() {
    this.value = maskPlate(this.value);
});
```

### `maskRG(value)`
Máscara para RG
```javascript
// Formato: 00.000.000-0
$('#rg').on('input', function() {
    this.value = maskRG(this.value);
});
```

---

## ✅ Validadores

### `validateCPF(cpf)`
Valida CPF com dígitos verificadores
```javascript
if (validateCPF('123.456.789-09')) {
    console.log('CPF válido!');
} else {
    Toast.error('CPF inválido!');
}
```

### `validateCNPJ(cnpj)`
Valida CNPJ com dígitos verificadores
```javascript
if (validateCNPJ('00.000.000/0000-00')) {
    console.log('CNPJ válido!');
} else {
    Toast.error('CNPJ inválido!');
}
```

### `validateEmail(email)`
Valida formato de email
```javascript
if (validateEmail('usuario@exemplo.com')) {
    console.log('Email válido!');
}
```

### `validateDate(date)`
Valida data no formato brasileiro
```javascript
if (validateDate('31/12/2025')) {
    console.log('Data válida!');
}
```

### `validateTime(time)`
Valida hora
```javascript
if (validateTime('14:30')) {
    console.log('Hora válida!');
}
```

### `validatePlate(plate)`
Valida placa de veículo (antiga ou Mercosul)
```javascript
if (validatePlate('ABC-1234') || validatePlate('ABC1D23')) {
    console.log('Placa válida!');
}
```

---

## 🌐 APIs Externas

### `buscarCep(cep, callback)`
Busca endereço completo pela API ViaCEP
```javascript
buscarCep('01310-100', function(data) {
    $('#logradouro').val(data.logradouro);
    $('#bairro').val(data.bairro);
    $('#cidade').val(data.localidade);
    $('#uf').val(data.uf);
});
```

**Retorna:**
```javascript
{
    "cep": "01310-100",
    "logradouro": "Avenida Paulista",
    "complemento": "",
    "bairro": "Bela Vista",
    "localidade": "São Paulo",
    "uf": "SP",
    "ibge": "3550308",
    "gia": "1004",
    "ddd": "11",
    "siafi": "7107"
}
```

### `buscarCnpj(cnpj, callback)`
Busca dados da empresa pela API ReceitaWS
```javascript
buscarCnpj('00000000000000', function(data) {
    $('#razaoSocial').val(data.razao_social);
    $('#nomeFantasia').val(data.nome_fantasia);
    $('#email').val(data.email);
    $('#telefone').val(data.telefone);
    $('#cep').val(data.cep);
    $('#logradouro').val(data.logradouro);
    $('#numero').val(data.numero);
    $('#complemento').val(data.complemento);
    $('#bairro').val(data.bairro);
    $('#cidade').val(data.municipio);
    $('#uf').val(data.uf);
});
```

**Retorna:**
```javascript
{
    "cnpj": "00.000.000/0000-00",
    "razao_social": "EMPRESA LTDA",
    "nome_fantasia": "EMPRESA",
    "situacao": "ATIVA",
    "abertura": "01/01/2000",
    "email": "contato@empresa.com",
    "telefone": "(11) 1234-5678",
    "cep": "01310-100",
    "logradouro": "Avenida Paulista",
    "numero": "1578",
    "complemento": "Sala 100",
    "bairro": "Bela Vista",
    "municipio": "São Paulo",
    "uf": "SP"
}
```

---

## 🔄 Conversões

### `moneyToFloat(value)`
Converte string monetária para float
```javascript
const valor = moneyToFloat('1.234,56'); // 1234.56
```

### `floatToMoney(value)`
Converte float para string monetária
```javascript
const texto = floatToMoney(1234.56); // "1.234,56"
```

### `dateToSql(value)`
Converte data BR para SQL
```javascript
const sqlDate = dateToSql('31/12/2025'); // "2025-12-31"
```

### `sqlToDate(value)`
Converte data SQL para BR
```javascript
const brDate = sqlToDate('2025-12-31'); // "31/12/2025"
```

### `formatMoney(value)`
Formata valor como moeda brasileira
```javascript
const texto = formatMoney(1234.56); // "R$ 1.234,56"
```

### `formatDate(date)`
Formata data para padrão brasileiro
```javascript
const texto = formatDate('2025-12-31'); // "31/12/2025"
```

---

## 🛠️ Utilitários

### `Toast.success(message)`
Notificação de sucesso
```javascript
Toast.success('Registro salvo com sucesso!');
```

### `Toast.error(message)`
Notificação de erro
```javascript
Toast.error('Erro ao salvar registro!');
```

### `Toast.warning(message)`
Notificação de aviso
```javascript
Toast.warning('Atenção! Verifique os dados.');
```

### `Toast.info(message)`
Notificação informativa
```javascript
Toast.info('Processamento iniciado.');
```

### `confirmDelete(message)`
Confirmação de exclusão com SweetAlert2
```javascript
const confirmed = await confirmDelete('Deseja realmente excluir?');
if (confirmed) {
    // Prosseguir com a exclusão
}
```

### `LoadingOverlay.show()` / `LoadingOverlay.hide()`
Exibe/oculta overlay de carregamento
```javascript
LoadingOverlay.show();
// ... operação demorada
LoadingOverlay.hide();
```

---

## 🎯 Exemplos de Uso

### Formulário de Pessoa Física
```html
<input type="text" id="cpf" placeholder="CPF">
<input type="text" id="rg" placeholder="RG">
<input type="text" id="telefone" placeholder="Telefone">
<input type="text" id="cep" placeholder="CEP">
<input type="email" id="email" placeholder="Email">
<input type="text" id="dataNascimento" placeholder="Data de Nascimento">

<script>
$('#cpf').on('input', function() {
    this.value = maskCPF(this.value);
}).on('blur', function() {
    if (!validateCPF(this.value)) {
        Toast.error('CPF inválido!');
    }
});

$('#rg').on('input', function() {
    this.value = maskRG(this.value);
});

$('#telefone').on('input', function() {
    this.value = maskPhone(this.value);
});

$('#cep').on('input', function() {
    this.value = maskZipCode(this.value);
}).on('blur', function() {
    buscarCep(this.value, function(data) {
        $('#endereco').val(data.logradouro);
        $('#bairro').val(data.bairro);
        $('#cidade').val(data.localidade);
        $('#uf').val(data.uf);
    });
});

$('#email').on('blur', function() {
    if (!validateEmail(this.value)) {
        Toast.error('Email inválido!');
    }
});

$('#dataNascimento').on('input', function() {
    this.value = maskDate(this.value);
}).on('blur', function() {
    if (!validateDate(this.value)) {
        Toast.error('Data inválida!');
    }
});
</script>
```

### Formulário de Empresa
```html
<input type="text" id="cnpj" placeholder="CNPJ">
<input type="text" id="razaoSocial" placeholder="Razão Social">
<input type="text" id="nomeFantasia" placeholder="Nome Fantasia">

<script>
$('#cnpj').on('input', function() {
    this.value = maskCNPJ(this.value);
}).on('blur', function() {
    const cnpj = this.value.replace(/\D/g, '');

    if (!validateCNPJ(cnpj)) {
        Toast.error('CNPJ inválido!');
        return;
    }

    buscarCnpj(cnpj, function(data) {
        Swal.fire({
            title: 'Dados encontrados!',
            text: 'Deseja preencher automaticamente?',
            icon: 'question',
            showCancelButton: true,
            confirmButtonText: 'Sim',
            cancelButtonText: 'Não'
        }).then((result) => {
            if (result.isConfirmed) {
                $('#razaoSocial').val(data.razao_social);
                $('#nomeFantasia').val(data.nome_fantasia);
                // ... demais campos
            }
        });
    });
});
</script>
```

### Formulário de Veículo
```html
<input type="text" id="placa" placeholder="Placa">

<script>
$('#placa').on('input', function() {
    this.value = maskPlate(this.value);
}).on('blur', function() {
    if (!validatePlate(this.value)) {
        Toast.error('Placa inválida!');
    }
});
</script>
```

### Campos Monetários
```html
<input type="text" id="valor" placeholder="Valor">
<input type="text" id="quantidade" placeholder="Quantidade">

<script>
$('#valor').on('input', function() {
    this.value = maskMoney(this.value);
});

$('#quantidade').on('input', function() {
    this.value = maskQuantity(this.value);
});

// Ao salvar, converter para float
const valorFloat = moneyToFloat($('#valor').val());
const quantidadeFloat = moneyToFloat($('#quantidade').val());
</script>
```

---

## 📊 Resumo de Todas as Funções

### Máscaras (14)
✅ `maskDocument()` - CPF/CNPJ automático
✅ `maskCPF()` - CPF
✅ `maskCNPJ()` - CNPJ
✅ `maskPhone()` - Telefone
✅ `maskZipCode()` - CEP
✅ `maskDate()` - Data
✅ `maskTime()` - Hora
✅ `maskDateTime()` - Data e Hora
✅ `maskMoney()` - Dinheiro
✅ `maskQuantity()` - Quantidade
✅ `maskPlate()` - Placa
✅ `maskRG()` - RG

### Validadores (6)
✅ `validateCPF()` - Valida CPF
✅ `validateCNPJ()` - Valida CNPJ
✅ `validateEmail()` - Valida Email
✅ `validateDate()` - Valida Data
✅ `validateTime()` - Valida Hora
✅ `validatePlate()` - Valida Placa

### APIs (2)
✅ `buscarCep()` - Busca endereço por CEP
✅ `buscarCnpj()` - Busca empresa por CNPJ

### Conversões (6)
✅ `moneyToFloat()` - String → Float
✅ `floatToMoney()` - Float → String
✅ `dateToSql()` - dd/mm/yyyy → yyyy-mm-dd
✅ `sqlToDate()` - yyyy-mm-dd → dd/mm/yyyy
✅ `formatMoney()` - Formata moeda
✅ `formatDate()` - Formata data

### Utilitários (6)
✅ `Toast.success()` - Notificação de sucesso
✅ `Toast.error()` - Notificação de erro
✅ `Toast.warning()` - Notificação de aviso
✅ `Toast.info()` - Notificação informativa
✅ `confirmDelete()` - Confirmação de exclusão
✅ `LoadingOverlay` - Overlay de carregamento

---

## 🔧 Arquivo de Origem

Todas essas funções estão implementadas em:
```
Systhema/public/assets/js/app.js
```

---

## 📝 Notas Importantes

1. **Máscaras são aplicadas em tempo real** (evento `input`)
2. **Validações devem ser feitas no `blur`** (quando o campo perde o foco)
3. **APIs externas mostram loading automático** via `LoadingOverlay`
4. **Todas as notificações usam SweetAlert2**
5. **Máscaras removem caracteres inválidos automaticamente**
6. **Validadores retornam `true` ou `false`**

---

## 🎓 Boas Práticas

✅ **Sempre valide CPF/CNPJ no backend também**
✅ **Use máscaras + validadores juntos**
✅ **Converta valores para float antes de salvar no banco**
✅ **Mostre feedback visual para o usuário (Toast)**
✅ **Use `buscarCep()` e `buscarCnpj()` para melhor UX**

---

**Desenvolvido com ❤️ pela equipe Systhema**
**Autor:** Romildo Conegundes (romildo@Systhema.com.br)
**Licença:** Proprietário
