# 🔧 Como Configurar OpenSSL 3.x no Servidor (SOLUÇÃO DEFINITIVA)

## ⚠️ Problema
O erro `error:0308010C:digital envelope routines::unsupported` ocorre porque o **OpenSSL 3.x** desabilitou algoritmos legados por padrão.

## ✅ SOLUÇÃO NO SERVIDOR (Recomendado - Resolve para toda aplicação)

### Opção 1: Configurar no PHP-FPM/Apache (Melhor para produção)

#### 1. Copiar arquivo de configuração para local padrão
```bash
# Copiar o arquivo openssl-legacy.cnf para um local acessível
sudo cp /caminho/do/projeto/src/Integrations/NFe/openssl-legacy.cnf /etc/ssl/openssl-legacy.cnf
# OU criar diretamente:
sudo nano /etc/ssl/openssl-legacy.cnf
```

Cole o conteúdo:
```ini
openssl_conf = openssl_init

[openssl_init]
providers = provider_sect

[provider_sect]
default = default_sect
legacy = legacy_sect

[default_sect]
activate = 1

[legacy_sect]
activate = 1
```

#### 2. Configurar no PHP-FPM (se usar PHP-FPM)
```bash
# Editar arquivo de configuração do PHP-FPM
sudo nano /etc/php/8.2/fpm/pool.d/www.conf
# (ajuste a versão do PHP conforme seu servidor)

# Adicionar ou modificar a linha:
env[OPENSSL_CONF] = /etc/ssl/openssl-legacy.cnf
```

#### 3. Configurar no Apache (se usar Apache + mod_php)
```bash
# Editar httpd.conf ou criar arquivo em conf.d
sudo nano /etc/apache2/conf-available/openssl-legacy.conf
# OU
sudo nano /etc/httpd/conf.d/openssl-legacy.conf
```

Adicionar:
```apache
# Configurar OPENSSL_CONF para suportar algoritmos legados
SetEnv OPENSSL_CONF /etc/ssl/openssl-legacy.cnf
```

Ativar (Apache 2.4+):
```bash
sudo a2enconf openssl-legacy
sudo systemctl restart apache2
```

#### 4. Configurar no Nginx + PHP-FPM
No arquivo de configuração do site (`/etc/nginx/sites-available/seu-site`):
```nginx
location ~ \.php$ {
    fastcgi_pass unix:/var/run/php/php8.2-fpm.sock;
    fastcgi_param OPENSSL_CONF /etc/ssl/openssl-legacy.cnf;
    # ... outras configurações
}
```

#### 5. Reiniciar serviços
```bash
# PHP-FPM
sudo systemctl restart php8.2-fpm

# Apache
sudo systemctl restart apache2

# Nginx
sudo systemctl restart nginx
```

---

### Opção 2: Configurar no php.ini (Funciona, mas menos eficiente)

```bash
# Editar php.ini
sudo nano /etc/php/8.2/fpm/php.ini
# OU
sudo nano /etc/php/8.2/apache2/php.ini
```

Adicionar na seção `[PHP]`:
```ini
[PHP]
; Configurar OpenSSL para suportar algoritmos legados
; Ajuste o caminho conforme seu servidor
openssl.cafile=/etc/ssl/openssl-legacy.cnf
```

**OU usar putenv no php.ini** (não recomendado, mas funciona):
```ini
; No início do php.ini, adicionar:
auto_prepend_file = /caminho/para/config-openssl.php
```

Criar `/caminho/para/config-openssl.php`:
```php
<?php
putenv('OPENSSL_CONF=/etc/ssl/openssl-legacy.cnf');
```

---

### Opção 3: Configurar como variável de ambiente do sistema (Para todos os processos)

```bash
# Editar /etc/environment
sudo nano /etc/environment

# Adicionar:
OPENSSL_CONF=/etc/ssl/openssl-legacy.cnf

# Ou adicionar no perfil do usuário do servidor web
sudo nano /etc/profile.d/openssl-legacy.sh
```

Conteúdo do arquivo:
```bash
#!/bin/bash
export OPENSSL_CONF=/etc/ssl/openssl-legacy.cnf
```

Tornar executável:
```bash
sudo chmod +x /etc/profile.d/openssl-legacy.sh
```

---

## ✅ SOLUÇÃO NO CÓDIGO (Já implementada - Funciona como fallback)

O código já está preparado para funcionar mesmo sem configuração no servidor:
- ✅ `CertificateHelper.php` já detecta OpenSSL 3.x
- ✅ Configura `OPENSSL_CONF` automaticamente quando necessário
- ✅ Usa `openssl_pkcs12_read()` que respeita `OPENSSL_CONF`

**O código já funciona, mas a solução no servidor é melhor!**

---

## 🧪 Como Testar

### 1. Verificar versão do OpenSSL no PHP
```bash
php -r "echo OPENSSL_VERSION_TEXT;"
```

### 2. Verificar se a configuração está ativa
```bash
php -r "echo getenv('OPENSSL_CONF');"
```

### 3. Testar certificado via código PHP
```php
<?php
// Criar arquivo teste-certificado.php
putenv('OPENSSL_CONF=/etc/ssl/openssl-legacy.cnf');

$certPath = '/caminho/para/certificado.pfx';
$senha = 'senha_do_certificado';
$certContent = file_get_contents($certPath);

$certData = [];
if (openssl_pkcs12_read($certContent, $certData, $senha)) {
    echo "✅ Certificado carregado com sucesso!\n";
    print_r(openssl_x509_parse($certData['cert']));
} else {
    echo "❌ Erro: " . openssl_error_string() . "\n";
}
```

---

## 📊 Comparação de Soluções

| Solução | Vantagens | Desvantagens |
|---------|-----------|--------------|
| **Servidor (PHP-FPM/Apache)** | ✅ Aplica para toda aplicação<br>✅ Mais eficiente<br>✅ Não precisa código adicional | ⚠️ Requer acesso root<br>⚠️ Precisa reiniciar serviços |
| **Código (já implementado)** | ✅ Funciona sem acesso root<br>✅ Portável entre servidores | ⚠️ Configura em cada requisição<br>⚠️ Menos eficiente |

---

## 🎯 RECOMENDAÇÃO FINAL

**Use a solução no servidor (Opção 1 - PHP-FPM/Apache)** porque:
1. ✅ Resolve para toda a aplicação de uma vez
2. ✅ Melhor performance
3. ✅ Mais profissional e escalável
4. ✅ O código já funciona como fallback se o servidor não estiver configurado

**O código atual já funciona**, mas configurar no servidor elimina a necessidade de configurar em cada requisição!

---

## 📞 Suporte

Se precisar de ajuda:
1. Verifique os logs: `tail -f /var/log/php8.2-fpm.log`
2. Teste o certificado: Acesse a página de empresas e clique em "Testar Certificado"
3. Verifique permissões: `ls -la /etc/ssl/openssl-legacy.cnf`
