# ✅ Checklist: O que precisa para Emitir NFS-e

## 📊 Status Atual do Sistema

### ✅ **Já Implementado:**
1. ✅ Botão de emissão no módulo de vendas
2. ✅ Interface de formulário com campos necessários
3. ✅ Geração do XML do RPS (Recife - padrão ABRASF)
4. ✅ Salvamento do XML localmente
5. ✅ Campos no banco de dados para armazenar dados da NFS-e
6. ✅ Configurações na empresa (código serviço, alíquota, etc.)

### ❌ **Falta Implementar:**
1. ❌ **Assinatura digital do XML** com certificado
2. ❌ **Envio ao webservice** da Prefeitura de Recife
3. ❌ **Processamento da resposta** do webservice
4. ❌ **Atualização do status** após emissão

---

## 🔧 O que precisa ser feito para COMPLETAR a emissão

### **1. ASSINAR O XML COM CERTIFICADO DIGITAL**

**Status Atual:** O XML é gerado, mas **NÃO é assinado**.

**O que fazer:**
- Usar certificado digital A1 ou A3 (e-CNPJ)
- Assinar o XML seguindo padrão XAdES
- Adicionar a tag `<Signature>` no XML

**Biblioteca recomendada:**
- Usar a mesma biblioteca que já usa para NF-e (NFePHP)
- Ou biblioteca específica para assinatura XML (openssl + XMLSec)

---

### **2. ENVIAR AO WEBSERVICE DA PREFEITURA DE RECIFE**

**Status Atual:** XML é gerado e salvo localmente, mas **NÃO é enviado** ao webservice.

**O que fazer:**
- Conectar ao webservice SOAP da Prefeitura de Recife
- Enviar o XML assinado
- Aguardar resposta

**URLs do Webservice (Recife):**

**Homologação:**
- WSDL: `https://nfsehml.capital.pe.gov.br/nfse/services/NFSeWS?wsdl`
- Endpoint: `https://nfsehml.capital.pe.gov.br/nfse/services/NFSeWS`

**Produção:**
- WSDL: `https://nfse.capital.pe.gov.br/nfse/services/NFSeWS?wsdl`
- Endpoint: `https://nfse.capital.pe.gov.br/nfse/services/NFSeWS`

**Método SOAP:** `EnviarLoteRps`

---

### **3. PROCESSAR A RESPOSTA DO WEBSERVICE**

**O que receber:**
- XML com resposta do webservice
- Número da NFS-e (se autorizado)
- Código de verificação
- Data de emissão
- Status (autorizado, rejeitado, etc.)

**O que fazer:**
- Ler o XML de resposta
- Extrair dados da NFS-e autorizada
- Salvar XML de resposta
- Atualizar banco de dados

---

### **4. ATUALIZAR STATUS E DADOS NO BANCO**

**Campos a atualizar na tabela `vendas`:**
- `nfse_status` → 'autorizado' ou 'rejeitado'
- `nfse_numero` → Número da NFS-e
- `nfse_codigo_verificacao` → Código de verificação
- `nfse_data_emissao` → Data/hora da emissão
- `nfse_xml_path` → Caminho do XML protocolado

---

## 📋 Requisitos Técnicos

### **1. Certificado Digital**
- ✅ Certificado A1 (arquivo .pfx) ou A3 (token)
- ✅ Senha do certificado
- ✅ Instalado/configurado no servidor

### **2. Dados da Empresa (já configurável no sistema)**
- ✅ CNPJ
- ✅ Inscrição Municipal
- ✅ Código do Serviço (Lista Municipal)
- ✅ Código de Tributação
- ✅ Alíquota de ISS
- ✅ Natureza da Operação
- ✅ Regime Tributário

### **3. Dados do Cliente/Tomador**
- ✅ CPF ou CNPJ
- ✅ Nome/Razão Social
- ✅ Endereço completo
- ✅ CEP

### **4. Dados do Serviço**
- ✅ Descrição
- ✅ Valor dos serviços
- ✅ Valores de retenções (se houver)
- ✅ ISS retido ou não

---

## 🚀 Passos para Implementação Completa

### **PASSO 1: Assinar XML** (2-3 dias)

```php
// Adicionar no NFSeRecifeService.php
public function assinarXml(string $xml, array $empresa): string
{
    // 1. Carregar certificado digital
    $certificado = $this->carregarCertificado($empresa);

    // 2. Assinar XML
    $xmlAssinado = $this->assinarComCertificado($xml, $certificado);

    // 3. Retornar XML assinado
    return $xmlAssinado;
}
```

**Bibliotecas necessárias:**
- OpenSSL (nativo PHP)
- Ou usar NFePHP para assinatura (se compatível)

---

### **PASSO 2: Enviar ao Webservice** (3-4 dias)

```php
// Adicionar no NFSeRecifeService.php
public function enviarLoteRps(string $xmlAssinado, bool $homologacao = false): array
{
    // 1. Configurar URL do webservice
    $wsdl = $homologacao
        ? 'https://nfsehml.capital.pe.gov.br/nfse/services/NFSeWS?wsdl'
        : 'https://nfse.capital.pe.gov.br/nfse/services/NFSeWS?wsdl';

    // 2. Criar cliente SOAP
    $client = new SoapClient($wsdl, [
        'trace' => true,
        'exceptions' => true,
        'soap_version' => SOAP_1_2,
        'cache_wsdl' => WSDL_CACHE_NONE,
    ]);

    // 3. Preparar XML para envio (pode precisar envolver em envelope)
    $xmlEnvelope = $this->prepararEnvelopeSoap($xmlAssinado);

    // 4. Chamar método EnviarLoteRps
    $response = $client->EnviarLoteRps([
        'xml' => $xmlEnvelope
    ]);

    // 5. Retornar resposta
    return $this->processarResposta($response);
}
```

**Observação:**
- Prefeitura de Recife usa padrão ABRASF
- Pode precisar adaptar formato do envelope SOAP
- Verificar documentação oficial da Prefeitura

---

### **PASSO 3: Processar Resposta** (1-2 dias)

```php
// Adicionar no NFSeRecifeService.php
private function processarResposta($respostaSoap): array
{
    // 1. Extrair XML da resposta
    $xmlResposta = $respostaSoap->return ?? $respostaSoap->xml ?? '';

    // 2. Parsear XML
    $dom = new DOMDocument();
    $dom->loadXML($xmlResposta);

    // 3. Verificar status (sucesso/erro)
    $status = $this->extrairStatus($dom);

    if ($status['sucesso']) {
        // 4. Extrair dados da NFS-e
        return [
            'success' => true,
            'numero_nfse' => $this->extrairNumeroNfse($dom),
            'codigo_verificacao' => $this->extrairCodigoVerificacao($dom),
            'data_emissao' => $this->extrairDataEmissao($dom),
            'xml_protocolado' => $xmlResposta,
        ];
    } else {
        // 5. Extrair erro
        return [
            'success' => false,
            'erro' => $status['mensagem'],
            'codigo_erro' => $status['codigo'],
        ];
    }
}
```

---

### **PASSO 4: Atualizar Controller** (1 dia)

```php
// Modificar VendasController::emitirNfse()
public function emitirNfse(): void
{
    // ... código existente ...

    // 1. Gerar XML
    $xml = $nfseService->gerarXmlRps($payload);

    // 2. Assinar XML (NOVO)
    $xmlAssinado = $nfseService->assinarXml($xml, $empresa);

    // 3. Enviar ao webservice (NOVO)
    $resultado = $nfseService->enviarLoteRps($xmlAssinado, $homologacao);

    // 4. Processar resultado (NOVO)
    if ($resultado['success']) {
        // Salvar XML protocolado
        $xmlProtocoladoPath = $nfseService->salvarXmlProtocolado(
            $resultado['xml_protocolado'],
            $empresa['cnpj'],
            $resultado['numero_nfse']
        );

        // Atualizar banco com dados da NFS-e autorizada
        $this->atualizarDadosVendaNFSe($vendaId, [
            'nfse_numero' => $resultado['numero_nfse'],
            'nfse_serie' => $serieRps,
            'nfse_status' => 'autorizado', // ✅ NFS-e autorizada!
            'nfse_codigo_verificacao' => $resultado['codigo_verificacao'],
            'nfse_data_emissao' => $resultado['data_emissao'],
            'nfse_xml_path' => $xmlProtocoladoPath,
        ]);

        $this->success('NFS-e emitida com sucesso!', [
            'numero_nfse' => $resultado['numero_nfse'],
            'codigo_verificacao' => $resultado['codigo_verificacao'],
        ]);
    } else {
        $this->error('Erro ao emitir NFS-e: ' . $resultado['erro']);
    }
}
```

---

## 📚 Documentação Necessária

### **1. Documentação da Prefeitura de Recife**
- Manual técnico do webservice
- Esquema XSD para validação
- Exemplos de requisição/resposta

### **2. Padrão ABRASF**
- Especificação técnica ABRASF
- Layout do XML
- Regras de validação

### **3. Assinatura Digital**
- Padrão XAdES para XML
- Certificado digital A1/A3

---

## ⏱️ Tempo Estimado de Implementação

| Tarefa | Tempo | Prioridade |
|--------|-------|------------|
| Assinar XML | 2-3 dias | 🔴 ALTA |
| Enviar ao webservice | 3-4 dias | 🔴 ALTA |
| Processar resposta | 1-2 dias | 🔴 ALTA |
| Atualizar controller | 1 dia | 🔴 ALTA |
| Testes e ajustes | 2-3 dias | 🟡 MÉDIA |
| **TOTAL** | **9-13 dias** | |

---

## 🎯 Resumo: O que precisa para emitir NFS-e?

### ✅ **Já tem:**
1. Botão no sistema ✅
2. Formulário com campos ✅
3. Geração de XML ✅
4. Configurações da empresa ✅

### ❌ **Falta:**
1. **Assinar XML** com certificado digital
2. **Enviar ao webservice** da Prefeitura
3. **Processar resposta** e salvar dados
4. **Atualizar status** após autorização

### 📝 **Requisitos:**
- Certificado digital (A1 ou A3)
- Documentação do webservice da Prefeitura de Recife
- Ambiente de homologação para testes
- Bibliotecas para SOAP e assinatura XML

---

## 💡 Próximos Passos

1. **Obter documentação** do webservice da Prefeitura de Recife
2. **Implementar assinatura** do XML
3. **Implementar envio** ao webservice
4. **Testar em homologação** antes de produção
5. **Atualizar controller** para processar resposta completa

Quer que eu implemente alguma dessas partes?
