# 🎯 Cheatsheet - Máscaras e Validadores

**Quick Reference para Desenvolvimento Rápido**

---

## 🎭 Máscaras Básicas

```javascript
// CPF/CNPJ Automático
$('#doc').on('input', function() { this.value = maskDocument(this.value); });

// CPF
$('#cpf').on('input', function() { this.value = maskCPF(this.value); });

// CNPJ
$('#cnpj').on('input', function() { this.value = maskCNPJ(this.value); });

// Telefone
$('#tel').on('input', function() { this.value = maskPhone(this.value); });

// CEP
$('#cep').on('input', function() { this.value = maskZipCode(this.value); });

// RG
$('#rg').on('input', function() { this.value = maskRG(this.value); });
```

---

## 📅 Máscaras de Data/Hora

```javascript
// Data (dd/mm/aaaa)
$('#data').on('input', function() { this.value = maskDate(this.value); });

// Hora (00:00)
$('#hora').on('input', function() { this.value = maskTime(this.value); });

// Data e Hora (dd/mm/aaaa 00:00)
$('#dt').on('input', function() { this.value = maskDateTime(this.value); });
```

---

## 💰 Máscaras de Valores

```javascript
// Moeda (1.234,56)
$('#valor').on('input', function() { this.value = maskMoney(this.value); });

// Quantidade (123,456)
$('#qtd').on('input', function() { this.value = maskQuantity(this.value); });
```

---

## 🚗 Outras Máscaras

```javascript
// Placa de Veículo
$('#placa').on('input', function() { this.value = maskPlate(this.value); });
```

---

## ✅ Validações Inline

```javascript
// Validar no blur (quando perder foco)
$('#cpf').on('blur', function() {
    if (!validateCPF(this.value)) {
        Toast.error('CPF inválido!');
    }
});

$('#cnpj').on('blur', function() {
    if (!validateCNPJ(this.value)) {
        Toast.error('CNPJ inválido!');
    }
});

$('#email').on('blur', function() {
    if (!validateEmail(this.value)) {
        Toast.error('Email inválido!');
    }
});

$('#data').on('blur', function() {
    if (!validateDate(this.value)) {
        Toast.error('Data inválida!');
    }
});

$('#placa').on('blur', function() {
    if (!validatePlate(this.value)) {
        Toast.error('Placa inválida!');
    }
});
```

---

## 🌐 APIs Externas (Busca Automática)

```javascript
// Buscar CEP e preencher endereço
$('#cep').on('blur', function() {
    buscarCep(this.value, function(data) {
        $('#endereco').val(data.logradouro);
        $('#bairro').val(data.bairro);
        $('#cidade').val(data.localidade);
        $('#uf').val(data.uf);
    });
});

// Buscar CNPJ e preencher empresa
$('#cnpj').on('blur', function() {
    const cnpj = this.value.replace(/\D/g, '');
    if (cnpj.length === 14) {
        buscarCnpj(cnpj, function(data) {
            $('#razao').val(data.razao_social);
            $('#fantasia').val(data.nome_fantasia);
            $('#email').val(data.email);
            $('#telefone').val(data.telefone);
            // ... outros campos
        });
    }
});
```

---

## 🔄 Conversões

```javascript
// Moeda → Float
const valor = moneyToFloat('1.234,56'); // 1234.56

// Float → Moeda
const texto = floatToMoney(1234.56); // "1.234,56"

// Data BR → SQL
const sql = dateToSql('31/12/2025'); // "2025-12-31"

// Data SQL → BR
const br = sqlToDate('2025-12-31'); // "31/12/2025"
```

---

## 🔔 Notificações

```javascript
// Sucesso
Toast.success('Salvo com sucesso!');

// Erro
Toast.error('Erro ao salvar!');

// Aviso
Toast.warning('Atenção!');

// Info
Toast.info('Processando...');

// Confirmação de exclusão
const ok = await confirmDelete('Deseja excluir?');
if (ok) {
    // Excluir
}
```

---

## 📝 Template Completo de Formulário

```html
<form id="meuForm">
    <input type="text" id="cpf" placeholder="CPF">
    <input type="text" id="telefone" placeholder="Telefone">
    <input type="text" id="cep" placeholder="CEP">
    <input type="text" id="data" placeholder="Data">
    <input type="text" id="valor" placeholder="Valor">
    <button type="submit">Salvar</button>
</form>

<script>
$(document).ready(function() {
    // Máscaras
    $('#cpf').on('input', function() { this.value = maskCPF(this.value); });
    $('#telefone').on('input', function() { this.value = maskPhone(this.value); });
    $('#cep').on('input', function() { this.value = maskZipCode(this.value); });
    $('#data').on('input', function() { this.value = maskDate(this.value); });
    $('#valor').on('input', function() { this.value = maskMoney(this.value); });

    // Validações
    $('#cpf').on('blur', function() {
        if (!validateCPF(this.value)) Toast.error('CPF inválido!');
    });

    $('#data').on('blur', function() {
        if (!validateDate(this.value)) Toast.error('Data inválida!');
    });

    // Busca CEP
    $('#cep').on('blur', function() {
        buscarCep(this.value, function(data) {
            // Preencher campos
        });
    });

    // Submit
    $('#meuForm').on('submit', function(e) {
        e.preventDefault();

        // Validar tudo antes de enviar
        if (!validateCPF($('#cpf').val())) {
            Toast.error('CPF inválido!');
            return;
        }

        // Converter valores
        const valorFloat = moneyToFloat($('#valor').val());
        const dataSql = dateToSql($('#data').val());

        // Enviar dados
        $.ajax({
            url: '/api/salvar',
            method: 'POST',
            data: {
                cpf: $('#cpf').val().replace(/\D/g, ''),
                valor: valorFloat,
                data: dataSql
            },
            success: function() {
                Toast.success('Salvo!');
            }
        });
    });
});
</script>
```

---

## 🎨 HTML com Atributos Data

```html
<!-- Aplica máscara automaticamente -->
<input type="text" data-mask="cpf" id="cpf">
<input type="text" data-mask="cnpj" id="cnpj">
<input type="text" data-mask="phone" id="telefone">
<input type="text" data-mask="cep" id="cep">
<input type="text" data-mask="date" id="data">
<input type="text" data-mask="money" id="valor">
<input type="text" data-mask="plate" id="placa">

<script>
// Auto-aplicar máscaras baseado em data-mask
$('[data-mask]').each(function() {
    const mask = $(this).data('mask');
    $(this).on('input', function() {
        switch(mask) {
            case 'cpf': this.value = maskCPF(this.value); break;
            case 'cnpj': this.value = maskCNPJ(this.value); break;
            case 'phone': this.value = maskPhone(this.value); break;
            case 'cep': this.value = maskZipCode(this.value); break;
            case 'date': this.value = maskDate(this.value); break;
            case 'money': this.value = maskMoney(this.value); break;
            case 'plate': this.value = maskPlate(this.value); break;
        }
    });
});
</script>
```

---

## 🚀 Inicialização Automática (Recomendado)

Adicione este código no `app.js` para aplicar máscaras automaticamente:

```javascript
// Auto-init máscaras baseado em classes
$(document).on('input', '.mask-cpf', function() { this.value = maskCPF(this.value); });
$(document).on('input', '.mask-cnpj', function() { this.value = maskCNPJ(this.value); });
$(document).on('input', '.mask-phone', function() { this.value = maskPhone(this.value); });
$(document).on('input', '.mask-cep', function() { this.value = maskZipCode(this.value); });
$(document).on('input', '.mask-date', function() { this.value = maskDate(this.value); });
$(document).on('input', '.mask-time', function() { this.value = maskTime(this.value); });
$(document).on('input', '.mask-money', function() { this.value = maskMoney(this.value); });
$(document).on('input', '.mask-plate', function() { this.value = maskPlate(this.value); });

// Uso:
// <input type="text" class="form-control mask-cpf">
```

---

## 📊 Tabela de Referência Rápida

| Função | Formato | Exemplo |
|--------|---------|---------|
| `maskCPF()` | 000.000.000-00 | 123.456.789-09 |
| `maskCNPJ()` | 00.000.000/0000-00 | 12.345.678/0001-90 |
| `maskPhone()` | (00) 00000-0000 | (11) 98765-4321 |
| `maskZipCode()` | 00000-000 | 01310-100 |
| `maskDate()` | dd/mm/aaaa | 31/12/2025 |
| `maskTime()` | 00:00 | 14:30 |
| `maskMoney()` | 1.234,56 | 1.234,56 |
| `maskPlate()` | ABC-1234 ou ABC1D23 | ABC-1234 |
| `maskRG()` | 00.000.000-0 | 12.345.678-9 |

---

## 🐛 Troubleshooting Rápido

**Máscara não funciona?**
```javascript
// Certifique-se de que app.js está carregado
console.log(typeof maskCPF); // Deve ser "function"
```

**Validação sempre retorna false?**
```javascript
// Remova a máscara antes de validar
const cpfLimpo = $('#cpf').val().replace(/\D/g, '');
validateCPF(cpfLimpo);
```

**API não retorna dados?**
```javascript
// Verifique o console
// Certifique-se de que jQuery está carregado
console.log(typeof $); // Deve ser "function"
```

---

**Desenvolvido por:** Romildo Conegundes
**Arquivo:** `Systhema/public/assets/js/app.js`
**Demo:** `Systhema/demo-mascaras.html`

