# Sistema de Bloqueio Universal por Inadimplência

## 📋 Visão Geral

O Sistema de Bloqueio Universal por Inadimplência permite que uma empresa bloqueie clientes inadimplentes em **TODAS as empresas** do sistema quando configurado. Isso é útil para grupos empresariais que querem controlar inadimplência de forma centralizada.

## ⚙️ Configuração

### 1. Ativar Bloqueio Universal

1. Acesse **Parâmetros** → **Financeiro**
2. Marque a opção **"Bloqueio Universal por Inadimplência"**
3. Configure os **dias de bloqueio** (0 = imediato após vencimento)

### 2. Parâmetros Disponíveis

- **`bloqueio_universal_inadimplente`**: Ativa/desativa o bloqueio universal
- **`dias_bloqueio_inadimplente`**: Dias de tolerância após vencimento (0-365)

## 🔧 Como Funciona

### Verificação Automática

1. **Cliente inadimplente** em qualquer empresa
2. **Sistema verifica** todas as empresas ativas
3. **Bloqueia automaticamente** o cliente em todas as empresas
4. **Exibe alerta** ao tentar operar com o cliente

### Operações Bloqueadas

- ✅ Criação de vendas
- ✅ Criação de pedidos
- ✅ Criação de orçamentos
- ✅ Edição de documentos financeiros

## 📁 Arquivos Criados

### Services
- `src/Services/InadimplenciaService.php` - Lógica principal do sistema

### Middleware
- `src/Middleware/InadimplenciaMiddleware.php` - Intercepta operações

### Helpers
- `src/Helpers/InadimplenciaHelper.php` - Funções auxiliares

### APIs
- `api-verificar-inadimplencia.php` - Endpoint para verificação em tempo real

### JavaScript
- Funções adicionadas em `public/assets/js/app.js`

## 🚀 Uso das Funções JavaScript

### Verificar Inadimplência

```javascript
// Verificar se cliente está inadimplente
verificarInadimplenciaUniversal(clienteId, function(resultado) {
    if (resultado.inadimplente) {
        console.log('Cliente inadimplente:', resultado.mensagem);
    }
});
```

### Verificar ao Selecionar Cliente

```javascript
// Verificar e exibir alerta automaticamente
verificarInadimplenciaAoSelecionarCliente(clienteId, function(resultado) {
    if (resultado.inadimplente) {
        // Cliente bloqueado - operação não permitida
    }
});
```

### Exibir Alerta Manual

```javascript
// Exibir alerta de inadimplência
exibirAlertaInadimplencia(resultado);
```

## 🔄 Fluxo de Funcionamento

### 1. Configuração
```
Empresa A → Parâmetros → Ativa "Bloqueio Universal"
```

### 2. Detecção
```
Cliente inadimplente em Empresa B
↓
Sistema verifica todas as empresas
↓
Bloqueia cliente em Empresa A, B, C...
```

### 3. Bloqueio
```
Usuário tenta vender para cliente
↓
Sistema verifica inadimplência
↓
Exibe alerta e bloqueia operação
```

## 📊 Estrutura do Banco

### Tabela `system_parameters`

```sql
-- Parâmetros de bloqueio universal
INSERT INTO system_parameters (param_key, param_value, param_category, description) VALUES
('bloqueio_universal_inadimplente', '0', 'financeiro', 'Bloqueio universal por inadimplência'),
('dias_bloqueio_inadimplente', '0', 'financeiro', 'Dias de inadimplência para bloqueio');
```

### Tabela `clients` (em cada tenant)

```sql
-- Campos para controle de bloqueio
ALTER TABLE clients ADD COLUMN status VARCHAR(20) DEFAULT 'active';
ALTER TABLE clients ADD COLUMN blocked_reason TEXT NULL;
ALTER TABLE clients ADD COLUMN blocked_at TIMESTAMP NULL;
```

## 🛡️ Segurança

### Verificações Implementadas

1. **Autenticação**: Usuário deve estar logado
2. **Autorização**: Verifica permissões por empresa
3. **Validação**: Dados de entrada validados
4. **Logs**: Todas as operações são logadas

### Middleware de Proteção

```php
// Aplicar middleware em rotas sensíveis
$router->group(['middleware' => ['AuthMiddleware', 'InadimplenciaMiddleware']], function($router) {
    $router->post('/vendas', 'VendasController@store');
    $router->post('/pedidos', 'PedidosController@store');
});
```

## 📈 Monitoramento

### Logs de Bloqueio

- **Data/Hora** do bloqueio
- **Cliente** bloqueado
- **Empresa** de origem da inadimplência
- **Motivo** do bloqueio
- **Usuário** que tentou a operação

### Relatórios Disponíveis

- Clientes inadimplentes por empresa
- Histórico de bloqueios universais
- Estatísticas de inadimplência

## 🔧 Manutenção

### Desbloquear Cliente

```php
// Via código
$inadimplenciaService = new InadimplenciaService();
$resultado = $inadimplenciaService->desbloquearClienteUniversal($clienteId);
```

### Verificar Status

```php
// Verificar se cliente está bloqueado
$status = InadimplenciaHelper::getStatusInadimplencia($clienteId);
```

## ⚠️ Considerações Importantes

### Performance

- Verificação é feita em tempo real
- Pode impactar performance com muitas empresas
- Cache implementado para otimização

### Configuração

- Cada empresa pode ativar/desativar independentemente
- Configuração é por empresa, não global
- Dias de bloqueio configuráveis por empresa

### Compatibilidade

- Funciona com sistema multi-tenant existente
- Não interfere com bloqueios individuais por empresa
- Compatível com todas as operações financeiras

## 🎯 Benefícios

1. **Controle Centralizado**: Bloqueia inadimplentes em todas as empresas
2. **Prevenção de Perdas**: Evita novas vendas para clientes problemáticos
3. **Flexibilidade**: Configurável por empresa
4. **Transparência**: Cliente sabe o motivo do bloqueio
5. **Automação**: Funciona automaticamente sem intervenção manual

## 📞 Suporte

Para dúvidas ou problemas com o sistema de bloqueio universal:

- **Email**: romildo@Systhema.com.br
- **Documentação**: Este arquivo
- **Logs**: Verificar arquivos de log do sistema
