-- =====================================================
-- Migration: Ajuste Tabelas PDV - Sessões e Movimentos
-- Descrição: Ajusta estrutura das tabelas pdv_sessoes e pdv_movimentos
--            para suportar registro completo de abertura e fechamento
-- Data: 2024
-- =====================================================

-- =====================================================
-- 1. AJUSTAR TABELA pdv_sessoes
-- =====================================================

-- Adicionar coluna valor_fechamento se não existir
SET @col_exists = (
    SELECT COUNT(*)
    FROM INFORMATION_SCHEMA.COLUMNS
    WHERE TABLE_SCHEMA = DATABASE()
    AND TABLE_NAME = 'pdv_sessoes'
    AND COLUMN_NAME = 'valor_fechamento'
);

SET @sql = IF(@col_exists = 0,
    'ALTER TABLE pdv_sessoes ADD COLUMN valor_fechamento DECIMAL(10,2) DEFAULT NULL COMMENT ''Valor total informado no fechamento'' AFTER saldo_atual',
    'SELECT ''Coluna valor_fechamento já existe'' AS info'
);

PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- Adicionar coluna diferenca se não existir
SET @col_exists = (
    SELECT COUNT(*)
    FROM INFORMATION_SCHEMA.COLUMNS
    WHERE TABLE_SCHEMA = DATABASE()
    AND TABLE_NAME = 'pdv_sessoes'
    AND COLUMN_NAME = 'diferenca'
);

SET @sql = IF(@col_exists = 0,
    'ALTER TABLE pdv_sessoes ADD COLUMN diferenca DECIMAL(10,2) DEFAULT NULL COMMENT ''Diferença entre saldo esperado e valor informado no fechamento'' AFTER valor_fechamento',
    'SELECT ''Coluna diferenca já existe'' AS info'
);

PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- Adicionar coluna observacoes_fechamento se não existir
SET @col_exists = (
    SELECT COUNT(*)
    FROM INFORMATION_SCHEMA.COLUMNS
    WHERE TABLE_SCHEMA = DATABASE()
    AND TABLE_NAME = 'pdv_sessoes'
    AND COLUMN_NAME = 'observacoes_fechamento'
);

SET @sql = IF(@col_exists = 0,
    'ALTER TABLE pdv_sessoes ADD COLUMN observacoes_fechamento TEXT DEFAULT NULL COMMENT ''Observações do fechamento de caixa'' AFTER diferenca',
    'SELECT ''Coluna observacoes_fechamento já existe'' AS info'
);

PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- Adicionar coluna closed_at se não existir
SET @col_exists = (
    SELECT COUNT(*)
    FROM INFORMATION_SCHEMA.COLUMNS
    WHERE TABLE_SCHEMA = DATABASE()
    AND TABLE_NAME = 'pdv_sessoes'
    AND COLUMN_NAME = 'closed_at'
);

SET @sql = IF(@col_exists = 0,
    'ALTER TABLE pdv_sessoes ADD COLUMN closed_at DATETIME DEFAULT NULL COMMENT ''Data/hora do fechamento do caixa'' AFTER updated_at',
    'SELECT ''Coluna closed_at já existe'' AS info'
);

PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- =====================================================
-- 2. VERIFICAR/CRIAR TABELA pdv_movimentos
-- =====================================================

CREATE TABLE IF NOT EXISTS pdv_movimentos (
    id INT AUTO_INCREMENT PRIMARY KEY,
    company_id INT NOT NULL,
    sessao_id INT NOT NULL,
    venda_id INT DEFAULT NULL,
    tipo ENUM('abertura', 'venda', 'suprimento', 'sangria', 'fechamento') NOT NULL,
    valor DECIMAL(10,2) NOT NULL,
    metodo_pagamento_id INT DEFAULT NULL,
    observacoes TEXT DEFAULT NULL,
    created_by INT DEFAULT NULL,
    created_at DATETIME NOT NULL,
    INDEX idx_company_id (company_id),
    INDEX idx_sessao_id (sessao_id),
    INDEX idx_venda_id (venda_id),
    INDEX idx_tipo (tipo),
    INDEX idx_metodo_pagamento_id (metodo_pagamento_id),
    INDEX idx_created_at (created_at),
    FOREIGN KEY (company_id) REFERENCES empresas(id) ON DELETE CASCADE,
    FOREIGN KEY (sessao_id) REFERENCES pdv_sessoes(id) ON DELETE CASCADE,
    FOREIGN KEY (metodo_pagamento_id) REFERENCES metodos_pagamento(id) ON DELETE SET NULL,
    FOREIGN KEY (created_by) REFERENCES users(id) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
COMMENT='Movimentações do PDV (abertura, vendas, suprimentos, sangrias, fechamento)';

-- Atualizar ENUM da coluna tipo para incluir 'abertura' e 'fechamento'
-- Nota: Este comando pode falhar se já existirem registros com valores que não estão no novo ENUM
-- Se a tabela já existir e tiver apenas os tipos antigos (venda, suprimento, sangria),
-- será necessário limpar/ajustar dados antes ou executar manualmente
SET @table_exists = (
    SELECT COUNT(*)
    FROM INFORMATION_SCHEMA.TABLES
    WHERE TABLE_SCHEMA = DATABASE()
    AND TABLE_NAME = 'pdv_movimentos'
);

SET @sql = IF(@table_exists > 0,
    'ALTER TABLE pdv_movimentos MODIFY COLUMN tipo ENUM(''abertura'', ''venda'', ''suprimento'', ''sangria'', ''fechamento'') NOT NULL COMMENT ''Tipo de movimentação: abertura, venda, suprimento, sangria, fechamento''',
    'SELECT ''Tabela pdv_movimentos criada com ENUM completo'' AS info'
);

PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- =====================================================
-- 3. ADICIONAR ÍNDICES SE NÃO EXISTIREM
-- =====================================================

-- Adicionar índice composto para consultas por sessão e tipo (se não existir)
SET @index_exists = (
    SELECT COUNT(*)
    FROM INFORMATION_SCHEMA.STATISTICS
    WHERE TABLE_SCHEMA = DATABASE()
    AND TABLE_NAME = 'pdv_movimentos'
    AND INDEX_NAME = 'idx_sessao_tipo'
);

SET @sql = IF(@index_exists = 0,
    'CREATE INDEX idx_sessao_tipo ON pdv_movimentos(sessao_id, tipo)',
    'SELECT ''Índice idx_sessao_tipo já existe'' AS info'
);

PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- =====================================================
-- 4. VERIFICAR ESTRUTURA FINAL
-- =====================================================

SELECT '=====================================================' AS '';
SELECT 'Migration concluída com sucesso!' AS resultado;
SELECT 'Verifique a estrutura das tabelas abaixo:' AS '';
SELECT '=====================================================' AS '';

-- Mostrar estrutura da tabela pdv_sessoes
SHOW COLUMNS FROM pdv_sessoes;

-- Mostrar estrutura da tabela pdv_movimentos
SHOW COLUMNS FROM pdv_movimentos;

